/*  ZA2 Driver for Linux
    Copyright (C) 1998,1999 Peter Wahl

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Send bug reports to:
    Peter Wahl
    wahl@uni-bonn.de
*/

#define _za2recplay_c
#include <asm/io.h>
#include <asm/dma.h>
#include <string.h>
#include "za2.h"
#include "za2hard.h"
#define BUFFER_FULL (MAX_BUFFER/dev->blocksize+1)

#define MP2_DUMMY 2
#define MP2_OVERHEAD_SIZE (MP2BLOCKSIZE+4+MP2_DUMMY)
#define MP2_BUFFERS ((int)(MAX_BUFFER/MP2_OVERHEAD_SIZE))
#define MP2_OVERHEAD_BUFFER (MP2_BUFFERS*MP2_OVERHEAD_SIZE)
#define MP2_BUFFER (MP2_BUFFERS*MP2BLOCKSIZE)

void __inline__ progdma(struct za2_dma *dmast,int counter)
{
  unsigned long flags;
  disable_dma(dmast->dma);                             /* before not executed when programming for playing ! */
  flags=claim_dma_lock();
  clear_dma_ff(dmast->dma);
  set_dma_addr(dmast->dma,virt_to_bus(dmast->dmaptr));
  set_dma_count(dmast->dma,counter);
  release_dma_lock(flags);
}

/***********************************************************************/
/*   INTERRUPT                                                         */
/***********************************************************************/
/* Interrupt-Handler */
static void intrHandler(int x, void * y, struct pt_regs * z)
{
  struct za2_status *dev=za2_st;
  int last,next,wakeup=0;
  unsigned long flags;
  if(dev->playing)
    {
      if(dev->filter==MPEG2)
	{
	  dev->outdma->currentDmaPtr+=MP2_OVERHEAD_SIZE;
	  dev->outdma->currentDmaPtr%=MP2_OVERHEAD_BUFFER;
	  disable_dma(dev->outdma->dma);
	  flags=claim_dma_lock();
	  clear_dma_ff(dev->outdma->dma);
	  set_dma_addr(dev->outdma->dma,virt_to_bus((char *)dev->outdma->dmaptr+dev->outdma->currentDmaPtr));
	  set_dma_count(dev->outdma->dma,MP2_OVERHEAD_SIZE);
	  release_dma_lock(flags);
	  enable_dma(dev->outdma->dma);
	}
      else
	{
	  last=dev->outdma->currentDmaPtr;
	  dev->outdma->currentDmaPtr+=dev->blocksize;
	  next=dev->outdma->currentDmaPtr;
	  dev->outdma->currentDmaPtr&=(MAX_BUFFER-1);
	  if(last>next)
	    {
	      if(dev->outdma->currentDataPtr>last)
		next=MAX_BUFFER-1;
	      else last=0;
	    }
	  /*Stoppen der DMA wenn Buffer leer !*/
	  if((dev->outdma->currentDataPtr>last) && (dev->outdma->currentDataPtr<=next))
	    {
	      disable_dma(dev->outdma->dma);
              dev->playing=0; za2setmode(dev);
	      outw(0,dev->za2_base); //<-- ?? (NEW - to be tested)
	      dev->outdma->firstBlock=1;
	      wakeup=1;
	    }
	}
    }
  if(dev->recording)
    {
      last=dev->indma->currentDmaPtr;
      dev->indma->currentDmaPtr+=dev->blocksize;
      next=dev->indma->currentDmaPtr;
      dev->indma->currentDmaPtr&=(MAX_BUFFER-1);
      if(last>next)
	{
	  if(dev->indma->currentDataPtr>last)
	    next=MAX_BUFFER-1;
	  else last=0;
	}
      /*Stoppen der DMA wenn Buffer leer !*/
      if((dev->indma->currentDataPtr>last) && (dev->indma->currentDataPtr<=next))
	{
	  disable_dma(dev->indma->dma);
          dev->recording=0; za2setmode(dev);
	  dev->indma->firstBlock=1;
	  wakeup=1;
	}
    }
  if(wakeup||dev->playing||dev->recording)
    {
      dev->interrupt++;
      ZA2_WAKEUP_PROCESS;
    }
}

/*** za2_get_block_size ***/
size_t za2_get_block_size(struct za2_status *dev)
{
  return(dev->blocksize);
}

/*** za2_open_play ***/
short za2_open_play(struct za2_status *dev)
{
  ZaNuMode(dev,1);
  dev->currentState = ZA2_OPEN_PLAY;
  dev->outdma->firstBlock=1;
  return 0;
}

/*** za2_open_rec ***/
short za2_open_rec(struct za2_status *dev)
{
  int stillLoop=MAX_LOOP;
  if(za2status(dev)&0x200)
    {
      while((za2status(dev)&0x200)&&(--stillLoop)) wait_za2(dev);
      if(!stillLoop)
	{
	  DEBUG(printk("ZA2: no clock input !\n"));
	  return ZA2_EXT_SYNC_NOT_VALID;
	}
    }
  if((dev->mode!=3) || (dev->srate!=48000))
    {
      if(dev->srate!=48000)
	{ 
	  dev->srate = 48000;
	  setza(dev);      /* some kind of workaround, but it actually works !! */
	}
      ZaNuMode(dev,3);
      if(setza(dev)) return -1;
    }
  dev->currentState = ZA2_OPEN_REC;
  return 0;
}

/* in principle, this one's void, but for reasons of symmetry, I set it to short */
/* we return alway zero ... */
short za2_open_recplay(struct za2_status *dev)
{
  /* puts ZA2 in mode 2 when possible, mode 1 when not; */ 
  /* sets currentState to show whether we've been successfull*/
  int stillLoop=MAX_LOOP;
  dev->outdma->firstBlock=1;
  /* not tested whether we recognize a valid signal ! */
  while((!(za2status(dev)&4))&&(--stillLoop)) wait_za2(dev);
  if(!stillLoop)
    {
      DEBUG(printk("ZA2: no clock input !\n")); /*no clock, no record*/
      za2_open_play(dev);
      dev->currentState|=ZA2_OPEN_RECPLAY;
      return(0);
    }
  if((dev->mode!=2) || (dev->srate!=48000))
    {
      if(dev->srate!=48000)
	{ 
	  dev->srate = 48000;
	  setza(dev);      /* some kind of workaround, but it actually works !! */
	}
      ZaNuMode(dev,2);
      if(setza(dev)) 
	{
	  ZaNuMode(dev,1); /*we don't get it synchronized, so record is not possible*/
	  dev->currentState=ZA2_OPEN_RECPLAY | ZA2_OPEN_PLAY;
	}
      else
	dev->currentState = ZA2_OPEN_RECPLAY | ZA2_OPEN_REC | ZA2_OPEN_PLAY;
    }
  else dev->currentState = ZA2_OPEN_RECPLAY | ZA2_OPEN_REC | ZA2_OPEN_PLAY;
  return 0;
}

/*  za2_start  */
short za2_start_play(struct za2_status *dev)
{
  if(dev->currentState&ZA2_OPEN_PLAY)
    {
      if(dev->swap)
	{
	  *dev->outdma->dmaptr = 0x7a61;             
	  *(dev->outdma->dmaptr+1) = 0;
	}
      else
	{
	  *dev->outdma->dmaptr = 0;             
	  *(dev->outdma->dmaptr+1) = 0x7a61;
	}
      dev->playing = 1;
      /* let's try */
      if(!dev->recording)
	{
	  GET_CPU_EXCLUSIVITY;
	  ZaOpenDSP(dev);
	  ZaSendDSPCMD(dev,ZA2_SYNC_OUTPUT,0);                /* get ready to sync output */
	  ZaSendDSPCMD(dev,ZA2_IRQ_RATE,(dev->blocksize)>>2); /* send the switch to new irqrate */
	  ZaSendDSPCMD(dev,ZA2_DAC_STATE, 0xb001);            /* enable DAC */
	  ZaCloseDSP(dev);
	  RELEASE_CPU_EXCLUSIVITY;
	}
      progdma(dev->outdma,MAX_BUFFER);                            /* setup the DMA controller */
      set_dma_mode(dev->outdma->dma,DMA_MODE_WRITE|DMA_AUTOINIT); /* read mem to output dma single mode, autoinit */
      za2setmode(dev); 
      enable_dma(dev->outdma->dma);          
      return 0;
    }
  return ZA2_DEVICE_NOT_IN_RECPLAY;
}

short za2_start_rec(struct za2_status *dev)
{
  unsigned int k = 0xffff,err;
  if(dev->currentState&ZA2_OPEN_REC)
    {
      dev->recording = 1;
      dev->indma->currentDmaPtr=0;
      dev->indma->currentDataPtr=0;
      dev->indma->firstBlock=1;
      dev->errors = 0;
      progdma(dev->indma,MAX_BUFFER);                           /* setup DMA controller for recording */
      set_dma_mode(dev->indma->dma,DMA_MODE_READ|DMA_AUTOINIT); /* enable controller, start recording */
      /* sync and start recording */
      if(!dev->playing)
	{
	  GET_CPU_EXCLUSIVITY;
	  ZaOpenDSP(dev);
	  ZaSendDSPCMD(dev,ZA2_IRQ_RATE,(dev->blocksize) >> 2);     /* send the switch to new irqrate */
	  if((err=ZaCloseDSP(dev)))
	    {
	      dev->recording = 0;
	      RELEASE_CPU_EXCLUSIVITY;
	      DEBUG(printk("ZA2: PLL is unable to lock (%d).\n",err));
	      return ZA2_NO_SYNC_RECEIVED;
	    }
	  while( (((unsigned short) inw(dev->za2_base)) == 0x7a61) && --k )
	    wait_za2(dev);
	  RELEASE_CPU_EXCLUSIVITY;
	}
      za2setmode(dev);
      enable_dma(dev->indma->dma);
      if(!k)
	return ZA2_NO_SYNC_RECEIVED;
      return 0;
    }
  return ZA2_DEVICE_NOT_IN_RECPLAY;
}

/*** za2_stop ***/
void za2_stop_play(struct za2_status *dev)
{
  int wait=100;
  if(dev->filter&7)
    {
      if(dev->playing)
	{
	  while((dev->playing) && wait)
	    {
	      ZA2_BLOCK_PROCESS;
	      wait--;
	    }
	  if(!wait)
	    {
	      DEBUG(printk("ZA2: za2_stop_play : Timeout !\n"));
	      disable_dma(dev->outdma->dma);
	    }
	}  
      if(!dev->recording)
	{
	  GET_CPU_EXCLUSIVITY;
	  ZaOpenDSP(dev);
	  ZaSendDSPCMD(dev,ZA2_SYNC_OUTPUT,1);                    /* set R=R and L=L again */
	  ZaCloseDSP(dev);
	  RELEASE_CPU_EXCLUSIVITY;
	}
    }
  else 
    disable_dma(dev->outdma->dma);
  dev->playing=0;
  za2setmode(dev);
  mute_za2(dev);
}

/* we assume that stop_rec is called after stop_play and */
/* that the card is inactive (at least not playing) when entering this procedure */
void za2_stop_rec(struct za2_status *dev)
{
  int wait=100;
  if(dev->recording)
    {
      set_dma_mode(dev->indma->dma,DMA_MODE_READ);            /* make input dma read with no autoinit */
      /* Wait for the DMA to reach the next block boundary */
      /* so we can disable ZA2 DRQs..                    */
      dev->interrupt=0;
      while(dev->recording && (!dev->interrupt) && wait)
	{
	  ZA2_BLOCK_PROCESS;
	  wait--;
	}
      DEBUG(if(!wait) printk("ZA2: za2_stop_rec : Timeout !\n"));
      disable_dma(dev->indma->dma);
      dev->recording=0;
      za2setmode(dev);
      ZaNuMode(dev,1);
    }
  if((dev->currentState&ZA2_OPEN_REC) && dev->errors)
    printk("ZA2: Errors during recording: %d\n",dev->errors);
}

/*** za2_close ***/
void za2_close(struct za2_status *dev)
{
  if(dev==NULL) return;
  dev->currentState = ZA2_CLOSE;
  za2_stop_play(dev);
  za2_stop_rec(dev);
  DEBUG(printk("ZA2: device closed\n"));
}

/*** za2_begin_write ***/
int za2_begin_write(struct za2_status *dev,void **pdata, size_t size)
{
  size_t ix;
  size_t freeSpace;                                 /* actual free space in DMA buffer */
  if(!(dev->currentState & ZA2_OPEN_PLAY))
    {
      DEBUG(printk("ZA2: write: not playing.\n"));
      return ZA2_DEVICE_NOT_IN_PLAY;
    }
  if(dev->recording && dev->singledma)
    {
      DEBUG(printk("ZA2: write: duplex not supported in single-dma-mode !\n"));
      return ZA2_DEVICE_NOT_IN_RECPLAY;
    }
  for(ix = 0; ix < BUFFER_FULL; ix++)
    {
      if(dev->outdma->firstBlock)                   /* calculate free space */
	{
	  dev->outdma->currentDmaPtr=0;
	  dev->outdma->currentDataPtr=4;
	  freeSpace = MAX_BUFFER - dev->outdma->currentDataPtr;
	  dev->outdma->firstBlock=0;
	}
      else 
	{
	  if((!dev->playing) && (dev->autoStart)) za2_start_play(dev);
	  if(dev->outdma->currentDataPtr > dev->outdma->currentDmaPtr)
	    {
	      freeSpace = MAX_BUFFER - dev->outdma->currentDataPtr;
	      if(!dev->outdma->currentDmaPtr)
		{
		  if(freeSpace>dev->blocksize)
		    freeSpace-=dev->blocksize;
		  else freeSpace=0; 
		}
	    }
	  else 
	    {
	      freeSpace = dev->outdma->currentDmaPtr - dev->outdma->currentDataPtr;
	      if(freeSpace>dev->blocksize) 
		freeSpace-=dev->blocksize;
	      else freeSpace=0;
	    }
	}
      if((freeSpace >= dev->blocksize) || 
	 ((dev->outdma->currentDataPtr>dev->outdma->currentDmaPtr) && freeSpace))
	/* if there is enough space, return the pointer */
	{
	  if(freeSpace>size) freeSpace=size;
	  *pdata=(void *)(((char *)dev->outdma->dmaptr)+dev->outdma->currentDataPtr);
	  dev->outdma->currentDataPtr += freeSpace;
	  dev->outdma->currentDataPtr &= (MAX_BUFFER-1);
	  return freeSpace;
	}
      else
	ZA2_BLOCK_PROCESS;
    }
  return ZA2_NO_IRQ_RECEIVED;
}

int za2_write_mp2(struct za2_status *dev,void **ptr,size_t size)
{
  int wait=20,i,free;
  if(dev->outdma->firstBlock) 
    {
      dev->playing=0;
      for(i=0;i<MP2_BUFFERS;i++)
	{
	  *(dev->outdma->dmaptr+i*(MP2_OVERHEAD_SIZE>>1))=0;
	  *(dev->outdma->dmaptr+i*(MP2_OVERHEAD_SIZE>>1)+1)=0x7a61;
	  *(dev->outdma->dmaptr+i*(MP2_OVERHEAD_SIZE>>1)+(MP2BLOCKSIZE>>1)+2)=0;
	}
      dev->outdma->currentDmaPtr=0;
      dev->outdma->firstBlock=0;
      disable_za2(dev);
      set_dma_mode(dev->outdma->dma,DMA_MODE_WRITE);
      free=MP2BLOCKSIZE;
      *ptr=(void *)((char *)dev->outdma->dmaptr+4);
      if(size<free) free=size;
      dev->outdma->currentDataPtr=free+4;
      return(free);
    }
  else
    {
      if((!dev->playing) && (dev->autoStart))
	{
	  progdma(dev->outdma,MP2_OVERHEAD_SIZE);
	  za2command(dev,dev->command&0xffcc);        /* enable the IRQ and output dma on ZA2, disable input DMA and input*/
	  enable_dma(dev->outdma->dma);
	  dev->playing=1;
	}
      while(wait)
	{
	  free=MP2_OVERHEAD_SIZE-(dev->outdma->currentDataPtr%MP2_OVERHEAD_SIZE)-MP2_DUMMY;
	  if(!free)
	    {
	      dev->outdma->currentDataPtr+=(4+MP2_DUMMY);
	      dev->outdma->currentDataPtr%=MP2_OVERHEAD_BUFFER;
	      free=MP2BLOCKSIZE;
	    }
	  if(size<free) free=size;
	  if((dev->outdma->currentDataPtr>(dev->outdma->currentDmaPtr+MP2_OVERHEAD_SIZE)) || (dev->outdma->currentDataPtr<dev->outdma->currentDmaPtr))
	    {
	      *ptr=(void *)((char *)dev->outdma->dmaptr+dev->outdma->currentDataPtr);
	      dev->outdma->currentDataPtr+=free;
	      return free;
	    }
	  else
	    ZA2_BLOCK_PROCESS;
	  wait--;
	}
    }
  DEBUG(printk("ZA2: MP2 Playback: Don't get IRQ !\n"));
  return -1;
}

/***  za2_begin_read ***/
int za2_begin_read(struct za2_status *dev,void** pdata, size_t size)
{
  short  err;
  size_t ix;
  size_t doneSpace;                                            /* actual free space in DMA buffer */
  if(!(dev->currentState & ZA2_OPEN_REC))
    {
      DEBUG(printk("ZA2: read: not recording !\n"));
      return ZA2_DEVICE_NOT_IN_REC;
    }
  if(dev->playing && dev->singledma)
    {
      DEBUG(printk("ZA2: read: duplex not supported in single-dma-mode!\n"));
      return ZA2_DEVICE_NOT_IN_RECPLAY;
    }
  if( (!dev->recording) && dev->autoStart )                    /* check for auto start condition */
    {
      err = za2_start_rec(dev); 
      if(err) return err;
    }
  for(ix = 0; ix < BUFFER_FULL; ix++)
    {
      if(dev->indma->firstBlock) 
	 { doneSpace=0; dev->indma->firstBlock=0; }
      else if(dev->indma->currentDataPtr > dev->indma->currentDmaPtr) /* calculate done space */
	{
	  doneSpace = MAX_BUFFER - dev->indma->currentDataPtr;
	  if(!dev->indma->currentDmaPtr)
	    {
	      if(doneSpace>dev->blocksize)
		doneSpace-=dev->blocksize;
	      else doneSpace=0;
	    }
	}
      else
	{
	  doneSpace = dev->indma->currentDmaPtr - dev->indma->currentDataPtr;
	  if(doneSpace>dev->blocksize)
	    doneSpace-=dev->blocksize;
	  else doneSpace=0;
	}
      if(!(za2status(dev)&4))
	if(dev->errors++>320000)                                 /* check for TX errors */
	  {
	    printk("ZA2: Too many invalid samples... recording halted\n");
	    *pdata = NULL;
	    return ZA2_ACCUMULATED_RX_ERROR;
	  }
      if((doneSpace >= dev->blocksize) ||
	 ((dev->indma->currentDataPtr>dev->indma->currentDmaPtr) && doneSpace))
	/* if there is enough space, return the pointer */
	{
	  if(doneSpace>size) doneSpace=size;
	  *pdata = (void *)(((char *)dev->indma->dmaptr)+dev->indma->currentDataPtr);
	  /*if(size<dev->blocksize) 
	    memset(((char *)(*pdata))+size,0,dev->blocksize-size);*/
	  dev->indma->currentDataPtr += doneSpace;
	  dev->indma->currentDataPtr &= (MAX_BUFFER-1);
	  return doneSpace;
	}
      else ZA2_BLOCK_PROCESS;
    }
  return ZA2_NO_IRQ_RECEIVED;
}



