/*  ZA2 Driver for Linux
    Copyright (C) 1998,1999 Peter Wahl

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Send bug reports to:
    Peter Wahl
    wahl@uni-bonn.de
*/

#define __NO_VERSION_
#include <linux/module.h>

#include "za2.h"
#include "za2hard.h"
#include <linux/malloc.h>
#include <linux/soundcard.h>
#include <asm/uaccess.h>
#include <asm/dma.h>
#include <asm/io.h>

#include "za2specials.h"

#define ZA2_RESYNC_DELAY 110

void __inline__ za2waitint(struct za2_status *dev)
{
  int err=100;
  if(dev->recording || dev->playing)
    {
      dev->interrupt=0;
      while((!dev->interrupt) && err--) ZA2_BLOCK_PROCESS;
    }
}

/* Resync ZA2 to resume playback after modification of DSP-variables.
   This causes the ZA2 to 'glitch'. Important parameter is the resume time
   in samples. It needs to be even but must not be a multiple of 4 (ZA-sync 
   is always on the right (? -> second sample) channel) */
void __inline__ sync_za2(struct za2_status *dev)
{
  short *dmapos;
  int count;
  unsigned long flags;
  if(dev->playing)
    {
      flags=claim_dma_lock();
      clear_dma_ff(dev->outdma->dma);
      count=get_dma_residue(dev->outdma->dma);
      release_dma_lock(flags);
      if(count<ZA2_RESYNC_DELAY) count=MAX_BUFFER;
      else count&=0xfffc;
      dmapos=(short *)(((char *)dev->outdma->dmaptr)+MAX_BUFFER-count+ZA2_RESYNC_DELAY);
          /*perhaps we should clean the intermediate data !*/
      if(dev->swap) dmapos++;
      *dmapos=0x7a61;
    }
}

void za2setvolume(struct za2_status *dev)
{
  za2waitint(dev);
  GET_CPU_EXCLUSIVITY;
  ZaOpenDSP(dev);
  ZaSendDSPCMD(dev,ZA2_VOLUME,dev->volume);
  if(dev->playing) ZaSendDSPCMD(dev,ZA2_SYNC_OUTPUT,0);
  ZaCloseDSP(dev);
  sync_za2(dev);
  za2setmode(dev);
  RELEASE_CPU_EXCLUSIVITY;
}

void za2startid(struct za2_status *dev)
{
  unsigned long flags;
  za2waitint(dev);
  GET_CPU_EXCLUSIVITY;
  ZaOpenDSP(dev);
  ZaSendDSPCMD(dev,ZA2_START_ID,300);
  if(dev->playing) ZaSendDSPCMD(dev,ZA2_SYNC_OUTPUT,0);
  //ZaSendDSPCMD(dev,0,0);
  ZaCloseDSP(dev);
  sync_za2(dev);
  za2setmode(dev);
  RELEASE_CPU_EXCLUSIVITY;
}

void za2setflags(struct za2_status *dev)
{
  if(!dev->aeso)
    {
      za2waitint(dev);
      GET_CPU_EXCLUSIVITY;
      ZaOpenDSP(dev);
      ZaSendDSPCMD(dev,ZA2_CBITS_0,0x003000*(1-dev->scms)+0x00c000*dev->emph);
      if(dev->playing) ZaSendDSPCMD(dev,ZA2_SYNC_OUTPUT,0);
      ZaCloseDSP(dev);
      sync_za2(dev);
      za2setmode(dev);
      RELEASE_CPU_EXCLUSIVITY;
    }
}

void za2setaeso(struct za2_status *dev)
{
  za2waitint(dev);
  GET_CPU_EXCLUSIVITY;
  setza(dev);
  za2setmode(dev);
  RELEASE_CPU_EXCLUSIVITY;
}

int za2readcbits(struct za2_status *dev)
{
  int i=200,cbits=0;
  while((za2status(dev)&0x100)&&i)
    {
      wait_za2(dev);
      i--;
    }
  while((!(za2status(dev)&0x100))&&i)
    {
      wait_za2(dev);
      i--;
    }
  if(!i) 
    {
      DEBUG(printk("ZA2: readcbits: Timeout !\n"));
      return 0;
    }
  for(i=0;i<32;i++)
    {
      if(za2status(dev)&0x40) 
	cbits|=1<<i;
      wait_za2(dev);
    }
  return cbits;
}

int __inline__ wait_sample(struct za2_status *dev,int last)
{
  int err=100;
  while(((za2status(dev)&8)==(last&8)) && err) err--;
  if(err)
    return za2status(dev);
  else 
    {
      DEBUG(printk("ZA2: wait_sample: Timeout !\n"));
      return -1;
    }
}

int za2readubits(struct za2_status *dev,char *bits)
{
  int c=0,b,err=1200,last;
  unsigned long int crc;
  last=za2status(dev);
  while((c<16)&&err)
    {
      last=wait_sample(dev,last);
      if(last&0x80) c=0;
      else c++;
      err--;
    }
  if(!err) 
    {
      DEBUG(printk("ZA2: readubits: Can't find start of UBITS !\n"));
      return -EIO;
    }
  for(c=0;c<98;c++)
    {
      bits[c]=1; err=100;
      last=za2status(dev);
      while((!(last&0x80))&&err)
	{
	  last=wait_sample(dev,last);
	  err--;
	}
      if(!err) 
	{
	  DEBUG(printk("ZA2: readubits: Timeout !\n"));
	  return -EIO;
	}
      for(b=2;b&0xff;b<<=1)
	{ 
	  last=wait_sample(dev,last);
	  if(last&0x80) bits[c]|=b;
	}
    }
  for(c=0,crc=0;c<80;c++)
    {
      crc = crc<<1;
      if (bits[c]&2)
	crc++;
      if (crc&0x10000)
	crc^=0x11021;
    }
  for(c=80;c<96;c++)
    {
      crc = crc<<1;
      if (!(bits[c]&2))
	crc++;
      if (crc&0x10000)
	crc^=0x11021;
    }
  if(crc) 
    {
      DEBUG(printk("ZA2: readcbits: CRC error in UBIT-stream.\n"));
      return -EIO;
    }
  else return 0;
}

int za2_mixer(struct za2_status *dev,unsigned int cmd, unsigned long *param)
{
  switch(cmd) 
    {
    case SOUND_MIXER_WRITE_VOLUME:
      if((*param>=0) && (*param<=100))
	{
	  dev->volume=(*param<<14)/100; /* scale volume from 0..100 to 0..16384 */
	  za2setvolume(dev);
	}
      else return -EINVAL;
      break;
    case SOUND_MIXER_READ_VOLUME:
      *param=(dev->volume*100)>>14;
      break;
    case SOUND_MIXER_WRITE_RECSRC:
      switch(*param)
	{
	case SOUND_MASK_DIGITAL1 : dev->digin=0; /* TOSLINK */
	  za2setmode(dev);
	  break;
	case SOUND_MASK_DIGITAL2 : dev->digin=1; /* COAX */
	  za2setmode(dev);
	  break;
	case SOUND_MASK_DIGITAL3 : dev->digin=2; /* AES/EBU */
	  za2setmode(dev);
	  break;
	default: return -EINVAL;
	}
      break;
    case SOUND_MIXER_READ_DEVMASK:
      *param=SOUND_MASK_DIGITAL1|SOUND_MASK_DIGITAL2|SOUND_MASK_DIGITAL3|SOUND_MASK_VOLUME;
      break;
    case SOUND_MIXER_READ_STEREODEVS:
    case SOUND_MIXER_READ_RECMASK:
      *param=SOUND_MASK_DIGITAL1|SOUND_MASK_DIGITAL2|SOUND_MASK_DIGITAL3;
      break;
    case SOUND_MIXER_READ_RECSRC:
      *param=(dev->digin)?(dev->digin<<18):1<<17; /*TOSLINK is 0=DIG1 !!*/
      break;
    case SOUND_MIXER_READ_CAPS:
      *param=SOUND_CAP_EXCL_INPUT;
      break;
    case SOUND_MIXER_ZA2SETFLAGS:                 /*set SCMS, EMPH, AESO*/
      if(*param&SOUND_MIXER_ZA2SCMS)
	dev->scms=1;
      else dev->scms=0;
      if(*param&SOUND_MIXER_ZA2EMPH)
	dev->emph=1;
      else dev->emph=0;
      if(*param&SOUND_MIXER_ZA2SWAP)
	dev->swap=1;
      else dev->swap=0;
      if((dev->aeso<<2)==(*param&SOUND_MIXER_ZA2AESO))
	za2setflags(dev);
      else
	{
	  if(*param&SOUND_MIXER_ZA2AESO)
	    dev->aeso=1;
	  else dev->aeso=0;
	  if(!dev->playing) za2setaeso(dev);
	}
      break;
    case SOUND_MIXER_ZA2GETFLAGS:                   /*get SCMS, EMPH, AESO*/
      *param=(dev->emph)?SOUND_MIXER_ZA2EMPH:0+
             (dev->scms)?SOUND_MIXER_ZA2SCMS:0+
             (dev->aeso)?SOUND_MIXER_ZA2AESO:0+
	     (dev->swap)?SOUND_MIXER_ZA2SWAP:0;
      break;
    case SOUND_MIXER_ZA2STARTID:
      za2startid(dev);
      break;
    case SOUND_MIXER_ZA2GETCBITS:
      *param=za2readcbits(dev);
      break;
    default:
      return -EINVAL;
    }
  return 0;
}

static int mixer_ioctl(struct inode * aInode, struct file * aFile,
		       unsigned int cmd, unsigned long arg)
{
  int err;
  long param = 0;
  char *bits;
  struct za2_status *dev=za2_st;
  /* extract the parameter */
  if(((cmd>>8)&0xff)=='M')
     if(cmd==SOUND_MIXER_ZA2GETUBITS)
       {
	 bits=(char *)kmalloc(ZA2UBITS+1,GFP_KERNEL);
	 if(!bits) return -ENOMEM;
	 err=za2readubits(dev,bits);
	 if(!err) copy_to_user((void *) arg,bits,ZA2UBITS);
	 kfree(bits);
	 return err;
       }
     else
       {
	 if(_SIOC_DIR(cmd)&_SIOC_WRITE)
	   {
	     if((err=verify_area(VERIFY_READ, (void *) arg, sizeof(long))))
	       return err;
	     get_user(param,(long *) arg);
	   }
	 /* do the command */
	 if((err=za2_mixer(dev,cmd,&param))) return err;
	 /* put back the parameter */
	 if(_SIOC_DIR(cmd)&_SIOC_READ) 
	   {
	     if((err=verify_area(VERIFY_WRITE, (void *) arg, sizeof(long))))
	       return err;
	     put_user(param, (long *) arg);
	   }
	 return 0;  /* all other settings are correct, but not handled */
       }
  else return -EINVAL;
}

static int mixer_open(struct inode * aInode, struct file * aFile)
{
  MOD_INC_USE_COUNT;
  return 0;
}

static int mixer_release(struct inode * aInode, struct file * aFile)
{
  MOD_DEC_USE_COUNT;
  return 0;
}

static struct file_operations za2mixer_fops = {
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  mixer_ioctl,
  NULL,
  mixer_open,
  NULL,
  mixer_release,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL
};
