/*  ZA2 Driver for Linux
    Copyright (C) 1998,1999 Peter Wahl

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Send bug reports to:
    Peter Wahl
    wahl@uni-bonn.de
*/


#include "za2.h"
#include <linux/sched.h>
#include <linux/module.h>
/*#include "soundmodule.h"*/
#include "sound_firmware.h"
#include "sound_config.h"
#include "dev_table.h"
#include "za2hard.h"
#include "za2fileops.h"
#include "za2mixer.h"
#include "za2recplay.h"

static struct za2_status za2_st[1]={{0,0,0x310,0,0,ZA2_CLOSE,0,0,NULL,NULL,0,0,0,0,0,0,0,0,48000,0x4000,1,0,0,0,0,0,0}};

static int za2_mskLen=0;
static char *za2_msk=NULL;

static int dmi=0,dmo=0;

/*Ldt bitweise die FPGA-Maske in ZA2*/
void doit(struct za2_status *dev,char *za2_ptr, unsigned num)
{
  int bitcount=128;
  while(num)
    {
      if((*za2_ptr)&bitcount)
	outb(0x02, dev->za2_base+6);
      else
	outb(0x0, dev->za2_base+6);
      bitcount>>=1;
      if(bitcount<1)
	{
	  bitcount=128;
	  za2_ptr++;
	}
      num--;
    }
}

/*Ldt FPGA-Maske*/
short upbit(struct za2_status *dev,unsigned char *za2_msk,int za2_mskLen)
{
  short err;
  ZaSetEnv(dev);
  outb(2, dev->za2_base+6);   /* turn off the reset */
  delay(10);
  if((za2_msk[6]!=0xf0)||(za2_msk[7]!=0x0f)||(za2_msk[8]!=0x00)||(za2_msk[9]!=0x33)||(za2_msk[10]!=0x30))
    {
      printk("ZA2: not a valid .bit file\n");
      return(ZA2_BAD_BIT_FILE);
    }
  if(za2_msk[11]!=0x33)
    return(ZA2_CANT_CONFIG_ZA2_WITH_ZA1); /* error */
  doit(dev,za2_msk+118,22228);
  delay(100);
  if( ((short) za2status(dev)) == -1)
    {
      err = ZA2_DID_NOT_CONFIG_CORRECTLY;
      printk("ZA2: Could not configure ZA2 !\n");
    }
  else
    {
      dev->currentState = ZA2_CLOSE;
      err = 0;
    }
  return err;
}

void freeDma(struct za2_dma *dmast)
{
  if(dmast->dma) free_dma(dmast->dma);
  if(dmast->dmaptr) kfree(dmast->dmaptr);
  kfree(dmast);
}

/* Gibt DMA-Buffer frei */
static void freeDmaBuffer(struct za2_status *dev)
{
  if(dev->indma) freeDma(dev->indma);
  if(!dev->singledma)
    if(dev->outdma) freeDma(dev->outdma);
  dev->outdma=dev->indma=NULL;
}

/*  Desciption: Install the interrupt handler */
static int instIntrHandler(struct za2_status *dev)
{
  if(dev->irq==0)
    {
      /* za2 card support irq 10, 11 and 12 */
      for(dev->irq = 10; dev->irq <= 12; dev->irq++)
	if(!request_irq((unsigned int) dev->irq, intrHandler, 0, "ZA2", NULL))
	  return 0;
    }
  else if(!request_irq((unsigned int) dev->irq, intrHandler, 0, "ZA2", NULL))
    return 0;
  printk("ZA2: error: unable to obtain interrupt\n");
  return -1;
}

short za2_init_dma(struct za2_dma *dmast)
{
  if(dmast->dma==0)
    {
      for(dmast->dma=5;dmast->dma<=7;dmast->dma++)
	if(!request_dma((unsigned int) dmast->dma, "ZA2")) break;
    }
  else if(request_dma((unsigned int) dmast->dma, "ZA2")) dmast->dma=8;
  if(dmast->dma==8) 
    {
      dmast->dma=0;
      return(ZA2_NO_DMA_AVAILABLE);
    }
  disable_dma(dmast->dma);
  dmast->dmaptr = (short *) kmalloc(MAX_BUFFER,GFP_DMA);
  if(!dmast->dmaptr)
    return(ZA2_NO_DMA_AVAILABLE);
  dmast->currentDataPtr=dmast->currentDmaPtr=0;
  dmast->firstBlock=1;
  return 0;
}

/* za2 hardware initialization */
short za2_init_recplay(struct za2_status *dev,int dmi,int dmo)
{
  if(dev->currentState != ZA2_CLOSE)
    return ZA2_DEVICE_ALREADY_OPEN;
  if(instIntrHandler(dev) != 0)
    return ZA2_NO_IRQ_AVAILABLE;
  dev->indma=(struct za2_dma *)kmalloc(sizeof(struct za2_dma),GFP_KERNEL);
  if(!dev->indma) return(ZA2_NO_MEMORY);
  dev->indma->dma=dmi;
  if(za2_init_dma(dev->indma))
    {
      printk("ZA2: No DMA-Memory!\n");
      return ZA2_NO_MEMORY;
    }
  if(dmi==dmo)
    {
      dev->singledma=1;
      dev->outdma=dev->indma;
      printk("Ok\nZA2: Operating in Single-DMA-Mode ...");
    }
  else
    {
      dev->singledma=0;
      dev->outdma=(struct za2_dma *)kmalloc(sizeof(struct za2_dma),GFP_KERNEL);
      if(!dev->outdma) return(ZA2_NO_MEMORY);
      dev->outdma->dma=dmo;
      if(za2_init_dma(dev->outdma))
	{
	  printk("ZA2: No DMA-Memory!\n");
	  return ZA2_NO_MEMORY;
	}
    }
  ZaSetEnv(dev); /* place dmo and dmi in command */ 
  return 0;
}

/* uninstIntrHandler */
void uninstIntrHandler(struct za2_status *dev)
{
  free_irq(dev->irq,NULL);
}

void free_firmware()
{
  if(za2_msk) vfree(za2_msk);
  za2_msk=NULL;
}

void free_resources(struct za2_status *dev)
{
  if(dev->irq) free_irq(dev->irq, NULL);
  freeDmaBuffer(dev);
  release_region(dev->za2_base, 8);
}

/*** za2_init ***/
int za2_init(void)
{  
  struct za2_status *dev=za2_st;
  /* Initialize the processor if necessary */
  if(((short)za2status(dev))!=-1)
    {
      printk("ZA2: FPGA-Mask already loaded.\n");
      dev->za2_initialized=1;
    }
  else if(dev->za2_initialized!=1)
    {
      printk("ZA2: Uploading FPGA Mask ...");
      if (za2_mskLen > 0)
	{
	  if (upbit(dev,za2_msk,za2_mskLen))
	    {
	      printk("Error\n");
	      free_firmware();
	      return 0;
	    }
	  free_firmware();
	}
      else 
	{
	  printk("Error\n");
	  return 0;
	}
      printk("Ok\n");
    }
  ZaSetEnv(dev);
  disable_za2(dev);
  switch(dev->command&3)
    {
    case 0: printk("ZA2: Using fiber optic input.\n"); break;
    case 1: printk("ZA2: Using coaxial input.\n"); break;
    case 2: printk("ZA2: Using AES/EBU input.\n"); break;
    case 3: printk("ZA2: Using AES/EBU input.\n"); break;
    }
  printk("ZA2: Initializing ZA2 ...");
  if(za2_init_recplay(dev,dmi,dmo))
    {
      printk("Error\n");
      free_resources(dev);
      return 0;
    }
  dmi=dmo=0;
  printk("Ok\n");
  printk("ZA2: Uploading DSP Code ...");
  if (loadfilter(dev,DSPOS)<0)
    {
      printk("Error\n");
      free_resources(dev);
    }
  printk("Ok\n");
  dev->minor=register_sound_dsp(&za2card_fops,-1);
  if(dev->minor<0)
    {
      printk("Error\n");
      free_resources(dev);
      return 0;
    }
  dev->mixer=register_sound_mixer(&za2mixer_fops,-1);
  if(dev->mixer<0)
    {
      printk("Error\n");
      unregister_sound_dsp(dev->minor);
      free_resources(dev);
      return 0;
    }
  return(dev->za2_initialized=2);
}

int probe_za2(struct address_info *hw_config)
{
  struct za2_status *dev=za2_st;
  if (check_region(hw_config->io_base, 8))
    return 0;
  dev->za2_base = hw_config->io_base;
  if(!za2_init()) return 0;
  hw_config->irq=dev->irq;
  hw_config->dma=dev->indma->dma; hw_config->dma2=dev->outdma->dma;
  request_region(hw_config->io_base, 8, "ZA2");
  hw_config->name="ZA2";
  return(dev->za2_initialized=1);
}

void attach_za2(struct address_info *hw_config)
{
}

void unload_za2(struct address_info *hw_config)
{
  struct za2_status *dev=za2_st;
  free_resources(dev);  
  unregister_sound_dsp(dev->minor);
  unregister_sound_mixer(dev->mixer);
}

#ifdef MODULE

MODULE_DESCRIPTION("ZA2 Driver");
MODULE_AUTHOR("Peter Wahl");

MODULE_PARM(digin,"i");
MODULE_PARM(io,"i");
MODULE_PARM(scms,"i");
MODULE_PARM(emph,"i");
MODULE_PARM(dmi,"i");
MODULE_PARM(irq,"i");
MODULE_PARM(dmo,"i");

EXPORT_NO_SYMBOLS;

int io = -1,scms = -1,emph = -1,aeso = -1,digin=-1,irq=-1;

struct address_info cfg;

int za2_check_values()
{
  if (io == -1)
    {
      printk(KERN_ERR "za2: io must be set.\n");
      return -EINVAL;
    }
  else if((io!=0x210)&&(io!=0x250)&&(io!=0x310)&&(io!=0x350))
    {
      printk(KERN_ERR "za2: io must be one of 0x210, 0x250, 0x310 and 0x350.\n");
      return -EINVAL;
    }
  if((irq!=-1) && ((irq<10) || (irq>12)))
    {
      printk(KERN_ERR "za2: irq must be one of 10, 11, 12\n");
      return -EINVAL;
    }
  if((dmi!=-1) && ((dmi<5) || (dmi>7)))
    {
      printk(KERN_ERR "za2: dmi must be one of 5, 6, 7\n");
      return -EINVAL;
    }
  if((dmo!=-1) && ((dmo<5) || (dmo>7)))
    {
      printk(KERN_ERR "za2: dmo must be one of 5, 6, 7\n");
      return -EINVAL;
    }
  if((digin!=-1) && ((digin<0) || (digin>3)))
    {
      printk(KERN_INFO "za2: digin should be one of 0, 1, 2, 3. Setting it to 0 (SPDIF).\n");
      digin=-1;
    }
  if((emph<-1) || (emph>1))
    {
      printk(KERN_INFO "za2: emph should be one of 0, 1. Setting it to 0 (no emphasis).\n");
      emph=-1;
    }
  if((scms<-1) || (scms>1))
    {
      printk(KERN_INFO "za2: scms should be one of 0, 1. Setting it to 0 (no SCMS).\n");
      scms=-1;
    }
  if((aeso<-1) || (aeso>1))
    {
      printk(KERN_INFO "za2: aeso should be one of 0, 1. Setting it to 0 (SPDIF).\n");
      aeso=-1;
    }
  return(0);
}

void za2_defaults(struct za2_status *dev)
{
  dmi=(dmi==-1)?0:dmi;                   /*Probe*/
  dmo=(dmo==-1)?0:dmo;                   /*Probe*/
  dev->digin=(digin==-1)?0:(digin&3);    /*TOSLINK*/
  dev->scms=(scms==-1)?0:(scms&1);       /*SCMS off*/
  dev->emph=(emph==-1)?0:(emph&1);       /*Emphasis off*/
  dev->aeso=(aeso==-1)?0:(aeso&1);       /*output mode*/
  dev->irq=(irq==-1)?0:irq;              /*Probe*/
}

/* Install a ZA2 */
int init_module(void)
{
  struct za2_status *dev=za2_st;
  int err;
  printk(KERN_INFO "ZA2 Driver, (c) 1998, 1999 Peter Wahl\n");
  err=za2_check_values();
  if(err) return(err);
  za2_defaults(dev);
  dev->za2_base = cfg.io_base = io;      /*(IO can't be probed)*/
  za2_mskLen = mod_firmware_load(ZA2FIRMWAREDIR "/za2mask.bit", (char **) &za2_msk);
  if(!za2_mskLen) return -ENOENT;
  if (probe_za2(&cfg) == 0)
    {
      free_firmware();
      return -ENODEV;
    }
  attach_za2(&cfg);
  return 0;
}

void cleanup_module(void)
{
  printk(KERN_INFO "ZA2 Driver unloaded.\n");
  unload_za2(&cfg);
}
#endif
