/*  ZA2 Driver for Linux
    Copyright (C) 1998,1999 Peter Wahl

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Send bug reports to:
    Peter Wahl
    wahl@uni-bonn.de
*/

#include <asm/io.h>

#define _za2hard_c
#include "sound_firmware.h"
#include "za2.h"
#include "za2hard.h"

static struct wait_queue *wq;
static spinlock_t za2_lock=SPIN_LOCK_UNLOCKED;
static unsigned long flags;

unsigned short __inline__ za2status(struct za2_status *dev)
{
  return((unsigned short)inw(dev->za2_base+2));
}

void __inline__ za2command(struct za2_status *dev,short val)
{
  outw(val,dev->za2_base+2);
}

void __inline__ disable_za2(struct za2_status *dev)
{
  za2command(dev,dev->command&0xff03);
  /*  if(dev->recording || dev->playing) outw(0, dev->za2_base); * added 11.3.99*/
}

void __inline__ mute_za2(struct za2_status *dev)
{
  outw(0,dev->za2_base);
}

void __inline__  za2setmode(struct za2_status *dev)
{
  int mode;
  mode=0xff03|((dev->recording)?0x3c:0)|((dev->playing)?0xcc:0);
  za2command(dev,dev->command&mode);
}

short wait_za2(struct za2_status *dev)
{
  int timeout=0;
  while((za2status(dev)&8) && (timeout++<40));
  while(!(za2status(dev)&8) && (timeout++<80));
  return(timeout==80);
}

void send_byte(struct za2_status *dev,short val)
{
  int timeout=0;
  while((za2status(dev)&0x400) && (timeout++<160));
  if(timeout<160)
    outw(val, dev->za2_base+4);
  DEBUG(else printk("ZA2: send_byte : Timeout !\n"));
}

void ZaSetEnv(struct za2_status *dev)
{
  dev->command = 0x1000 + (dev->mode << 8) + dev->digin + ((dev->irq - 9) << 2);
  if(dev->indma)
    dev->command+=((dev->indma->dma&3)<<4);
  if(dev->outdma)
    dev->command+=((dev->outdma->dma&3)<<6);
}

/*  zaGetSamplingRate */
unsigned int zaGetSamplingRate(struct za2_status *dev)
{
  if(dev->mode&2)                                      /*Check only in record and sync-mode*/
    {
      switch(za2status(dev)&0x830)
	{
	case 0x000:
          DEBUG(printk("Out of range...\n"));
	  dev->srate = 48000;
	  break;
	case 0x010:
          DEBUG(printk("48Khz (rough)\n"));
	  dev->srate = 48000;
	  break;
	case 0x020:
	  DEBUG(printk("44.1Khz (rough)\n"));
	  dev->srate = 44100;
	  break;
	case 0x030:
	  DEBUG(printk("32Khz (rough)\n"));
	  dev->srate = 32000;
	  break;
	case 0x800:
	  DEBUG(printk("48Khz\n"));
	  dev->srate = 48000;
	  break;
	case 0x810:
	  DEBUG(printk("44.1Khz\n"));
	  dev->srate = 44100;
	  break;
	case 0x820:
	  DEBUG(printk("44.056Khz\n"));
	  dev->srate = 44056;
	  break;
	case 0x830:
	  DEBUG(printk("32Khz\n"));
	  dev->srate = 32000;
	  break;
	}
      if((dev->filter&7) && (dev->filter!=DSPOS)) 
	{
	  DEBUG(printk("resampled"));
	  dev->srate = 44100;
	}
      if(dev->filter==MPEG2)
	DEBUG(printk("MPEG2"));
    }
  else DEBUG(printk("%d Hz\n",dev->srate));
  return dev->srate;
}

/*** ZaOpenDSP ***/
static void ZaOpenDSP(struct za2_status *dev)
{
  za2command(dev,0x3000 + ((dev->mode&2) << 8) + dev->digin);  /* raise CS */
  wait_za2(dev);
  outw(0x0, dev->za2_base+4);                                         /* clock it.. 8 times actually */
  wait_za2(dev);
  za2command(dev,0x1000 + ((dev->mode&2) << 8) + dev->digin);  /* drop CS again, keep reset hi */
  wait_za2(dev);
  outw(0x0, dev->za2_base+4);                                         /* send out read bit and dummy address */
  wait_za2(dev);
}

/*** SendDSPCMD ***/
static void ZaSendDSPCMD(struct za2_status *dev,unsigned short cmd, unsigned int data)
{
  wait_za2(dev);
  outw(0, dev->za2_base+4);
  wait_za2(dev);
  outw(0, dev->za2_base+4);
  wait_za2(dev);
  outw(0, dev->za2_base+4);
  wait_za2(dev);
  outw(cmd, dev->za2_base+4);
  cmd = ((data&0xff0000)>>16);
  wait_za2(dev);
  outw(cmd, dev->za2_base+4);
  cmd = ((data&0xff00)>>8);
  wait_za2(dev);
  outw(cmd, dev->za2_base+4);
  cmd = (data&0xff);
  wait_za2(dev);
  outw(cmd, dev->za2_base+4);
}

/*** ZaCloseDSP ** (Return!=0 -> Error)*/
static short ZaCloseDSP(struct za2_status *dev)
{
  int wait;
  short last,samp,ercflag=0,r;

  za2command(dev,0x1000 + ((dev->mode&2) << 8) + dev->digin);             /* be sure we have a clock */
  ercflag += wait_za2(dev);
  outw(0x00ff, dev->za2_base+4);                                          /* signal end of block...? */
  ercflag += wait_za2(dev);
  outw(0x00ff, dev->za2_base+4);                                          /* signal end of block...? */
  ercflag += wait_za2(dev);
  za2command(dev,0x3000 + ((dev->mode&2) << 8) + dev->digin);             /* raise CS */
  ercflag += wait_za2(dev);
  outw(0x0, dev->za2_base+4);                                             /* clock it.. 8 times actually */
  ercflag += wait_za2(dev);
  za2command(dev,0x1000 + ((dev->mode&2) << 8) + dev->digin);             /* drop CS again, keep reset hi */
  ercflag += wait_za2(dev);
  outw(0x1, dev->za2_base+4);                                             /* send out read bit and dummy address */
  ercflag += wait_za2(dev);
  za2command(dev,0x1000 + ((1+(dev->mode&2)) << 8) + dev->digin);         /*  let the PC read info */
  if (ercflag) return(7);                                                 /* 7 means Clock is dead.. */
  wait = 0;
  while(!(za2status(dev)&0x200))                                         /* make sure REQ is inactive */
    {
      if(wait++>1000)
	return(8);                                                        /* 8 is 'REQ never went inactive' */
      while(za2status(dev)&0x400);                                       /* wait for host port * DEADLOCK! */
      outw(0, dev->za2_base+4);                                           /* clock it 8 times */
    }
  za2command(dev,0x1000 + (dev->mode << 8) + dev->digin);                 /* back to normal */
  wait = 0;
  while(za2status(dev)&0x200)                                             /* wait for REQ to be active */
    {
      if(wait++>2000)                                                     /* could be a long wait.. not sure why */
	return(9);                                                        /* 9 is 'REQ never came back!' */
      wait_za2(dev);
    }
  if (dev->mode&1)                                                        /* wait for the 'za' sync for mode 1 and 3 */
    {
      wait = 0;
      samp = 0;
      last=(za2status(dev)&8); 
      while(samp!=0x7a61)
	{
	  while((za2status(dev)&8)==last);                                /* DEADLOCK ! */
	  last = (za2status(dev)&8);
	  samp = (short) inw(dev->za2_base);
	  if(wait++>100)
	    {
	      if (za2status(dev)&0x200)
		return(3);
	      else
		return(1);
	    }
	}
      for(r=0;r<16;r++)
	{
	  while((za2status(dev)&8)==last);
	  last = (za2status(dev)&8);
	  if( ((unsigned short) inw(dev->za2_base)) != 0x7a61)
	    {
	      if (za2status(dev)&0x200)
		return(4);
	      else
		return(2);
	    }
	}
    }
  return(0);
}

/*** ZaNuMode ***/
static void ZaNuMode(struct za2_status *dev,short numode)
{
  int stillLoop = MAX_LOOP;
  if((dev->mode&2)!=(numode&2) || (!dev->mode))
  {
    za2command(dev,0x1000+((1+(numode&2)) << 8)+dev->digin);
    wait_za2(dev);
    wait_za2(dev);
    while((za2status(dev)&0x200)&&(--stillLoop)) wait_za2(dev);
    DEBUG(if(!stillLoop) printk("ZA2: Got no sync !\n"));
  }
  za2command(dev,0x1000+(numode << 8)+dev->digin);
  dev->mode = numode;
  dev->command &= 0xfcff;
  dev->command += (dev->mode << 8);
}

/* Ldt DSP-Code auf Karte */
short up_sim(struct za2_status *dev,char *code,int len,int program)
{
  unsigned char *za2_ptr=code;
  unsigned short r,adrs,length;
  long tcheck,checksum = 0;
  int pp = 0;
  ZaSetEnv(dev);
  za2command(dev,0x7000);     /* raise BOOT */
  wait_za2(dev);
  za2command(dev,0x5000);     /* select SPI mode, drop CS */
  wait_za2(dev);
  za2command(dev,0x4000);     /* drop RESET */
  wait_za2(dev);
  za2command(dev,0x5000);     /* raise RESET, initiate download */
  wait_za2(dev); 
  za2command(dev,0x1000);     /* drop BOOT */
  wait_za2(dev); 
  delay(100);                 /* pause */
  /* begin of upload process */
  za2command(dev,0x3000);     /* raise CS */
  wait_za2(dev);
  outw(0, dev->za2_base+4);   /* clock it.. 8 times actually */
  wait_za2(dev);
  za2command(dev,0x1000);     /* drop CS again, keep reset hi */
  wait_za2(dev);
  outw(0, dev->za2_base+4);   /* send out write bit and dummy address */
  wait_za2(dev);
  GET_CPU_EXCLUSIVITY;
  while(pp<len)
  {
    send_byte(dev,*za2_ptr);
    adrs=((*za2_ptr) << 8);
    za2_ptr++; pp++;
    send_byte(dev,*za2_ptr);
    adrs+=*za2_ptr;
    za2_ptr++; pp++;
    if(adrs==0xffff)
    {
      send_byte(dev,*(za2_ptr++)); pp++;
      send_byte(dev,*(za2_ptr++)); pp++;
      send_byte(dev,*(za2_ptr++)); pp++;
      wait_za2(dev);
      if(!(za2status(dev)&0x200))
	{
	  disable_za2(dev);	  
	  RELEASE_CPU_EXCLUSIVITY;
	  printk("ZA2: Upload of DSP-OS failed.\n");
	  return ZA2_DOWNLOAD_FAILED;
	}
      switch(program)
	{
	case RESAMPLE_48244 :
	  DEBUG(printk("48Khz "));
	  ZaNuMode(dev,3);
	  dev->filter=RESAMPLE_48244;
	  dev->blocksize=ZA2_BLOCK_SIZE;
	  DEBUG(printk("to 44.1Khz resampler loaded\n"));
	  break;
	case RESAMPLE_32244 :
	  DEBUG(printk("32Khz "));
	  ZaNuMode(dev,3);
	  dev->filter=RESAMPLE_32244;
	  dev->blocksize=ZA2_BLOCK_SIZE;
	  DEBUG(printk("to 44.1Khz resampler loaded\n"));
	  break;
	case DSPOS :
	  DEBUG(printk("DSP "));
	  dev->mode=0;
	  if(dev->currentState != ZA2_OPEN_REC)   /*to be tested !!*/
	    ZaNuMode(dev,1);
	  dev->filter=DSPOS;
	  dev->blocksize=ZA2_BLOCK_SIZE;
	  DEBUG(printk("Operating System loaded\n"));
	  break;
	case MPEG2 :
	  DEBUG(printk("MPEG2 "));
	  ZaNuMode(dev,1);
	  dev->filter=MPEG2;
	  dev->blocksize=MP2BLOCKSIZE;
	  DEBUG(printk("Decoder loaded\n"));
	  break;
	default :
	  DEBUG(printk("User program "));
	  ZaNuMode(dev,1);
	  dev->filter=USER;
	  DEBUG(printk("%d loaded (Mode 1)",program));
	}
      dev->volume = 16384;
      dev->srate = 48000;
      disable_za2(dev);
      mute_za2(dev);
      RELEASE_CPU_EXCLUSIVITY;
      return 0;
    }
    checksum+=(long)adrs;
    send_byte(dev,*za2_ptr);
    length=((*za2_ptr) << 8);
    za2_ptr++; pp++;
    send_byte(dev,*za2_ptr);
    length+=*za2_ptr;
    za2_ptr++; pp++;
    checksum+=(long)length;
    length= length/3;
    for (r=0;r<length;r++)
    {
      send_byte(dev,*za2_ptr);
      tcheck=((long)(*za2_ptr) << 16);
      za2_ptr++; pp++;
      send_byte(dev,*za2_ptr);
      tcheck+=((long)(*za2_ptr) << 8);
      za2_ptr++; pp++;
      send_byte(dev,*za2_ptr);     
      tcheck+=(long)(*za2_ptr);
      za2_ptr++; pp++;
      checksum += tcheck;
    }
  }
  RELEASE_CPU_EXCLUSIVITY;
  return ZA2_UPSIM_UNKNOW_ERROR;
}

/* Return: >0: nothing happened, 0: loaded successfully, <0: error */
int loadfilter(struct za2_status *dev,int nr)
{
  char *firmware;
  int len;
  if(dev->filter!=nr)
    {
      DEBUG(printk("Loading Firmware..."));
      switch(nr)
	{
	case DSPOS : 
	  DEBUG(printk("DSPOS "));
	  len = mod_firmware_load(ZA2FIRMWAREDIR "/dsposx2.sim", (char **) &firmware);
	  break;
	case RESAMPLE_48244 :
	  DEBUG(printk("48244 "));
	  len = mod_firmware_load(ZA2FIRMWAREDIR "/48244.sim", (char **) &firmware);
	  break;
	case RESAMPLE_32244 :
	  DEBUG(printk("32244 "));
	  len = mod_firmware_load(ZA2FIRMWAREDIR "/32244.sim", (char **) &firmware);
	  break;
	case MPEG2 :
	  DEBUG(printk("MPEG2 "));
	  len = mod_firmware_load(ZA2FIRMWAREDIR "/mpeg.sim", (char **) &firmware);
	  break;
	default : 
	  return ZA2_NEED_TO_RELOAD_FILTER;
	}
      if(!firmware)
	{
	  DEBUG(printk("ZA2: Error: Cannot load new filter !\n"));
	  return ZA2_NEED_TO_RELOAD_FILTER;
	}
      DEBUG(printk("Ok\n"));
      if(up_sim(dev,firmware,len,nr))
	{
	  DEBUG(printk("ZA2: Error while uploading dsp-code\n"));
	  dev->filter=INVALID;
	  return ZA2_NEED_TO_RELOAD_FILTER;
	}
      vfree(firmware);
    }
  else return(1);
  return(0);
}

/*** ZaCalcSampleRate ***/
/*Festkommaarithmetik: 4 Stellen hinter dem Komma (in Hertz gerechnet)*/
/*Genauigkeit ist noch nicht perfekt (z.B. bei 44056 Hz)              */
/*Verbesserungsmglichkeit: optimierte Multiplikationsroutine ?       */
#define ONE 10000
static unsigned int ZaCalcSampleRate(int nsrate)
{
  unsigned short r,bestr=0;
  int z;
  int cm1;
  int div,mlt;
  int nuy,besty=(int)(9.9*ONE),nux;
  int bestz = 0;
  nsrate*=ONE;
  for(r = 1; r < 512; r++)
    {
      nux = ((nsrate / 75 ) * r)/1280;
      nuy = nux%ONE;                 /* Nachkommastellen */
      z=nux-nuy;                     /* Vorkommastellen */
      if(nuy > ((int)(0.5*ONE)))
	{
	  nuy = ((int)(1.0*ONE)) - nuy;
	  z+=ONE;
	}
      if((z>ONE)&&(r>1))
	{
	  nuy = nsrate - ((z*1280)/r)*75;
	  if (nuy<0) nuy*=-1;
	  if (nuy<besty)
	    {
	      besty = nuy;
	      bestr = r;
	      bestz = z;
	    }
	}
    }
  div = bestr;
  mlt = bestz/ONE;
  cm1 = ((div-1)<<10)+mlt-1;
  return(cm1);
}

/* setza */
short setza(struct za2_status *dev)
{
  short  ix;                                            /* for loop */
  short  err = 0;                                       /* respond from send command (sync or not) */
  int tmpData = 0;                                      /* temporary variable */
  ZaOpenDSP(dev);
  ZaSendDSPCMD(dev,ZA2_DAC_STATE, 0x9001);              /* disable DAC*/
  ZaSendDSPCMD(dev,ZA2_SAMPLE_RATE,ZaCalcSampleRate(dev->srate));
  if(dev->aeso)
    {
      ZaSendDSPCMD(dev,ZA2_USER_BIT, 0);                  /* no user bits in AES */
      tmpData = 0x000300;
      if((dev->srate>=44000) && (dev->srate<44200))
	tmpData += 0x300000;                        /* label as 44.1khz */
      else if(dev->srate<44000)
	tmpData += 0xf00000;                        /* label as 32khz */
      else
	tmpData += 0xc00000;                        /* label as 48khz */
      if(dev->emph)
	tmpData += 0x00f000;                        /* with emphasis */
      ZaSendDSPCMD(dev,ZA2_CBITS_0, tmpData);             /* pro mode */
      ZaSendDSPCMD(dev,ZA2_CBITS_1, 0);
      ZaSendDSPCMD(dev,ZA2_CBITS_2, 0);
      ZaSendDSPCMD(dev,ZA2_CBITS_3, 0);
    }
  else /*SPDIF*/
    {
      ZaSendDSPCMD(dev,ZA2_CBITS_0, 0x003000*(1-dev->scms)+0x00c000*dev->emph);
      ZaSendDSPCMD(dev,ZA2_CBITS_1, 0xc00f00);            /* original DAT source */
      ZaSendDSPCMD(dev,ZA2_CBITS_2, 0x000000);            /* source/channel num */
      if((dev->srate>=44000) && (dev->srate<44200))
	{
	  ZaSendDSPCMD(dev,ZA2_USER_BIT, 1323);
	  ZaSendDSPCMD(dev,ZA2_CBITS_3, 0x000000);        /* label as 44.1khz */
	}
      else if(dev->srate<44000)
	{
	  ZaSendDSPCMD(dev,ZA2_USER_BIT, 1920);
	  ZaSendDSPCMD(dev,ZA2_CBITS_3, 0x000f00);        /* label as 32khz */
	}
      else
	{
	  ZaSendDSPCMD(dev,ZA2_USER_BIT, 1440);
	  ZaSendDSPCMD(dev,ZA2_CBITS_3, 0x000c00);        /* label as 48khz */
	}
    }
  ZaSendDSPCMD(dev,ZA2_SOFT_RESET, 1);          /* soft reset... */
  ZaSendDSPCMD(dev,ZA2_DO_IT, 0xffffff);        /* do it.. */
  for(ix = 0; ix < 8000; ix++)
    wait_za2(dev);
  if(dev->currentState == ZA2_CLOSE)
    {
      ZaOpenDSP(dev);
      ZaSendDSPCMD(dev,ZA2_DAC_STATE, 0xb001);  /* enable DAC */  
      err = ZaCloseDSP(dev);
      disable_za2(dev);
    }
  return err;
}

/*** zaSetSamplingRate ***/
short zaSetSamplingRate(struct za2_status *dev,int newsrate)
{
  if( (dev->currentState == ZA2_CLOSE) ||
      (!(dev->currentState&ZA2_OPEN_REC) && !dev->playing) )
    {
      if( (newsrate < 20000) || (newsrate > 60000) )
	return ZA2_VALUE_OUT_OF_BOUND;
      
      dev->srate = newsrate;
      setza(dev);
      return 0;
    }
  if(dev->playing)
    return ZA2_DEVICE_IS_PLAYING;
  return ZA2_DEVICE_NOT_IN_PLAY;
}

/*** za2PrintErr ***/
int za2PrintErr(short err) 
{
  switch(err) 
    {
    case ZA2_UNKNOW_ERROR:
      printk("ZA2: Unknow error\n"); break;
    case ZA2_BAD_BIT_FILE:
      printk("ZA2: invalid fpga mask\n"); break;
    case ZA2_CANT_CONFIG_ZA1_WITH_ZA2:
      printk("ZA2: cannot configure za1 with za2 code\n"); break;
    case ZA2_CANT_CONFIG_ZA2_WITH_ZA1:
      printk("ZA2: cannot configure za2 with za1 code\n"); break;
    case ZA2_ALREADY_INITIALISED:
      printk("ZA2: the card is already initialized\n"); break;
    case ZA2_DID_NOT_CONFIG_CORRECTLY:
      printk("ZA2: the card did not configure correctly\n"); break;
    case ZA2_NO_MEMORY:
      printk("ZA2: no memory left for buffers\n"); break;
    case ZA2_NO_HARDWARE_CONFIG:
      printk("ZA2: no hardware configuration present\n"); break;
    case ZA2_NOT_INITIALISED:
      printk("ZA2: the card is not initialized\n"); break;
    case ZA2_DOWNLOAD_FAILED:
      printk("ZA2: uploading of the DSP code failed\n"); break;
    case ZA2_UPSIM_UNKNOW_ERROR:
      printk("ZA2: unknow error while uploading DSP code\n"); break;
    case ZA2_SIM_OUT_OF_SPACE:
      printk("ZA2: Out of space while uploading DSP code\n"); break;
    case ZA2_HAS_NO_DSP_OS:
      printk("ZA2: the card has no DSP code loaded\n"); break;
    case ZA2_EXT_SYNC_NOT_VALID:
      printk("ZA2: the PLL cannot lock on incoming signal\n"); break;
    case ZA2_NEED_TO_RELOAD_FILTER:
      printk("ZA2: za2 needs to reload the filter code\n"); break;
    case ZA2_UNKNOW_MODE:
      printk("ZA2: unknow mode\n"); break;
    case ZA2_UNKNOW_PROTOCOL:
      printk("ZA2: unknow protocol\n"); break;
    case ZA2_UNKNOW_PARAM:
      printk("ZA2: parameter is unknow\n"); break;
    case ZA2_VALUE_OUT_OF_BOUND:
      printk("ZA2: value is out of bound\n"); break;
    case ZA2_DEVICE_ALREADY_OPEN:
      printk("ZA2: device is already opened\n"); break;
    case ZA2_DEVICE_NOT_OPEN:
      printk("ZA2: device is not opened\n"); break;
    case ZA2_DEVICE_NOT_IN_PLAY:
      printk("ZA2: device is not in play mode\n"); break;
    case ZA2_DEVICE_NOT_IN_REC:
      printk("ZA2: device is not in record mode\n"); break;
    case ZA2_DEVICE_NOT_IN_RECPLAY:
      printk("ZA2: device is not in record/play mode\n"); break;
    case ZA2_DEVICE_NOT_RECORDING:
      printk("ZA2: device is not recording\n"); break;
    case ZA2_DEVICE_NOT_PLAYING:
      printk("ZA2: device is not playing\n"); break;
    case ZA2_DEVICE_IS_PLAYING:
      printk("ZA2: device is already playing\n"); break;
    case ZA2_DEVICE_IS_RECORDING:
      printk("ZA2: device is aleady recording\n"); break;
    case ZA2_NO_INPUT_SIGNAL:
      printk("ZA2: No input signal present\n"); break;
    case ZA2_NO_SYNC_RECEIVED:
      printk("ZA2: PLL cannot lock, no sync received\n"); break;
    case ZA2_ACCUMULATED_RX_ERROR:
      printk("ZA2: too many transmission errors accumulated\n"); break;
    case ZA2_OUT_OF_SPACE:
      printk("ZA2: Out of space\n"); break;
    case ZA2_NO_IRQ_RECEIVED:
      printk("ZA2: No interrupt received, bad IRQ or DMA\n"); break;
    case ZA2_FEATURE_NOT_SUPPORTED:
      printk("ZA2: this feature/parameter is not supported\n"); break;
    default:
      printk("ZA2: Unknow error\n"); break;
    }
  return -1;
}
