/*  ZA2 Driver for Linux
    Copyright (C) 1998,1999 Peter Wahl

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Send bug reports to:
    Peter Wahl
    wahl@uni-bonn.de
*/

#include "za2.h"
#include <asm/uaccess.h>
#include "sound_config.h"
#include "za2hard.h"
#include "za2recplay.h"
#include "za2mixer.h"

/***  za2_read ***/ 
static ssize_t za2_read(struct file *file, char * buf, size_t count, loff_t *lp)
{
  struct za2_status *dev=za2_st;
  unsigned int minor=MINOR(file->f_dentry->d_inode->i_rdev);
  void *ptr;
  int actualCount = 0,c=count;
  if(minor==dev->minor)
    while(c)
      {
	if( (actualCount =  za2_begin_read(dev,&ptr, (size_t) c)) < 0)   /* get a pointer on dma buffer (this can block) */
	  return za2PrintErr(actualCount);
	copy_to_user(buf, ptr, actualCount);                                  /* copy actual data */ 
	c -= actualCount;
	buf += actualCount;
      }
  return count;
}

/***  za2_write ***/
static ssize_t za2_write(struct file *file, const char * buf, size_t count, loff_t *lp) 
{
  struct za2_status *dev=za2_st;
  unsigned int minor = MINOR(file->f_dentry->d_inode->i_rdev);
  void *ptr;
  int actualCount = 0,c=count;
  if(minor==dev->minor)
    {
      while(c)
	{
	  switch(dev->filter)
	    {
	    case MPEG2:
	      if( (actualCount = za2_write_mp2(dev,&ptr, (size_t) c)) < 0)  /* get a pointer on dma buffer (this can block) */
		return za2PrintErr(actualCount);
	      break;
	      default:
		if( (actualCount = za2_begin_write(dev,&ptr, (size_t) c)) < 0)  /* get a pointer on dma buffer (this can block) */
		  return za2PrintErr(actualCount);
	    }
	  copy_from_user(ptr, buf, actualCount);                    	      /* copy actual data */
	  c -= actualCount;
	  buf += actualCount;
	}
    }
  return count;
}

/*Handles IOCTL Requests*/
static int za2_ioctl(struct inode * aInode, struct file * aFile,
                     unsigned int cmd, unsigned long arg)
{
  int  err   = 0,inputrate,buf;
  long param = 0;
  struct za2_status *dev=za2_st;
  if(((cmd>>8)&0xff) == 'P') 
    {
      /* extract the parameter */
      if(_SIOC_DIR(cmd)&_SIOC_WRITE) 
	{
	  err = verify_area(VERIFY_READ, (void *) arg, sizeof(long));
	  if(err)
	    return za2PrintErr(err);
	  get_user(param,(long *) arg);
	}
      /* do the command */
      switch(cmd) 
	{
	case SNDCTL_DSP_RESET: 
	case SNDCTL_DSP_SYNC: 
	  break;
	case SNDCTL_DSP_SPEED: 
	  /* in play mode (i.e. not rec nor recplay with record enabled), */ 
	  /* we always change the rate */
	  if(!(dev->currentState&ZA2_OPEN_REC))
	    {
	      err = zaSetSamplingRate(dev,param);
	      param = zaGetSamplingRate(dev);
	    }
	  /* in record mode, we check for matching rates                       */
	  /* If the rates don't match, we try to load filters (not in recplay) */
	  else if(dev->currentState==ZA2_OPEN_REC) 
	    { 
	      if((inputrate=zaGetSamplingRate(dev)) == param) err=loadfilter(dev,DSPOS);
	      else if((inputrate==32000)&&(param==44100)) err=loadfilter(dev,RESAMPLE_32244);
	      else if((inputrate==48000)&&(param==44100)) err=loadfilter(dev,RESAMPLE_48244);
	      else { param=inputrate; err=1; }        /*else return current frequency*/
	      if((err==0) && (dev->currentState&ZA2_OPEN_REC))
		{
		  dev->currentState=ZA2_CLOSE;
		  za2_open_rec(dev);
		}
	    }
	  /*  in recplay, we only return the sampling rate of the input,  */ 
	  /*  if there is any  */
	  else param = zaGetSamplingRate(dev); 
	  if(err<0) /* unsupported rate or error */
	    return za2PrintErr(err);
	  break;
	case SNDCTL_DSP_STEREO: 
	  if(param != 1)
	    {
	      printk("ZA2: ioctl(SNDCTL_DSP_STEREO,%ld) not supported.\n",param);
	      return ZA2_FEATURE_NOT_SUPPORTED;
	    }
	  break;
	case SNDCTL_DSP_GETBLKSIZE: 
	  param = za2_get_block_size(dev);
	  break;
	case SNDCTL_DSP_SETFMT:
	  switch(param)
	    {
	    case AFMT_MPEG : 
	      if(dev->currentState==ZA2_OPEN_PLAY) 
		loadfilter(dev,MPEG2);
	      else return za2PrintErr(ZA2_FEATURE_NOT_SUPPORTED);
	      break;
	    case AFMT_S16_LE :
	      if(!(dev->filter&7)) loadfilter(dev,DSPOS);                 /* not resampler or dspos */
	      break;
	    case AFMT_QUERY :
	      if(dev->filter&7)
		param=AFMT_S16_LE;
	      else param=AFMT_MPEG;
	      break;
	    default :
	      printk("ZA2: ioctl(SNDCTL_DSP_SETFMT,%lx) not supported.\n",param);
	      return ZA2_FEATURE_NOT_SUPPORTED;
	    }
	  break;
	case SNDCTL_DSP_CHANNELS: 
	  if(param == 1)
	    {
	      printk("ZA2: ioctl(SNDCTL_DSP_CHANNELS,%ld) not supported.\n",param);
	      return ZA2_FEATURE_NOT_SUPPORTED;
	    }
	  param = 2;
	  break;
	case SNDCTL_DSP_POST: 
	  printk("ZA2: ioctl(SNDCTL_DSP_POST) not supported.\n");
	  return ZA2_FEATURE_NOT_SUPPORTED;
	case SNDCTL_DSP_SUBDIVIDE: 
	  if((param==1) || (param==2) || (param==4))
	    dev->blocksize=ZA2_BLOCK_SIZE/param;
	  else return -EINVAL;
	  break;
	case SNDCTL_DSP_SETFRAGMENT:
	  buf=1<<param;
	  if((buf<((MAX_BUFFER+1)>>1)) && (buf>128)) 
	    dev->blocksize=buf;
	  else return -EINVAL;
	  break;
	case SNDCTL_DSP_GETFMTS: 
	  param = AFMT_S16_LE|AFMT_MPEG;
	  break;
	default:
	  printk("ZA2: tried to set %x.\n",cmd);
	  return -1;
	}
      /* put back the parameter */
      if(_SIOC_DIR(cmd)&_SIOC_READ) 
	{
	  err = verify_area(VERIFY_WRITE, (void *) arg, sizeof(long));
	  if(err)
	    return err;
	  put_user(param, (long *) arg);
	}
      return 0;  /* all other settings are correct, but not handled */
    }
  else return mixer_ioctl(aInode,aFile,cmd,arg);
}

/***  za2_open ***/
static int za2_open(struct inode * aInode, struct file * aFile)
{
  int err;
  unsigned int minor = MINOR(aInode->i_rdev);
  struct za2_status *dev=za2_st;
  if(dev->currentState)
    {
      DEBUG(printk("ZA2: Device busy.\n"));
      return -EBUSY;
    }
  if(!dev->za2_initialized) 
    {
      DEBUG(printk("ZA2: Not initialized.\n"));
      return -1;
    }
  if(minor==dev->minor)
    {
      if( aFile->f_flags & O_RDWR) 
	{
	  loadfilter(dev,DSPOS);
	  dev->currentState=ZA2_OPEN_RECPLAY;
	  za2_open_recplay(dev); 
	  DEBUG(printk("ZA2: open for playing/recording, 16 bit\n"));
	}
      else if( (aFile->f_flags & O_ACCMODE) == O_RDONLY ) 
	{
	  loadfilter(dev,DSPOS);
	  err = za2_open_rec(dev);
	  DEBUG(printk("ZA2: open for recording, 16 bit\n"));
	  if(err) 
	    {
	      DEBUG(printk("ZA2: open rec: err = %d\n", err));
	      return -1;
	    }
	}
      else if( (aFile->f_flags & O_ACCMODE) == O_WRONLY) 
	{
	  loadfilter(dev,DSPOS);
	  err = za2_open_play(dev);
	  DEBUG(printk("ZA2: open for playing, 16 bit\n"));
	  if(err) 
	    {
	      DEBUG(printk("ZA2: open play: err = %d\n", err));
	      return -1;
	    }
	}
      else 
	{
	  DEBUG(printk("ZA2: Unknown open flag\n"));
	  return -1;
	}
    }
  else
    {
      DEBUG(printk("ZA2: Unknown minor device\n"));
      return -1;
    }
  MOD_INC_USE_COUNT;
  return 0;
}

/***  za2_release ***/
static int za2_release(struct inode * aInode, struct file * aFile)
{
  za2_close(za2_st);
  MOD_DEC_USE_COUNT;
  return(0);
}

static struct file_operations za2card_fops = {
  NULL,
  za2_read,
  za2_write,
  NULL,
  NULL,
  za2_ioctl,
  NULL,
  za2_open,
  NULL,
  za2_release,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL
};
