/*
 *  Copyright (c) by Martin Pahl <pahl@tnt.uni-hannover.de>
 *  Routines for Zefiro ZA2
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define __SND_OSS_COMPAT__
#include "driver.h"
#include "za2.h"

static const unsigned char fpga_bit_stream[] = {
#ifdef CONFIG_ZA2_XZA2R1
#include "xza2r1.dcl"
#else
#ifdef CONFIG_ZA2_XZA2R1N
#include "xza2r1n.dcl"
#else
#ifdef CONFIG_ZA2_XZA2R2
#include "xza2r2.dcl"
#else
#ifdef CONFIG_ZA2_XZA2R2N
#include "xza2r2n.dcl"
#else
/*#include "xza2rd.dcl"*/
#include "za2_xza2rdx3.dcl"
#endif
#endif
#endif
#endif
};

unsigned char prg1[] = {
#include "za2_dspos.inc"
};

struct snd_stru_za2_dspos dspos_default =
{
	prg1,
	sizeof(prg1),
	"dspos"
};

static void snd_za2_interrupt(snd_pcm_t *pcm)
{
	snd_pcm1_t *pcm1;
        za2_t *codec;

        pcm1 = (snd_pcm1_t *) pcm->private_data;
        codec = (za2_t *) pcm1->private_data;
	
	if (codec->trigger_value & 1)
		pcm1->playback.ack(pcm1);
	if (codec->trigger_value & 2)
		pcm1->record.ack(pcm1);
#if 0
	printk("ZA2: interrupt done\n");
#endif
}

static int snd_za2_init(snd_pcm_t *pcm, int enable)
{
	snd_pcm1_t *pcm1;
	za2_t *codec;
	int err;
	pcm1=(snd_pcm1_t*)pcm->private_data;
	codec=(za2_t*)pcm1->private_data;
	if (enable) {
		printk("ZA2: uploading fpga mask...");
		if ((err=snd_za2_upbit(codec, fpga_bit_stream))!=0) {
			if (err!=za2_ERR_ALREADY_INITIALIZED) {
				printk("failed\nZA2: error(%d) uploading fpga mask\n",err);
				return -EIO;
			}
			printk("Ok\n");
		}
		printk("ZA2: uploading micro-code ...");
		if ((err=snd_za2_upsim(codec))!=0) {
			printk("failed\nZA2: error %d uploading micro-code\n",err);
			return -EIO;
		}
		printk("Ok\n");
	}
	return 0;
}

static int snd_za2_playback_open(snd_pcm1_t *pcm1)
{
	snd_card_t *card;
	za2_t *codec;
	int err;

	card = pcm1->card;
	codec = (za2_t*)pcm1->private_data;
	printk("ZA2: playback_open: pcm1 mode=0x%x, format=0x%x, real_rate=%d\n",
	       pcm1->playback.mode,
	       pcm1->playback.format,
	       pcm1->playback.real_rate);
	snd_za2_ZaNuMode(codec,za2_MODE_FREE_PLAY);
	if ((err=snd_pcm1_dma_alloc(pcm1, SND_PCM1_PLAYBACK, 
				    codec->dma_out16ptr, "ZA2 (playback)"))<0)
		return err;
	return 0;
}

static void snd_za2_playback_close(snd_pcm1_t *pcm1)
{
	za2_t *codec;
	
	codec=(za2_t*)pcm1->private_data;
	snd_pcm1_dma_free(pcm1, SND_PCM1_PLAYBACK, codec->dma_out16ptr);
}

static int snd_za2_playback_ioctl(snd_pcm1_t *pcm1,
				  unsigned int cmd, unsigned long *arg)
{
	printk("snd_za2_playback_ioctl: cmd=%d\n",cmd);
	switch (cmd) {
	case SND_PCM1_IOCTL_RATE:
		pcm1->playback.real_rate = pcm1->playback.rate;
		if (pcm1->playback.real_rate < pcm1->playback.hw.min_rate)
			pcm1->playback.real_rate = pcm1->playback.hw.min_rate;
		if (pcm1->playback.real_rate > pcm1->playback.hw.max_rate)
			pcm1->playback.real_rate = pcm1->playback.hw.max_rate;
		printk("snd_za2_playback_ioctl: realrate=%d rate=%d\n",
		       pcm1->playback.real_rate,pcm1->playback.rate);
		return 0;
	}
	return -ENXIO;
		
}

static void snd_za2_playback_prepare(snd_pcm1_t *pcm1,
 				     unsigned char *buffer,
				     unsigned int size,
				     unsigned int offset,
				     unsigned int count)
{
	za2_t *codec;
	snd_pcm1_channel_t *pchn;
	long irq_rate;
	snd_pcm_t *pcm;

	printk("ZA2: playback_prepare: pcm1 mode=0x%x, format=%d, real_rate=%d\n",
	       pcm1->playback.mode,
	       pcm1->playback.format,
	       pcm1->playback.real_rate);
	pchn = &pcm1->playback;
	/*! Set sampling-rate !*/
	codec = (za2_t*) pcm1->private_data;
	if (!snd_za2_initialized(codec))
		snd_za2_upsim(codec);
	snd_za2_ZaSetEnv(codec);

	pcm = codec->pcm;
	irq_rate = pchn->block_size >>2;
	snd_za2_ZaOpenDSP(codec);
	snd_za2_ZaSendDSP(codec, za2_REG_SYNC_OUTPUT,0);
	snd_za2_ZaSendDSP(codec, za2_REG_DAC_STATE, 0xb001);
	snd_za2_ZaSendDSP(codec, za2_REG_IRQ_RATE, irq_rate);
	printk("ZA2: CloseDSP=%d, irq_rate=0x%x(%d) \n",
	       snd_za2_ZaCloseDSP(codec),
	       irq_rate,
	       pchn->block_size);

	printk("ZA2: playback_prepare command=0x%x(0x%x)\n",
	       codec->command,codec->command&za2_NBITS_IDMA);
	snd_dma_program(codec->dma_out16, buffer, size, 
			DMA_MODE_WRITE | DMA_MODE_AUTOINIT);
	printk("ZA2: playback_prepare (Done) dma=%d, buffer=0x%x(%x), size=0x%x(%d), mode=%x\n",
	       codec->dma_out16, buffer, virt_to_bus((void*)buffer), size, size, 
	       DMA_MODE_WRITE | DMA_MODE_AUTOINIT);
	
}

static unsigned int snd_za2_playback_pointer(snd_pcm1_t *pcm1, 
					     unsigned int used_size)
{
	za2_t *codec;
	
	codec=(za2_t*)pcm1->private_data;
	printk("ZA2: playback_pointer\n");
	
	return used_size-snd_dma_residue(codec->dma_out16);
}

static void snd_za2_playback_trigger(snd_pcm1_t * pcm1, int up)
{
	za2_t *codec;
	unsigned long flags;
	
	codec = (za2_t*) pcm1->private_data;
	codec->trigger_value=(up?1:0);
	printk("ZA2: playback_trigger (up=%d, command=0x%x)\n",
	       up,codec->command&za2_NBITS_IDMA);
	if (up) {
		/*! Disable DMA-IN, no duplex yet !*/
		outw(codec->command&za2_NBITS_IDMA,ZA2P(codec,STATUS));
		/*dma_outb(0, DMA2_CMD_REG);*/ /* enable controller */

	}
	else {
		snd_cli(&flags);
		outw(codec->command & za2_BITS_INIT,ZA2P(codec,STATUS));
		snd_za2_ZaOpenDSP(codec);
		/* set R=R and L=L again */
		snd_za2_ZaSendDSP(codec,za2_REG_SYNC_OUTPUT,1);  
		printk("ZA2: CloseDSP=%d\n",snd_za2_ZaCloseDSP(codec));
		outw(codec->command & za2_BITS_INIT,ZA2P(codec,STATUS));
		outw(0,ZA2P(codec,SAMPLES));
		snd_sti(&flags);
	}
	printk("ZA2: playback_trigger (Done)\n");
}

static int snd_za2_record_open(snd_pcm1_t *pcm1)
{
	snd_card_t *card;
	za2_t *codec;
	int err;

	card = pcm1->card;
	codec = (za2_t*)pcm1->private_data;
	if ((err=snd_pcm1_dma_alloc(pcm1, SND_PCM1_RECORD, 
				    codec->dma_in16ptr, "ZA2 (playback)"))<0)
		return err;
	return 0;
}

static void snd_za2_record_close(snd_pcm1_t *pcm1)
{
	za2_t *codec;
	
	codec=(za2_t*)pcm1->private_data;
	snd_pcm1_dma_free(pcm1, SND_PCM1_RECORD, codec->dma_in16ptr);
}

static int snd_za2_record_ioctl(snd_pcm1_t *pcm1,
				unsigned int cmd, unsigned long *arg)
{
	switch (cmd) {
	case SND_PCM1_IOCTL_RATE:
		pcm1->record.real_rate = pcm1->record.rate;
		if (pcm1->record.real_rate < pcm1->record.hw.min_rate)
			pcm1->record.real_rate = pcm1->record.hw.min_rate;
		if (pcm1->record.real_rate > pcm1->record.hw.max_rate)
			pcm1->record.real_rate = pcm1->record.hw.max_rate;
		return 0;
	}
	return -ENXIO;
		
}
static void snd_za2_record_prepare(snd_pcm1_t *pcm1,
				   unsigned char *buffer,
				   unsigned int size,
				   unsigned int offset,
				   unsigned int count)
{
	unsigned long flags;
	za2_t *codec;
	snd_pcm1_channel_t *pchn;
	long irq_rate;
	snd_pcm_t *pcm;
	int k;

	pchn = &pcm1->record;
	/*! set sampling-rate !*/
	codec = (za2_t*) pcm1->private_data;
	if (!snd_za2_initialized(codec))
		snd_za2_upsim(codec);
	snd_za2_ZaSetEnv(codec);

	pcm = codec->pcm;
	irq_rate = pcm1->record.hw.min_fragment << 1;
	snd_cli(&flags);
	snd_za2_ZaOpenDSP(codec);
	snd_za2_ZaSendDSP(codec, za2_REG_IRQ_RATE, irq_rate);
	if (snd_za2_ZaCloseDSP(codec)) {
		snd_sti(&flags);
		printk("snd_za2_record_prepare: PLL is unable to lock\n");
		/*! Error handling !*/
		return;
	}
		
	k=65536;
	while( (((unsigned short) inw(ZA2P(codec,SAMPLES))) == za2_ZA_ID) 
	       && --k ) 
		;

	snd_sti(&flags);

	snd_dma_program(codec->dma_in16, buffer, size, 
			DMA_MODE_READ | DMA_MODE_AUTOINIT);
	if (!k) {
		/*! Error handling (no sync received) !*/
		return;
	}
}

static unsigned int snd_za2_record_pointer(snd_pcm1_t *pcm1, 
					   unsigned int used_size)
{
	za2_t *codec;
	
	codec=(za2_t*)pcm1->private_data;
	
	return used_size-snd_dma_residue(codec->dma_in16);
}

static void snd_za2_record_trigger(snd_pcm1_t *pcm1, int up)
{
	za2_t *codec;
	
	codec = (za2_t*) pcm1->private_data;
	codec->trigger_value=(up?2:0);
	if (up) {
		/*! Disable DMA-OUT, no duplex yet !*/
		outw(codec->command&za2_NBITS_ODMA,ZA2P(codec,STATUS));
	}
	else {

		outw(0,ZA2P(codec,SAMPLES));
		outw(codec->command & za2_BITS_INIT,ZA2P(codec,STATUS));
	}
}

static void snd_za2_free(void *private_data);

static struct snd_stru_pcm1_hardware snd_za2_playback =
{
	NULL, /* private data */
	NULL, /* private free */
	SND_PCM1_HW_AUTODMA | SND_PCM1_HW_16BITONLY, /* flags */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_S16_LE, /* formats */
	SND_PCM_FMT_S16_LE, /* hardware formats */
	0,     /* align value */
	6,     /* minimal fragment */
	20000, /* min. rate */
	60000, /* max. rate */
	2,
	snd_za2_playback_open,
	snd_za2_playback_close,
	snd_za2_playback_ioctl,
	snd_za2_playback_prepare,
	snd_za2_playback_trigger,
	snd_za2_playback_pointer,
	snd_pcm1_playback_dma_ulaw,
	snd_pcm1_dma_move,
	snd_pcm1_playback_dma_neutral
};

static struct snd_stru_pcm1_hardware snd_za2_record =
{
	NULL, /* private data */
	NULL, /* private free */
	SND_PCM1_HW_AUTODMA | SND_PCM1_HW_16BITONLY, /* flags */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_S16_LE, /* formats */
	SND_PCM_FMT_S16_LE, /* hardware formats */
	0,     /* align value */
	6,    /* minimal fragment */
	20000, /* min. rate */
	60000, /* max. rate */
	2,
	snd_za2_record_open,
	snd_za2_record_close,
	snd_za2_record_ioctl,
	snd_za2_record_prepare,
	snd_za2_record_trigger,
	snd_za2_record_pointer,
	snd_pcm1_record_dma_ulaw,
	snd_pcm1_dma_move,
	NULL
};

static void snd_za2_free(void *private_data)
{
	snd_free(private_data, sizeof(za2_t));
}

snd_pcm_t *snd_za2_new_device(snd_card_t *card,
			      unsigned short port,
			      snd_irq_t *irqptr,
			      snd_dma_t *dma_in16ptr,
			      snd_dma_t *dma_out16ptr,
			      unsigned short digin,
			      unsigned short za2mode)
{
	snd_pcm_t *pcm;
        snd_pcm1_t *pcm1;
        za2_t *codec;
 
	pcm = snd_pcm1_new_device(card, "ZA2");
	if (!pcm)
		return NULL;
	pcm1 = (snd_pcm1_t *) pcm->private_data;
	codec = (za2_t*) snd_calloc(sizeof(za2_t));
	if (!codec)
		return NULL;
	snd_spin_prepare(codec,reg);
	codec->pcm = pcm;
	codec->card = pcm1->card;
	codec->port = port;
	codec->irqptr = irqptr;
	codec->irq = irqptr->irq;
	codec->dma_in16ptr = dma_in16ptr;
	codec->dma_in16 = dma_in16ptr->dma;
	codec->dma_out16ptr = dma_out16ptr;
	codec->dma_out16 = dma_out16ptr->dma;
	codec->digin = digin;
	codec->za2mode = za2mode;
	codec->dspos = &dspos_default;
	memcpy(&pcm1->playback.hw, &snd_za2_playback, sizeof(snd_za2_playback));
	memcpy(&pcm1->record.hw, &snd_za2_record, sizeof(snd_za2_record));
	pcm1->private_data = codec;
	pcm1->private_free = snd_za2_free;
	pcm->info_flags = SND_PCM_INFO_CODEC | SND_PCM_INFO_MMAP |
            SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_RECORD;
	sprintf(pcm->name,"ZA2");
#ifdef USELESS
	if (snd_za2_probe(pcm1) < 0) {
		snd_pcm_free(pcm);
		return NULL;
	}
#endif
	return pcm;
}

