/////////////////////////////////////////////////////////////////
//
/////////////////////////////////////////////////////////////////

#define SND_MAIN_OBJECT_FILE

#include "../include/driver.h"
#include "../include/initval.h"
#include "../include/midi.h"

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;	/* Index 1-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;	/* ID for this card */
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for MotuMTPAV MIDI." );
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for MotuMTPAV MIDI." );

//int snd_mtpav_use_inc;

#define MTPAV_IOBASE 0x378
#define MTPAV_IRQ    7

#define SIGS_BYTE 0x08
#define SIGS_RFD 0x80
#define SIGS_IRQ 0x40
#define SIGS_IN0 0x10
#define SIGS_IN1 0x20

#define SIGC_WRITE 0x04
#define SIGC_READ 0x08
#define SIGC_INTEN 0x10

#define DREG 0
#define SREG 1
#define CREG 2


	
#define MTPAV_MODE_INPUT_OPENED  0x01
#define MTPAV_MODE_OUTPUT_OPENED  0x02
#define MTPAV_MODE_INPUT_TRIGGERED  0x04

//////////////////////////////////////////////////////////////////

typedef unsigned char U8;
typedef unsigned short int U16;
typedef unsigned long int U32;
typedef signed char S8;
typedef signed short int S16;
typedef signed long int S32;
typedef unsigned char UBOOL;

void note_on_port( U8 port, U8 chan, U8 note, U8 vel );
void note_off_port( U8 port, U8 chan, U8 note );

///////////////////////////////////////////////////////////////////

typedef struct Smtp
{

   snd_card_t *card;
   snd_port_t *rioport;
   snd_rawmidi_t *rmidi;
   snd_irq_t *irq;
   spinlock_t spinlock;
   U8 mode;
   
} TSmtp;

TSmtp *new_TSmtp( void )
{  TSmtp *ncrd = (TSmtp *) snd_kcalloc( sizeof( TSmtp ), GFP_KERNEL );

   if( ncrd != NULL )
   {    ncrd->card = NULL;
        ncrd->rioport = NULL;
        ncrd->rmidi = NULL;
        ncrd->irq = NULL;
        ncrd->mode = 0;
   }
   return ncrd;
}

void free_TSmtp( TSmtp *crd )
{    snd_kfree( crd );
}

///////////////////////////////////////////////////////////////////////////////

TSmtp *mtp_card;

U8 mgetreg( U16 reg )
{
	U8 rval = 0;

	if( reg == SREG )
	{	rval = inb( MTPAV_IOBASE+SREG );
		rval = (rval & 0xf8);
	}
	else if( reg == CREG )
	{	rval = inb( MTPAV_IOBASE+CREG );
		rval = (rval & 0x1c);
	}

	return rval;
}

///////////////////////////////////////////////////////////////////////////////

void mputreg( U16 reg, U8 val )
{
	if( reg == DREG )
	{	outb( val, (MTPAV_IOBASE+DREG) );
	}
	else if( reg == CREG )
	{	outb( val, (MTPAV_IOBASE+CREG) );
	}
	
}

//////////////////////////////////////////////////////////////////

void read_bus_handler( void )
{	
	U8 clrread, setread;
   U8 mtp_read_byte;
   U8 read_sr0, read_sr1, read_sr2, read_sr3;
   
   U8 sbyt = mgetreg( SREG );

   if( sbyt & SIGS_BYTE )
	{	U8 cbyt = mgetreg( CREG );
		clrread = cbyt & ( SIGC_READ ^ 0xff );
		setread = cbyt | SIGC_READ;
						
		mputreg( CREG, setread );
		read_sr0 = mgetreg( SREG );
		mputreg( CREG, clrread );

   	mputreg( CREG, setread );
		read_sr1 = mgetreg( SREG );
		mputreg( CREG, clrread );

		mputreg( CREG, setread );
		read_sr2 = mgetreg( SREG );
		mputreg( CREG, clrread );

   	mputreg( CREG, setread );
   	read_sr3 = mgetreg( SREG );
   	mputreg( CREG, clrread );

   	mtp_read_byte = ((read_sr0 & (SIGS_IN0|SIGS_IN1) ) /16 );
	   mtp_read_byte |= (((read_sr1 & (SIGS_IN0|SIGS_IN1) ) /16)<<2);
	   mtp_read_byte |= (((read_sr2 & (SIGS_IN0|SIGS_IN1) ) /16)<<4);
	   mtp_read_byte |= (((read_sr3 & (SIGS_IN0|SIGS_IN1) ) /16)<<6);

      if( mtp_card->mode & MTPAV_MODE_INPUT_TRIGGERED )
          mtp_card->rmidi->input.data( mtp_card->rmidi, & mtp_read_byte, 1);
   }
}

///////////////////////////////////////////////////////////////////////////////

void write_byte( U8 outbyte )
{
	U8 tcbyt;
	U8 byte;
	U8 clrwrite;
	U8 setwrite;

	U8 sbyt = mgetreg( SREG );

	while( ! (sbyt & SIGS_RFD) ) // wait for RFD hi
	{	sbyt = mgetreg( SREG );
	}

	/////////////////
	
	tcbyt = mgetreg( CREG );
	clrwrite = tcbyt & ( SIGC_WRITE ^ 0xff );
	setwrite = tcbyt | SIGC_WRITE;

	/////////////////
   // this stage is loopable!
   
	{  mputreg( DREG, outbyte );
      mputreg( CREG, clrwrite );	// clear write bit
      mputreg( CREG, setwrite );	// set write bit
   }

}

void note_on_port( U8 port, U8 chan, U8 note, U8 vel )
{	write_byte( 0xf5 );
   write_byte( port );
   write_byte( 0x90 | chan );
   write_byte( note );
   write_byte( vel );
}

///////////////////////////////////////////////////////////

void note_off_port( U8 port, U8 chan, U8 note )
{	write_byte( 0xf5 );
   write_byte( port );
   write_byte( 0x90 | chan );
   write_byte( note );
   write_byte( 0 );
}

//////////////////////////////////////////////////////////////////

static void snd_mtpav_use_inc(snd_card_t * card)
{
	MOD_INC_USE_COUNT;
}

static void snd_mtpav_use_dec(snd_card_t * card)
{
	MOD_DEC_USE_COUNT;
}

static void snd_mtpav_irqh(int irq, void *dev_id, struct pt_regs *regs)
{
	U32 flags;

   	if( mtp_card != 0 )
		spin_lock_irqsave( & mtp_card->spinlock, flags );
   
   	read_bus_handler();
   
	if( mtp_card != 0 )
		spin_unlock_irqrestore( & mtp_card->spinlock, flags );

}

//////////////////////////////////////////////////////////////////////////

static int snd_mtpav_input_open(snd_rawmidi_t * rmidi)
{  U32 flags;
   spin_lock_irqsave( & mtp_card->spinlock, flags);
   mtp_card->mode |= MTPAV_MODE_INPUT_OPENED;
   spin_unlock_irqrestore( & mtp_card->spinlock, flags);
   return 0;
}

static int snd_mtpav_input_close(snd_rawmidi_t * rmidi)
{  U32 flags;
   spin_lock_irqsave( & mtp_card->spinlock, flags );
   mtp_card->mode &= (~MTPAV_MODE_INPUT_OPENED);
	spin_unlock_irqrestore( & mtp_card->spinlock, flags);
   return 0;
}

static void snd_mtpav_input_trigger(snd_rawmidi_t * rmidi, int up)
{
   U32 flags;

   spin_lock_irqsave( & mtp_card->spinlock, flags);

   if (up)
		mtp_card->mode |= MTPAV_MODE_INPUT_TRIGGERED;
	else
		mtp_card->mode &= ~MTPAV_MODE_INPUT_TRIGGERED;

   spin_unlock_irqrestore( & mtp_card->spinlock, flags);

}

///////////////////////////////////////////////////////////////////////////////////
static int snd_mtpav_output_open(snd_rawmidi_t * rmidi)
{  U32 flags;
   spin_lock_irqsave( & mtp_card->spinlock, flags);
   mtp_card->mode |= MTPAV_MODE_OUTPUT_OPENED;
   spin_unlock_irqrestore( & mtp_card->spinlock, flags);
   return 0;
};

static int snd_mtpav_output_close(snd_rawmidi_t * rmidi)
{  U32 flags;
   spin_lock_irqsave( & mtp_card->spinlock, flags );
   mtp_card->mode &= (~MTPAV_MODE_INPUT_OPENED);
	spin_unlock_irqrestore( & mtp_card->spinlock, flags);
   return 0;
};

static void snd_mtpav_output_write(snd_rawmidi_t * rmidi)
{  U8 outbyte;

   if( rmidi->output.data( rmidi, &outbyte, 1 ) != 1 )
   {
   }
   else
   {  write_byte( outbyte );
   }

}

///////////////////////////////////////////////////////////////////////////////

static struct snd_stru_rawmidi_direction_hw snd_mtpav_output =
{
	SND_RAWMIDI_HW_POLL,	// flags
	NULL,			/// private_data
	NULL,			/// private_free
	snd_mtpav_output_open,	/// open
	snd_mtpav_output_close,	/// close
	NULL,			/// trigger
	{snd_mtpav_output_write},	// io.write
	NULL,			// abort/

};

static struct snd_stru_rawmidi_direction_hw snd_mtpav_input =
{
	0,			// flags
	NULL,			// private_data
	NULL,			// private_free
	snd_mtpav_input_open,	// open
	snd_mtpav_input_close,	// close
	snd_mtpav_input_trigger,	// trigger
	{NULL},			// io.read
	NULL,			// abort

};

#define MTPDEV 0

int mtp_card_reg( TSmtp *mtpc )
{	char longname_buffer[80];
	int port;
	int detected_ports = 0;
   snd_port_t *rioport;
   U32 flags;
   int possible_irqs[2];
   UBOOL irqtst;

   ///////////////////////
   
   mtpc->card = snd_card_new(snd_index[MTPDEV], snd_id[MTPDEV], snd_mtpav_use_inc, snd_mtpav_use_dec);
   
   //snd_printd ("snd_serial_probe snd_card_new'ed\n");
   
   if (mtpc->card == NULL)
	return -ENOMEM;
	
	mtpc->card->type = SND_CARD_TYPE_MTPAV;
	strcpy(mtpc->card->abbreviation, "mtpav");
	strcpy(mtpc->card->shortname, "mtpav on parallel port" );
	memset(longname_buffer, 0, sizeof(longname_buffer));
	sprintf(longname_buffer, "mtpav on parallel port at" );

   /////////////////////////////
   
   mtpc->rmidi = snd_rawmidi_new_device(mtpc->card, "MTPAV MIDI");
   if (mtpc->rmidi == NULL)
   	return NULL;

	mtpc->spinlock = SPIN_LOCK_UNLOCKED;
	
	memcpy(& mtpc->rmidi->output.hw, & snd_mtpav_output, sizeof(snd_mtpav_output));
	memcpy(& mtpc->rmidi->input.hw, & snd_mtpav_input, sizeof(snd_mtpav_input));

   ///////////////////////////////////////////
   // get ISA resources
   
   spin_lock_irqsave(  & mtpc->spinlock, flags );
	if( snd_register_ioport( mtpc->card, MTPAV_IOBASE, 3, "MotuMTPAV MIDI", & mtpc->rioport ) < 0)
   {	snd_printd("snd_uart16550_register_resources snd_register_ioport failed, port\n");
		spin_unlock_irqrestore( &mtpc->spinlock, flags );
		return -EBUSY;
	}

   possible_irqs[0] = 7;
   possible_irqs[1] = -1;
   snd_printd("snd_mtpav_register_resources requesting irq=%d\n", MTPAV_IRQ );
   irqtst = snd_register_interrupt( mtpc->card, "MOTU MTPAV", MTPAV_IRQ, SND_IRQ_TYPE_ISA, snd_mtpav_irqh, (void *) mtpc->rmidi, possible_irqs, & mtpc->irq );
   printk("mtpav irqtst: %d\n", irqtst );

   spin_unlock_irqrestore( & mtpc->spinlock, flags );

   //////////////////////////////////////////////
   
   mtpc->rmidi->info_flags |= SND_RAWMIDI_INFO_OUTPUT | SND_RAWMIDI_INFO_INPUT | SND_RAWMIDI_INFO_DUPLEX;
	strcpy( mtpc->rmidi->name, "MotuMIDI" );

    ////////////////////////////
      
	if (__snd_rawmidi_register(mtpc->rmidi, detected_ports) < 0)
   {  printk("__snd_rawmidi_register failed\n");
		snd_rawmidi_free(mtpc->rmidi);
		//continue;
	}

	snd_card_register(mtpc->card);

	return 0;
}

//////////////////////////////////////////////////////////////////

void mtp_card_unreg( TSmtp *mtpc )
{  //snd_printd ("cleanup_module serial.o card unregistering\n");
	snd_unregister_ioports( mtpc->card );
   snd_unregister_interrupts( mtpc->card );
   snd_card_unregister(mtpc->card);
   __snd_rawmidi_unregister(mtpc->rmidi);
   snd_card_free(mtpc->card);
	//snd_serial_free(serial_card);
}

//////////////////////////////////////////////////////////////////

int init_module( void )
{
   U32 tport;
   U32 tchan;

   mtp_card = new_TSmtp();
   if( mtp_card == NULL )
       return -ENOMEM;

   mtp_card_reg( mtp_card );
   
   mputreg( CREG, SIGC_INTEN );

   for( tport = 1; tport<=8; tport++ )
   {    for( tchan=0; tchan<16; tchan++ )
        {    note_on_port( tport, tchan, 0x20, 0x7f );
        }
        for( tchan=0; tchan<16; tchan++ )
        {    note_off_port( tport, tchan, 0x20 );
        }
   }

   return 0;
}

//////////////////////////////////////////////////////////////////

void cleanup_module( void )
{    mtp_card_unreg( mtp_card );
     free_TSmtp( mtp_card );
}
