<?php
# functions.php
# Copyright (C) 2007,2008 Robert Alex Marder
# This file is part of RAM-CMS
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 3
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
# or see <http://www.gnu.org/licenses/>.
#
# This file contains all the functions used by RAM-CMS.
# The RAM-CMS function library.
#


# Purpose: determine wwhen the requested page was last updated, and format as string.
# Santax: str $updated_show = ramcms_last_update_time ( str $page )
# Where: 
# $updated_show is the last update time as a date string, or 'unknown'.
# $page is the page requested.
#
function ramcms_last_update_time($page)
{
	# get the last updated time from the file of the page we want.
	# we check the file creation/change time and the file modified time and 
	# take the newer of the two. This way we have foolproof last update time.
	# the latest update time of the requested file gets put in [[updated]].
	
	# get the file modification time.
	$filemtime=filemtime($page);
	# get the file creation time.
	$filectime=filectime($page);
	
	# determine if file creation time or file modified time is newer (higher)
	if($filectime > $filemtime)
	{
		# file creation time is newer.
		$updated=$filectime;
	}
	else
	{
		# file modification time is newer.
		$updated=$filemtime;
	}
	
	# if for some reason we failed to figure out when we were last 
	# updated, fix $updated_show to be "unknown" as opposed 
	# to nothing.
	# if we did figure out what the last update was, convert 
	# $updated_show from a UNIX timestamp into something humans 
	# prefer to read.
	if($updated==false)
	{
		$updated_show='unknown';
	}
	else
	{
		# format example: 28 Sep 2007
		# TODO: make the date format configurable in config.php
		$updated_show=date('d M Y',$updated);
	}
	
	# return the last update time of the file, to be put into [[updated]].
	return $updated_show;
}

# Purpose: to output http headers based on a set of criteria
# Santax: null ramcms_send_http_headers ( bool $nocache )
# Where: 
# $nocache is if we should send anti-cache headers.
#
function ramcms_send_http_headers($nocache)
{
	# send some http headers.
	
	# please leave this here.
	header('X-CMS: RAM-CMS');
	
	if($nocache==true)
	{
		header('Cache-Control: no-store, no-cache, must-revalidate, post-check=0, pre-check=0');
		header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');
		header('Pragma: no-cache');
	}
	else
	{
		# cache content for 1 hour.
		header('Cache-Control: max-age=3600, proxy-revalidate');
		# only browsers may cache this.
		header('Pragma: private');
	}
}

# Purpose: to strip unwanted whitespace from $body so as to reduce bandwidth.
# Santax: str $body = ramcms_remove_body_whitespace ( str $body )
# Where: 
# $body is the $body of the requested page.
#
function ramcms_remove_body_whitespace($body)
{
	# using strstr twice is faster than using one regex (I think).
	
	# if <PRE> or <pre> is found in $body then preserve whitespace.
	# otherwise remove all tabs and newlines.
	if(strstr($body,'<PRE')==false && strstr($body,'<pre')==false && strstr($body,'<TEXTAREA')==false && strstr($body,'<textarea')==false)
	{
		# strip all newlines and tabs.
		$body=str_replace("\t",'',str_replace("\n",'',str_replace("\r",'',$body)));
	}
	return $body;
}

# Purpose: to replace [[string]] items with specific $string equivalents.
# Santax: 
# Where: 
#
function ramcms_page_template_parser($page_template,$title,$site_name,$header,$updated_show,$body,$copyright)
{
	# parse the $page_template information.
	$page_template=str_replace('[[body]]',$body,$page_template);
	$page_template=str_replace('[[copyright]]',$copyright,$page_template);
	$page_template=str_replace('[[header]]',$header,$page_template);
	$page_template=str_replace('[[nav]]',file_get_contents('nav.txt'),$page_template);
	$page_template=str_replace('[[site]]',$site_name,$page_template);
	$page_template=str_replace('[[title]]',$title,$page_template);
	$page_template=str_replace('[[updated]]',$updated_show,$page_template);
	$page_template=str_replace('[[ver]]','0.2.0',$page_template);
	
	return $page_template;
}

# Purpose: to parse BBCode tags in $body.
# Santax: str $body = ramcms_bbcode_parser ( str $body )
# Where: 
# $body is the $body of the requested page.
#
function ramcms_bbcode_parser($body)
{
	# TODO: make a better BBCode parser.
	
	# bold
	$body=str_replace('[b]','<b>',$body);
	$body=str_replace('[/b]','</b>',$body);
	
	# italics
	$body=str_replace('[i]','<i>',$body);
	$body=str_replace('[/i]','</i>',$body);
	
	# underline
	$body=str_replace('[u]','<u>',$body);
	$body=str_replace('[/u]','</u>',$body);
	
	# strikethrough
	$body=str_replace('[s]','<strike>',$body);
	$body=str_replace('[/s]','</strike>',$body);
	
	# teletype
	$body=str_replace('[tt]','<tt>',$body);
	$body=str_replace('[/tt]','</tt>',$body);
	
	# preformatted text
	$body=str_replace('[pre]','<pre>',$body);
	$body=str_replace('[/pre]','</pre>',$body);
	
	# subscript
	$body=str_replace('[sub]','<sub>',$body);
	$body=str_replace('[/sub]','</sub>',$body);
	
	# superscript
	$body=str_replace('[sup]','<sup>',$body);
	$body=str_replace('[/sup]','</sup>',$body);
	
	# big
	$body=str_replace('[big]','<big>',$body);
	$body=str_replace('[/big]','</big>',$body);
	
	# small
	$body=str_replace('[small]','<small>',$body);
	$body=str_replace('[/small]','</small>',$body);
	
	# horizontal rule
	$body=str_replace('[hr]','<hr>',$body);
	
	# center
	$body=str_replace('[center]','<center>',$body);
	$body=str_replace('[/center]','</center>',$body);
	
	# paragraphs
	$body=str_replace('[p]','<p>',$body);
	$body=str_replace('[/p]','</p>',$body);
	
	# line breaks
	$body=str_replace('[br]','<br>',$body);
	
	# list items
	$body=str_replace('[li]','<li>',$body);
	$body=str_replace('[/li]','</li>',$body);
	
	# unordered lists
	$body=str_replace('[ul]','<ul>',$body);
	$body=str_replace('[/ul]','</ul>',$body);
	
	# ordered lists
	$body=str_replace('[ol]','<ol>',$body);
	$body=str_replace('[/ol]','</ol>',$body);
	
	# headings
	$body=preg_replace('/\[h([1-6])\](.+)\[\/h([1-6])\]/U','<h${1}>${2}</h${1}>',$body);
	
	### Complex BBCode ###
	
	# handles [url]http://example.com/blah..[/url]
	$body=preg_replace('/\[url\](.+)\[\/url\]/U','<a href="${1}">${1}</a>',$body);
	
	# handles [url=http://example.com/blah..]Click Blah[/url]
	$pattern='/\[url=(.+)\](.+)\[\/url\]/U';
	$replacement='<a href="${1}">${2}</a>';
	$body=preg_replace($pattern,$replacement,$body);
	
	# handles [color=colorname]text here[/color]
	$body=preg_replace('/\[color=(.+)\](.+)\[\/color\]/U','<font color="${1}">${2}</font>',$body);
	
	# handles [img=imagefile]
	$body=preg_replace('/\[img=(.+)\]/U','<img alt="" src="${1}">',$body);
	
	# handles [!-- this is a comment --]
	$body=preg_replace('/\[!--(.+)--]/U','',$body);
	
	return $body;
}

#############################################################################
# RAM's PHP Compat Kit Functions - see http://freeprogs.us.tt/?page=phpcompat
#############################################################################

# file_get_contents() function for PHP < 4.3.0
# requires PHP >= 3.0
# Grade: B
# Only supports the first argument (filename)
if(function_exists('file_get_contents')==false)
{
	function file_get_contents($filename)
	{
		$fp=fopen($filename,'r');
		unset($filename);
		if($fp)
		{
			$data='';
			while(!feof($fp))
			{
				$data=$data.fread($fp,1024);
			}
			fclose($fp);
			return $data;
		}
		return false;
	}
}

?>