#include "mhi.h"

/** MHi Class:
 * Classe MHI, contem a implementacao do algoritmo Motion History Images (MHI).
 * No construtor da classe as variveis sao setadas e o numero de frames eh
 * recebido por parametro inteiro.
 * @author: Robson Pierangeli Godinho e Michel Melo da Silva
 * @date: 2011/03/01
 * @last: 2011/03/01
 */
Mhi::Mhi(int nFrames)
{
    // #frames
    Nframes = nFrames;

    // ring image buffer
    buf = 0;
    last = 0;

    // temporary images
    mhi = 0;     // MHI
    mask = 0;    // valid orientation mask
    storage = 0; // temporary storage
}

/** Update MHI:
 * Faz o calculo do MHI usando a instancia do objeto que foi usado para chamar
 * este metodo. Recebe como parametro de entrada uma imagem (src) no formato RGB
 * que sera usada para calcular o MHI, um inteiro indicando qual o valor a ser
 * usado no threshold e um double para indicar a duracao do MHI.
 * @return: void
 * @param: IplImage* -> imagem usada para realizar o MHI.
 *         int threshold-> valor utilizado para fazer o threshould.
 *         double MHI_DURATION -> valor utilizado para calculo no cvCvtScale.
 * @author: Robson Pierangeli Godinho e Michel Melo da Silva
 * @date: 2011/03/01
 * @last: 2012/10/18
 */
void Mhi::update_mhi( IplImage* src, IplImage* dst_color, IplImage* dst_gray,
                           int threshold, double MHI_DURATION)
{
    double timestamp = (double)clock()/CLOCKS_PER_SEC; // get current time in seconds

    CvSize size = cvSize(src->width,src->height); // get current frame size

    IplImage* silh;

    int i, j, idx1 = last, idx2;
    int red,green,blue;
    unsigned char aux;

    // allocate images at the beginning or
    // reallocate them if the frame size is changed
    if( !mhi || mhi->width != size.width || mhi->height != size.height )
    {
        if( buf == 0 )
        {
            buf = (IplImage**)malloc(Nframes*sizeof(buf[0]));
            memset( buf, 0, Nframes*sizeof(buf[0]));
        }

        for( i = 0; i < Nframes; i++ )
        {
            cvReleaseImage( &buf[i] );
            buf[i] = cvCreateImage( size, IPL_DEPTH_8U, 1 );
            cvZero( buf[i] );
        }
        cvReleaseImage( &mhi );
        cvReleaseImage( &mask );

        mhi = cvCreateImage( size, IPL_DEPTH_32F, 1 );
        cvZero( mhi ); // clear MHI at the beginning
        mask = cvCreateImage( size, IPL_DEPTH_8U, 1 );
    }

    cvCvtColor( src, buf[last], CV_BGR2GRAY ); // convert frame to grayscale

    idx2 = (last + 1) % Nframes; // index of (last - (N-1))th frame
    last = idx2;

    silh = buf[idx2];
    cvAbsDiff( buf[idx1], buf[idx2], silh ); // get difference between frames

    cvThreshold( silh, silh, threshold, 1, CV_THRESH_BINARY ); // and threshold it
    cvUpdateMotionHistory( silh, mhi, timestamp, MHI_DURATION ); // update MHI

    cvCvtScale( mhi, mask, 255./MHI_DURATION,
                (MHI_DURATION - timestamp) * 255.0 / MHI_DURATION );

    cvZero( dst_color );
    cvMerge( mask, mask, mask, 0, dst_color ); // compõe a imagem BGRA

    if( !storage )
        storage = cvCreateMemStorage(0);
    else
        cvClearMemStorage(storage);

    // Percorre todos os dados da imagem.
    for (j = 0; j < dst_color->height; j++)
    {
      for (i = 0; i < dst_color->widthStep; i = i + 3)
      {

        aux = dst_color->imageData[i + (j*dst_color->widthStep)];
        dst_gray->imageData[i/3 + (j*dst_gray->widthStep)] = aux;

        if(aux < 64)
        {
            red = 0;
            green = (aux*4);
            blue = 255;
        } else
        {
           if(aux < 128)
            {
                red = 0;
                green = 255;
                blue = (255 - (aux-64)*4);
            }
            else
            {
              if(aux < 192)
              {
                 red = ((aux-128) * 4);
                 green = 255;
                 blue = 0;
              }
              else
              {
                red = 255;
                green = (255 - (aux-192) * 4);
                blue = 0;
              }
            }
        }
        dst_color->imageData[i + 2 + (j * dst_color->widthStep)] = red;
        dst_color->imageData[i + 1 + (j * dst_color->widthStep)] = green;
        dst_color->imageData[i + (j * dst_color->widthStep)] = blue;
      }
    }

}
