/* Arg_parser - POSIX/GNU command-line argument parser. (C++ version)
   Copyright (C) 2006-2025 Antonio Diaz Diaz.

   This program is free software: you have unlimited permission to
   copy, distribute, and modify it.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/
/*
   Exit status: 0 for a normal exit, 1 for environmental problems
   (file not found, invalid command-line options, I/O errors, etc), 2 to
   indicate a corrupt or invalid input file, 3 for an internal consistency
   error (e.g., bug) which caused arg_parser to panic.
*/

#include <climits>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <string>
#include <vector>

#include "arg_parser.h"


namespace {

const char * const program_name = "arg_parser";
const char * const program_year = "2025";
const char * invocation_name = program_name;		// default value


void show_help( const bool verbose )
  {
  std::printf( "Arg_parser - POSIX/GNU command-line argument parser. (C++ version)\n"
               "\nArg_parser is an argument parser that follows POSIX and GNU conventions for\n"
               "command-line arguments. There exist C++ and C versions of Arg_parser. The\n"
               "C++ version is implemented as a C++ class, while the C version is\n"
               "implemented as a single struct plus associated functions. Both are simpler,\n"
               "easier to use, and safer than 'getopt_long'.\n"
               "\nFor maximum stability, Arg_parser is self-contained. It extracts all the\n"
               "information it needs from its arguments to avoid referring to them later.\n"
               "This avoids index-out-of-bounds errors and allows the parser object to be\n"
               "passed as argument to other functions for further analysis.\n"
               "\nArg_parser does not modify its arguments (argc, argv), nor uses any global\n"
               "variables.\n"
               "\nThe C++ version of Arg_parser can also parse options from configuration\n"
               "files.\n"
               "\nThe C++ version of Arg_parser is provided in the files 'arg_parser.h' and\n"
               "'arg_parser.cc'. To learn how to use Arg_parser in your C++ programs, see\n"
               "the C++ example in the manual and the example file 'main.cc' in the source\n"
               "tarball.\n"
               "\nUsage: %s [options]\n", invocation_name );
  std::printf( "\nOptions:\n"
               "  -h, --help                   display this help and exit\n"
               "  -V, --version                output version information and exit\n"
               "  -a, --append                 example of option with no argument\n"
               "  -b, --block=<arg>            example of option with required argument\n"
               "  -c, --casual[=<arg>]         example of option with optional argument\n"
               "  -e, --empty=<arg>            example of option with maybe empty argument\n"
               "  -o <arg>                     example of short only option\n"
               "      --long-only              example of long only option\n"
               "  -q, --quiet                  quiet operation\n"
               "  -u, --uncaught               example of intentional bug\n"
               "  -v, --verbose                verbose operation\n" );
  if( verbose )
    std::printf( "  -H, --hidden                 example of hidden option (shown with -v -h)\n" );
  std::printf( "\nReport bugs to arg-parser-bug@nongnu.org\n"
               "Arg_parser home page: http://www.nongnu.org/arg-parser/arg_parser.html\n" );
  }


void show_version()
  {
  std::printf( "%s %s\n", program_name, PROGVERSION );
  std::printf( "Copyright (C) %s Antonio Diaz Diaz.\n", program_year );
  std::printf( "License 2-clause BSD.\n"
               "This is free software: you are free to change and redistribute it.\n"
               "There is NO WARRANTY, to the extent permitted by law.\n" );
  }


void show_error( const char * const msg, const int errcode = 0,
                 const bool help = false )
  {
  if( msg && msg[0] )
    std::fprintf( stderr, "%s: %s%s%s\n", program_name, msg,
                  ( errcode > 0 ) ? ": " : "",
                  ( errcode > 0 ) ? std::strerror( errcode ) : "" );
  if( help )
    std::fprintf( stderr, "Try '%s --help' for more information.\n",
                  invocation_name );
  }


void internal_error( const char * const msg )
  {
  std::fprintf( stderr, "%s: internal error: %s\n", program_name, msg );
  std::exit( 3 );
  }


void print_opt( const char * const arg, const char * const option_name,
                const int code )
  {
  std::printf( "option '%s'", option_name );
  if( arg[0] || code == 'e' ) std::printf( " with argument '%s'", arg );
  if( code == INT_MIN || code == INT_MAX ) std::printf( " (code 0x%X)", code );
  std::fputc( '\n', stdout );
  }

} // end namespace


int main( const int argc, const char * const argv[] )
  {
  bool verbose = false;
  if( argc > 0 ) invocation_name = argv[0];

  enum { opt_lo = 256 };
  const Arg_parser::Option options[] =
    {
    // code, long_name, has_arg (no/yes/maybe/yme)
    { 'a', "append",       Arg_parser::no  },
    { 'b', "block",        Arg_parser::yes },
    { 'c', "casual",       Arg_parser::maybe },
    { 'e', "empty",        Arg_parser::yme },
    { 'h', "help",         Arg_parser::no  },
    { 'H', "hidden",       Arg_parser::no  },
    { 'o', 0,              Arg_parser::yes },
    { 'q', "quiet",        Arg_parser::no  },
    { 'u', "uncaught",     Arg_parser::no  },
    { 'v', "verbose",      Arg_parser::no  },
    { 'V', "version",      Arg_parser::no  },
    { opt_lo, "long-only", Arg_parser::no  },
    { INT_MIN, "int-min",  Arg_parser::no  },
    { INT_MAX, "int-max",  Arg_parser::no  },
    { 0, 0,                Arg_parser::no  } };

  const Arg_parser parser( argc, argv, options );
  if( parser.error().size() )				// bad option
    { show_error( parser.error().c_str(), 0, true ); return 1; }

  int argind = 0;
  for( ; argind < parser.arguments(); ++argind )
    {
    const int code = parser.code( argind );
    if( !code ) break;				// no more options
    const char * const pn = parser.parsed_name( argind ).c_str();
    const char * const arg = parser.argument( argind ).c_str();
    switch( code )
      {
      case 'a': print_opt( arg, pn, code ); break;
      case 'b': print_opt( arg, pn, code ); break;
      case 'c': print_opt( arg, pn, code ); break;
      case 'e': print_opt( arg, pn, code ); break;
      case 'h': show_help( verbose ); return 0;
      case 'H': print_opt( arg, pn, code ); break;
      case 'o': print_opt( arg, pn, code ); break;
      case 'q': print_opt( arg, pn, code ); verbose = false; break;
      // case 'u': break;			// intentionally not caught
      case 'v': print_opt( arg, pn, code ); verbose = true; break;
      case 'V': show_version(); return 0;
      case opt_lo: print_opt( arg, pn, code ); break;
      case INT_MIN: print_opt( arg, pn, code ); break;
      case INT_MAX: print_opt( arg, pn, code ); break;
      default: internal_error( "uncaught option." );
      }
    } // end process options

  for( ; argind < parser.arguments(); ++argind )
    {
    const char * const arg = parser.argument( argind ).c_str();
    std::printf( "non-option argument '%s'\n", arg );
    }

  if( !parser.arguments() ) std::fputs( "Hello, world!\n", stdout );

  return 0;
  }
