/*
*      visnotes.c
*      
*      Copyright 2010 Robert Schmidt <rschmi@gmx.net>
*      
*      This program is free software; you can redistribute it and/or modify
*      it under the terms of the GNU General Public License as published by
*      the Free Software Foundation; either version 2 of the License, or
*      (at your option) any later version.
*      
*      This program is distributed in the hope that it will be useful,
*      but WITHOUT ANY WARRANTY; without even the implied warranty of
*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*      GNU General Public License for more details.
*      
*      You should have received a copy of the GNU General Public License
*      along with this program; if not, write to the Free Software
*      Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
*      MA 02110-1301, USA.
*/


#define  PURPLE_PLUGINS

#include <glib.h>
#include <stdlib.h>
#include <time.h>
#include <dbus/dbus-glib.h>

#include "VisualNotifications.h"
#include "Notifications.h"

#include "connection.h"
#include "conversation.h"
#include "debug.h"
#include "plugin.h"
#include "version.h"

#define VISNOTE_PLUGIN_ID "core-schmidt-visnotes"
#define VISNOTE_PLUGIN_NAME "visnotes -- Pidgin-Notifications"
#define VISNOTE_PLUGIN_VERSION "0.2"
#define VISNOTE_PLUGIN_SUMMARY "Notifications for pidgin"
#define VISNOTE_PLUGIN_DESC "Informs the user about status changes and new messages through Notifications."
#define VISNOTE_PLUGIN_AUTHOR "Robert R. Schmidt <rschmi@gmx.net>"

#define SUMMARY_MESSAGE 0
#define SUMMARY_STATE_ON 1
#define SUMMARY_STATE_OFF 2


static GList *just_signed_on_accounts = NULL;
static GError *error;
static DBusGProxy *proxy; 
static gint timeout;

static gchar *best_alias(PurpleBuddy *buddy);

gboolean (*call_notify)(DBusGProxy *proxy, const gchar *IN_app_name, const guint IN_replaces_id, 
			const gchar *IN_app_icon, const gchar *IN_summary,
			const gchar *IN_body, const gchar **IN_actions, const GHashTable* IN_hints, 
			const gint IN_timeout, guint *OUT_arg0, GError **error) = NULL;
		
gboolean call_notify_kde(DBusGProxy *proxy, const gchar *IN_app_name, const guint IN_replaces_id, 
			const gchar *IN_app_icon, const gchar *IN_summary,
			const gchar *IN_body, const gchar **IN_actions, const GHashTable* IN_hints, 
			const gint IN_timeout, guint *OUT_arg0, GError **error)
{
	return org_kde_VisualNotifications_notify(proxy, IN_app_name, IN_replaces_id, "", IN_app_icon,
			IN_summary, IN_body, IN_actions, IN_hints, IN_timeout, OUT_arg0, error);
}

static void
notify(gint summary, PurpleBuddy *buddy, gchar *message)
{
	guint result = 0;
	gint note_id = 0;
	const gchar **actions = {NULL};
	gchar *sum, *body, *event_id, *stime;
	size_t len, tlen;
	GArray *arr = NULL;
	GHashTable *hash = g_hash_table_new (NULL, NULL);
	time_t now = time(NULL);
	struct tm tim = *(localtime(&now));
	GValue val = {0, };
	
	gconstpointer data = purple_buddy_icon_get_data(purple_buddy_get_icon(buddy), &len);
	
	stime = g_malloc(9);	
	strftime(stime, 9, "%H:%M:%S", &tim);
	
	if (data) {
		purple_debug_info(VISNOTE_PLUGIN_ID, "%s (%s), use icon, len = %d\n", best_alias(buddy), buddy->name, (guint)len);
		arr = g_array_new(FALSE, FALSE, sizeof(guchar));
		g_array_append_vals(arr, data, (guint)len);
		g_value_init(&val, DBUS_TYPE_G_UCHAR_ARRAY);
		g_value_take_boxed(&val, arr);
		g_hash_table_insert(hash, "image_data", &val);
	} else {		
		purple_debug_info(VISNOTE_PLUGIN_ID, "%s (%s), no icon\n", best_alias(buddy), buddy->name);
	}
	

	switch (summary) {
		case SUMMARY_MESSAGE:
			sum = g_strdup_printf("User wrote message (%s)", stime);
			body = g_strdup_printf("%s: <b>%s</b>\n<i>[%s]</i>", best_alias(buddy), message, buddy->name);
			purple_debug_info(VISNOTE_PLUGIN_ID, "message: '%s'\n", message);
			break;
		case SUMMARY_STATE_ON:
			sum = g_strdup_printf("User state changed (%s)", stime);
			body = g_strdup_printf("<b>%s</b> signed <b>on</b>.\n<i>[%s]</i>", best_alias(buddy), buddy->name);
			break;
		case SUMMARY_STATE_OFF:
			sum = g_strdup_printf("User state changed (%s)", stime);
			body = g_strdup_printf("<b>%s</b> signed <b>off</b>.\n<i>[%s]</i>", best_alias(buddy), buddy->name);
			break;
		default:
			purple_debug_error(VISNOTE_PLUGIN_ID, "illegal summary type (%d)\n", summary);
			return;
	}
	
	error = NULL;
	
	purple_debug_info(VISNOTE_PLUGIN_ID, "notify()\n");
	if (!(call_notify(proxy,
		"Pidgin",			// IN_app_name
		note_id,			// IN_replaces_id
		"pidgin", 			// IN_app_icon
		sum,				// IN_summary
		body,				// IN_body
		actions,			// IN_actions
		hash,				// IN_hints
		timeout,			// IN_timeout (msec!!)
		&result,
		&error)))
	{
		if (error->domain == DBUS_GERROR && error->code == DBUS_GERROR_REMOTE_EXCEPTION)
			purple_debug_error(VISNOTE_PLUGIN_ID, "Caught remote method exception %s: %s\n", dbus_g_error_get_name(error), error->message);
		else
			purple_debug_error(VISNOTE_PLUGIN_ID, "GError: %s\n", error->message);
	}
	else {
		purple_debug_info(VISNOTE_PLUGIN_ID, "result = %d\n", result);		
	}
	
	
	if (data)		
		g_array_free(arr, TRUE);
	g_hash_table_destroy(hash);
	g_free(actions);
	g_free(sum);
	g_free(body);
	g_free(event_id);
	g_free(stime);
}

static gchar *
best_alias(PurpleBuddy *buddy)
{
	if (buddy->alias != NULL)
		return buddy->alias;
	else if (buddy->server_alias != NULL)
		return buddy->server_alias;
	else
		return buddy->name;
}

static PurpleBuddy *
get_buddy(PurpleAccount *account, char *sender)
{
	PurpleBuddy *buddy = purple_find_buddy(account, sender);
	return buddy;
}

static gboolean
event_connection_throttle_cb(gpointer data)
{
	PurpleAccount *account;
	
	account = (PurpleAccount *) data;
	
	if (!account)
		return FALSE;
	
	if (!purple_account_get_connection(account)) {
		just_signed_on_accounts = g_list_remove(just_signed_on_accounts, account);
		return FALSE;
	}
	
	if (!purple_account_is_connected(account))
		return TRUE;
	
	just_signed_on_accounts = g_list_remove(just_signed_on_accounts, account);
	return FALSE;
}

static void
event_connection_throttle(PurpleConnection *conn, gpointer data)
{
	PurpleAccount *account;
	
	/* TODO: this function gets called after buddy signs on for GTalk
	users who have themselves as a buddy */
	purple_debug_info(VISNOTE_PLUGIN_ID, "event_connection_throttle() called\n");
	
	if (!conn)
		return;
	
	account = purple_connection_get_account(conn);
	if (!account)
		return;
	
	just_signed_on_accounts = g_list_prepend(just_signed_on_accounts, account);
	g_timeout_add_seconds(1 , event_connection_throttle_cb, (gpointer) account);
}

static void
buddy_signed_on(PurpleBuddy *buddy, void *data)
{	
	if (g_list_find(just_signed_on_accounts, buddy->account))
		return;
	
	notify(SUMMARY_STATE_ON, buddy, "");
}

static void
buddy_signed_off(PurpleBuddy *buddy, void *data)
{
	notify(SUMMARY_STATE_OFF, buddy, "");
}

static void
received_im_msg(PurpleAccount *account, char *sender, char *buffer,
				PurpleConversation *conv, PurpleMessageFlags flags, void *data)
{
	notify(SUMMARY_MESSAGE, get_buddy(account, sender), buffer);
}

static void
received_chat_msg(PurpleAccount *account, char *sender, char *buffer,
				  PurpleConversation *conv, PurpleMessageFlags flags, void *data)
{
	notify(SUMMARY_MESSAGE, get_buddy(account, sender), buffer);
}


static gboolean
plugin_load(PurplePlugin *plugin)
{
	void *blist_handle = purple_blist_get_handle();
	void *conn_handle = purple_connections_get_handle();
	void *conv_handle = purple_conversations_get_handle();
	
	DBusGConnection *connection;
	
	purple_signal_connect(blist_handle, "buddy-signed-on",
						  plugin, PURPLE_CALLBACK(buddy_signed_on), NULL);
	purple_signal_connect(blist_handle, "buddy-signed-off",
						  plugin, PURPLE_CALLBACK(buddy_signed_off), NULL);
	purple_signal_connect(conn_handle, "signed-on",
						  plugin, PURPLE_CALLBACK(event_connection_throttle), NULL);
	purple_signal_connect(conv_handle, "received-im-msg",
						  plugin, PURPLE_CALLBACK(received_im_msg), NULL);
	purple_signal_connect(conv_handle, "received-chat-msg",
						  plugin, PURPLE_CALLBACK(received_chat_msg), NULL);
						  
	g_type_init();
	error = NULL;
	if (!(connection = dbus_g_bus_get(DBUS_BUS_SESSION, &error))) {
		purple_debug_error(VISNOTE_PLUGIN_ID, 
				"Failed to open connection to bus: %s\n", error->message);
		g_error_free(error);
		return FALSE;
	}
	
	if (!(proxy = dbus_g_proxy_new_for_name_owner(connection, // kde <= 4.3 ?
		"org.kde.VisualNotifications",  	// name/service
		"/VisualNotifications", 			// path
		"org.kde.VisualNotifications",		// interface
		&error)))
	{
		purple_debug_error(VISNOTE_PLUGIN_ID, "Failed to connect to org.kde.VisualNotifications\n"); // kde 4.4
		purple_debug_error(VISNOTE_PLUGIN_ID, "%s\n", error->message);
		if (!(proxy = dbus_g_proxy_new_for_name_owner(connection,
			"org.freedesktop.Notifications",
			"/org/freedesktop/Notifications",
			"org.freedesktop.Notifications",
			&error)))
		{	
			purple_debug_error(VISNOTE_PLUGIN_ID, "Failed to connect to org.freedesktop.Notifications\n");
			purple_debug_error(VISNOTE_PLUGIN_ID, "%s\n", error->message);			
			g_error_free(error);
			return FALSE;
		} else {
			call_notify = &org_freedesktop_Notifications_notify;
			purple_debug_info(VISNOTE_PLUGIN_ID, "connected to org.freedesktop.Notifications\n");
		}
	} else {
		call_notify = &call_notify_kde;
		purple_debug_info(VISNOTE_PLUGIN_ID, "connected to org.kde.VisualNotifications\n");
	}

	timeout = 7000;
	
    return TRUE;
}

static gboolean
plugin_unload(PurplePlugin *plugin)
{
	// equal for disconnecting all signals separately
	purple_signals_disconnect_by_handle(plugin);
	
	if (error != NULL)
		g_error_free(error);
		
	return TRUE;
}


static PurplePluginInfo info =
{
	PURPLE_PLUGIN_MAGIC,
	PURPLE_MAJOR_VERSION,
	PURPLE_MINOR_VERSION,
	PURPLE_PLUGIN_STANDARD,	
	NULL,
	0,
	NULL,
	PURPLE_PRIORITY_DEFAULT,
	
	VISNOTE_PLUGIN_ID,
	VISNOTE_PLUGIN_NAME,
	VISNOTE_PLUGIN_VERSION,
	VISNOTE_PLUGIN_SUMMARY,
	VISNOTE_PLUGIN_DESC,
	VISNOTE_PLUGIN_AUTHOR,
	NULL,
	
	plugin_load,
	plugin_unload,
	NULL,
	
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static void
init_plugin(PurplePlugin *plugin)
{
	
}


PURPLE_INIT_PLUGIN(visnotes, init_plugin, info);
