/**
 * Copyright (c) 2010-2016, Abel Hegedus, IncQuery Labs Ltd.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.viatra.query.tooling.ui.queryregistry.index;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.inject.Inject;
import java.util.Collection;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.viatra.query.patternlanguage.emf.specification.SpecificationBuilder;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Pattern;
import org.eclipse.viatra.query.patternlanguage.emf.vql.PatternLanguagePackage;
import org.eclipse.viatra.query.runtime.api.IPatternMatch;
import org.eclipse.viatra.query.runtime.api.IQuerySpecification;
import org.eclipse.viatra.query.runtime.api.ViatraQueryMatcher;
import org.eclipse.viatra.query.runtime.extensibility.IQuerySpecificationProvider;
import org.eclipse.viatra.query.runtime.registry.IConnectorListener;
import org.eclipse.viatra.query.runtime.registry.IQuerySpecificationRegistry;
import org.eclipse.viatra.query.runtime.registry.connector.AbstractRegistrySourceConnector;
import org.eclipse.viatra.query.runtime.util.ViatraQueryLoggingUtil;
import org.eclipse.xtend.lib.annotations.AccessorType;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtend.lib.annotations.FinalFieldsConstructor;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.resource.IResourceDescription;
import org.eclipse.xtext.resource.IResourceDescriptions;
import org.eclipse.xtext.ui.notification.IStateChangeEventBroker;
import org.eclipse.xtext.ui.resource.IResourceSetProvider;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * @author Abel Hegedus
 */
@SuppressWarnings("all")
public class XtextIndexBasedRegistryUpdater {
  @FinalFieldsConstructor
  private static final class QueryRegistryUpdaterListener implements IResourceDescription.Event.Listener {
    private final XtextIndexBasedRegistryUpdater updater;

    @Override
    public void descriptionsChanged(final IResourceDescription.Event event) {
      final Consumer<IResourceDescription.Delta> _function = (IResourceDescription.Delta delta) -> {
        final IResourceDescription oldDesc = delta.getOld();
        final IResourceDescription newDesc = delta.getNew();
        final String uri = delta.getUri().toString();
        boolean _isPlatformResource = delta.getUri().isPlatformResource();
        boolean _not = (!_isPlatformResource);
        if (_not) {
          return;
        }
        final String projectName = delta.getUri().segment(1);
        final IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
        final boolean projectExists = project.exists();
        if ((!projectExists)) {
          return;
        }
        final String connectorId = (XtextIndexBasedRegistryUpdater.DYNAMIC_CONNECTOR_ID_PREFIX + projectName);
        try {
          if ((oldDesc != null)) {
            if (((newDesc == null) || (!project.isOpen()))) {
              final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector connector = this.updater.connectorMap.get(connectorId);
              if ((connector != null)) {
                connector.clearProviders(uri);
                boolean _isEmpty = connector.descriptionToProvider.isEmpty();
                if (_isEmpty) {
                  this.updater.connectedRegistry.removeSource(connector);
                  this.updater.connectorMap.remove(connectorId);
                }
              }
            } else {
              this.processResourceDescription(delta, newDesc, connectorId, projectName);
            }
          } else {
            if (((newDesc != null) && (!IterableExtensions.isEmpty(newDesc.getExportedObjectsByType(PatternLanguagePackage.Literals.PATTERN))))) {
              this.processResourceDescription(delta, newDesc, connectorId, projectName);
            }
          }
        } catch (final Throwable _t) {
          if (_t instanceof Exception) {
            final Exception ex = (Exception)_t;
            final Logger logger = ViatraQueryLoggingUtil.getLogger(XtextIndexBasedRegistryUpdater.class);
            StringConcatenation _builder = new StringConcatenation();
            _builder.append("Could not update registry based on Xtext index for ");
            _builder.append(uri);
            logger.error(_builder, ex);
          } else {
            throw Exceptions.sneakyThrow(_t);
          }
        }
      };
      event.getDeltas().forEach(_function);
    }

    public Object processResourceDescription(final IResourceDescription.Delta delta, final IResourceDescription desc, final String connectorId, final String projectName) {
      Object _xifexpression = null;
      boolean _containsKey = this.updater.connectorMap.containsKey(connectorId);
      if (_containsKey) {
        _xifexpression = this.updater.workspaceListener.connectorsToUpdate.put(desc.getURI(), desc);
      } else {
        boolean _xifexpression_1 = false;
        boolean _isEmpty = IterableExtensions.isEmpty(desc.getExportedObjectsByType(PatternLanguagePackage.Literals.PATTERN));
        boolean _not = (!_isEmpty);
        if (_not) {
          boolean _xblockexpression = false;
          {
            final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector connector = new XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector(connectorId);
            this.updater.connectorMap.put(connectorId, connector);
            boolean _haveEObjectDescriptionsChanged = delta.haveEObjectDescriptionsChanged();
            if (_haveEObjectDescriptionsChanged) {
              final ResourceSet resourceSet = this.updater.createResourceSet(projectName);
              final Consumer<IEObjectDescription> _function = (IEObjectDescription it) -> {
                final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSpecificationProvider provider = new XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSpecificationProvider(desc, it, resourceSet);
                connector.addProvider(desc.getURI().toString(), provider);
              };
              desc.getExportedObjectsByType(PatternLanguagePackage.Literals.PATTERN).forEach(_function);
            }
            _xblockexpression = this.updater.connectedRegistry.addSource(connector);
          }
          _xifexpression_1 = _xblockexpression;
        }
        _xifexpression = Boolean.valueOf(_xifexpression_1);
      }
      return _xifexpression;
    }

    public QueryRegistryUpdaterListener(final XtextIndexBasedRegistryUpdater updater) {
      super();
      this.updater = updater;
    }
  }

  @FinalFieldsConstructor
  private static final class PatternDescriptionBasedSpecificationProvider implements IPatternBasedSpecificationProvider {
    private final IResourceDescription resourceDesc;

    private final IEObjectDescription description;

    private final ResourceSet resourceSet;

    private IQuerySpecification<?> specification;

    @Override
    public String getFullyQualifiedName() {
      return this.description.getQualifiedName().toString();
    }

    @Override
    public IQuerySpecification<?> get() {
      if ((this.specification == null)) {
        final Pattern pattern = this.findPatternForDescription();
        final SpecificationBuilder builder = new SpecificationBuilder();
        this.specification = builder.getOrCreateSpecification(pattern);
      }
      return this.specification;
    }

    @Override
    public IQuerySpecification<?> getSpecification(final SpecificationBuilder builder) {
      final Pattern pattern = this.findPatternForDescription();
      final IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>> spec = builder.getOrCreateSpecification(pattern);
      return spec;
    }

    public Pattern findPatternForDescription() {
      EObject pattern = this.description.getEObjectOrProxy();
      boolean _eIsProxy = pattern.eIsProxy();
      if (_eIsProxy) {
        pattern = EcoreUtil.resolve(pattern, this.resourceSet);
      }
      boolean _eIsProxy_1 = pattern.eIsProxy();
      if (_eIsProxy_1) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Cannot load specification ");
        String _fullyQualifiedName = this.getFullyQualifiedName();
        _builder.append(_fullyQualifiedName);
        _builder.append(" from Xtext index");
        throw new IllegalStateException(_builder.toString());
      }
      return ((Pattern) pattern);
    }

    @Override
    public String getSourceProjectName() {
      return this.resourceDesc.getURI().segment(1);
    }

    @Override
    public URI getSpecificationSourceURI() {
      return this.description.getEObjectURI();
    }

    public PatternDescriptionBasedSpecificationProvider(final IResourceDescription resourceDesc, final IEObjectDescription description, final ResourceSet resourceSet) {
      super();
      this.resourceDesc = resourceDesc;
      this.description = description;
      this.resourceSet = resourceSet;
    }
  }

  private static final class PatternDescriptionBasedSourceConnector extends AbstractRegistrySourceConnector {
    private final Multimap<String, IQuerySpecificationProvider> descriptionToProvider;

    public PatternDescriptionBasedSourceConnector(final String identifier) {
      super(identifier, false);
      this.descriptionToProvider = HashMultimap.<String, IQuerySpecificationProvider>create();
    }

    public void addProvider(final String resourceUri, final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSpecificationProvider provider) {
      this.descriptionToProvider.put(resourceUri, provider);
      final Consumer<IConnectorListener> _function = (IConnectorListener it) -> {
        it.querySpecificationAdded(this, provider);
      };
      this.listeners.forEach(_function);
    }

    public Collection<IQuerySpecificationProvider> clearProviders(final String resourceUri) {
      Collection<IQuerySpecificationProvider> _xblockexpression = null;
      {
        final Consumer<IQuerySpecificationProvider> _function = (IQuerySpecificationProvider provider) -> {
          final Consumer<IConnectorListener> _function_1 = (IConnectorListener it) -> {
            it.querySpecificationRemoved(this, provider);
          };
          this.listeners.forEach(_function_1);
        };
        this.descriptionToProvider.get(resourceUri).forEach(_function);
        _xblockexpression = this.descriptionToProvider.removeAll(resourceUri);
      }
      return _xblockexpression;
    }

    @Override
    protected void sendQuerySpecificationsToListener(final IConnectorListener listener) {
      final Consumer<IQuerySpecificationProvider> _function = (IQuerySpecificationProvider it) -> {
        listener.querySpecificationAdded(this, it);
      };
      this.descriptionToProvider.values().forEach(_function);
    }
  }

  @FinalFieldsConstructor
  private static final class WorkspaceBuildCompletedListener implements IResourceChangeListener {
    private final Map<URI, IResourceDescription> connectorsToUpdate = CollectionLiterals.<URI, IResourceDescription>newHashMap();

    private final XtextIndexBasedRegistryUpdater updater;

    @Override
    public void resourceChanged(final IResourceChangeEvent event) {
      final int type = event.getType();
      if ((type == IResourceChangeEvent.POST_CHANGE)) {
        boolean _isEmpty = this.connectorsToUpdate.isEmpty();
        if (_isEmpty) {
          return;
        }
        final ImmutableMap<URI, IResourceDescription> update = ImmutableMap.<URI, IResourceDescription>copyOf(this.connectorsToUpdate);
        final BiConsumer<URI, IResourceDescription> _function = (URI uri, IResourceDescription descr) -> {
          try {
            this.connectorsToUpdate.remove(uri);
            final String projectName = uri.segment(1);
            final String connectorId = (XtextIndexBasedRegistryUpdater.DYNAMIC_CONNECTOR_ID_PREFIX + projectName);
            final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector connector = this.updater.connectorMap.get(connectorId);
            connector.clearProviders(uri.toString());
            final ResourceSet resourceSet = this.updater.createResourceSet(projectName);
            final Iterable<IEObjectDescription> patternObjects = descr.getExportedObjectsByType(PatternLanguagePackage.Literals.PATTERN);
            final Consumer<IEObjectDescription> _function_1 = (IEObjectDescription it) -> {
              final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSpecificationProvider provider = new XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSpecificationProvider(descr, it, resourceSet);
              connector.addProvider(uri.toString(), provider);
            };
            patternObjects.forEach(_function_1);
          } catch (final Throwable _t) {
            if (_t instanceof Exception) {
              final Exception ex = (Exception)_t;
              final Logger logger = ViatraQueryLoggingUtil.getLogger(XtextIndexBasedRegistryUpdater.class);
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("Could not update registry based on Xtext index for ");
              _builder.append(uri);
              logger.error(_builder, ex);
            } else {
              throw Exceptions.sneakyThrow(_t);
            }
          }
        };
        update.forEach(_function);
      }
    }

    public WorkspaceBuildCompletedListener(final XtextIndexBasedRegistryUpdater updater) {
      super();
      this.updater = updater;
    }
  }

  public static final String DYNAMIC_CONNECTOR_ID_PREFIX = "dynamic:";

  private final IStateChangeEventBroker source;

  private final IResourceDescriptions descriptions;

  private final IResourceSetProvider resourceSetProvider;

  private final XtextIndexBasedRegistryUpdater.QueryRegistryUpdaterListener listener;

  private final Map<String, XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector> connectorMap;

  private final XtextIndexBasedRegistryUpdater.WorkspaceBuildCompletedListener workspaceListener;

  @Accessors(AccessorType.PROTECTED_GETTER)
  private IQuerySpecificationRegistry connectedRegistry;

  @Inject
  public XtextIndexBasedRegistryUpdater(final IStateChangeEventBroker source, final IResourceDescriptions descriptions, final IResourceSetProvider resSetProvider) {
    super();
    this.source = source;
    this.descriptions = descriptions;
    this.resourceSetProvider = resSetProvider;
    XtextIndexBasedRegistryUpdater.WorkspaceBuildCompletedListener _workspaceBuildCompletedListener = new XtextIndexBasedRegistryUpdater.WorkspaceBuildCompletedListener(this);
    this.workspaceListener = _workspaceBuildCompletedListener;
    XtextIndexBasedRegistryUpdater.QueryRegistryUpdaterListener _queryRegistryUpdaterListener = new XtextIndexBasedRegistryUpdater.QueryRegistryUpdaterListener(this);
    this.listener = _queryRegistryUpdaterListener;
    this.connectorMap = Maps.<String, XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector>newTreeMap();
  }

  public void connectIndexToRegistry(final IQuerySpecificationRegistry registry) {
    if ((this.connectedRegistry == null)) {
      this.connectedRegistry = registry;
      final Consumer<IResourceDescription> _function = (IResourceDescription resourceDesc) -> {
        boolean _isPlatformResource = resourceDesc.getURI().isPlatformResource();
        boolean _not = (!_isPlatformResource);
        if (_not) {
          return;
        }
        final Iterable<IEObjectDescription> patternObjects = resourceDesc.getExportedObjectsByType(PatternLanguagePackage.Literals.PATTERN);
        boolean _isEmpty = IterableExtensions.isEmpty(patternObjects);
        if (_isEmpty) {
          return;
        }
        final String uri = resourceDesc.getURI().toString();
        final String projectName = resourceDesc.getURI().segment(1);
        final IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
        final boolean projectExists = project.exists();
        if (((!projectExists) || (!project.isOpen()))) {
          return;
        }
        final String connectorId = (XtextIndexBasedRegistryUpdater.DYNAMIC_CONNECTOR_ID_PREFIX + projectName);
        XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector connector = this.connectorMap.get(connectorId);
        if ((connector == null)) {
          XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector _patternDescriptionBasedSourceConnector = new XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector(connectorId);
          connector = _patternDescriptionBasedSourceConnector;
          this.connectorMap.put(connectorId, connector);
        }
        final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector conn = connector;
        final ResourceSet resourceSet = this.createResourceSet(projectName);
        final Consumer<IEObjectDescription> _function_1 = (IEObjectDescription it) -> {
          final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSpecificationProvider provider = new XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSpecificationProvider(resourceDesc, it, resourceSet);
          conn.addProvider(uri, provider);
        };
        patternObjects.forEach(_function_1);
      };
      this.descriptions.getAllResourceDescriptions().forEach(_function);
      final Consumer<XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector> _function_1 = (XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector connector) -> {
        registry.addSource(connector);
      };
      this.connectorMap.values().forEach(_function_1);
      this.source.addListener(this.listener);
      ResourcesPlugin.getWorkspace().addResourceChangeListener(this.workspaceListener);
    }
  }

  public IQuerySpecificationRegistry disconnectIndexFromRegistry() {
    IQuerySpecificationRegistry _xifexpression = null;
    if ((this.connectedRegistry != null)) {
      IQuerySpecificationRegistry _xblockexpression = null;
      {
        final Consumer<XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector> _function = (XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector it) -> {
          this.connectedRegistry.removeSource(it);
        };
        this.connectorMap.values().forEach(_function);
        this.source.removeListener(this.listener);
        this.connectorMap.clear();
        _xblockexpression = this.connectedRegistry = null;
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }

  public ResourceSet createResourceSet(final String projectName) {
    final IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
    final IProject project = root.getProject(projectName);
    final ResourceSet resourceSet = this.resourceSetProvider.get(project);
    return resourceSet;
  }

  @Pure
  protected IQuerySpecificationRegistry getConnectedRegistry() {
    return this.connectedRegistry;
  }
}
