"use strict";
/*********************************************************************
 * Copyright (c) 2023 Kichwa Coders Canada Inc. and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const GDBBackend_1 = require("../GDBBackend");
const MIParser_1 = require("../MIParser");
const sinon = require("sinon");
const logger_1 = require("@vscode/debugadapter/lib/logger");
describe('MI Parser Test Suite', function () {
    let gdbBackendMock;
    let loggerErrorSpy;
    let parser;
    beforeEach(function () {
        return __awaiter(this, void 0, void 0, function* () {
            gdbBackendMock = sinon.createStubInstance(GDBBackend_1.GDBBackend);
            loggerErrorSpy = sinon.spy(logger_1.logger, 'error');
            parser = new MIParser_1.MIParser(gdbBackendMock);
        });
    });
    afterEach(function () {
        try {
            sinon.assert.notCalled(loggerErrorSpy);
        }
        finally {
            sinon.restore();
        }
    });
    it('simple result-record', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const callback = sinon.spy();
            parser.queueCommand(5, callback);
            parser.parseLine('5^done');
            sinon.assert.calledOnceWithExactly(callback, 'done', {});
        });
    });
    it('simple result-record with multi-digit token', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const callback = sinon.spy();
            parser.queueCommand(1234, callback);
            parser.parseLine('1234^done');
            sinon.assert.calledOnceWithExactly(callback, 'done', {});
        });
    });
    it('simple result-record for unknown token number', function () {
        return __awaiter(this, void 0, void 0, function* () {
            parser.parseLine('5^done');
            sinon.assert.calledOnceWithExactly(loggerErrorSpy, 'GDB response with no command: 5');
            loggerErrorSpy.resetHistory();
        });
    });
    it('simple result-record for no token number', function () {
        return __awaiter(this, void 0, void 0, function* () {
            parser.parseLine('^done');
            sinon.assert.calledOnceWithExactly(loggerErrorSpy, 'GDB response with no command: ');
            loggerErrorSpy.resetHistory();
        });
    });
    it('simple console-stream-output', function () {
        return __awaiter(this, void 0, void 0, function* () {
            parser.parseLine('~"message"');
            sinon.assert.calledOnceWithExactly(gdbBackendMock.emit, 'consoleStreamOutput', 'message', 'stdout');
        });
    });
    it('simple target-stream-output', function () {
        return __awaiter(this, void 0, void 0, function* () {
            parser.parseLine('@"message"');
            sinon.assert.calledOnceWithExactly(gdbBackendMock.emit, 'consoleStreamOutput', 'message', 'stdout');
        });
    });
    it('simple log-stream-output', function () {
        return __awaiter(this, void 0, void 0, function* () {
            parser.parseLine('&"message"');
            sinon.assert.calledOnceWithExactly(gdbBackendMock.emit, 'consoleStreamOutput', 'message', 'log');
        });
    });
    it('simple notify-async-output', function () {
        return __awaiter(this, void 0, void 0, function* () {
            parser.parseLine('=message,object={value="1234"}');
            sinon.assert.calledOnceWithExactly(gdbBackendMock.emit, 'notifyAsync', 'message', {
                object: {
                    value: '1234',
                },
            });
        });
    });
    it('simple exec-async-output', function () {
        return __awaiter(this, void 0, void 0, function* () {
            parser.parseLine('*message,object={value="1234"}');
            sinon.assert.calledOnceWithExactly(gdbBackendMock.emit, 'execAsync', 'message', {
                object: {
                    value: '1234',
                },
            });
        });
    });
    it('simple status-async-output', function () {
        return __awaiter(this, void 0, void 0, function* () {
            parser.parseLine('+message,object={value="1234"}');
            sinon.assert.calledOnceWithExactly(gdbBackendMock.emit, 'statusAsync', 'message', {
                object: {
                    value: '1234',
                },
            });
        });
    });
    it('simple non-MI output', function () {
        return __awaiter(this, void 0, void 0, function* () {
            // this is when the output line doesn't match any of
            // expected output syntax so we just log it back to the
            // user. This can happen when the inferior's stdout
            // is the same as gdb's stdout.
            parser.parseLine('other');
            sinon.assert.calledOnceWithExactly(gdbBackendMock.emit, 'consoleStreamOutput', 
            // XXX: This tests for how this code has always been implemented,
            // but it isn't particularly useful to do this. Fixing it is low
            // priority because users should avoid having inferior stdout
            // being on the MI stdout as it leads to parsing errors
            'other\n', 'stdout');
        });
    });
    it('structure that starts with a curly bracket and contains values but not keys', function () {
        return __awaiter(this, void 0, void 0, function* () {
            parser.parseLine('+message,bkpt={number="1",type="breakpoint",thread-groups=["i1"],script={"p }123","p 321","p 789"}}');
            sinon.assert.calledOnceWithExactly(gdbBackendMock.emit, 'statusAsync', 'message', {
                bkpt: {
                    number: '1',
                    type: 'breakpoint',
                    'thread-groups': ['i1'],
                    script: { '0': 'p }123', '1': 'p 321', '2': 'p 789' },
                },
            });
        });
    });
});
//# sourceMappingURL=miparser.spec.js.map