/*******************************************************************************
 * Copyright (c) 2000, 2017 IBM Corporation and others.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 *
 *******************************************************************************/
package org.eclipse.dltk.debug.ui.actions;

import org.eclipse.debug.ui.IDebugView;
import org.eclipse.dltk.debug.ui.DLTKDebugUIPlugin;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.widgets.Event;
import org.eclipse.ui.IActionDelegate2;
import org.eclipse.ui.IViewActionDelegate;
import org.eclipse.ui.IViewPart;

public abstract class ViewFilterAction extends ViewerFilter
		implements IViewActionDelegate, IActionDelegate2 {

	private IViewPart fView;
	private IAction fAction;
	private IPropertyChangeListener fListener = new Updater();

	class Updater implements IPropertyChangeListener {

		@Override
		public void propertyChange(PropertyChangeEvent event) {
			if (event.getProperty().equals(getPreferenceKey())
					|| event.getProperty().equals(getCompositeKey())) {
				fAction.setChecked(getPreferenceValue());
			}
		}
	}

	public ViewFilterAction() {
		super();
	}

	@Override
	public void init(IViewPart view) {
		fView = view;
		fAction.setChecked(getPreferenceValue());
		run(fAction);
		getPreferenceStore().addPropertyChangeListener(fListener);
	}

	@Override
	public void init(IAction action) {
		fAction = action;
	}

	@Override
	public void dispose() {
		getPreferenceStore().removePropertyChangeListener(fListener);
	}

	@Override
	public void runWithEvent(IAction action, Event event) {
		run(action);
	}

	@Override
	public void run(IAction action) {
		StructuredViewer viewer = getStructuredViewer();
		ViewerFilter[] filters = viewer.getFilters();
		ViewerFilter filter = null;
		for (int i = 0; i < filters.length; i++) {
			if (filters[i] == this) {
				filter = filters[i];
				break;
			}
		}
		boolean checked = action.isChecked();
		if (filter == null && !checked) {
			viewer.addFilter(this);
			viewer.refresh(false);

			viewer.setInput(viewer.getInput());

		} else if (checked && filter != null) {
			// only refresh is removing - adding will refresh automatically
			viewer.removeFilter(this);
			viewer.refresh(false);
			viewer.setInput(viewer.getInput());
		}
		IPreferenceStore store = getPreferenceStore();
		String key = getView().getSite().getId() + "." + getPreferenceKey(); //$NON-NLS-1$
		store.setValue(key, checked);
	}

	@Override
	public void selectionChanged(IAction action, ISelection selection) {
	}

	protected IPreferenceStore getPreferenceStore() {
		return DLTKDebugUIPlugin.getDefault().getPreferenceStore();
	}

	/**
	 * Returns the value of this filters preference (on/off) for the given view.
	 *
	 * @param part
	 * @return boolean
	 */
	protected boolean getPreferenceValue() {
		String key = getCompositeKey();
		IPreferenceStore store = getPreferenceStore();
		boolean value = false;
		if (store.contains(key)) {
			value = store.getBoolean(key);
		} else {
			value = store.getBoolean(getPreferenceKey());
		}
		return value;
	}

	/**
	 * Returns the key for this action's preference
	 *
	 * @return String
	 */
	protected abstract String getPreferenceKey();

	/**
	 * Returns the key used by this action to store its preference
	 * value/setting. Based on a base key (suffix) and part id (prefix).
	 *
	 * @return preference store key
	 */
	protected String getCompositeKey() {
		String baseKey = getPreferenceKey();
		String viewKey = getView().getSite().getId();
		return viewKey + "." + baseKey; //$NON-NLS-1$
	}

	protected IViewPart getView() {
		return fView;
	}

	protected StructuredViewer getStructuredViewer() {
		IDebugView view = getView().getAdapter(IDebugView.class);
		if (view != null) {
			Viewer viewer = view.getViewer();
			if (viewer instanceof StructuredViewer) {
				return (StructuredViewer) viewer;
			}
		}
		return null;
	}

	/**
	 * Returns whether this action is selected/checked.
	 *
	 * @return whether this action is selected/checked
	 */
	protected boolean getValue() {
		return fAction.isChecked();
	}
}
