package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Version}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableVersion.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableVersion implements Version {
  private final String apiVersion;
  private final String arch;
  private final @Nullable String buildTime;
  private final String gitCommit;
  private final String goVersion;
  private final String kernelVersion;
  private final String os;
  private final String version;

  private ImmutableVersion(
      String apiVersion,
      String arch,
      @Nullable String buildTime,
      String gitCommit,
      String goVersion,
      String kernelVersion,
      String os,
      String version) {
    this.apiVersion = apiVersion;
    this.arch = arch;
    this.buildTime = buildTime;
    this.gitCommit = gitCommit;
    this.goVersion = goVersion;
    this.kernelVersion = kernelVersion;
    this.os = os;
    this.version = version;
  }

  /**
   * @return The value of the {@code apiVersion} attribute
   */
  @JsonProperty("ApiVersion")
  @Override
  public String apiVersion() {
    return apiVersion;
  }

  /**
   * @return The value of the {@code arch} attribute
   */
  @JsonProperty("Arch")
  @Override
  public String arch() {
    return arch;
  }

  /**
   * @return The value of the {@code buildTime} attribute
   */
  @JsonProperty("BuildTime")
  @Override
  public @Nullable String buildTime() {
    return buildTime;
  }

  /**
   * @return The value of the {@code gitCommit} attribute
   */
  @JsonProperty("GitCommit")
  @Override
  public String gitCommit() {
    return gitCommit;
  }

  /**
   * @return The value of the {@code goVersion} attribute
   */
  @JsonProperty("GoVersion")
  @Override
  public String goVersion() {
    return goVersion;
  }

  /**
   * @return The value of the {@code kernelVersion} attribute
   */
  @JsonProperty("KernelVersion")
  @Override
  public String kernelVersion() {
    return kernelVersion;
  }

  /**
   * @return The value of the {@code os} attribute
   */
  @JsonProperty("Os")
  @Override
  public String os() {
    return os;
  }

  /**
   * @return The value of the {@code version} attribute
   */
  @JsonProperty("Version")
  @Override
  public String version() {
    return version;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Version#apiVersion() apiVersion} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for apiVersion
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableVersion withApiVersion(String value) {
    String newValue = Objects.requireNonNull(value, "apiVersion");
    if (this.apiVersion.equals(newValue)) return this;
    return new ImmutableVersion(
        newValue,
        this.arch,
        this.buildTime,
        this.gitCommit,
        this.goVersion,
        this.kernelVersion,
        this.os,
        this.version);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Version#arch() arch} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for arch
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableVersion withArch(String value) {
    String newValue = Objects.requireNonNull(value, "arch");
    if (this.arch.equals(newValue)) return this;
    return new ImmutableVersion(
        this.apiVersion,
        newValue,
        this.buildTime,
        this.gitCommit,
        this.goVersion,
        this.kernelVersion,
        this.os,
        this.version);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Version#buildTime() buildTime} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for buildTime (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableVersion withBuildTime(@Nullable String value) {
    if (Objects.equals(this.buildTime, value)) return this;
    return new ImmutableVersion(
        this.apiVersion,
        this.arch,
        value,
        this.gitCommit,
        this.goVersion,
        this.kernelVersion,
        this.os,
        this.version);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Version#gitCommit() gitCommit} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for gitCommit
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableVersion withGitCommit(String value) {
    String newValue = Objects.requireNonNull(value, "gitCommit");
    if (this.gitCommit.equals(newValue)) return this;
    return new ImmutableVersion(
        this.apiVersion,
        this.arch,
        this.buildTime,
        newValue,
        this.goVersion,
        this.kernelVersion,
        this.os,
        this.version);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Version#goVersion() goVersion} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for goVersion
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableVersion withGoVersion(String value) {
    String newValue = Objects.requireNonNull(value, "goVersion");
    if (this.goVersion.equals(newValue)) return this;
    return new ImmutableVersion(
        this.apiVersion,
        this.arch,
        this.buildTime,
        this.gitCommit,
        newValue,
        this.kernelVersion,
        this.os,
        this.version);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Version#kernelVersion() kernelVersion} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for kernelVersion
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableVersion withKernelVersion(String value) {
    String newValue = Objects.requireNonNull(value, "kernelVersion");
    if (this.kernelVersion.equals(newValue)) return this;
    return new ImmutableVersion(
        this.apiVersion,
        this.arch,
        this.buildTime,
        this.gitCommit,
        this.goVersion,
        newValue,
        this.os,
        this.version);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Version#os() os} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for os
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableVersion withOs(String value) {
    String newValue = Objects.requireNonNull(value, "os");
    if (this.os.equals(newValue)) return this;
    return new ImmutableVersion(
        this.apiVersion,
        this.arch,
        this.buildTime,
        this.gitCommit,
        this.goVersion,
        this.kernelVersion,
        newValue,
        this.version);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Version#version() version} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for version
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableVersion withVersion(String value) {
    String newValue = Objects.requireNonNull(value, "version");
    if (this.version.equals(newValue)) return this;
    return new ImmutableVersion(
        this.apiVersion,
        this.arch,
        this.buildTime,
        this.gitCommit,
        this.goVersion,
        this.kernelVersion,
        this.os,
        newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableVersion} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableVersion
        && equalsByValue((ImmutableVersion) another);
  }

  private boolean equalsByValue(ImmutableVersion another) {
    return apiVersion.equals(another.apiVersion)
        && arch.equals(another.arch)
        && Objects.equals(buildTime, another.buildTime)
        && gitCommit.equals(another.gitCommit)
        && goVersion.equals(another.goVersion)
        && kernelVersion.equals(another.kernelVersion)
        && os.equals(another.os)
        && version.equals(another.version);
  }

  /**
   * Computes a hash code from attributes: {@code apiVersion}, {@code arch}, {@code buildTime}, {@code gitCommit}, {@code goVersion}, {@code kernelVersion}, {@code os}, {@code version}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + apiVersion.hashCode();
    h += (h << 5) + arch.hashCode();
    h += (h << 5) + Objects.hashCode(buildTime);
    h += (h << 5) + gitCommit.hashCode();
    h += (h << 5) + goVersion.hashCode();
    h += (h << 5) + kernelVersion.hashCode();
    h += (h << 5) + os.hashCode();
    h += (h << 5) + version.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code Version} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Version{"
        + "apiVersion=" + apiVersion
        + ", arch=" + arch
        + ", buildTime=" + buildTime
        + ", gitCommit=" + gitCommit
        + ", goVersion=" + goVersion
        + ", kernelVersion=" + kernelVersion
        + ", os=" + os
        + ", version=" + version
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Version} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Version instance
   */
  public static ImmutableVersion copyOf(Version instance) {
    if (instance instanceof ImmutableVersion) {
      return (ImmutableVersion) instance;
    }
    return ImmutableVersion.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableVersion ImmutableVersion}.
   * <pre>
   * ImmutableVersion.builder()
   *    .apiVersion(String) // required {@link Version#apiVersion() apiVersion}
   *    .arch(String) // required {@link Version#arch() arch}
   *    .buildTime(String | null) // nullable {@link Version#buildTime() buildTime}
   *    .gitCommit(String) // required {@link Version#gitCommit() gitCommit}
   *    .goVersion(String) // required {@link Version#goVersion() goVersion}
   *    .kernelVersion(String) // required {@link Version#kernelVersion() kernelVersion}
   *    .os(String) // required {@link Version#os() os}
   *    .version(String) // required {@link Version#version() version}
   *    .build();
   * </pre>
   * @return A new ImmutableVersion builder
   */
  public static ImmutableVersion.Builder builder() {
    return new ImmutableVersion.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableVersion ImmutableVersion}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_API_VERSION = 0x1L;
    private static final long INIT_BIT_ARCH = 0x2L;
    private static final long INIT_BIT_GIT_COMMIT = 0x4L;
    private static final long INIT_BIT_GO_VERSION = 0x8L;
    private static final long INIT_BIT_KERNEL_VERSION = 0x10L;
    private static final long INIT_BIT_OS = 0x20L;
    private static final long INIT_BIT_VERSION = 0x40L;
    private long initBits = 0x7fL;

    private String apiVersion;
    private String arch;
    private String buildTime;
    private String gitCommit;
    private String goVersion;
    private String kernelVersion;
    private String os;
    private String version;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Version} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Version instance) {
      Objects.requireNonNull(instance, "instance");
      this.apiVersion(instance.apiVersion());
      this.arch(instance.arch());
      @Nullable String buildTimeValue = instance.buildTime();
      if (buildTimeValue != null) {
        buildTime(buildTimeValue);
      }
      this.gitCommit(instance.gitCommit());
      this.goVersion(instance.goVersion());
      this.kernelVersion(instance.kernelVersion());
      this.os(instance.os());
      this.version(instance.version());
      return this;
    }

    /**
     * Initializes the value for the {@link Version#apiVersion() apiVersion} attribute.
     * @param apiVersion The value for apiVersion 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("ApiVersion")
    public final Builder apiVersion(String apiVersion) {
      this.apiVersion = Objects.requireNonNull(apiVersion, "apiVersion");
      initBits &= ~INIT_BIT_API_VERSION;
      return this;
    }

    /**
     * Initializes the value for the {@link Version#arch() arch} attribute.
     * @param arch The value for arch 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Arch")
    public final Builder arch(String arch) {
      this.arch = Objects.requireNonNull(arch, "arch");
      initBits &= ~INIT_BIT_ARCH;
      return this;
    }

    /**
     * Initializes the value for the {@link Version#buildTime() buildTime} attribute.
     * @param buildTime The value for buildTime (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("BuildTime")
    public final Builder buildTime(@Nullable String buildTime) {
      this.buildTime = buildTime;
      return this;
    }

    /**
     * Initializes the value for the {@link Version#gitCommit() gitCommit} attribute.
     * @param gitCommit The value for gitCommit 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("GitCommit")
    public final Builder gitCommit(String gitCommit) {
      this.gitCommit = Objects.requireNonNull(gitCommit, "gitCommit");
      initBits &= ~INIT_BIT_GIT_COMMIT;
      return this;
    }

    /**
     * Initializes the value for the {@link Version#goVersion() goVersion} attribute.
     * @param goVersion The value for goVersion 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("GoVersion")
    public final Builder goVersion(String goVersion) {
      this.goVersion = Objects.requireNonNull(goVersion, "goVersion");
      initBits &= ~INIT_BIT_GO_VERSION;
      return this;
    }

    /**
     * Initializes the value for the {@link Version#kernelVersion() kernelVersion} attribute.
     * @param kernelVersion The value for kernelVersion 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("KernelVersion")
    public final Builder kernelVersion(String kernelVersion) {
      this.kernelVersion = Objects.requireNonNull(kernelVersion, "kernelVersion");
      initBits &= ~INIT_BIT_KERNEL_VERSION;
      return this;
    }

    /**
     * Initializes the value for the {@link Version#os() os} attribute.
     * @param os The value for os 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Os")
    public final Builder os(String os) {
      this.os = Objects.requireNonNull(os, "os");
      initBits &= ~INIT_BIT_OS;
      return this;
    }

    /**
     * Initializes the value for the {@link Version#version() version} attribute.
     * @param version The value for version 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Version")
    public final Builder version(String version) {
      this.version = Objects.requireNonNull(version, "version");
      initBits &= ~INIT_BIT_VERSION;
      return this;
    }

    /**
     * Builds a new {@link ImmutableVersion ImmutableVersion}.
     * @return An immutable instance of Version
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableVersion build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableVersion(apiVersion, arch, buildTime, gitCommit, goVersion, kernelVersion, os, version);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_API_VERSION) != 0) attributes.add("apiVersion");
      if ((initBits & INIT_BIT_ARCH) != 0) attributes.add("arch");
      if ((initBits & INIT_BIT_GIT_COMMIT) != 0) attributes.add("gitCommit");
      if ((initBits & INIT_BIT_GO_VERSION) != 0) attributes.add("goVersion");
      if ((initBits & INIT_BIT_KERNEL_VERSION) != 0) attributes.add("kernelVersion");
      if ((initBits & INIT_BIT_OS) != 0) attributes.add("os");
      if ((initBits & INIT_BIT_VERSION) != 0) attributes.add("version");
      return "Cannot build Version, some of required attributes are not set " + attributes;
    }
  }
}
