/**
 * Copyright (c) 2016, 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.executionframework.debugger;

import com.google.common.base.Objects;
import java.util.ArrayList;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.function.BiPredicate;
import java.util.function.Supplier;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gemoc.dsl.debug.ide.event.IDSLDebugEventProcessor;
import org.eclipse.gemoc.executionframework.debugger.Activator;
import org.eclipse.gemoc.executionframework.debugger.GenericSequentialModelDebugger;
import org.eclipse.gemoc.executionframework.engine.core.EngineStoppedException;
import org.eclipse.gemoc.trace.commons.model.trace.Dimension;
import org.eclipse.gemoc.trace.commons.model.trace.MSEOccurrence;
import org.eclipse.gemoc.trace.commons.model.trace.State;
import org.eclipse.gemoc.trace.commons.model.trace.Step;
import org.eclipse.gemoc.trace.commons.model.trace.TracedObject;
import org.eclipse.gemoc.trace.commons.model.trace.Value;
import org.eclipse.gemoc.trace.gemoc.api.IMultiDimensionalTraceAddon;
import org.eclipse.gemoc.trace.gemoc.api.ITraceExplorer;
import org.eclipse.gemoc.trace.gemoc.api.ITraceViewListener;
import org.eclipse.gemoc.trace.gemoc.api.ITraceViewNotifier;
import org.eclipse.gemoc.xdsmlframework.api.core.IExecutionEngine;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

@SuppressWarnings("all")
public class OmniscientGenericSequentialModelDebugger extends GenericSequentialModelDebugger implements ITraceViewListener {
  private ITraceExplorer<Step<?>, State<?, ?>, TracedObject<?>, Dimension<?>, Value<?>> traceExplorer;
  
  private int steppingOverStackFrameIndex = (-1);
  
  private int steppingReturnStackFrameIndex = (-1);
  
  private final List<EObject> callerStack = new ArrayList<EObject>();
  
  private final List<Step<?>> previousCallStack = new ArrayList<Step<?>>();
  
  public OmniscientGenericSequentialModelDebugger(final IDSLDebugEventProcessor target, final IExecutionEngine engine) {
    super(target, engine);
  }
  
  private void pushStackFrame(final String threadName, final Step<?> step) {
    final GenericSequentialModelDebugger.MSEFrameInformation info = this.getMSEFrameInformation(step);
    this.pushStackFrame(threadName, info.prettyLabel, info.caller, info.caller);
    this.callerStack.add(0, info.caller);
  }
  
  @Override
  public void popStackFrame(final String threadName) {
    super.popStackFrame(threadName);
    this.callerStack.remove(0);
  }
  
  @Override
  public void aboutToExecuteStep(final IExecutionEngine executionEngine, final Step<?> step) {
    final MSEOccurrence mseOccurrence = step.getMseoccurrence();
    if ((mseOccurrence != null)) {
      final boolean shallContinue = this.control(this.threadName, step);
      if ((!shallContinue)) {
        throw new EngineStoppedException("Debug thread has stopped.");
      }
    }
  }
  
  @Override
  public void resume() {
    if ((!this.executionTerminated)) {
      boolean _isInReplayMode = this.traceExplorer.isInReplayMode();
      if (_isInReplayMode) {
        this.traceExplorer.loadLastState();
      }
      super.resume();
    }
  }
  
  @Override
  public void resume(final String threadName) {
    if ((!this.executionTerminated)) {
      boolean _isInReplayMode = this.traceExplorer.isInReplayMode();
      if (_isInReplayMode) {
        this.traceExplorer.loadLastState();
      }
      super.resume(threadName);
    }
  }
  
  @Override
  public void terminate() {
    super.terminate();
    Activator _default = Activator.getDefault();
    final Supplier<OmniscientGenericSequentialModelDebugger> _function = new Supplier<OmniscientGenericSequentialModelDebugger>() {
      @Override
      public OmniscientGenericSequentialModelDebugger get() {
        return null;
      }
    };
    _default.setDebuggerSupplier(_function);
  }
  
  @Override
  protected void setupStepOverPredicateBreak() {
    abstract class __OmniscientGenericSequentialModelDebugger_1 implements BiPredicate<IExecutionEngine, Step<?>> {
      Step<?> steppedOver;
    }
    
    if ((this.steppingOverStackFrameIndex != (-1))) {
      final IExecutionEngine seqEngine = ((IExecutionEngine) this.engine);
      final List<Step<?>> stack = this.traceExplorer.getCallStack();
      int _size = stack.size();
      int _minus = (_size - this.steppingOverStackFrameIndex);
      final int idx = (_minus - 1);
      __OmniscientGenericSequentialModelDebugger_1 ___OmniscientGenericSequentialModelDebugger_1 = new __OmniscientGenericSequentialModelDebugger_1() {
        {
          steppedOver = stack.get(idx);
        }
        @Override
        public boolean test(final IExecutionEngine t, final Step<?> u) {
          Deque<Step<?>> _currentStack = seqEngine.getCurrentStack();
          boolean _contains = _currentStack.contains(this.steppedOver);
          return (!_contains);
        }
      };
      this.addPredicateBreak(___OmniscientGenericSequentialModelDebugger_1);
      this.steppingOverStackFrameIndex = (-1);
    } else {
      super.setupStepOverPredicateBreak();
    }
  }
  
  @Override
  public void stepInto(final String threadName) {
    if ((this.traceExplorer.isInReplayMode() || this.executionTerminated)) {
      if (((!this.traceExplorer.stepInto()) && (!this.executionTerminated))) {
        this.traceExplorer.loadLastState();
        super.stepInto(threadName);
      }
    } else {
      super.stepInto(threadName);
    }
  }
  
  @Override
  public void stepOver(final String threadName) {
    if ((this.traceExplorer.isInReplayMode() || this.executionTerminated)) {
      if (((!this.traceExplorer.stepOver()) && (!this.executionTerminated))) {
        this.steppingOverStackFrameIndex = (this.nbStackFrames - 1);
        this.traceExplorer.loadLastState();
        super.stepOver(threadName);
      }
    } else {
      super.stepOver(threadName);
    }
  }
  
  @Override
  protected void setupStepReturnPredicateBreak() {
    abstract class __OmniscientGenericSequentialModelDebugger_2 implements BiPredicate<IExecutionEngine, Step<?>> {
      Step<?> steppedReturn;
    }
    
    if ((this.steppingReturnStackFrameIndex != (-1))) {
      final IExecutionEngine seqEngine = ((IExecutionEngine) this.engine);
      final List<Step<?>> stack = this.traceExplorer.getCallStack();
      int _size = stack.size();
      int _minus = (_size - this.steppingReturnStackFrameIndex);
      final int idx = (_minus - 1);
      __OmniscientGenericSequentialModelDebugger_2 ___OmniscientGenericSequentialModelDebugger_2 = new __OmniscientGenericSequentialModelDebugger_2() {
        {
          steppedReturn = stack.get(idx);
        }
        @Override
        public boolean test(final IExecutionEngine t, final Step<?> u) {
          Deque<Step<?>> _currentStack = seqEngine.getCurrentStack();
          boolean _contains = _currentStack.contains(this.steppedReturn);
          return (!_contains);
        }
      };
      this.addPredicateBreak(___OmniscientGenericSequentialModelDebugger_2);
      this.steppingReturnStackFrameIndex = (-1);
    } else {
      super.setupStepReturnPredicateBreak();
    }
  }
  
  @Override
  public void stepReturn(final String threadName) {
    if ((this.traceExplorer.isInReplayMode() || this.executionTerminated)) {
      if (((!this.traceExplorer.stepReturn()) && (!this.executionTerminated))) {
        this.steppingReturnStackFrameIndex = (this.nbStackFrames - 2);
        this.traceExplorer.loadLastState();
        super.stepReturn(threadName);
      }
    } else {
      super.stepReturn(threadName);
    }
  }
  
  public void stepBackInto() {
    this.traceExplorer.stepBackInto();
  }
  
  public void stepBackOver() {
    this.traceExplorer.stepBackOver();
  }
  
  public void stepBackOut() {
    this.traceExplorer.stepBackOut();
  }
  
  public boolean canStepBackInto() {
    return this.traceExplorer.canStepBackInto();
  }
  
  public boolean canStepBackOver() {
    return this.traceExplorer.canStepBackOver();
  }
  
  public boolean canStepBackOut() {
    return this.traceExplorer.canStepBackOut();
  }
  
  @Override
  public boolean validateVariableValue(final String threadName, final String variableName, final String value) {
    boolean _isInReplayMode = this.traceExplorer.isInReplayMode();
    if (_isInReplayMode) {
      Status _status = new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Illegal variable value set");
      ErrorDialog.openError(null, "Illegal variable value set", 
        "Cannot set the value of a variable when in replay mode", _status);
      return false;
    }
    return super.validateVariableValue(threadName, variableName, value);
  }
  
  @Override
  public void engineStarted(final IExecutionEngine executionEngine) {
    final Activator activator = Activator.getDefault();
    final Supplier<OmniscientGenericSequentialModelDebugger> _function = new Supplier<OmniscientGenericSequentialModelDebugger>() {
      @Override
      public OmniscientGenericSequentialModelDebugger get() {
        return OmniscientGenericSequentialModelDebugger.this;
      }
    };
    activator.setDebuggerSupplier(_function);
    super.engineStarted(executionEngine);
    final Set<IMultiDimensionalTraceAddon> traceAddons = executionEngine.<IMultiDimensionalTraceAddon>getAddonsTypedBy(IMultiDimensionalTraceAddon.class);
    Iterator<IMultiDimensionalTraceAddon> _iterator = traceAddons.iterator();
    final IMultiDimensionalTraceAddon traceAddon = _iterator.next();
    ITraceExplorer _traceExplorer = traceAddon.getTraceExplorer();
    this.traceExplorer = _traceExplorer;
    final ITraceViewNotifier.TraceViewCommand _function_1 = new ITraceViewNotifier.TraceViewCommand() {
      @Override
      public void execute() {
        OmniscientGenericSequentialModelDebugger.this.update();
      }
    };
    this.traceExplorer.registerCommand(this, _function_1);
  }
  
  @Override
  public void engineAboutToStop(final IExecutionEngine engine) {
    this.traceExplorer.loadLastState();
    super.engineAboutToStop(engine);
  }
  
  @Override
  public void engineStopped(final IExecutionEngine executionEngine) {
    final Activator activator = Activator.getDefault();
    activator.setDebuggerSupplier(null);
    super.engineStopped(executionEngine);
  }
  
  @Override
  public void updateStack(final String threadName, final EObject instruction) {
    int i = 0;
    while ((((i < this.previousCallStack.size()) && (i < this.traceExplorer.getCallStack().size())) && 
      Objects.equal(this.previousCallStack.get(i), this.traceExplorer.getCallStack().get(i)))) {
      i++;
    }
    for (int j = i; (j < this.previousCallStack.size()); j++) {
      this.popStackFrame(threadName);
    }
    for (int j = i; (j < this.traceExplorer.getCallStack().size()); j++) {
      List<Step<?>> _callStack = this.traceExplorer.getCallStack();
      Step<?> _get = _callStack.get(j);
      this.pushStackFrame(threadName, _get);
    }
    boolean _isEmpty = this.callerStack.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      EObject _get = this.callerStack.get(0);
      this.setCurrentInstruction(threadName, _get);
    }
    this.previousCallStack.clear();
    List<Step<?>> _callStack = this.traceExplorer.getCallStack();
    this.previousCallStack.addAll(_callStack);
  }
  
  @Override
  public void update() {
    if ((this.executedModelRoot != null)) {
      try {
        boolean _isEmpty = this.callerStack.isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          final Function1<EObject, Boolean> _function = new Function1<EObject, Boolean>() {
            @Override
            public Boolean apply(final EObject it) {
              return Boolean.valueOf(true);
            }
          };
          EObject _findFirst = IterableExtensions.<EObject>findFirst(this.callerStack, _function);
          this.updateData(this.threadName, _findFirst);
        } else {
        }
      } catch (final Throwable _t) {
        if (_t instanceof IllegalStateException) {
          final IllegalStateException e = (IllegalStateException)_t;
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
  }
}
