/*
 * Copyright (c) 2017-2022, ARM Limited and Contributors. All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <assert.h>
#include <errno.h>
#include <string.h>

#include <platform_def.h>

#include <arch_helpers.h>
#include <common/debug.h>
#include <drivers/ufs.h>
#include <drivers/io/io_block.h>
#include <drivers/io/io_driver.h>
#include <drivers/io/io_fip.h>
#include <drivers/io/io_memmap.h>
#include <drivers/io/io_storage.h>
#include <drivers/partition/partition.h>
#include <lib/mmio.h>
#include <lib/semihosting.h>
#include <tools_share/firmware_image_package.h>

#include "hikey960_def.h"
#include "hikey960_private.h"

struct plat_io_policy {
	uintptr_t *dev_handle;
	uintptr_t image_spec;
	int (*check)(const uintptr_t spec);
};

static const io_dev_connector_t *ufs_dev_con, *fip_dev_con;
static uintptr_t ufs_dev_handle, fip_dev_handle;

static int check_ufs(const uintptr_t spec);
static int check_fip(const uintptr_t spec);
size_t ufs_read_lun3_blks(int lba, uintptr_t buf, size_t size);
size_t ufs_write_lun3_blks(int lba, const uintptr_t buf, size_t size);

static io_block_spec_t ufs_fip_spec;

static const io_block_spec_t ufs_gpt_spec = {
	.offset		= 0,
	.length		= PLAT_PARTITION_BLOCK_SIZE *
			  (PLAT_PARTITION_MAX_ENTRIES / 4 + 2),
};

/* Fastboot serial number stored within first UFS device blocks */
static const io_block_spec_t ufs_fastboot_spec = {
	.offset         = UFS_BASE,
	.length         = 1 << 20,
};

static const io_block_dev_spec_t ufs_dev_spec = {
	/* It's used as temp buffer in block driver. */
	.buffer		= {
		.offset	= HIKEY960_UFS_DATA_BASE,
		.length	= HIKEY960_UFS_DATA_SIZE,
	},
	.ops		= {
		.read	= ufs_read_lun3_blks,
		.write	= ufs_write_lun3_blks,
	},
	.block_size	= UFS_BLOCK_SIZE,
};

static const io_uuid_spec_t scp_bl2_uuid_spec = {
	.uuid = UUID_SCP_FIRMWARE_SCP_BL2,
};

static const io_uuid_spec_t bl31_uuid_spec = {
	.uuid = UUID_EL3_RUNTIME_FIRMWARE_BL31,
};

static const io_uuid_spec_t bl32_uuid_spec = {
	.uuid = UUID_SECURE_PAYLOAD_BL32,
};

static const io_uuid_spec_t bl32_extra1_uuid_spec = {
	.uuid = UUID_SECURE_PAYLOAD_BL32_EXTRA1,
};

static const io_uuid_spec_t bl32_extra2_uuid_spec = {
	.uuid = UUID_SECURE_PAYLOAD_BL32_EXTRA2,
};

#ifdef SPD_spmd
static const io_uuid_spec_t bl32_tos_fw_spec = {
	.uuid = UUID_TOS_FW_CONFIG,
};
#endif

static const io_uuid_spec_t bl33_uuid_spec = {
	.uuid = UUID_NON_TRUSTED_FIRMWARE_BL33,
};

#if TRUSTED_BOARD_BOOT
static const io_uuid_spec_t trusted_key_cert_uuid_spec = {
	.uuid = UUID_TRUSTED_KEY_CERT,
};

static const io_uuid_spec_t scp_fw_key_cert_uuid_spec = {
	.uuid = UUID_SCP_FW_KEY_CERT,
};

static const io_uuid_spec_t soc_fw_key_cert_uuid_spec = {
	.uuid = UUID_SOC_FW_KEY_CERT,
};

static const io_uuid_spec_t tos_fw_key_cert_uuid_spec = {
	.uuid = UUID_TRUSTED_OS_FW_KEY_CERT,
};

static const io_uuid_spec_t nt_fw_key_cert_uuid_spec = {
	.uuid = UUID_NON_TRUSTED_FW_KEY_CERT,
};

static const io_uuid_spec_t scp_fw_cert_uuid_spec = {
	.uuid = UUID_SCP_FW_CONTENT_CERT,
};

static const io_uuid_spec_t soc_fw_cert_uuid_spec = {
	.uuid = UUID_SOC_FW_CONTENT_CERT,
};

static const io_uuid_spec_t tos_fw_cert_uuid_spec = {
	.uuid = UUID_TRUSTED_OS_FW_CONTENT_CERT,
};

static const io_uuid_spec_t nt_fw_cert_uuid_spec = {
	.uuid = UUID_NON_TRUSTED_FW_CONTENT_CERT,
};
#endif /* TRUSTED_BOARD_BOOT */

static const struct plat_io_policy policies[] = {
	[FIP_IMAGE_ID] = {
		&ufs_dev_handle,
		(uintptr_t)&ufs_fip_spec,
		check_ufs
	},
	[SCP_BL2_IMAGE_ID] = {
		&fip_dev_handle,
		(uintptr_t)&scp_bl2_uuid_spec,
		check_fip
	},
	[BL31_IMAGE_ID] = {
		&fip_dev_handle,
		(uintptr_t)&bl31_uuid_spec,
		check_fip
	},
	[BL32_IMAGE_ID] = {
		&fip_dev_handle,
		(uintptr_t)&bl32_uuid_spec,
		check_fip
	},
	[BL32_EXTRA1_IMAGE_ID] = {
		&fip_dev_handle,
		(uintptr_t)&bl32_extra1_uuid_spec,
		check_fip
	},
	[BL32_EXTRA2_IMAGE_ID] = {
		&fip_dev_handle,
		(uintptr_t)&bl32_extra2_uuid_spec,
		check_fip
	},

#ifdef SPD_spmd
	[TOS_FW_CONFIG_ID] = {
		&fip_dev_handle,
		(uintptr_t)&bl32_tos_fw_spec,
		check_fip
	},
#endif

	[BL33_IMAGE_ID] = {
		&fip_dev_handle,
		(uintptr_t)&bl33_uuid_spec,
		check_fip
	},
#if TRUSTED_BOARD_BOOT
	[TRUSTED_KEY_CERT_ID] = {
		&fip_dev_handle,
		(uintptr_t)&trusted_key_cert_uuid_spec,
		check_fip
	},
	[SCP_FW_KEY_CERT_ID] = {
		&fip_dev_handle,
		(uintptr_t)&scp_fw_key_cert_uuid_spec,
		check_fip
	},
	[SOC_FW_KEY_CERT_ID] = {
		&fip_dev_handle,
		(uintptr_t)&soc_fw_key_cert_uuid_spec,
		check_fip
	},
	[TRUSTED_OS_FW_KEY_CERT_ID] = {
		&fip_dev_handle,
		(uintptr_t)&tos_fw_key_cert_uuid_spec,
		check_fip
	},
	[NON_TRUSTED_FW_KEY_CERT_ID] = {
		&fip_dev_handle,
		(uintptr_t)&nt_fw_key_cert_uuid_spec,
		check_fip
	},
	[SCP_FW_CONTENT_CERT_ID] = {
		&fip_dev_handle,
		(uintptr_t)&scp_fw_cert_uuid_spec,
		check_fip
	},
	[SOC_FW_CONTENT_CERT_ID] = {
		&fip_dev_handle,
		(uintptr_t)&soc_fw_cert_uuid_spec,
		check_fip
	},
	[TRUSTED_OS_FW_CONTENT_CERT_ID] = {
		&fip_dev_handle,
		(uintptr_t)&tos_fw_cert_uuid_spec,
		check_fip
	},
	[NON_TRUSTED_FW_CONTENT_CERT_ID] = {
		&fip_dev_handle,
		(uintptr_t)&nt_fw_cert_uuid_spec,
		check_fip
	},
#endif /* TRUSTED_BOARD_BOOT */
	[GPT_IMAGE_ID] = {
		&ufs_dev_handle,
		(uintptr_t)&ufs_gpt_spec,
		check_ufs
	},
};

static int check_ufs(const uintptr_t spec)
{
	int result;
	uintptr_t local_handle;

	result = io_dev_init(ufs_dev_handle, (uintptr_t)NULL);
	if (result == 0) {
		result = io_open(ufs_dev_handle, spec, &local_handle);
		if (result == 0)
			io_close(local_handle);
	}
	return result;
}

static int check_fip(const uintptr_t spec)
{
	int result;
	uintptr_t local_image_handle;

	/* See if a Firmware Image Package is available */
	result = io_dev_init(fip_dev_handle, (uintptr_t)FIP_IMAGE_ID);
	if (result == 0) {
		result = io_open(fip_dev_handle, spec, &local_image_handle);
		if (result == 0) {
			VERBOSE("Using FIP\n");
			io_close(local_image_handle);
		}
	}
	return result;
}

int hikey960_load_serialno(uint64_t *serno)
{
	int result;
	size_t len = 0;
	uintptr_t local_handle;
	uint64_t buf[HIKEY960_SERIAL_NUMBER_SIZE / sizeof(uint64_t)];

	if (serno == NULL) {
		return -1;
	}

	result = io_dev_init(ufs_dev_handle, (uintptr_t)NULL);
	if (result != 0) {
		return result;
	}

	result = io_open(ufs_dev_handle,
		(uintptr_t)&ufs_fastboot_spec, &local_handle);
	if (result != 0) {
		return result;
	}

	result = io_seek(local_handle, IO_SEEK_SET,
		HIKEY960_SERIAL_NUMBER_LBA * UFS_BLOCK_SIZE);
	if (result != 0) {
		goto closing;
	}

	result = io_read(local_handle, (uintptr_t)buf,
		HIKEY960_SERIAL_NUMBER_SIZE, &len);
	if (result != 0) {
		goto closing;
	}

	if (len != HIKEY960_SERIAL_NUMBER_SIZE) {
		result = -1;
		goto closing;
	}

	/* UEFI fastboot app stores a 16 bytes blob       */
	/* We extract only relevant 8 bytes serial number */
	*serno = buf[1];

closing:
	io_close(local_handle);
	return result;
}

void hikey960_io_setup(void)
{
	int result;

	result = register_io_dev_block(&ufs_dev_con);
	assert(result == 0);

	result = register_io_dev_fip(&fip_dev_con);
	assert(result == 0);

	result = io_dev_open(ufs_dev_con, (uintptr_t)&ufs_dev_spec,
			     &ufs_dev_handle);
	assert(result == 0);

	result = io_dev_open(fip_dev_con, (uintptr_t)NULL, &fip_dev_handle);
	assert(result == 0);

	/* Ignore improbable errors in release builds */
	(void)result;
}

int hikey960_set_fip_addr(unsigned int image_id, const char *name)
{
	const partition_entry_t *entry;

	if (ufs_fip_spec.length == 0) {
		partition_init(GPT_IMAGE_ID);
		entry = get_partition_entry(name);
		if (entry == NULL) {
			ERROR("Could NOT find the %s partition!\n", name);
			return -ENOENT;
		}
		ufs_fip_spec.offset = entry->start;
		ufs_fip_spec.length = entry->length;
	}
	return 0;
}

/* Return an IO device handle and specification which can be used to access
 * an image. Use this to enforce platform load policy
 */
int plat_get_image_source(unsigned int image_id, uintptr_t *dev_handle,
			  uintptr_t *image_spec)
{
	int result;
	const struct plat_io_policy *policy;

	assert(image_id < ARRAY_SIZE(policies));

	policy = &policies[image_id];
	result = policy->check(policy->image_spec);
	assert(result == 0);

	*image_spec = policy->image_spec;
	*dev_handle = *(policy->dev_handle);

	return result;
}

size_t ufs_read_lun3_blks(int lba, uintptr_t buf, size_t size)
{
	return ufs_read_blocks(3, lba, buf, size);
}

size_t ufs_write_lun3_blks(int lba, const uintptr_t buf, size_t size)
{
	return ufs_write_blocks(3, lba, buf, size);
}
