// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

// This is a programmatically generated file from Regex.GenerateUnicodeTables.
// It provides serialized BDD Unicode category definitions for System.Environment.Version = 8.0.0

using System.Globalization;

namespace System.Text.RegularExpressions.Symbolic
{
    internal static class UnicodeCategoryRanges
    {
        /// <summary>Serialized BDD representation of the set of all whitespace characters.</summary>
        public static ReadOnlySpan<byte> SerializedWhitespaceBDD => new byte[] { 0x3, 0x4, 0x0, 0x0, 0x7, 0x0, 0x0, 0x0, 0x8, 0x0, 0x10, 0x0, 0x0, 0x1, 0x10, 0x0, 0x31, 0x0, 0x0, 0x1, 0x8, 0x0, 0x21, 0x8, 0x0, 0x1, 0x18, 0x0, 0x1, 0x10, 0x0, 0x11, 0x18, 0x0, 0x2, 0x20, 0x0, 0x52, 0x0, 0x0, 0x52, 0x30, 0x0, 0x2, 0x38, 0x0, 0x82, 0x0, 0x0, 0x2, 0x48, 0x0, 0x62, 0x50, 0x0, 0x3, 0x58, 0x0, 0xC3, 0x0, 0x0, 0xD3, 0x0, 0x0, 0xE3, 0x8, 0x0, 0x3, 0x78, 0x0, 0x3, 0x80, 0x0, 0x13, 0x1, 0x0, 0x4, 0x90, 0x0, 0x34, 0x1, 0x0, 0x4, 0xA0, 0x0, 0x4, 0xA8, 0x0, 0x4, 0xB0, 0x0, 0x4, 0xB8, 0x0, 0x4, 0xC0, 0x0, 0x5, 0xC8, 0x0, 0x5, 0xD0, 0x0, 0xB5, 0xE1, 0x0, 0x95, 0xE9, 0x0, 0xE5, 0xF9, 0x0, 0x6, 0x0, 0x1, 0x16, 0x12, 0x1, 0x6, 0x18, 0x1, 0x6, 0x20, 0x1, 0x7, 0x28, 0x1, 0x7, 0x30, 0x1, 0x57, 0x2, 0x0, 0x77, 0x42, 0x1, 0x8, 0x48, 0x1, 0x8, 0x50, 0x1, 0x8, 0x58, 0x1, 0x8, 0x60, 0x1, 0x9, 0x68, 0x1, 0x9, 0x70, 0x1, 0xF9, 0x2, 0x0, 0x9, 0x80, 0x1, 0xA, 0x88, 0x1, 0xA, 0x90, 0x1, 0x3A, 0x3, 0x0, 0xA, 0xA0, 0x1, 0xB, 0xA8, 0x1, 0xB, 0xB0, 0x1, 0xB, 0xB8, 0x1, 0xB, 0xC0, 0x1, 0x9C, 0xD3, 0x1, 0xBC, 0xE3, 0x1, 0xDD, 0xF3, 0x1, 0xE, 0xF8, 0x1, 0xF, 0x0, 0x2, };

        /// <summary>Gets the serialized BDD representations of any defined UnicodeCategory.</summary>
        public static ReadOnlySpan<byte> GetSerializedCategory(UnicodeCategory category) =>
            (int)category switch
            {
                0 => SerializedCategory0_UppercaseLetter,
                1 => SerializedCategory1_LowercaseLetter,
                2 => SerializedCategory2_TitlecaseLetter,
                3 => SerializedCategory3_ModifierLetter,
                4 => SerializedCategory4_OtherLetter,
                5 => SerializedCategory5_NonSpacingMark,
                6 => SerializedCategory6_SpacingCombiningMark,
                7 => SerializedCategory7_EnclosingMark,
                8 => SerializedCategory8_DecimalDigitNumber,
                9 => SerializedCategory9_LetterNumber,
                10 => SerializedCategory10_OtherNumber,
                11 => SerializedCategory11_SpaceSeparator,
                12 => SerializedCategory12_LineSeparator,
                13 => SerializedCategory13_ParagraphSeparator,
                14 => SerializedCategory14_Control,
                15 => SerializedCategory15_Format,
                16 => SerializedCategory16_Surrogate,
                17 => SerializedCategory17_PrivateUse,
                18 => SerializedCategory18_ConnectorPunctuation,
                19 => SerializedCategory19_DashPunctuation,
                20 => SerializedCategory20_OpenPunctuation,
                21 => SerializedCategory21_ClosePunctuation,
                22 => SerializedCategory22_InitialQuotePunctuation,
                23 => SerializedCategory23_FinalQuotePunctuation,
                24 => SerializedCategory24_OtherPunctuation,
                25 => SerializedCategory25_MathSymbol,
                26 => SerializedCategory26_CurrencySymbol,
                27 => SerializedCategory27_ModifierSymbol,
                28 => SerializedCategory28_OtherSymbol,
                29 => SerializedCategory29_OtherNotAssigned,
                _ => default,
            };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.UppercaseLetter.</summary>
        private static ReadOnlySpan<byte> SerializedCategory0_UppercaseLetter => new byte[] { 0x3, 0x4, 0x0, 0x0, 0x9, 0x0, 0x0, 0x0, 0x20, 0x0, 0x10, 0x0, 0x0, 0x21, 0x20, 0x0, 0x11, 0x60, 0x0, 0x1, 0x60, 0x0, 0x1, 0x40, 0x0, 0x21, 0x0, 0x0, 0x11, 0x40, 0x0, 0x31, 0x0, 0x0, 0x21, 0x60, 0x0, 0x11, 0x0, 0x0, 0x31, 0x40, 0x0, 0x1, 0x20, 0x0, 0x31, 0x20, 0x0, 0x2, 0x80, 0x0, 0x12, 0xA0, 0x0, 0x62, 0x0, 0x0, 0x2, 0xE0, 0x0, 0x82, 0xE0, 0x0, 0x2, 0xC0, 0x0, 0x12, 0x40, 0x0, 0x22, 0x20, 0x0, 0x42, 0x20, 0x1, 0x82, 0x40, 0x0, 0x62, 0x40, 0x1, 0xB2, 0x60, 0x0, 0x22, 0x0, 0x1, 0x2, 0x40, 0x0, 0x72, 0x40, 0x0, 0x2, 0x0, 0x1, 0xC2, 0x0, 0x0, 0x62, 0x40, 0x0, 0x52, 0x60, 0x0, 0xD2, 0x20, 0x1, 0x2, 0x40, 0x1, 0x2, 0x20, 0x0, 0xE2, 0x20, 0x1, 0x22, 0x0, 0x0, 0xE2, 0xA0, 0x0, 0x62, 0x80, 0x0, 0xE2, 0x40, 0x1, 0xA2, 0x0, 0x1, 0x72, 0x20, 0x0, 0xE2, 0x20, 0x0, 0x82, 0x0, 0x0, 0x52, 0x80, 0x0, 0xF2, 0x20, 0x0, 0x42, 0x20, 0x0, 0x62, 0x60, 0x0, 0x32, 0xE0, 0x1, 0x52, 0x60, 0x1, 0xD2, 0x0, 0x0, 0x72, 0x80, 0x1, 0xA2, 0x0, 0x0, 0x12, 0xE0, 0x1, 0x92, 0x80, 0x0, 0x42, 0x60, 0x0, 0xB2, 0x80, 0x1, 0x92, 0xC0, 0x0, 0x82, 0x60, 0x0, 0x32, 0x0, 0x1, 0x32, 0xA0, 0x0, 0x92, 0x40, 0x0, 0xF2, 0xE0, 0x0, 0x92, 0xE0, 0x1, 0xC2, 0xA0, 0x0, 0x92, 0x60, 0x1, 0x62, 0xE0, 0x1, 0x3, 0x21, 0x0, 0x13, 0x20, 0x2, 0x3, 0x40, 0x2, 0x33, 0x81, 0x2, 0x53, 0xC1, 0x2, 0x23, 0xE0, 0x2, 0x83, 0x41, 0x0, 0x23, 0x20, 0x3, 0xA3, 0x41, 0x0, 0xB3, 0x1, 0x0, 0x23, 0x80, 0x3, 0xD3, 0x41, 0x0, 0xE3, 0x41, 0x0, 0x3, 0xE0, 0x3, 0xA3, 0xA1, 0x3, 0x3, 0x22, 0x4, 0x23, 0x62, 0x4, 0x3, 0x80, 0x4, 0x3, 0xA2, 0x4, 0x63, 0xE2, 0x4, 0x83, 0x22, 0x5, 0xA3, 0x62, 0x5, 0xC3, 0x2, 0x0, 0x53, 0x2, 0x0, 0x13, 0x0, 0x0, 0x33, 0x0, 0x0, 0xD3, 0x2, 0x0, 0xE3, 0xC2, 0x3, 0xF3, 0x22, 0x0, 0x3, 0xA0, 0x5, 0x23, 0x1, 0x6, 0x3, 0x20, 0x6, 0x23, 0x63, 0x6, 0xC3, 0x61, 0x2, 0x43, 0xA3, 0x6, 0x23, 0xC0, 0x6, 0x73, 0x3, 0x0, 0x53, 0x2, 0x7, 0x93, 0xC3, 0x5, 0x73, 0x23, 0x3, 0x23, 0x40, 0x7, 0xB3, 0xA3, 0x3, 0x23, 0x80, 0x7, 0xD3, 0x63, 0x0, 0xE3, 0xA3, 0x6, 0x73, 0xE0, 0x7, 0xC3, 0x3, 0x8, 0x13, 0x44, 0x8, 0x33, 0x84, 0x8, 0x53, 0x44, 0x0, 0xC3, 0x61, 0x0, 0x33, 0x40, 0x0, 0x64, 0xE4, 0x8, 0x84, 0x4, 0x0, 0x94, 0x44, 0x9, 0xB4, 0x84, 0x9, 0xD4, 0xC4, 0x9, 0xF4, 0x44, 0x0, 0x14, 0x45, 0x0, 0x4, 0x40, 0xA, 0x34, 0x85, 0xA, 0x54, 0xC5, 0xA, 0x4, 0x20, 0x0, 0x4, 0xE0, 0xA, 0x4, 0x0, 0x9, 0x84, 0x25, 0xB, 0xA4, 0x65, 0xB, 0xD4, 0x85, 0xB, 0xD4, 0x5, 0x0, 0x4, 0xC0, 0xB, 0xF4, 0x5, 0xC, 0x4, 0xC6, 0xB, 0x14, 0x46, 0x0, 0x24, 0x26, 0x0, 0x14, 0x0, 0x0, 0x34, 0x26, 0x0, 0x4, 0x80, 0xC, 0x54, 0x26, 0x0, 0x74, 0x44, 0x0, 0x24, 0xC0, 0xC, 0x24, 0xE0, 0xC, 0x84, 0x46, 0x0, 0x94, 0x46, 0xD, 0x4, 0x60, 0xD, 0x74, 0x84, 0xD, 0xD4, 0x6, 0x0, 0x4, 0xC0, 0xD, 0xF4, 0x46, 0x0, 0x4, 0x47, 0x0, 0x14, 0x47, 0xE, 0x34, 0x87, 0xE, 0x54, 0xC7, 0xE, 0x74, 0x47, 0x0, 0x24, 0x0, 0xF, 0x94, 0x47, 0x0, 0x14, 0x23, 0x0, 0xA5, 0x7, 0x0, 0xB5, 0x87, 0xF, 0xD5, 0xC7, 0xF, 0xF5, 0x47, 0x0, 0x5, 0x5, 0x0, 0x5, 0x0, 0x10, 0x15, 0x48, 0x0, 0x25, 0x48, 0x0, 0x35, 0x8, 0x0, 0x45, 0x28, 0x0, 0x5, 0xA0, 0x10, 0x5, 0xC0, 0x10, 0x75, 0x8, 0x11, 0x95, 0xA8, 0xB, 0xA5, 0x8, 0x0, 0xB5, 0x88, 0x11, 0xE5, 0xA5, 0x11, 0x25, 0xC0, 0x11, 0xF5, 0x8, 0x12, 0x15, 0x29, 0x0, 0x15, 0x0, 0x0, 0x5, 0x40, 0x12, 0x5, 0x60, 0x12, 0x45, 0x49, 0x0, 0x25, 0xA0, 0x12, 0x25, 0xC0, 0x12, 0x25, 0x0, 0x0, 0x75, 0x9, 0x13, 0x95, 0x49, 0x13, 0xB5, 0x9, 0x0, 0x5, 0x80, 0x13, 0xD5, 0x49, 0x0, 0xE5, 0xE9, 0x13, 0x5, 0x2A, 0x14, 0x25, 0x6A, 0x14, 0x45, 0x4A, 0x0, 0x5, 0xA0, 0x14, 0x5, 0x40, 0xF, 0x6, 0xC0, 0x14, 0x76, 0xA, 0x15, 0x96, 0x4A, 0x15, 0x6, 0x60, 0x15, 0xC6, 0xA, 0x0, 0xD6, 0x4A, 0x0, 0xE6, 0xEA, 0x15, 0x6, 0x0, 0x16, 0x16, 0x4B, 0x16, 0x36, 0x8B, 0x16, 0x56, 0xCB, 0x16, 0x26, 0xE0, 0x16, 0x6, 0x0, 0x17, 0x96, 0x4B, 0x17, 0xB6, 0x4B, 0x17, 0xC6, 0xAB, 0x17, 0xE6, 0xEB, 0x17, 0x6, 0xEC, 0x15, 0x16, 0x4C, 0x18, 0x36, 0xC, 0x0, 0x46, 0xAC, 0x18, 0x66, 0xEC, 0x18, 0x86, 0x2C, 0x19, 0xA6, 0xC, 0x0, 0xB6, 0xC, 0x0, 0x7, 0x80, 0x19, 0xD7, 0xCC, 0x19, 0xF7, 0xC, 0x1A, 0x17, 0x4D, 0x1A, 0x37, 0x8D, 0x1A, 0x57, 0xCD, 0x1A, 0x77, 0x4D, 0x0, 0x87, 0xD, 0x0, 0x97, 0xD, 0x0, 0xA7, 0xD, 0x0, 0x7, 0x60, 0x1B, 0xC7, 0xAD, 0x1B, 0xE7, 0xED, 0x1B, 0x7, 0x0, 0x1C, 0x17, 0x4E, 0x1C, 0x37, 0x8E, 0x1C, 0x58, 0xE, 0x0, 0x68, 0xEE, 0x1C, 0x8, 0x0, 0x1D, 0x98, 0xE, 0x0, 0xA8, 0x6E, 0x1D, 0x8, 0x80, 0x1D, 0xD8, 0xE, 0x0, 0x8, 0xC0, 0x1D, 0xF8, 0xE, 0x1E, 0x18, 0x4F, 0x1E, 0x38, 0x8F, 0x1E, 0x59, 0xF, 0x0, 0x69, 0xF, 0x0, 0x9, 0xE0, 0x1E, 0x9, 0x0, 0x1F, 0x99, 0x4F, 0x1F, 0xB9, 0x8F, 0x1F, 0x9, 0xA0, 0x1F, 0xE9, 0xEF, 0x1F, 0xA, 0x10, 0x0, 0x1A, 0x10, 0x0, 0x2A, 0x10, 0x0, 0xA, 0x60, 0x20, 0x4A, 0x10, 0x0, 0xA, 0xA0, 0x20, 0x6A, 0xF0, 0x20, 0x8B, 0x10, 0x0, 0xB, 0x20, 0x21, 0xAB, 0x70, 0x21, 0xCB, 0xB0, 0x21, 0xB, 0xC0, 0x21, 0xFC, 0x10, 0x0, 0xC, 0x0, 0x22, 0xC, 0x20, 0x22, 0x2C, 0x71, 0x22, 0x4D, 0x11, 0x0, 0x5D, 0x11, 0x0, 0x6D, 0xF1, 0x22, 0x8E, 0x31, 0x23, 0xE, 0x40, 0x23, 0xBF, 0x91, 0x23, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.LowercaseLetter.</summary>
        private static ReadOnlySpan<byte> SerializedCategory1_LowercaseLetter => new byte[] { 0x3, 0x4, 0x0, 0x0, 0x9, 0x0, 0x0, 0x0, 0x20, 0x0, 0x10, 0x0, 0x0, 0x21, 0x20, 0x0, 0x11, 0x60, 0x0, 0x31, 0x0, 0x0, 0x1, 0x40, 0x0, 0x21, 0x0, 0x0, 0x1, 0x60, 0x0, 0x31, 0x20, 0x0, 0x31, 0x40, 0x0, 0x11, 0x40, 0x0, 0x21, 0x60, 0x0, 0x1, 0x20, 0x0, 0x11, 0x0, 0x0, 0x2, 0x80, 0x0, 0x12, 0xA0, 0x0, 0x12, 0xC0, 0x0, 0x42, 0x20, 0x0, 0x2, 0xE0, 0x0, 0x2, 0x0, 0x1, 0x82, 0x0, 0x0, 0x2, 0x20, 0x1, 0x2, 0x40, 0x0, 0x2, 0x60, 0x0, 0x32, 0x0, 0x0, 0x62, 0x20, 0x1, 0xA2, 0x60, 0x0, 0x22, 0x0, 0x0, 0xB2, 0x40, 0x0, 0x32, 0x40, 0x1, 0x32, 0xC0, 0x0, 0x92, 0x60, 0x0, 0x92, 0x0, 0x0, 0xA2, 0x20, 0x0, 0x2, 0xC0, 0x0, 0x72, 0x60, 0x0, 0x2, 0x20, 0x0, 0xC2, 0x60, 0x0, 0x72, 0x40, 0x0, 0xD2, 0x20, 0x1, 0x72, 0x0, 0x0, 0x82, 0xC0, 0x1, 0xF2, 0x0, 0x0, 0xE2, 0x20, 0x1, 0x62, 0x0, 0x0, 0xF2, 0x0, 0x1, 0xC2, 0xE0, 0x1, 0xF2, 0x20, 0x0, 0xC2, 0x20, 0x0, 0xE2, 0x20, 0x0, 0x52, 0x60, 0x0, 0x52, 0x80, 0x0, 0xC2, 0x40, 0x0, 0x22, 0x0, 0x1, 0x72, 0x60, 0x1, 0x92, 0x20, 0x0, 0x52, 0xC0, 0x1, 0x12, 0x0, 0x0, 0xE2, 0xC0, 0x0, 0x62, 0x60, 0x0, 0x52, 0x20, 0x0, 0x62, 0x40, 0x0, 0xB2, 0xC0, 0x1, 0x12, 0x60, 0x0, 0x92, 0x60, 0x1, 0xA2, 0x40, 0x0, 0x22, 0x20, 0x1, 0x52, 0xA0, 0x1, 0x22, 0xE0, 0x0, 0x92, 0x80, 0x1, 0x82, 0xA0, 0x0, 0x92, 0x0, 0x1, 0xE2, 0xE0, 0x0, 0xC2, 0x0, 0x1, 0x3, 0x21, 0x0, 0x13, 0x20, 0x2, 0x3, 0x40, 0x2, 0x3, 0x60, 0x2, 0x43, 0x21, 0x0, 0x53, 0xC1, 0x2, 0x73, 0x61, 0x0, 0x83, 0x21, 0x3, 0x33, 0x40, 0x3, 0xB3, 0x61, 0x0, 0x33, 0x80, 0x3, 0xD3, 0x61, 0x0, 0xE3, 0x21, 0x2, 0x33, 0xE0, 0x3, 0x33, 0x0, 0x4, 0x93, 0x61, 0x0, 0x13, 0x62, 0x0, 0x23, 0x62, 0x4, 0x3, 0x80, 0x4, 0xD3, 0xA1, 0x4, 0x63, 0xE2, 0x4, 0x83, 0x22, 0x5, 0x3, 0x40, 0x5, 0xB3, 0x82, 0x5, 0xD3, 0x42, 0x5, 0xE3, 0x2, 0x0, 0xF3, 0x2, 0x0, 0x3, 0x0, 0x6, 0x3, 0x20, 0x0, 0x3, 0x20, 0x6, 0x63, 0x41, 0x6, 0x33, 0x23, 0x0, 0x3, 0x60, 0x6, 0x33, 0x82, 0x6, 0x13, 0x21, 0x0, 0x23, 0x1, 0x0, 0x63, 0x22, 0x0, 0x33, 0x3, 0x0, 0x53, 0x23, 0x0, 0x63, 0xE3, 0x6, 0x3, 0xE2, 0x2, 0x83, 0x23, 0x7, 0x33, 0x40, 0x7, 0x33, 0x21, 0x0, 0xB3, 0x3, 0x0, 0x3, 0x80, 0x2, 0xC3, 0xA3, 0x7, 0x13, 0xC0, 0x7, 0x33, 0xE0, 0x7, 0x3, 0x24, 0x8, 0x33, 0x40, 0x8, 0x33, 0x44, 0x0, 0x43, 0xC4, 0x2, 0x53, 0xC4, 0x8, 0x73, 0x24, 0x4, 0x83, 0x24, 0x9, 0xA3, 0x44, 0x8, 0xB3, 0x64, 0x0, 0xF3, 0x41, 0x0, 0x23, 0x60, 0x0, 0x13, 0x60, 0x2, 0x3, 0x40, 0x4, 0xC4, 0xA4, 0x9, 0xE4, 0xE4, 0x9, 0x14, 0x0, 0xA, 0xC4, 0x24, 0x0, 0x14, 0x0, 0x0, 0x14, 0x5, 0x0, 0x24, 0x65, 0xA, 0x44, 0xA5, 0xA, 0x64, 0xE5, 0xA, 0x84, 0x65, 0x0, 0x94, 0x45, 0xB, 0xB4, 0x65, 0x0, 0x4, 0x80, 0xB, 0x4, 0xA0, 0xB, 0xE4, 0xE5, 0xB, 0x4, 0x26, 0xC, 0x4, 0x40, 0xC, 0x4, 0x60, 0xC, 0x44, 0xA6, 0xC, 0xE4, 0xC5, 0xC, 0x74, 0x6, 0xD, 0x94, 0xE6, 0xC, 0xA4, 0x6, 0xD, 0xB4, 0x6, 0xD, 0x84, 0x86, 0xD, 0xC4, 0x6, 0xD, 0xD4, 0x66, 0x0, 0xE4, 0xE6, 0xD, 0x4, 0x0, 0xE, 0x4, 0x0, 0xA, 0x14, 0x7, 0x0, 0x24, 0x27, 0x0, 0x4, 0x60, 0x0, 0x34, 0x60, 0xE, 0x34, 0x80, 0xE, 0x54, 0x67, 0x0, 0x64, 0xE7, 0xE, 0x14, 0x0, 0xF, 0x94, 0x7, 0x0, 0xA4, 0x7, 0x0, 0x4, 0x20, 0x0, 0xB4, 0x27, 0x0, 0x14, 0x80, 0xF, 0xD4, 0x67, 0x0, 0xE4, 0x67, 0x0, 0xF4, 0x7, 0x10, 0x14, 0x48, 0x10, 0x34, 0x88, 0x10, 0x54, 0x68, 0x0, 0x34, 0xC0, 0x10, 0x74, 0x68, 0x0, 0x84, 0x28, 0x0, 0x54, 0x6, 0x0, 0x94, 0x8, 0x0, 0x5, 0x40, 0x11, 0x5, 0x60, 0x11, 0xC5, 0xA8, 0x11, 0xE5, 0x8, 0x0, 0xF5, 0x8, 0x12, 0x15, 0x49, 0x12, 0x35, 0x69, 0x0, 0x45, 0x9, 0x0, 0x5, 0xA0, 0x12, 0x65, 0x69, 0x0, 0x75, 0x29, 0x0, 0x85, 0x69, 0x0, 0x95, 0x29, 0x0, 0x5, 0x40, 0x13, 0x5, 0x60, 0x13, 0xC5, 0xA9, 0x13, 0xE5, 0xE9, 0x13, 0x5, 0xA, 0xD, 0x15, 0x4A, 0x14, 0x85, 0x66, 0x14, 0x35, 0x80, 0x14, 0x5, 0xA0, 0x11, 0x55, 0xA, 0x0, 0x65, 0x2A, 0x0, 0x5, 0xE0, 0x14, 0x85, 0xA, 0x0, 0x95, 0xCA, 0x11, 0x15, 0x0, 0x0, 0xA5, 0x6A, 0x0, 0x35, 0x60, 0x15, 0x35, 0x80, 0x15, 0x35, 0x20, 0x0, 0xD5, 0xCA, 0x15, 0xF5, 0xA, 0x16, 0x15, 0xB, 0x0, 0x25, 0x6B, 0x16, 0x15, 0x80, 0x16, 0x55, 0x6B, 0x0, 0x65, 0xEB, 0x16, 0x85, 0x2B, 0x17, 0xA5, 0x6B, 0x17, 0xC5, 0x6B, 0x0, 0xD5, 0xCB, 0x17, 0xF5, 0xB, 0x0, 0xA5, 0x8, 0x0, 0x6, 0xC, 0x0, 0x6, 0x20, 0x18, 0x6, 0x20, 0x0, 0x26, 0x6C, 0x18, 0x46, 0xAC, 0x18, 0x66, 0xEC, 0x18, 0x6, 0x0, 0x19, 0x96, 0xC, 0x0, 0x6, 0x40, 0x19, 0xB6, 0x6C, 0x0, 0xC6, 0x6C, 0x18, 0x6, 0xA0, 0x19, 0xE6, 0xEC, 0x19, 0x6, 0x2D, 0x1A, 0x26, 0x6D, 0x1A, 0x36, 0x80, 0x1A, 0x6, 0xA0, 0x1A, 0x66, 0xED, 0x1A, 0x6, 0x0, 0x1B, 0x96, 0xD, 0x0, 0xA6, 0xD, 0x0, 0xB6, 0x8D, 0x1B, 0xD6, 0xCD, 0x1B, 0xF6, 0x6D, 0x18, 0x6, 0x2E, 0x1C, 0x26, 0xE, 0x0, 0x6, 0x60, 0x1C, 0x46, 0xAE, 0x1C, 0x66, 0xEE, 0x1C, 0x86, 0x2E, 0x1D, 0xA6, 0x6E, 0x1D, 0xC6, 0xE, 0x0, 0x7, 0xA0, 0x1D, 0x7, 0xC0, 0x1D, 0xF7, 0xE, 0x1E, 0x17, 0x4F, 0x1E, 0x37, 0x8F, 0x1E, 0x7, 0xA0, 0x1E, 0x67, 0xEF, 0x1E, 0x87, 0x2F, 0x1F, 0xA7, 0x6F, 0x1F, 0xC7, 0x6F, 0x0, 0xD7, 0xCF, 0x1F, 0xF7, 0xF, 0x0, 0x7, 0x10, 0x0, 0x17, 0x10, 0x0, 0xF7, 0x4F, 0x20, 0x37, 0x90, 0x20, 0x57, 0xD0, 0x20, 0x77, 0x10, 0x21, 0x97, 0x50, 0x21, 0xB7, 0x90, 0x21, 0xD8, 0x10, 0x0, 0xE8, 0x10, 0x0, 0xF8, 0x10, 0x0, 0x8, 0x31, 0x22, 0x28, 0x71, 0x22, 0x48, 0x11, 0x0, 0x58, 0xD1, 0x22, 0x78, 0x11, 0x23, 0x98, 0x11, 0x0, 0x8, 0x40, 0x23, 0xB8, 0x91, 0x23, 0xD8, 0xD1, 0x23, 0xF8, 0x11, 0x24, 0x19, 0x12, 0x0, 0x29, 0x12, 0x0, 0x39, 0x12, 0x0, 0x49, 0x12, 0x0, 0x9, 0xA0, 0x24, 0x9, 0xC0, 0x24, 0x79, 0x12, 0x25, 0x99, 0x52, 0x25, 0x9, 0x60, 0x25, 0xC9, 0xB2, 0x25, 0xEA, 0xF2, 0x25, 0xA, 0x0, 0x26, 0x1A, 0x13, 0x0, 0x2A, 0x13, 0x0, 0xA, 0x60, 0x26, 0x4A, 0x13, 0x0, 0xA, 0xA0, 0x26, 0x6A, 0xF3, 0x26, 0x8B, 0x13, 0x0, 0x9B, 0x53, 0x27, 0xBB, 0x93, 0x27, 0xDB, 0xD3, 0x27, 0xB, 0xE0, 0x27, 0xC, 0x14, 0x0, 0xC, 0x20, 0x28, 0xC, 0x40, 0x28, 0x3C, 0x94, 0x28, 0x5D, 0x14, 0x0, 0x6D, 0x14, 0x0, 0x7D, 0x14, 0x29, 0x9E, 0x54, 0x29, 0xE, 0x60, 0x29, 0xCF, 0xB4, 0x29, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.TitlecaseLetter.</summary>
        private static ReadOnlySpan<byte> SerializedCategory2_TitlecaseLetter => new byte[] { 0x2, 0x4, 0x0, 0x6, 0x0, 0x0, 0x4, 0x10, 0x0, 0x1, 0x8, 0x21, 0x0, 0x31, 0x8, 0x1, 0xC, 0x42, 0x0, 0x2, 0x14, 0x2, 0x18, 0x72, 0x0, 0x83, 0x0, 0x13, 0x0, 0x3, 0x24, 0xA3, 0x2C, 0xC4, 0x0, 0x4, 0x30, 0xC4, 0x34, 0xE4, 0x0, 0x4, 0x3C, 0x5, 0x45, 0x25, 0x35, 0x35, 0x51, 0x56, 0x59, 0x76, 0x1, 0x87, 0x1, 0x97, 0x1, 0xA8, 0x1, 0xB8, 0x1, 0xC9, 0x1, 0x9, 0x74, 0xEA, 0x1, 0xA, 0x7C, 0xB, 0x2, 0xB, 0x84, 0x2C, 0x8E, 0xD, 0x90, 0xE, 0x94, 0xF, 0x98, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.ModifierLetter.</summary>
        private static ReadOnlySpan<byte> SerializedCategory3_ModifierLetter => new byte[] { 0x3, 0x4, 0x0, 0x0, 0x9, 0x0, 0x0, 0x0, 0x20, 0x0, 0x10, 0x0, 0x0, 0x11, 0x0, 0x0, 0x1, 0x40, 0x0, 0x1, 0x60, 0x0, 0x31, 0x0, 0x0, 0x21, 0x0, 0x0, 0x1, 0x20, 0x0, 0x21, 0x20, 0x0, 0x21, 0x60, 0x0, 0x11, 0x60, 0x0, 0x42, 0x0, 0x0, 0x2, 0xA0, 0x0, 0x2, 0xC0, 0x0, 0x12, 0x0, 0x0, 0x52, 0xE0, 0x0, 0x62, 0x0, 0x0, 0x82, 0x0, 0x0, 0x72, 0x0, 0x0, 0x2, 0x20, 0x1, 0x52, 0x80, 0x0, 0x92, 0x0, 0x0, 0x52, 0x0, 0x0, 0x92, 0x20, 0x0, 0xA2, 0x0, 0x0, 0xB2, 0x0, 0x0, 0x2, 0xE0, 0x0, 0x92, 0x80, 0x1, 0x52, 0x20, 0x0, 0x12, 0xE0, 0x0, 0x2, 0x40, 0x1, 0x2, 0x0, 0x1, 0x22, 0x0, 0x0, 0x42, 0x20, 0x1, 0xD3, 0x0, 0x0, 0x3, 0xC0, 0x1, 0xF3, 0x0, 0x0, 0x3, 0x1, 0x0, 0x3, 0x20, 0x2, 0x23, 0x1, 0x0, 0x3, 0x60, 0x2, 0x43, 0x1, 0x0, 0x53, 0xC1, 0x2, 0xE3, 0x0, 0x0, 0x13, 0x80, 0x2, 0x73, 0x1, 0x0, 0x83, 0x1, 0x0, 0x93, 0x1, 0x0, 0x3, 0x40, 0x2, 0xA3, 0x1, 0x0, 0xB3, 0x1, 0x0, 0xC3, 0xA1, 0x3, 0xE3, 0x21, 0x0, 0x43, 0xE1, 0x1, 0xF3, 0x1, 0x0, 0x3, 0x22, 0x0, 0x3, 0x80, 0x2, 0x3, 0x80, 0x3, 0x3, 0xE0, 0x1, 0xE3, 0x0, 0x3, 0x13, 0x2, 0x0, 0x13, 0xE2, 0x2, 0x3, 0x60, 0x3, 0x13, 0x2, 0x3, 0x23, 0xC2, 0x3, 0x3, 0xA0, 0x2, 0x13, 0x60, 0x4, 0x44, 0x2, 0x0, 0x54, 0x2, 0x0, 0x4, 0xC0, 0x4, 0x74, 0x2, 0x0, 0x84, 0x2, 0x0, 0x94, 0x2, 0x0, 0x4, 0x40, 0x5, 0x4, 0x60, 0x5, 0xC4, 0x2, 0x0, 0x4, 0xA0, 0x5, 0xE4, 0x2, 0x0, 0xF4, 0x2, 0x0, 0xB4, 0x2, 0x0, 0x4, 0x0, 0x6, 0x14, 0x3, 0x0, 0x24, 0x3, 0x0, 0x34, 0x3, 0x0, 0x44, 0x3, 0x0, 0x54, 0x3, 0x0, 0x4, 0x40, 0x6, 0x64, 0x3, 0x0, 0x74, 0x3, 0x0, 0x84, 0x3, 0x0, 0xD4, 0x22, 0x7, 0x14, 0xE0, 0x4, 0x4, 0x40, 0x7, 0x4, 0x60, 0x7, 0xA4, 0x3, 0x0, 0x4, 0x3, 0x0, 0xC4, 0x3, 0x0, 0x4, 0xA0, 0x7, 0xE4, 0x3, 0x0, 0xF4, 0x3, 0x0, 0x4, 0x0, 0x8, 0x4, 0xA0, 0x4, 0x64, 0x2, 0x0, 0x14, 0x4, 0x0, 0x4, 0x40, 0x8, 0x34, 0x84, 0x8, 0x14, 0x0, 0x0, 0x5, 0xA0, 0x8, 0x65, 0x4, 0x0, 0x75, 0x4, 0x9, 0x95, 0x44, 0x9, 0xB5, 0x84, 0x9, 0xD5, 0x4, 0x0, 0x5, 0xC0, 0x9, 0x5, 0xE0, 0x9, 0x5, 0x0, 0xA, 0x15, 0x5, 0x0, 0x5, 0x40, 0xA, 0x35, 0x5, 0x0, 0x5, 0x80, 0xA, 0x55, 0x5, 0x0, 0x5, 0xC0, 0xA, 0x75, 0x5, 0xB, 0xC5, 0x4, 0x0, 0x5, 0x5, 0x0, 0x5, 0x20, 0xB, 0xA5, 0x5, 0x0, 0x15, 0x60, 0xB, 0xC5, 0x25, 0x0, 0xD5, 0x5, 0x0, 0xE5, 0x5, 0x0, 0x5, 0xE0, 0xB, 0x5, 0x0, 0xC, 0x15, 0x6, 0x0, 0x5, 0x60, 0x9, 0x25, 0x6, 0x0, 0x5, 0xE0, 0x8, 0x35, 0x86, 0xC, 0x55, 0x6, 0x0, 0x65, 0x6, 0x0, 0x5, 0xE0, 0xC, 0x5, 0x0, 0xD, 0x95, 0x6, 0x0, 0xA5, 0x66, 0xD, 0xC5, 0x6, 0x0, 0x6, 0xA0, 0xD, 0xE6, 0x6, 0x0, 0xF6, 0x6, 0x0, 0x6, 0x7, 0x0, 0x16, 0x7, 0x0, 0x26, 0x67, 0xE, 0xE6, 0x86, 0xE, 0x6, 0xA0, 0xE, 0x66, 0xE7, 0xE, 0x86, 0x7, 0x0, 0x6, 0x20, 0xF, 0xA6, 0x67, 0xF, 0x6, 0x80, 0xF, 0x6, 0xA0, 0xF, 0xD6, 0x7, 0x0, 0xE6, 0x7, 0x0, 0x6, 0xE0, 0xF, 0x6, 0x8, 0x0, 0x6, 0x20, 0x10, 0x26, 0x68, 0x10, 0x6, 0x80, 0x10, 0x56, 0x8, 0x0, 0x66, 0x8, 0x0, 0x76, 0x8, 0x0, 0x86, 0x8, 0x0, 0x96, 0x8, 0x0, 0xA6, 0x8, 0x0, 0x6, 0x60, 0x11, 0xC6, 0x8, 0x0, 0xD6, 0x8, 0x0, 0xE6, 0x8, 0x0, 0xF6, 0x8, 0x0, 0x6, 0x9, 0x0, 0x16, 0x49, 0x12, 0x37, 0x89, 0x12, 0x7, 0xA0, 0x12, 0x67, 0x89, 0x12, 0x77, 0x9, 0x0, 0x87, 0x29, 0x13, 0xA7, 0x69, 0x13, 0xC7, 0x9, 0x0, 0x7, 0xA0, 0x13, 0xE7, 0xE9, 0x13, 0x7, 0x0, 0x14, 0x7, 0x20, 0x14, 0x7, 0x40, 0x14, 0x37, 0x8A, 0x14, 0x57, 0xCA, 0x14, 0x7, 0x80, 0x13, 0x77, 0xA, 0x0, 0x7, 0x0, 0x15, 0x97, 0xA, 0x0, 0xA7, 0xA, 0x0, 0x7, 0x80, 0x15, 0xD7, 0xCA, 0x15, 0xF7, 0xA, 0x0, 0x7, 0x2B, 0x16, 0x7, 0x40, 0x16, 0x7, 0x60, 0x16, 0x47, 0xB, 0x0, 0x58, 0xB, 0x0, 0x68, 0xEB, 0x16, 0x88, 0xB, 0x0, 0x98, 0x4B, 0x17, 0x8, 0x60, 0x17, 0x8, 0x80, 0x17, 0x8, 0xA0, 0x17, 0x8, 0xC0, 0x17, 0xF8, 0xB, 0x18, 0x8, 0x20, 0x18, 0x28, 0x6C, 0x18, 0x8, 0x80, 0x18, 0x8, 0xA0, 0x18, 0x68, 0xC, 0x0, 0x8, 0xE0, 0x18, 0x8, 0x60, 0x15, 0x88, 0x2C, 0x19, 0xA8, 0x6C, 0x19, 0xC8, 0xC, 0x0, 0xD8, 0xCC, 0x19, 0xF9, 0xC, 0x0, 0x9, 0x2D, 0x1A, 0x29, 0x6D, 0x1A, 0x9, 0x80, 0x1A, 0x9, 0xA0, 0x1A, 0x69, 0xED, 0x1A, 0x9, 0x0, 0x1B, 0x9, 0x20, 0x1B, 0xA9, 0x6D, 0x1B, 0xC9, 0xD, 0x0, 0x9, 0xA0, 0x1B, 0xE9, 0xD, 0x0, 0x9, 0xE0, 0x1B, 0x9, 0x2E, 0x1C, 0x29, 0xE, 0x0, 0x3A, 0xE, 0x0, 0xA, 0x80, 0x1C, 0x5A, 0xCE, 0x1C, 0xA, 0xE0, 0x1C, 0x8A, 0xE, 0x0, 0xA, 0x20, 0x1D, 0xAA, 0x6E, 0x1D, 0xCA, 0xAE, 0x1D, 0xEA, 0xEE, 0x1D, 0xA, 0x2F, 0x1E, 0x2B, 0xF, 0x0, 0x3B, 0x8F, 0x1E, 0xB, 0xA0, 0x1E, 0x6B, 0xEF, 0x1E, 0x8B, 0x2F, 0x1F, 0xAB, 0x6F, 0x1F, 0xCC, 0xF, 0x0, 0xC, 0xA0, 0x1F, 0xEC, 0xEF, 0x1F, 0xC, 0x30, 0x20, 0x2D, 0x10, 0x0, 0x3D, 0x10, 0x0, 0x4D, 0xB0, 0x20, 0x6E, 0xF0, 0x20, 0xE, 0x0, 0x21, 0x9F, 0x50, 0x21, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.OtherLetter.</summary>
        private static ReadOnlySpan<byte> SerializedCategory4_OtherLetter => new byte[] { 0x3, 0x4, 0x0, 0x0, 0xA, 0x0, 0x0, 0x0, 0x40, 0x0, 0x10, 0x0, 0x0, 0x1, 0x80, 0x0, 0x11, 0x0, 0x0, 0x21, 0x40, 0x0, 0x1, 0x40, 0x0, 0x11, 0xC0, 0x0, 0x11, 0x80, 0x0, 0x31, 0x0, 0x0, 0x31, 0x40, 0x0, 0x21, 0x0, 0x0, 0x21, 0xC0, 0x0, 0x1, 0xC0, 0x0, 0x31, 0x80, 0x0, 0x42, 0x40, 0x1, 0x12, 0x40, 0x1, 0x62, 0x40, 0x0, 0x72, 0x40, 0x0, 0x12, 0x0, 0x2, 0x52, 0x0, 0x0, 0x42, 0x40, 0x0, 0x92, 0x40, 0x0, 0x2, 0x40, 0x0, 0x12, 0x80, 0x2, 0x2, 0xC0, 0x1, 0x92, 0xC0, 0x2, 0x22, 0x40, 0x0, 0x12, 0x40, 0x2, 0x32, 0x0, 0x0, 0x2, 0x80, 0x1, 0x62, 0x0, 0x2, 0x2, 0x0, 0x3, 0x42, 0x80, 0x2, 0x2, 0x80, 0x0, 0x72, 0x0, 0x2, 0xD2, 0x80, 0x3, 0x52, 0x0, 0x3, 0x12, 0x80, 0x1, 0x2, 0x0, 0x1, 0x62, 0x40, 0x1, 0xF2, 0x40, 0x0, 0x12, 0x0, 0x0, 0xD2, 0x80, 0x2, 0xA2, 0x0, 0x0, 0xC2, 0x0, 0x0, 0x2, 0x40, 0x1, 0x82, 0x0, 0x0, 0xA2, 0x80, 0x1, 0x42, 0x0, 0x0, 0xD2, 0x40, 0x0, 0x92, 0x0, 0x2, 0x72, 0x40, 0x2, 0x82, 0x0, 0x1, 0x72, 0x40, 0x1, 0xE2, 0x0, 0x0, 0x2, 0x40, 0x2, 0x32, 0x40, 0x0, 0x92, 0x40, 0x3, 0xE2, 0x40, 0x0, 0x12, 0xC0, 0x2, 0x62, 0x0, 0x0, 0xE2, 0x80, 0x1, 0x2, 0x40, 0x3, 0xD2, 0x0, 0x0, 0xE2, 0xC0, 0x1, 0x82, 0x40, 0x0, 0x52, 0x80, 0x1, 0x82, 0x80, 0x2, 0x2, 0x80, 0x3, 0xB2, 0x0, 0x0, 0x82, 0x40, 0x2, 0x12, 0xC0, 0x3, 0xB2, 0x40, 0x0, 0x22, 0x0, 0x2, 0xD2, 0x40, 0x2, 0x52, 0x80, 0x0, 0xF2, 0x40, 0x1, 0x2, 0x80, 0x2, 0x3, 0x41, 0x4, 0x23, 0x41, 0x0, 0x33, 0x41, 0x0, 0x13, 0x0, 0x5, 0x13, 0x40, 0x5, 0x63, 0x41, 0x0, 0x13, 0xC0, 0x5, 0x83, 0x41, 0x0, 0x3, 0x40, 0x0, 0x13, 0x40, 0x4, 0x13, 0x40, 0x6, 0x3, 0x80, 0x6, 0x13, 0xC0, 0x6, 0xC3, 0x81, 0x4, 0xD3, 0x41, 0x0, 0xE3, 0x1, 0x0, 0xA3, 0x41, 0x0, 0x93, 0x81, 0x4, 0x3, 0x0, 0x6, 0x3, 0xC0, 0x7, 0x3, 0x0, 0x8, 0x3, 0x40, 0x8, 0xF3, 0x41, 0x0, 0x23, 0x2, 0x0, 0x3, 0xC0, 0x8, 0x43, 0x42, 0x9, 0x63, 0x2, 0x8, 0x83, 0xC1, 0x9, 0x83, 0x42, 0x0, 0x93, 0x2, 0x0, 0x13, 0x80, 0xA, 0x13, 0xC0, 0xA, 0x3, 0x80, 0x4, 0x3, 0xC0, 0x4, 0x13, 0x1, 0x0, 0xC3, 0x42, 0x4, 0x73, 0xC2, 0x2, 0x93, 0x41, 0xB, 0xD3, 0x2, 0x0, 0xE3, 0x2, 0x0, 0xF3, 0x2, 0x0, 0x13, 0x0, 0xC, 0x13, 0x43, 0x0, 0xD3, 0x82, 0x4, 0x23, 0x3, 0x0, 0x3, 0x80, 0xB, 0x83, 0x2, 0xC, 0x13, 0xC2, 0xC, 0x43, 0x3, 0x0, 0x3, 0x3, 0x0, 0x53, 0x1, 0xA, 0x63, 0x1, 0xC, 0x3, 0x80, 0x5, 0x33, 0x42, 0x0, 0x13, 0xC0, 0x9, 0x3, 0x0, 0xA, 0x73, 0x41, 0x0, 0xD3, 0x82, 0x6, 0x83, 0x2, 0x5, 0xA3, 0x42, 0x0, 0x13, 0x40, 0xD, 0x13, 0x80, 0x4, 0x23, 0x41, 0xD, 0x53, 0x43, 0x0, 0x53, 0x83, 0x4, 0xE3, 0x82, 0x6, 0x13, 0x80, 0xD, 0x53, 0x41, 0x9, 0x73, 0xC3, 0x4, 0x63, 0x1, 0x0, 0x43, 0x41, 0x0, 0xB3, 0x2, 0x0, 0x83, 0x43, 0xE, 0x13, 0x80, 0xE, 0x73, 0xC2, 0xE, 0x3, 0x40, 0xE, 0xC3, 0x43, 0xF, 0x13, 0x81, 0x4, 0xD3, 0x82, 0xF, 0xF3, 0x43, 0x0, 0x13, 0x40, 0x7, 0x73, 0xC1, 0xA, 0x3, 0x0, 0x10, 0x13, 0x4, 0x0, 0x23, 0xC4, 0x10, 0x73, 0x81, 0xD, 0xF3, 0x3, 0x0, 0x23, 0x44, 0x0, 0x73, 0x1, 0xC, 0x43, 0x84, 0x8, 0xB3, 0x43, 0xD, 0x43, 0x44, 0x11, 0x3, 0x80, 0x11, 0x73, 0x4, 0x0, 0x23, 0x4, 0x12, 0x13, 0x40, 0x12, 0xA3, 0x2, 0xC, 0x63, 0x4, 0x0, 0x13, 0x40, 0xF, 0xA3, 0x4, 0xC, 0x3, 0x0, 0x4, 0xB3, 0x4, 0x0, 0xA3, 0x1, 0x13, 0x13, 0x3, 0xC, 0x23, 0x83, 0x6, 0x23, 0x44, 0x13, 0xA3, 0x82, 0xD, 0x13, 0x0, 0xA, 0x3, 0x42, 0x0, 0xE3, 0x4, 0x0, 0x53, 0x1, 0x0, 0x3, 0x0, 0xF, 0xF3, 0x1, 0x5, 0x3, 0x80, 0xC, 0xF3, 0x4, 0x0, 0x13, 0x2, 0x0, 0x4, 0x45, 0x4, 0x14, 0x45, 0x0, 0x24, 0x45, 0x0, 0x34, 0x5, 0x15, 0x54, 0x45, 0x0, 0x64, 0x5, 0x0, 0x74, 0x5, 0x0, 0x4, 0x0, 0x16, 0x94, 0x45, 0x0, 0x14, 0x0, 0x0, 0xA4, 0x5, 0x0, 0xB4, 0x45, 0x0, 0x14, 0x0, 0x17, 0xD4, 0x85, 0x17, 0xF4, 0x5, 0x0, 0x4, 0x46, 0x0, 0x14, 0x80, 0x14, 0x74, 0x45, 0x0, 0x14, 0x40, 0x18, 0x14, 0x80, 0x18, 0x4, 0xC0, 0x18, 0x4, 0x80, 0x4, 0x44, 0x6, 0x8, 0x54, 0x86, 0x19, 0x74, 0x6, 0x1A, 0x94, 0x46, 0x0, 0xA4, 0x46, 0x0, 0x4, 0xC0, 0x1A, 0x4, 0x0, 0x1B, 0xD4, 0x86, 0x1B, 0x34, 0x46, 0x0, 0x14, 0xC0, 0x1B, 0x4, 0x0, 0x1C, 0x14, 0x40, 0x1C, 0x14, 0x80, 0x1C, 0x34, 0x7, 0x0, 0x24, 0x46, 0x0, 0x14, 0x40, 0x16, 0x14, 0x0, 0x1D, 0x54, 0x7, 0x0, 0x4, 0x80, 0x1D, 0x4, 0x40, 0x1C, 0x4, 0xC0, 0x1D, 0x4, 0x0, 0x1E, 0x14, 0xC0, 0x15, 0x84, 0x45, 0x0, 0x4, 0x40, 0x15, 0x64, 0x45, 0x0, 0x4, 0x40, 0x14, 0x34, 0x45, 0x0, 0x14, 0x40, 0x1E, 0xA4, 0x47, 0x0, 0x14, 0xC0, 0x14, 0xB4, 0x47, 0x0, 0xC4, 0x7, 0x0, 0x4, 0x40, 0x1F, 0x4, 0x47, 0x0, 0xE4, 0x7, 0x0, 0xF4, 0x7, 0x20, 0x24, 0x47, 0x20, 0x4, 0x80, 0x18, 0x24, 0x87, 0x20, 0x14, 0x80, 0x16, 0x4, 0xC0, 0x20, 0x44, 0x48, 0x0, 0x4, 0x0, 0x18, 0x44, 0x88, 0x14, 0x14, 0x47, 0x0, 0x14, 0x40, 0x21, 0x14, 0x80, 0x1B, 0xC4, 0x46, 0x0, 0x14, 0x80, 0x21, 0x74, 0x8, 0x22, 0x94, 0x48, 0x0, 0xA4, 0x88, 0x19, 0x64, 0xC6, 0x14, 0x14, 0xC0, 0x22, 0x4, 0x40, 0x0, 0x64, 0x46, 0x0, 0xC4, 0x48, 0x0, 0xD4, 0x8, 0x23, 0xE4, 0x48, 0x0, 0x14, 0xC0, 0x23, 0x4, 0xC9, 0x23, 0x4, 0x40, 0x24, 0x24, 0xC9, 0x24, 0x44, 0x9, 0x0, 0x64, 0x87, 0x19, 0x4, 0x40, 0x25, 0x14, 0x80, 0x25, 0x4, 0xC0, 0x21, 0x74, 0x9, 0x21, 0x84, 0x49, 0x26, 0x14, 0x80, 0x26, 0xB4, 0x49, 0x0, 0xC4, 0x49, 0x0, 0xD4, 0x49, 0x1E, 0x24, 0xC7, 0x16, 0xE4, 0xC9, 0x1D, 0xF4, 0x49, 0x0, 0x4, 0x4A, 0x28, 0x24, 0xCA, 0x16, 0x34, 0xA, 0x0, 0x44, 0x8A, 0x17, 0x4, 0x4A, 0x29, 0x4, 0xC0, 0x16, 0x64, 0xA, 0x0, 0x74, 0x8A, 0x17, 0x4, 0xA, 0x2A, 0x74, 0x8, 0x0, 0x4, 0x46, 0x2A, 0xA4, 0xCA, 0x2A, 0xC4, 0xCA, 0x16, 0xD4, 0xA, 0x0, 0xE4, 0x8A, 0x17, 0xF4, 0xA, 0x2C, 0x14, 0xCB, 0x16, 0x24, 0xCB, 0x2C, 0x44, 0xB, 0x0, 0x54, 0xB, 0x0, 0x64, 0x8B, 0x17, 0xF4, 0xCA, 0x2D, 0x84, 0xCB, 0x16, 0xD4, 0xCA, 0x1D, 0x94, 0x8B, 0x17, 0xF4, 0x8A, 0x2E, 0x94, 0xC5, 0x16, 0xB4, 0xB, 0x0, 0x74, 0x4A, 0x0, 0x4, 0x0, 0x2F, 0x14, 0x80, 0x19, 0x4, 0x80, 0x19, 0xC4, 0x4A, 0x1C, 0x14, 0x40, 0x20, 0x4, 0xA, 0x0, 0xD4, 0x8B, 0x2F, 0xF4, 0x4B, 0x0, 0x34, 0x85, 0x2F, 0x4, 0x0, 0x30, 0x34, 0x86, 0x1E, 0x14, 0xC, 0x0, 0x24, 0xC, 0x0, 0x5, 0x0, 0x31, 0x55, 0x8C, 0x31, 0x75, 0xC, 0x0, 0x85, 0x4C, 0x0, 0x95, 0xC, 0x0, 0xA5, 0xCC, 0x32, 0x15, 0x0, 0x33, 0x15, 0x40, 0x33, 0x65, 0x4C, 0x0, 0x15, 0x80, 0x33, 0xF5, 0x4C, 0x0, 0x15, 0x0, 0x34, 0x15, 0x8D, 0x34, 0x5, 0xC0, 0x34, 0x45, 0x4D, 0x0, 0x55, 0x8D, 0x35, 0x75, 0x4D, 0x0, 0x85, 0x4D, 0x0, 0x95, 0x8D, 0x36, 0xB5, 0xD, 0x37, 0xD5, 0x4D, 0x0, 0xE5, 0xCD, 0x37, 0x5, 0x4E, 0x0, 0x15, 0xE, 0x0, 0x25, 0xCE, 0x38, 0x85, 0xC, 0x39, 0x55, 0x8E, 0x39, 0x75, 0xE, 0x0, 0x85, 0x4E, 0x3A, 0x85, 0x4E, 0x0, 0x85, 0x8D, 0x3A, 0xB5, 0xE, 0x0, 0x5, 0x0, 0x3B, 0xD5, 0x4E, 0x0, 0x15, 0x0, 0x0, 0xE5, 0xE, 0x0, 0xF5, 0xE, 0x3C, 0x15, 0x4F, 0x33, 0x5, 0x80, 0x3C, 0x15, 0xC0, 0x3C, 0xD5, 0xC, 0x0, 0x15, 0x0, 0x3D, 0x55, 0x8F, 0x3D, 0x75, 0x4F, 0x0, 0x85, 0x4F, 0x3E, 0x15, 0x0, 0x3E, 0xA5, 0xCF, 0x3E, 0x5, 0x80, 0x4, 0x25, 0x1, 0x3F, 0xB5, 0x4C, 0x0, 0xD5, 0xF, 0x0, 0xE5, 0xF, 0x0, 0x15, 0xCF, 0x3F, 0x5, 0x50, 0x0, 0x15, 0x90, 0x40, 0x5, 0xC0, 0x40, 0x85, 0x8E, 0x3D, 0x5, 0x0, 0x41, 0x15, 0x0, 0x3F, 0x5, 0x40, 0x41, 0x5, 0x4F, 0x0, 0x65, 0x50, 0x33, 0x5, 0x40, 0x31, 0x75, 0x50, 0x0, 0x85, 0x50, 0x42, 0xA5, 0xD0, 0x42, 0x5, 0x80, 0x3E, 0xC5, 0xD0, 0x33, 0xF5, 0x4C, 0x43, 0xE5, 0x50, 0x0, 0x15, 0xC0, 0x43, 0x5, 0x51, 0x0, 0x5, 0x40, 0x44, 0x5, 0x80, 0x44, 0x15, 0xC0, 0x44, 0x15, 0x0, 0x45, 0x55, 0x91, 0x45, 0x15, 0xC0, 0x45, 0x5, 0x0, 0x46, 0x95, 0x91, 0x46, 0xB5, 0x51, 0x0, 0x5, 0x0, 0x47, 0x25, 0x4F, 0x47, 0x5, 0x80, 0x47, 0x5, 0xC0, 0x47, 0x5, 0x52, 0x48, 0x5, 0x40, 0x3B, 0x25, 0x12, 0x3E, 0x5, 0x0, 0x39, 0x35, 0x12, 0x49, 0x55, 0x92, 0x49, 0x75, 0x12, 0x4A, 0x95, 0x92, 0x4A, 0xB5, 0x12, 0x4B, 0xD5, 0x92, 0x4B, 0xF5, 0x12, 0x4C, 0x5, 0x40, 0x4C, 0x25, 0xD3, 0x4C, 0x45, 0x53, 0x4D, 0x65, 0xD3, 0x4D, 0x85, 0x53, 0x4C, 0x65, 0x53, 0x4E, 0xA5, 0xD3, 0x4E, 0xC5, 0x53, 0x4F, 0xE5, 0xD3, 0x4F, 0x5, 0x54, 0x50, 0x25, 0xD4, 0x50, 0x45, 0xD4, 0x38, 0x5, 0x0, 0x38, 0x15, 0x40, 0x51, 0x65, 0x94, 0x42, 0x5, 0xC0, 0x41, 0x75, 0x94, 0x39, 0x85, 0x54, 0x0, 0x15, 0x40, 0x52, 0x15, 0x91, 0x52, 0xB5, 0x14, 0x53, 0xD5, 0x94, 0x53, 0xF5, 0x54, 0x33, 0x5, 0x0, 0x54, 0x5, 0x0, 0x40, 0x15, 0x15, 0x0, 0x35, 0xC, 0x0, 0x26, 0xD5, 0x54, 0x46, 0x15, 0x0, 0x56, 0x55, 0x0, 0x66, 0x15, 0x0, 0x76, 0x15, 0x56, 0x96, 0x95, 0x56, 0xB6, 0x15, 0x57, 0xD6, 0x95, 0x57, 0xF6, 0x55, 0x0, 0x6, 0x56, 0x0, 0x16, 0x96, 0x58, 0x36, 0x16, 0x0, 0x6, 0x0, 0x59, 0x56, 0x96, 0x59, 0x76, 0x16, 0x5A, 0x96, 0x96, 0x5A, 0xB6, 0x16, 0x5B, 0xD6, 0x96, 0x5B, 0xF6, 0x16, 0x5C, 0x16, 0x97, 0x5C, 0x36, 0x17, 0x5D, 0x56, 0x97, 0x5D, 0x76, 0x17, 0x5E, 0x16, 0x40, 0x5E, 0x6, 0x40, 0x0, 0xA6, 0xD7, 0x5E, 0x16, 0x0, 0x5F, 0xD6, 0x97, 0x5F, 0xF6, 0x17, 0x60, 0x16, 0x98, 0x60, 0x36, 0x98, 0x5E, 0x6, 0x0, 0x61, 0x56, 0x18, 0x0, 0x66, 0xD8, 0x61, 0x36, 0x17, 0x62, 0x96, 0x98, 0x62, 0xB6, 0x18, 0x63, 0xD6, 0x98, 0x63, 0xF6, 0x18, 0x64, 0x16, 0x99, 0x64, 0x36, 0x19, 0x5D, 0x46, 0xD9, 0x5B, 0x56, 0x99, 0x65, 0x76, 0x19, 0x66, 0x96, 0x59, 0x0, 0x16, 0xC0, 0x5F, 0x6, 0x80, 0x66, 0xB6, 0x19, 0x67, 0xD6, 0x99, 0x67, 0xF6, 0x59, 0x0, 0x6, 0x0, 0x68, 0x16, 0x9A, 0x68, 0x6, 0xC0, 0x68, 0x46, 0x5A, 0x69, 0x66, 0xDA, 0x69, 0x86, 0x5A, 0x6A, 0xA6, 0xDA, 0x6A, 0xC6, 0x5A, 0x6B, 0xE6, 0xDA, 0x6B, 0x6, 0x5B, 0x6C, 0x26, 0xDB, 0x6C, 0x46, 0x5B, 0x6D, 0x66, 0xDB, 0x6D, 0x86, 0x5B, 0x6E, 0xA6, 0xDB, 0x6E, 0xC6, 0x5B, 0x6F, 0xE6, 0xDB, 0x6F, 0x6, 0x5C, 0x70, 0x26, 0xDC, 0x70, 0x46, 0x5C, 0x71, 0x66, 0x5C, 0x0, 0x76, 0x1C, 0x5D, 0x86, 0x1C, 0x0, 0x6, 0x40, 0x72, 0xA6, 0xDC, 0x72, 0x6, 0x0, 0x73, 0xD7, 0x9C, 0x73, 0xF7, 0x1C, 0x74, 0x17, 0x9D, 0x74, 0x37, 0x1D, 0x75, 0x57, 0x9D, 0x75, 0x77, 0x5D, 0x0, 0x87, 0x5D, 0x76, 0xA7, 0xDD, 0x76, 0xC7, 0x5D, 0x77, 0xE7, 0xDD, 0x77, 0x7, 0x1E, 0x0, 0x17, 0x9E, 0x78, 0x37, 0x5E, 0x0, 0x17, 0x0, 0x79, 0x57, 0x5E, 0x0, 0x67, 0xDE, 0x79, 0x87, 0x5E, 0x7A, 0xA7, 0xDE, 0x7A, 0x7, 0x0, 0x7B, 0xD7, 0x9E, 0x7B, 0xF7, 0x1E, 0x7C, 0x7, 0x40, 0x7C, 0x27, 0xDF, 0x7C, 0x47, 0x5F, 0x7D, 0x67, 0xDF, 0x7D, 0x87, 0x5F, 0x7E, 0x17, 0x80, 0x7E, 0xB7, 0x1F, 0x7F, 0xD7, 0x9F, 0x7F, 0xF7, 0x1F, 0x80, 0x17, 0xA0, 0x80, 0x37, 0x20, 0x81, 0x57, 0xA0, 0x81, 0x77, 0x20, 0x82, 0x97, 0xA0, 0x82, 0xB7, 0x20, 0x83, 0xD7, 0xA0, 0x83, 0xF7, 0x20, 0x84, 0x17, 0xA1, 0x84, 0x37, 0x21, 0x85, 0x57, 0x21, 0x0, 0x67, 0x21, 0x0, 0x77, 0x21, 0x0, 0x87, 0x21, 0x0, 0x98, 0xA1, 0x86, 0xB8, 0x61, 0x0, 0xC8, 0x61, 0x87, 0x18, 0x0, 0x0, 0xE8, 0x61, 0x0, 0xF8, 0x21, 0x88, 0x18, 0xA2, 0x88, 0x38, 0x22, 0x89, 0x18, 0x40, 0x89, 0x18, 0x80, 0x89, 0x78, 0x62, 0x0, 0x88, 0x62, 0x8A, 0xA8, 0x22, 0x0, 0xB8, 0x22, 0x0, 0x8, 0x0, 0x8B, 0xD8, 0xA2, 0x8B, 0xF8, 0x22, 0x8C, 0x18, 0xA3, 0x8C, 0x18, 0xC0, 0x8C, 0x48, 0x63, 0x8D, 0x18, 0x80, 0x8D, 0x78, 0x23, 0x8E, 0x98, 0xA3, 0x8E, 0xB8, 0x23, 0x8F, 0xD8, 0xA3, 0x8F, 0xF8, 0x23, 0x90, 0x18, 0x24, 0x0, 0x8, 0x80, 0x90, 0x38, 0x24, 0x91, 0x59, 0xA4, 0x91, 0x79, 0x24, 0x92, 0x99, 0x64, 0x0, 0xA9, 0xE4, 0x92, 0xC9, 0x64, 0x93, 0x19, 0x80, 0x93, 0x19, 0xC0, 0x93, 0x9, 0x0, 0x94, 0x9, 0x40, 0x94, 0x9, 0x80, 0x94, 0x9, 0xC0, 0x94, 0x49, 0x65, 0x95, 0x69, 0xE5, 0x95, 0x89, 0x65, 0x96, 0xA9, 0xE5, 0x96, 0xC9, 0x65, 0x97, 0xE9, 0xE5, 0x97, 0x9, 0x66, 0x98, 0x2A, 0xE6, 0x98, 0x4A, 0x66, 0x0, 0x1A, 0x40, 0x99, 0x6A, 0xE6, 0x99, 0x8A, 0x66, 0x0, 0x1A, 0x40, 0x9A, 0xAA, 0x26, 0x0, 0xA, 0xC0, 0x9A, 0xCA, 0x66, 0x9B, 0xEA, 0xE6, 0x9B, 0xA, 0x67, 0x9C, 0x2A, 0xE7, 0x9C, 0x4B, 0x27, 0x0, 0xB, 0x40, 0x9D, 0x6B, 0xE7, 0x9D, 0x8B, 0x67, 0x0, 0x1B, 0x40, 0x9E, 0xAB, 0xE7, 0x9E, 0xCB, 0x67, 0x9F, 0xEB, 0xE7, 0x9F, 0xC, 0x28, 0x0, 0x1C, 0x68, 0x0, 0x1C, 0x80, 0xA0, 0x1C, 0xC0, 0xA0, 0x4C, 0x68, 0xA1, 0x6C, 0xE8, 0xA1, 0x8D, 0x68, 0xA2, 0xAD, 0x68, 0x0, 0x1D, 0xC0, 0xA2, 0xCD, 0x68, 0xA3, 0xEE, 0xE8, 0xA3, 0xE, 0x69, 0xA4, 0x2F, 0xE9, 0xA4, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.NonSpacingMark.</summary>
        private static ReadOnlySpan<byte> SerializedCategory5_NonSpacingMark => new byte[] { 0x3, 0x4, 0x0, 0x0, 0xA, 0x0, 0x0, 0x0, 0x40, 0x0, 0x10, 0x0, 0x0, 0x21, 0x0, 0x0, 0x1, 0xC0, 0x0, 0x1, 0x80, 0x0, 0x1, 0x40, 0x0, 0x11, 0x0, 0x0, 0x31, 0x80, 0x0, 0x11, 0x80, 0x0, 0x31, 0x0, 0x0, 0x21, 0xC0, 0x0, 0x21, 0x40, 0x0, 0x11, 0xC0, 0x0, 0x31, 0x40, 0x0, 0x42, 0x0, 0x0, 0x52, 0x80, 0x1, 0x52, 0x0, 0x0, 0x72, 0x0, 0x0, 0x2, 0x40, 0x1, 0x82, 0x80, 0x1, 0x92, 0x80, 0x2, 0x62, 0x0, 0x0, 0x2, 0xC0, 0x2, 0xD2, 0x80, 0x3, 0x82, 0xC0, 0x2, 0x2, 0x40, 0x3, 0x2, 0xC0, 0x1, 0xB2, 0x0, 0x0, 0x72, 0x40, 0x0, 0x82, 0x0, 0x0, 0xC2, 0x0, 0x0, 0x12, 0x0, 0x0, 0x2, 0x0, 0x3, 0x72, 0x0, 0x2, 0x2, 0x80, 0x1, 0xE2, 0x40, 0x1, 0x62, 0x40, 0x0, 0x12, 0x80, 0x2, 0x12, 0x40, 0x3, 0x82, 0x40, 0x0, 0x32, 0xC0, 0x1, 0x72, 0xC0, 0x3, 0x2, 0x40, 0x0, 0x12, 0xC0, 0x2, 0x2, 0x0, 0x1, 0x62, 0x40, 0x3, 0xA2, 0x0, 0x0, 0xD2, 0x40, 0x0, 0xF2, 0x40, 0x0, 0x92, 0x40, 0x0, 0xE2, 0x80, 0x0, 0x2, 0x40, 0x2, 0x2, 0x80, 0x3, 0xB2, 0x40, 0x3, 0xF2, 0xC0, 0x2, 0x12, 0x80, 0x3, 0xF2, 0x0, 0x0, 0x2, 0x0, 0x2, 0x62, 0x0, 0x2, 0xE2, 0x0, 0x0, 0xC2, 0x0, 0x1, 0x62, 0x80, 0x3, 0x82, 0xC0, 0x1, 0xA2, 0x40, 0x0, 0x32, 0x0, 0x0, 0x12, 0x40, 0x1, 0x22, 0x0, 0x2, 0x62, 0xC0, 0x2, 0x92, 0x0, 0x0, 0x72, 0x80, 0x2, 0x12, 0x0, 0x2, 0xF2, 0x80, 0x3, 0x52, 0xC0, 0x1, 0x72, 0x40, 0x2, 0xB2, 0x0, 0x3, 0x62, 0x0, 0x1, 0x12, 0xC0, 0x3, 0x72, 0x80, 0x3, 0xE2, 0x40, 0x0, 0xF2, 0x0, 0x3, 0x3, 0x1, 0x0, 0x13, 0x81, 0x4, 0x3, 0x0, 0x4, 0x33, 0x1, 0x0, 0x3, 0x0, 0x5, 0x53, 0x81, 0x5, 0x73, 0x1, 0x0, 0x73, 0x1, 0x6, 0x3, 0x0, 0x3, 0x93, 0x1, 0x0, 0x3, 0x80, 0x4, 0x73, 0x80, 0x6, 0x3, 0xC0, 0x6, 0x3, 0x0, 0x7, 0x13, 0x40, 0x7, 0xE3, 0xC1, 0x7, 0xD3, 0x1, 0x7, 0x3, 0xC0, 0x4, 0x73, 0x1, 0x8, 0x83, 0x1, 0x1, 0xF3, 0x1, 0x0, 0xE3, 0x41, 0x8, 0xD3, 0x1, 0x0, 0x23, 0x2, 0x0, 0x33, 0x2, 0x0, 0x3, 0x0, 0x9, 0x13, 0x40, 0x9, 0x63, 0x42, 0x0, 0xC3, 0xC1, 0x5, 0x13, 0xC1, 0x9, 0x13, 0x0, 0xA, 0x3, 0x40, 0xA, 0x13, 0x2, 0x0, 0xA3, 0x2, 0x4, 0xB3, 0xC2, 0x8, 0x3, 0x0, 0xB, 0xD3, 0x2, 0x0, 0x3, 0x80, 0xB, 0xF3, 0x2, 0xC, 0x13, 0x43, 0x0, 0x23, 0x43, 0x0, 0x33, 0x43, 0xB, 0x63, 0x2, 0xD, 0x13, 0x3, 0x4, 0x53, 0x43, 0x7, 0x63, 0x83, 0xB, 0x43, 0xC2, 0xD, 0x43, 0x1, 0x0, 0x3, 0x0, 0x8, 0x83, 0x3, 0x0, 0x23, 0x1, 0xB, 0x93, 0x3, 0x4, 0xE3, 0x81, 0xE, 0x3, 0xC0, 0xE, 0x3, 0x0, 0xF, 0xD3, 0x3, 0x0, 0x23, 0x1, 0x0, 0x23, 0x81, 0xF, 0x3, 0xC0, 0xF, 0x3, 0x4, 0x0, 0xC3, 0xC0, 0x9, 0x93, 0x43, 0x0, 0xD3, 0x43, 0x10, 0x13, 0x43, 0xE, 0x43, 0x81, 0x10, 0xC3, 0x1, 0x0, 0x13, 0x3, 0x0, 0x63, 0xC2, 0x10, 0x13, 0x43, 0x7, 0xB3, 0xC1, 0x10, 0x3, 0x0, 0x11, 0xE3, 0x2, 0x0, 0x23, 0xC1, 0xF, 0x53, 0x4, 0x0, 0x33, 0x1, 0x4, 0x63, 0x4, 0x0, 0x73, 0x44, 0x5, 0x3, 0x3, 0x0, 0x3, 0x80, 0x1, 0x23, 0x1, 0x9, 0x83, 0x4, 0x0, 0x13, 0x41, 0x12, 0x3, 0x80, 0x8, 0x3, 0x80, 0x12, 0xB3, 0x4, 0x13, 0x3, 0x40, 0xE, 0x13, 0x41, 0xE, 0xD3, 0x4, 0x0, 0x13, 0x80, 0x13, 0x13, 0x0, 0x0, 0x63, 0x3, 0x0, 0xF3, 0x4, 0x14, 0xE3, 0xC4, 0x7, 0x13, 0xC0, 0x7, 0xB3, 0x41, 0x0, 0x73, 0xC4, 0xC, 0x33, 0xC3, 0x7, 0x3, 0x40, 0x14, 0x13, 0x40, 0xE, 0x3, 0x40, 0xB, 0x4, 0x40, 0x0, 0x24, 0x5, 0x0, 0x4, 0xC0, 0x14, 0x44, 0x45, 0x15, 0x4, 0x80, 0x15, 0x74, 0x5, 0x0, 0x84, 0x5, 0x0, 0x4, 0x40, 0x16, 0xA4, 0xC5, 0x16, 0x4, 0x0, 0x17, 0xD4, 0x5, 0x0, 0x4, 0x80, 0x17, 0xF4, 0x5, 0x18, 0x4, 0x40, 0x18, 0x24, 0x46, 0x0, 0x4, 0xC0, 0x18, 0x4, 0x0, 0x19, 0x4, 0x40, 0x19, 0xF4, 0x5, 0x0, 0x64, 0x6, 0x0, 0x74, 0x6, 0x1A, 0x94, 0x6, 0x0, 0x4, 0x80, 0x1A, 0x84, 0x6, 0x0, 0xF4, 0x5, 0x1A, 0xB4, 0x6, 0x1B, 0xD4, 0x6, 0x0, 0xE4, 0xC6, 0x1B, 0x4, 0x7, 0x0, 0x14, 0x87, 0x1C, 0xF4, 0xC5, 0x1C, 0x4, 0x0, 0x1D, 0x4, 0xC0, 0x17, 0x54, 0x87, 0x1D, 0x4, 0xC0, 0x1D, 0x84, 0x7, 0x0, 0x4, 0x40, 0x1D, 0x4, 0x40, 0x1E, 0xA4, 0x7, 0x0, 0xB4, 0x7, 0x1F, 0xD4, 0x7, 0x0, 0xE4, 0x7, 0x0, 0xF4, 0x7, 0x20, 0x4, 0x40, 0x20, 0x4, 0x80, 0x20, 0x4, 0xC0, 0x20, 0x44, 0x48, 0x21, 0x64, 0x8, 0x0, 0x74, 0x8, 0x0, 0x84, 0x8, 0x0, 0x94, 0x8, 0x0, 0xA4, 0xC8, 0x22, 0xC4, 0xC8, 0x1A, 0xD4, 0x8, 0x0, 0xE4, 0x48, 0x22, 0x4, 0x0, 0x15, 0xD4, 0x46, 0x0, 0xF4, 0x8, 0x24, 0x14, 0x9, 0x0, 0x24, 0x9, 0x0, 0x34, 0x9, 0x0, 0x4, 0x0, 0x25, 0x54, 0x9, 0x0, 0x4, 0x80, 0x25, 0x74, 0x9, 0x0, 0x84, 0x49, 0x26, 0x4, 0xC0, 0x21, 0x4, 0x80, 0x26, 0xB4, 0x9, 0x0, 0x4, 0x0, 0x27, 0xD4, 0x9, 0x0, 0x24, 0x88, 0x27, 0xF4, 0x9, 0x0, 0x4, 0x0, 0x28, 0x4, 0x40, 0x28, 0x24, 0x88, 0x26, 0x24, 0xCA, 0x21, 0x4, 0xC0, 0x28, 0x4, 0x0, 0x29, 0x54, 0xA, 0x0, 0x64, 0x85, 0x29, 0x24, 0xC5, 0x21, 0x74, 0xA, 0x2A, 0x94, 0xA, 0x0, 0x14, 0x80, 0x2A, 0x14, 0x80, 0x28, 0xB4, 0xA, 0x0, 0xC4, 0xA, 0x0, 0x4, 0x40, 0x2B, 0xE4, 0xA, 0x0, 0x44, 0x88, 0x1D, 0xB4, 0xC6, 0x2B, 0x4, 0x0, 0x2C, 0x14, 0x0, 0x0, 0x64, 0x5, 0x0, 0x4, 0x40, 0x2C, 0x24, 0xB, 0x0, 0xB4, 0x6, 0x0, 0x14, 0x80, 0x1D, 0x4, 0xB, 0x0, 0x4, 0xC0, 0x2C, 0xA4, 0x47, 0x0, 0x44, 0xB, 0x0, 0x4, 0x40, 0x2D, 0x5, 0xC0, 0x2D, 0x85, 0xB, 0x0, 0x95, 0x8B, 0x2E, 0xB5, 0xB, 0x0, 0xC5, 0x4B, 0x2F, 0xE5, 0xB, 0x0, 0xF5, 0xB, 0x0, 0x5, 0x4C, 0x30, 0x5, 0x80, 0x30, 0x35, 0xC, 0x0, 0x45, 0x4C, 0x31, 0x65, 0xCC, 0x31, 0x85, 0xC, 0x0, 0x5, 0x40, 0x32, 0xA5, 0xC, 0x0, 0x5, 0xC0, 0x32, 0xC5, 0xC, 0x0, 0x15, 0x0, 0x0, 0xD5, 0xC, 0x0, 0xE5, 0xC, 0x0, 0xF5, 0xC, 0x34, 0x15, 0x8D, 0x34, 0x35, 0xD, 0x0, 0x45, 0xD, 0x0, 0x55, 0x8D, 0x35, 0x75, 0xD, 0x36, 0x95, 0x8D, 0x36, 0x5, 0xC0, 0x36, 0xC5, 0xD, 0x0, 0xD5, 0x8D, 0x37, 0x5, 0xC0, 0x37, 0x5, 0xE, 0x0, 0x15, 0x8E, 0x38, 0x5, 0xC0, 0x38, 0x5, 0x0, 0x39, 0x55, 0xE, 0x0, 0x65, 0xCE, 0x39, 0x5, 0x0, 0x3A, 0x5, 0x40, 0x3A, 0xA5, 0xCE, 0x3A, 0xC5, 0xE, 0x0, 0x5, 0x40, 0x3B, 0xE5, 0xCE, 0x3B, 0x5, 0xF, 0x0, 0x15, 0x8F, 0x3C, 0x5, 0xC0, 0x3C, 0x45, 0xF, 0x0, 0x5, 0x40, 0x3D, 0x65, 0xF, 0x0, 0x5, 0xC0, 0x3D, 0x5, 0x80, 0x2E, 0x85, 0x4F, 0x3E, 0xA5, 0x8F, 0x35, 0x85, 0xCF, 0x3E, 0xC5, 0x8F, 0x2E, 0x85, 0x4F, 0x3F, 0xE5, 0xCF, 0x3F, 0x5, 0x0, 0x40, 0x5, 0x0, 0x36, 0x85, 0x4F, 0x40, 0x25, 0xD0, 0x40, 0xC5, 0xB, 0x41, 0x55, 0x90, 0x41, 0x75, 0x50, 0x3E, 0xC5, 0x8B, 0x2E, 0x85, 0xF, 0x42, 0x95, 0x50, 0x30, 0xA5, 0xD0, 0x42, 0x5, 0x0, 0x43, 0x5, 0x40, 0x43, 0xE5, 0xD0, 0x43, 0x5, 0x11, 0x0, 0x15, 0x11, 0x0, 0x5, 0x80, 0x44, 0x35, 0x11, 0x45, 0x55, 0x91, 0x45, 0x75, 0x11, 0x46, 0x5, 0x40, 0x46, 0x5, 0x80, 0x46, 0xB5, 0x11, 0x47, 0x5, 0x40, 0x47, 0x65, 0x4B, 0x0, 0x6, 0x80, 0x2D, 0x6, 0x80, 0x47, 0xF6, 0x11, 0x0, 0x6, 0x52, 0x48, 0x26, 0xD2, 0x48, 0x46, 0x52, 0x49, 0x66, 0xD2, 0x49, 0x86, 0x12, 0x0, 0x6, 0x40, 0x4A, 0xA6, 0xD2, 0x4A, 0xC6, 0x12, 0x0, 0x6, 0x40, 0x4B, 0x6, 0x80, 0x4B, 0xF6, 0x12, 0x0, 0x6, 0x13, 0x0, 0x16, 0x13, 0x0, 0x26, 0x13, 0x0, 0x16, 0x0, 0x0, 0x36, 0x13, 0x0, 0x6, 0x0, 0x4D, 0x56, 0x93, 0x4D, 0x76, 0x13, 0x4E, 0x96, 0x93, 0x4E, 0xB6, 0x13, 0x4F, 0x6, 0x40, 0x4F, 0x6, 0x80, 0x4F, 0x6, 0xC0, 0x4F, 0x6, 0x54, 0x50, 0x66, 0x8E, 0x50, 0x36, 0x14, 0x0, 0x6, 0x0, 0x51, 0x56, 0x94, 0x51, 0x76, 0x14, 0x52, 0x96, 0x94, 0x52, 0xB6, 0x14, 0x53, 0xD6, 0x94, 0x53, 0xF6, 0x14, 0x54, 0x16, 0x95, 0x54, 0x36, 0x15, 0x55, 0x56, 0x95, 0x55, 0x76, 0x15, 0x56, 0x96, 0x15, 0x55, 0xA6, 0xD5, 0x56, 0xC6, 0xD5, 0x56, 0xD6, 0x95, 0x57, 0xF6, 0x15, 0x58, 0x16, 0x96, 0x58, 0x36, 0x16, 0x59, 0x56, 0x96, 0x59, 0x76, 0x16, 0x5A, 0x96, 0x16, 0x0, 0xA6, 0xD6, 0x5A, 0xC6, 0x56, 0x5B, 0x6, 0x80, 0x5B, 0xF6, 0x56, 0x0, 0x7, 0x0, 0x5C, 0x7, 0x40, 0x5C, 0x27, 0x17, 0x0, 0x37, 0x17, 0x5D, 0x57, 0x97, 0x5D, 0x77, 0x17, 0x5E, 0x97, 0x97, 0x5E, 0xB7, 0x17, 0x5F, 0xD7, 0x97, 0x5F, 0xF7, 0x17, 0x0, 0x7, 0x18, 0x0, 0x17, 0x18, 0x0, 0x27, 0xD8, 0x60, 0x47, 0x58, 0x61, 0x67, 0xD8, 0x61, 0x7, 0x0, 0x62, 0x97, 0x98, 0x62, 0xB7, 0x18, 0x63, 0x7, 0x40, 0x63, 0xE7, 0xD8, 0x63, 0x7, 0x59, 0x64, 0x27, 0xD9, 0x64, 0x47, 0x59, 0x65, 0x67, 0xD9, 0x65, 0x87, 0x59, 0x66, 0xA7, 0xD9, 0x66, 0xC7, 0x59, 0x67, 0xE7, 0xD9, 0x67, 0x7, 0x5A, 0x68, 0x27, 0xDA, 0x68, 0x47, 0x1A, 0x0, 0x57, 0x1A, 0x0, 0x7, 0x80, 0x69, 0x8, 0xC0, 0x69, 0x88, 0x1A, 0x0, 0x98, 0x9A, 0x6A, 0xB8, 0x1A, 0x6B, 0x8, 0x40, 0x6B, 0x8, 0x80, 0x6B, 0xF8, 0x1A, 0x6C, 0x8, 0x40, 0x6C, 0x28, 0xDB, 0x6C, 0x48, 0x5B, 0x6D, 0x68, 0xDB, 0x6D, 0x88, 0x1B, 0x0, 0x98, 0x1B, 0x0, 0x8, 0x80, 0x6E, 0xB8, 0x1B, 0x6F, 0xD8, 0x9B, 0x6F, 0xF8, 0x1B, 0x70, 0x18, 0x9C, 0x70, 0x38, 0x1C, 0x71, 0x58, 0x9C, 0x71, 0x78, 0x1C, 0x0, 0x89, 0x1C, 0x0, 0x99, 0x1C, 0x0, 0xA9, 0xDC, 0x72, 0xC9, 0x1C, 0x0, 0x9, 0x40, 0x73, 0x9, 0x80, 0x73, 0x9, 0xC0, 0x73, 0x9, 0x0, 0x74, 0x19, 0x9D, 0x74, 0x39, 0x1D, 0x0, 0x49, 0x5D, 0x75, 0x69, 0xDD, 0x75, 0x89, 0x5D, 0x76, 0xA9, 0xDD, 0x76, 0xC9, 0x1D, 0x0, 0xDA, 0x9D, 0x77, 0xA, 0xC0, 0x77, 0xA, 0x1E, 0x0, 0xA, 0x40, 0x78, 0x2A, 0x1E, 0x0, 0xA, 0xC0, 0x78, 0x4A, 0x5E, 0x79, 0x6A, 0xDE, 0x79, 0x8A, 0x5E, 0x7A, 0xAA, 0xDE, 0x7A, 0xCB, 0x1E, 0x0, 0xDB, 0x9E, 0x7B, 0xB, 0xC0, 0x7B, 0xB, 0x5F, 0x7C, 0x2B, 0xDF, 0x7C, 0x4B, 0x5F, 0x7D, 0x6C, 0x1F, 0x0, 0xC, 0xC0, 0x7D, 0x8C, 0x5F, 0x7E, 0xAC, 0xDF, 0x7E, 0xCD, 0x1F, 0x0, 0xDD, 0x1F, 0x0, 0xED, 0xDF, 0x7F, 0xE, 0x60, 0x80, 0xE, 0x80, 0x80, 0x3F, 0x20, 0x81, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.SpacingCombiningMark.</summary>
        private static ReadOnlySpan<byte> SerializedCategory6_SpacingCombiningMark => new byte[] { 0x3, 0x4, 0x0, 0x0, 0x9, 0x0, 0x0, 0x0, 0x20, 0x0, 0x10, 0x0, 0x0, 0x1, 0x40, 0x0, 0x21, 0x60, 0x0, 0x11, 0x40, 0x0, 0x31, 0x0, 0x0, 0x1, 0x60, 0x0, 0x11, 0x0, 0x0, 0x31, 0x40, 0x0, 0x1, 0x20, 0x0, 0x21, 0x0, 0x0, 0x31, 0x20, 0x0, 0x11, 0x60, 0x0, 0x21, 0x20, 0x0, 0x42, 0xA0, 0x0, 0x62, 0xE0, 0x0, 0x82, 0x0, 0x0, 0x92, 0xE0, 0x0, 0x82, 0xE0, 0x0, 0x42, 0x40, 0x1, 0x72, 0x0, 0x0, 0x2, 0x80, 0x0, 0xB2, 0x0, 0x0, 0x2, 0xE0, 0x0, 0x2, 0x20, 0x1, 0x2, 0x20, 0x0, 0x12, 0x0, 0x0, 0x2, 0x60, 0x1, 0xA2, 0xE0, 0x0, 0x92, 0x0, 0x0, 0x2, 0x0, 0x1, 0x22, 0x20, 0x1, 0x2, 0x80, 0x1, 0x92, 0x0, 0x1, 0x42, 0xA0, 0x1, 0xE2, 0xE0, 0x0, 0x42, 0x0, 0x0, 0x2, 0xE0, 0x1, 0xE2, 0x0, 0x0, 0x42, 0x60, 0x0, 0x32, 0x0, 0x0, 0x2, 0xA0, 0x0, 0x12, 0xA0, 0x1, 0x2, 0xC0, 0x1, 0xF2, 0xE0, 0x0, 0xD2, 0x20, 0x0, 0xC2, 0x0, 0x0, 0x42, 0x20, 0x1, 0xA2, 0x20, 0x0, 0xB2, 0x20, 0x0, 0xA2, 0x20, 0x1, 0x42, 0xE0, 0x0, 0x42, 0xC0, 0x0, 0x92, 0x80, 0x0, 0x52, 0x0, 0x0, 0x2, 0xC0, 0x0, 0x42, 0xC0, 0x1, 0x92, 0xA0, 0x0, 0x72, 0x80, 0x0, 0x62, 0xC0, 0x1, 0x3, 0x21, 0x2, 0x3, 0x40, 0x2, 0x33, 0x1, 0x0, 0x43, 0x1, 0x0, 0x23, 0x1, 0x0, 0x3, 0xA0, 0x2, 0x63, 0x1, 0x0, 0x3, 0xE0, 0x2, 0x93, 0x0, 0x3, 0x3, 0x20, 0x3, 0x3, 0x40, 0x3, 0x3, 0x60, 0x3, 0x13, 0x80, 0x3, 0x3, 0xA0, 0x3, 0x3, 0xC0, 0x3, 0xF3, 0x1, 0x0, 0x3, 0xC0, 0x2, 0x3, 0x0, 0x4, 0x3, 0x0, 0x3, 0xB3, 0x81, 0x3, 0x13, 0xC2, 0x2, 0x23, 0x62, 0x4, 0x3, 0x40, 0x4, 0x3, 0x80, 0x4, 0x53, 0x42, 0x2, 0x3, 0xC0, 0x4, 0x3, 0xE0, 0x4, 0x83, 0x22, 0x5, 0x3, 0x40, 0x5, 0xB3, 0x2, 0x0, 0x73, 0x81, 0x5, 0xD3, 0xC2, 0x5, 0x73, 0xE1, 0x5, 0xF3, 0x1, 0x6, 0x13, 0x3, 0x0, 0x23, 0xC3, 0x3, 0x33, 0x83, 0x6, 0x3, 0xE0, 0x3, 0x53, 0x1, 0x4, 0x53, 0x3, 0x0, 0x13, 0xA0, 0x3, 0x63, 0xE3, 0x6, 0x3, 0x0, 0x7, 0x93, 0x43, 0x6, 0x3, 0x3, 0x0, 0x3, 0x40, 0x7, 0x3, 0xA0, 0x5, 0x63, 0x63, 0x7, 0x53, 0x81, 0x7, 0x93, 0xE2, 0x2, 0xD3, 0xE3, 0x2, 0x4, 0xC0, 0x7, 0xF4, 0x3, 0x8, 0x14, 0x4, 0x0, 0x4, 0x40, 0x8, 0x34, 0x84, 0x8, 0x4, 0xA0, 0x8, 0x64, 0x4, 0x0, 0x4, 0xE0, 0x8, 0x84, 0x4, 0x0, 0x4, 0x20, 0x9, 0xA4, 0x4, 0x0, 0x4, 0x60, 0x9, 0x4, 0x80, 0x9, 0x4, 0xA0, 0x9, 0xE4, 0xE4, 0x9, 0x4, 0x25, 0xA, 0x84, 0x44, 0xA, 0x4, 0x60, 0xA, 0x4, 0x80, 0xA, 0x4, 0xA0, 0xA, 0x64, 0x5, 0x0, 0x4, 0xE0, 0xA, 0x84, 0x25, 0xB, 0xA4, 0x5, 0x0, 0xB4, 0x5, 0x0, 0xC4, 0xA5, 0xB, 0x4, 0xC0, 0xB, 0xF4, 0x5, 0x0, 0x74, 0x5, 0x0, 0xF4, 0x3, 0x0, 0x4, 0x26, 0xC, 0x4, 0x40, 0xC, 0x34, 0x6, 0x0, 0x44, 0xA6, 0xC, 0x44, 0x4, 0x0, 0xD4, 0x4, 0x0, 0x64, 0x86, 0x8, 0x4, 0x0, 0x9, 0xE4, 0xE4, 0xC, 0x74, 0x4, 0x0, 0x84, 0x26, 0xD, 0xA4, 0x6, 0x0, 0x4, 0x60, 0xD, 0x4, 0x80, 0xD, 0xE4, 0xA4, 0xD, 0xE4, 0xC4, 0xD, 0x4, 0xE0, 0xD, 0x4, 0x0, 0xE, 0x4, 0x4, 0x0, 0x15, 0x7, 0x0, 0x25, 0x7, 0x0, 0x35, 0x87, 0xE, 0x55, 0x7, 0x0, 0x5, 0xC0, 0xE, 0x75, 0x7, 0xF, 0x5, 0x20, 0xF, 0x5, 0x40, 0xF, 0xB5, 0x87, 0xF, 0xD5, 0x7, 0x0, 0xE5, 0x7, 0x0, 0xF5, 0x7, 0x0, 0x5, 0x8, 0x0, 0x15, 0x8, 0x0, 0x25, 0x68, 0x10, 0x5, 0x80, 0x10, 0x55, 0xC8, 0x10, 0x75, 0x8, 0x11, 0x5, 0x20, 0x11, 0xA5, 0x8, 0x0, 0x5, 0x60, 0x11, 0xC5, 0x8, 0x0, 0xD5, 0xC8, 0x11, 0x5, 0xE0, 0x11, 0x5, 0x29, 0x12, 0x25, 0x9, 0x0, 0x35, 0x9, 0x0, 0x45, 0x9, 0x0, 0x95, 0xA7, 0x12, 0x5, 0xC0, 0x12, 0x5, 0xE0, 0x12, 0x45, 0xC9, 0x12, 0x85, 0x29, 0x13, 0xA5, 0xC9, 0x12, 0x5, 0x60, 0x13, 0x5, 0x80, 0x13, 0x5, 0xA0, 0x13, 0x5, 0xC0, 0x13, 0x5, 0xE0, 0x13, 0x45, 0x9, 0xF, 0x5, 0x0, 0x14, 0x15, 0xA, 0xF, 0x26, 0xA, 0x0, 0x36, 0xA, 0x0, 0x46, 0xAA, 0x14, 0x66, 0xEA, 0x14, 0x86, 0xA, 0x0, 0x96, 0x4A, 0x15, 0x6, 0x60, 0x15, 0x6, 0x80, 0x15, 0xD6, 0xA, 0x0, 0x6, 0xC0, 0x15, 0xF6, 0xA, 0x16, 0x16, 0x4B, 0x16, 0x36, 0x8B, 0x16, 0x6, 0xA0, 0x16, 0x66, 0xEB, 0x16, 0x6, 0x0, 0x17, 0x6, 0x20, 0x17, 0xA6, 0x6B, 0x17, 0xC6, 0xAB, 0x17, 0xE6, 0xEB, 0x17, 0x6, 0x2C, 0x18, 0x26, 0x6C, 0x18, 0x46, 0xAC, 0x18, 0x66, 0xAC, 0x17, 0x76, 0x6C, 0x18, 0x86, 0x2C, 0x19, 0x66, 0x2A, 0x19, 0x76, 0x2C, 0x18, 0xA6, 0x6C, 0x19, 0xC7, 0xC, 0x0, 0xD7, 0xCC, 0x19, 0xF7, 0xC, 0x1A, 0x17, 0x4D, 0x1A, 0x7, 0x60, 0x1A, 0x47, 0xAD, 0x1A, 0x67, 0xED, 0x1A, 0x7, 0x0, 0x1B, 0x7, 0x20, 0x1B, 0xA7, 0x6D, 0x1B, 0xC7, 0xAD, 0x1B, 0x7, 0xC0, 0x1B, 0xF7, 0xD, 0x1C, 0x17, 0x4E, 0x1C, 0x37, 0x8E, 0x1C, 0x57, 0xCE, 0x1C, 0x77, 0xE, 0x1D, 0x98, 0x4E, 0x1D, 0xB8, 0x8E, 0x1D, 0x8, 0xA0, 0x1D, 0x8, 0xC0, 0x1D, 0xF8, 0xE, 0x1E, 0x18, 0xF, 0x0, 0x28, 0xF, 0x0, 0x8, 0x60, 0x1E, 0x48, 0xF, 0x0, 0x58, 0xCF, 0x1E, 0x78, 0xF, 0x1F, 0x98, 0xF, 0x0, 0xA9, 0x6F, 0x1F, 0x9, 0x80, 0x1F, 0x9, 0xA0, 0x1F, 0xE9, 0xEF, 0x1F, 0x9, 0x10, 0x0, 0x9, 0x20, 0x20, 0x29, 0x70, 0x20, 0x49, 0xB0, 0x20, 0xA, 0xC0, 0x20, 0xA, 0xE0, 0x20, 0x8A, 0x30, 0x21, 0xAA, 0x70, 0x21, 0xCA, 0xB0, 0x21, 0xEB, 0x10, 0x0, 0xB, 0xE0, 0x21, 0xB, 0x31, 0x22, 0x2B, 0x11, 0x0, 0xC, 0x60, 0x22, 0x4C, 0x11, 0x0, 0x5C, 0xD1, 0x22, 0x7D, 0x11, 0x0, 0x8D, 0x31, 0x23, 0xE, 0x40, 0x23, 0xE, 0x60, 0x23, 0xCF, 0xB1, 0x23, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.EnclosingMark.</summary>
        private static ReadOnlySpan<byte> SerializedCategory7_EnclosingMark => new byte[] { 0x2, 0x4, 0x0, 0x6, 0x0, 0x0, 0x4, 0x10, 0x0, 0x21, 0x4, 0x1, 0x8, 0x11, 0x8, 0x11, 0xC, 0x21, 0x0, 0x1, 0x4, 0x2, 0x10, 0x52, 0x18, 0x72, 0x0, 0x82, 0x0, 0x2, 0x24, 0x3, 0x28, 0x3, 0x2C, 0xC3, 0x0, 0xD3, 0x0, 0xE3, 0x0, 0xF4, 0x0, 0x4, 0x40, 0x14, 0x1, 0x24, 0x1, 0x4, 0x4C, 0x45, 0x1, 0x55, 0x59, 0x75, 0x1, 0x5, 0x60, 0x96, 0x1, 0xA6, 0x1, 0x6, 0x6C, 0x6, 0x70, 0x7, 0x74, 0xE7, 0x1, 0xF7, 0x1, 0x7, 0x2, 0x8, 0x84, 0x8, 0x88, 0x8, 0x8C, 0x8, 0x90, 0x59, 0x2, 0x9, 0x98, 0x79, 0x2, 0x9, 0xA0, 0x9A, 0x2, 0xA, 0xA8, 0xA, 0xAC, 0xCA, 0x2, 0xB, 0xB4, 0xB, 0xB8, 0xFB, 0x2, 0xB, 0xC0, 0xC, 0xC4, 0xC, 0xC8, 0x3C, 0xD3, 0x5D, 0x3, 0x6D, 0xDF, 0xE, 0xE0, 0xE, 0xE4, 0xAF, 0xEF, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.DecimalDigitNumber.</summary>
        private static ReadOnlySpan<byte> SerializedCategory8_DecimalDigitNumber => new byte[] { 0x3, 0x4, 0x0, 0x0, 0x7, 0x0, 0x0, 0x1, 0x8, 0x0, 0x11, 0x0, 0x0, 0x2, 0x10, 0x0, 0x32, 0x0, 0x0, 0x43, 0x8, 0x0, 0x13, 0x28, 0x0, 0x64, 0x0, 0x0, 0x4, 0x30, 0x0, 0x4, 0x38, 0x0, 0x5, 0x40, 0x0, 0x85, 0x0, 0x0, 0x5, 0x48, 0x0, 0x95, 0x0, 0x0, 0x5, 0x30, 0x0, 0x5, 0x50, 0x0, 0xA5, 0x0, 0x0, 0x6, 0x58, 0x0, 0xC6, 0x0, 0x0, 0xB6, 0x0, 0x0, 0x86, 0x0, 0x0, 0x6, 0x68, 0x0, 0x6, 0x70, 0x0, 0xF6, 0x0, 0x0, 0x6, 0x60, 0x0, 0x6, 0x78, 0x0, 0x6, 0x1, 0x0, 0xE6, 0x0, 0x0, 0xD6, 0x0, 0x0, 0x16, 0x1, 0x0, 0x7, 0x90, 0x0, 0x37, 0x1, 0x0, 0x7, 0xA0, 0x0, 0x57, 0xB1, 0x0, 0x47, 0x1, 0x0, 0x7, 0xB8, 0x0, 0x7, 0xC0, 0x0, 0x97, 0xA1, 0x0, 0xA7, 0x1, 0x0, 0x47, 0xD9, 0x0, 0xC7, 0x1, 0x0, 0x27, 0xE9, 0x0, 0xD7, 0x1, 0x0, 0x37, 0xE1, 0x0, 0x7, 0xC8, 0x0, 0xF8, 0x1, 0x0, 0x8, 0xA, 0x1, 0x28, 0x1A, 0x1, 0x8, 0x20, 0x1, 0x8, 0x28, 0x1, 0x68, 0x3A, 0x1, 0x88, 0xFA, 0x0, 0x98, 0x2, 0x0, 0x8, 0x50, 0x1, 0x48, 0xA2, 0x0, 0xE8, 0x1, 0x0, 0xB8, 0x62, 0x1, 0x8, 0x68, 0x1, 0xE9, 0x2, 0x0, 0xF9, 0x82, 0x1, 0x19, 0x3, 0x0, 0x9, 0x90, 0x1, 0x39, 0xA3, 0x1, 0x59, 0x3, 0x0, 0x9, 0xB0, 0x1, 0x79, 0xF3, 0x0, 0xE9, 0xC1, 0x1, 0x99, 0x3, 0x0, 0x9, 0xD0, 0x1, 0xBA, 0x3, 0x0, 0xA, 0xE0, 0x1, 0xDA, 0x3, 0x0, 0xEA, 0xFB, 0x1, 0xA, 0xC, 0x2, 0x2A, 0x1C, 0x2, 0x4A, 0x2C, 0x2, 0x6B, 0x4, 0x0, 0x7B, 0x44, 0x2, 0x9B, 0x54, 0x2, 0xBB, 0x64, 0x2, 0xDC, 0x4, 0x0, 0xC, 0x70, 0x2, 0xFC, 0x84, 0x2, 0x1D, 0x5, 0x0, 0x2D, 0x5, 0x0, 0xD, 0x98, 0x2, 0x4E, 0xAD, 0x2, 0xE, 0xB0, 0x2, 0x7F, 0xC5, 0x2, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.LetterNumber.</summary>
        private static ReadOnlySpan<byte> SerializedCategory9_LetterNumber => new byte[] { 0x3, 0x4, 0x0, 0x0, 0x7, 0x0, 0x0, 0x0, 0x8, 0x0, 0x10, 0x0, 0x0, 0x11, 0x0, 0x0, 0x21, 0x8, 0x0, 0x1, 0x8, 0x0, 0x11, 0x18, 0x0, 0x31, 0x0, 0x0, 0x1, 0x10, 0x0, 0x42, 0x0, 0x0, 0x2, 0x28, 0x0, 0x2, 0x30, 0x0, 0x12, 0x38, 0x0, 0x82, 0x0, 0x0, 0x2, 0x48, 0x0, 0x72, 0x28, 0x0, 0x13, 0x50, 0x0, 0xB3, 0x0, 0x0, 0xC3, 0x68, 0x0, 0x3, 0x70, 0x0, 0xF3, 0x80, 0x0, 0x3, 0x78, 0x0, 0xA3, 0x0, 0x0, 0x4, 0x88, 0x0, 0x24, 0x99, 0x0, 0x4, 0xA0, 0x0, 0x4, 0xA8, 0x0, 0x64, 0xB9, 0x0, 0x85, 0x1, 0x0, 0x95, 0xD1, 0x0, 0x5, 0xD8, 0x0, 0x15, 0x0, 0x0, 0xC5, 0x1, 0x0, 0xD6, 0x1, 0x0, 0x6, 0xF0, 0x0, 0x6, 0xF8, 0x0, 0x6, 0x2, 0x0, 0x16, 0x2, 0x0, 0x27, 0x2, 0x0, 0x7, 0x18, 0x1, 0x47, 0x2A, 0x1, 0x67, 0x2, 0x0, 0x8, 0x38, 0x1, 0x8, 0x40, 0x1, 0x98, 0x2, 0x0, 0x8, 0x50, 0x1, 0xB9, 0x2, 0x0, 0x9, 0x60, 0x1, 0x9, 0x68, 0x1, 0xE9, 0x2, 0x0, 0xFA, 0x2, 0x0, 0xA, 0x80, 0x1, 0xA, 0x88, 0x1, 0x2A, 0x3, 0x0, 0xB, 0x98, 0x1, 0xB, 0xA0, 0x1, 0xB, 0xA8, 0x1, 0xB, 0xB0, 0x1, 0xC, 0xB8, 0x1, 0x8C, 0xCB, 0x1, 0xAC, 0x3, 0x0, 0xBD, 0x3, 0x0, 0xCD, 0xEB, 0x1, 0xE, 0xF0, 0x1, 0xE, 0xF8, 0x1, 0xF, 0xC, 0x2, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.OtherNumber.</summary>
        private static ReadOnlySpan<byte> SerializedCategory10_OtherNumber => new byte[] { 0x3, 0x4, 0x0, 0x0, 0x8, 0x0, 0x0, 0x10, 0x0, 0x0, 0x0, 0x10, 0x0, 0x1, 0x10, 0x0, 0x11, 0x20, 0x0, 0x11, 0x0, 0x0, 0x1, 0x20, 0x0, 0x1, 0x30, 0x0, 0x31, 0x0, 0x0, 0x31, 0x10, 0x0, 0x42, 0x10, 0x0, 0x12, 0x50, 0x0, 0x2, 0x40, 0x0, 0x42, 0x60, 0x0, 0x72, 0x0, 0x0, 0x2, 0x10, 0x0, 0x62, 0x0, 0x0, 0x12, 0x60, 0x0, 0x2, 0x70, 0x0, 0x12, 0x80, 0x0, 0x2, 0x90, 0x0, 0x82, 0x10, 0x0, 0x2, 0x80, 0x0, 0xA2, 0x10, 0x0, 0x2, 0xA0, 0x0, 0x12, 0x0, 0x0, 0xA2, 0x70, 0x0, 0x2, 0x60, 0x0, 0x3, 0xB0, 0x0, 0x13, 0xC0, 0x0, 0xD3, 0x10, 0x0, 0x13, 0x0, 0x0, 0x3, 0xE0, 0x0, 0xF3, 0x0, 0x0, 0x3, 0x0, 0x1, 0x13, 0x10, 0x1, 0x23, 0x1, 0x0, 0x3, 0x11, 0x0, 0x33, 0x1, 0x0, 0xD3, 0x40, 0x1, 0x53, 0x1, 0x0, 0x63, 0x11, 0x0, 0xC3, 0x0, 0x0, 0x73, 0x11, 0x0, 0x83, 0x1, 0x0, 0x3, 0x90, 0x1, 0x3, 0x20, 0x1, 0xD3, 0xA0, 0x1, 0xB3, 0xC1, 0x1, 0xD4, 0x1, 0x0, 0xE4, 0x1, 0x0, 0x4, 0xF0, 0x1, 0xE4, 0x1, 0x2, 0x14, 0x2, 0x0, 0x24, 0x2, 0x0, 0x34, 0x12, 0x0, 0x44, 0x2, 0x0, 0x14, 0x50, 0x2, 0x64, 0x12, 0x0, 0x4, 0x70, 0x2, 0x14, 0x0, 0x0, 0x84, 0x2, 0x0, 0x94, 0x2, 0x0, 0xF4, 0x1, 0x0, 0xA4, 0xB2, 0x2, 0x34, 0x52, 0x2, 0xC4, 0x2, 0x0, 0xD4, 0x2, 0x0, 0xE4, 0x2, 0x0, 0xF4, 0x2, 0x0, 0x4, 0x3, 0x0, 0x14, 0x3, 0x0, 0x25, 0x3, 0x0, 0x35, 0x43, 0x3, 0x5, 0x50, 0x3, 0x45, 0x3, 0x0, 0x5, 0x60, 0x3, 0x75, 0x3, 0x0, 0x5, 0x80, 0x3, 0x95, 0x3, 0x0, 0xA5, 0x3, 0x0, 0x5, 0xB0, 0x3, 0x15, 0x0, 0x0, 0x5, 0xC0, 0x3, 0x5, 0xD0, 0x3, 0x5, 0x40, 0x3, 0xE5, 0x3, 0x0, 0x5, 0xF0, 0x3, 0x5, 0x4, 0x0, 0x15, 0x4, 0x0, 0x25, 0x4, 0x0, 0x35, 0x44, 0x4, 0x45, 0x4, 0x0, 0x55, 0x4, 0x0, 0x65, 0x4, 0x0, 0x75, 0x4, 0x0, 0x85, 0x4, 0x0, 0x6, 0x90, 0x4, 0x6, 0xA0, 0x4, 0xB6, 0xC4, 0x4, 0x6, 0xD0, 0x4, 0xE6, 0x4, 0x0, 0x6, 0xF0, 0x4, 0x6, 0x5, 0x0, 0x16, 0x25, 0x5, 0x36, 0x5, 0x0, 0x6, 0x40, 0x5, 0x56, 0x5, 0x0, 0x6, 0x60, 0x5, 0x76, 0x5, 0x0, 0x86, 0x5, 0x0, 0x96, 0x5, 0x0, 0xA6, 0x5, 0x0, 0x6, 0xB0, 0x5, 0xC6, 0x5, 0x0, 0xD6, 0x5, 0x0, 0xE6, 0x5, 0x0, 0xF6, 0x5, 0x0, 0x6, 0x6, 0x0, 0x6, 0x10, 0x6, 0x7, 0x20, 0x6, 0x37, 0x46, 0x6, 0x57, 0x6, 0x0, 0x67, 0x6, 0x0, 0x77, 0x86, 0x6, 0x97, 0xA6, 0x6, 0xB7, 0xC6, 0x6, 0xD7, 0xE6, 0x6, 0xF7, 0x6, 0x0, 0x7, 0x7, 0x0, 0x7, 0x10, 0x7, 0x7, 0x20, 0x7, 0x7, 0x30, 0x7, 0x7, 0x40, 0x7, 0x57, 0x67, 0x7, 0x77, 0x7, 0x0, 0x87, 0x7, 0x0, 0x8, 0x90, 0x7, 0x8, 0xA0, 0x7, 0xB8, 0x7, 0x0, 0x8, 0xC0, 0x7, 0xD8, 0x7, 0x0, 0x8, 0xE0, 0x7, 0xF8, 0x7, 0x8, 0x18, 0x8, 0x0, 0x28, 0x8, 0x0, 0x38, 0x8, 0x0, 0x48, 0x8, 0x0, 0x58, 0x68, 0x8, 0x78, 0x8, 0x0, 0x88, 0x8, 0x0, 0x8, 0x90, 0x8, 0x9, 0xA0, 0x8, 0xB9, 0xC8, 0x8, 0x9, 0xD0, 0x8, 0xE9, 0xF8, 0x8, 0x9, 0x0, 0x9, 0x9, 0x10, 0x9, 0x29, 0x9, 0x0, 0x39, 0x9, 0x0, 0x49, 0x59, 0x9, 0x69, 0x79, 0x9, 0x9, 0x80, 0x9, 0xA, 0x90, 0x9, 0xA, 0xA0, 0x9, 0xBA, 0x9, 0x0, 0xCA, 0xD9, 0x9, 0xA, 0xE0, 0x9, 0xFA, 0x9, 0xA, 0x1A, 0x2A, 0xA, 0xA, 0x30, 0xA, 0x4B, 0xA, 0x0, 0xB, 0x50, 0xA, 0x6B, 0x7A, 0xA, 0x8B, 0x9A, 0xA, 0xAB, 0xBA, 0xA, 0xC, 0xC0, 0xA, 0xDC, 0xEA, 0xA, 0xFC, 0xA, 0xB, 0x1D, 0xB, 0x0, 0x2D, 0x3B, 0xB, 0xE, 0x40, 0xB, 0xE, 0x50, 0xB, 0x6F, 0x7B, 0xB, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.SpaceSeparator.</summary>
        private static ReadOnlySpan<byte> SerializedCategory11_SpaceSeparator => new byte[] { 0x2, 0x4, 0x0, 0x6, 0x0, 0x0, 0x4, 0x10, 0x0, 0x1, 0x8, 0x31, 0x0, 0x21, 0x4, 0x1, 0x8, 0x2, 0x10, 0x52, 0x0, 0x2, 0x18, 0x2, 0x1C, 0x3, 0x20, 0x93, 0x0, 0xA3, 0x4, 0x3, 0x2C, 0x4, 0x30, 0xD4, 0x0, 0x4, 0x34, 0x4, 0x38, 0x4, 0x3C, 0x5, 0x40, 0x5, 0x44, 0x25, 0x4D, 0x5, 0x1, 0x45, 0x1, 0x6, 0x54, 0x66, 0x5D, 0x6, 0x60, 0x6, 0x64, 0x7, 0x68, 0x7, 0x6C, 0xA7, 0x1, 0xC7, 0x75, 0x8, 0x78, 0x8, 0x7C, 0x8, 0x80, 0x8, 0x84, 0x9, 0x88, 0x9, 0x8C, 0x49, 0x2, 0x9, 0x94, 0xA, 0x98, 0xA, 0x9C, 0x8A, 0x2, 0xA, 0xA4, 0xB, 0xA8, 0xB, 0xAC, 0xB, 0xB0, 0xB, 0xB4, 0xEC, 0xBE, 0xC, 0xC7, 0x2D, 0xCF, 0xE, 0xD0, 0xF, 0xD4, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.LineSeparator.</summary>
        private static ReadOnlySpan<byte> SerializedCategory12_LineSeparator => new byte[] { 0x2, 0x4, 0x0, 0x5, 0x0, 0x0, 0x2, 0x1, 0x4, 0x2, 0x6, 0x43, 0x0, 0x4, 0xA, 0x65, 0x0, 0x6, 0xE, 0x7, 0x10, 0x8, 0x12, 0x9, 0x14, 0xA, 0x16, 0xB, 0x18, 0xC, 0x1A, 0xED, 0x0, 0xE, 0x1E, 0xF, 0x20, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.ParagraphSeparator.</summary>
        private static ReadOnlySpan<byte> SerializedCategory13_ParagraphSeparator => new byte[] { 0x2, 0x4, 0x0, 0x5, 0x0, 0x10, 0x0, 0x1, 0x4, 0x2, 0x6, 0x43, 0x0, 0x4, 0xA, 0x65, 0x0, 0x6, 0xE, 0x7, 0x10, 0x8, 0x12, 0x9, 0x14, 0xA, 0x16, 0xB, 0x18, 0xC, 0x1A, 0xED, 0x0, 0xE, 0x1E, 0xF, 0x20, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.Control.</summary>
        private static ReadOnlySpan<byte> SerializedCategory14_Control => new byte[] { 0x2, 0x4, 0x0, 0x5, 0x0, 0x10, 0x0, 0x21, 0x0, 0x32, 0x0, 0x43, 0x0, 0x54, 0x0, 0x5, 0x2, 0x65, 0x0, 0x6, 0xE, 0x86, 0xE, 0x97, 0x14, 0x8, 0x16, 0x9, 0x18, 0xA, 0x1A, 0xB, 0x1C, 0xC, 0x1E, 0xD, 0x20, 0xE, 0x22, 0xF, 0x24, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.Format.</summary>
        private static ReadOnlySpan<byte> SerializedCategory15_Format => new byte[] { 0x3, 0x4, 0x0, 0x0, 0x7, 0x0, 0x0, 0x10, 0x0, 0x0, 0x0, 0x8, 0x0, 0x11, 0x10, 0x0, 0x21, 0x0, 0x0, 0x11, 0x18, 0x0, 0x31, 0x8, 0x0, 0x11, 0x0, 0x0, 0x31, 0x0, 0x0, 0x1, 0x8, 0x0, 0x1, 0x10, 0x0, 0x1, 0x18, 0x0, 0x2, 0x20, 0x0, 0x52, 0x0, 0x0, 0x62, 0x8, 0x0, 0x72, 0x40, 0x0, 0x12, 0x28, 0x0, 0x92, 0x0, 0x0, 0x2, 0x48, 0x0, 0x2, 0x50, 0x0, 0xB2, 0x0, 0x0, 0xC2, 0x0, 0x0, 0xA2, 0x8, 0x0, 0xD3, 0x0, 0x0, 0xE3, 0x0, 0x0, 0x13, 0x78, 0x0, 0x3, 0x1, 0x0, 0x13, 0x1, 0x0, 0x23, 0x1, 0x0, 0x3, 0x98, 0x0, 0x3, 0xA0, 0x0, 0x53, 0x1, 0x0, 0x63, 0x1, 0x0, 0x3, 0xB8, 0x0, 0x84, 0x1, 0x0, 0x94, 0x1, 0x0, 0x4, 0xD0, 0x0, 0x4, 0xD8, 0x0, 0x4, 0xE0, 0x0, 0x4, 0xE8, 0x0, 0x4, 0xF0, 0x0, 0xF4, 0x1, 0x0, 0x4, 0xC8, 0x0, 0x4, 0x2, 0x0, 0x14, 0x12, 0x1, 0x4, 0x0, 0x1, 0x35, 0x2, 0x0, 0x45, 0x2, 0x0, 0x55, 0x2, 0x0, 0x65, 0x3A, 0x1, 0x5, 0x40, 0x1, 0x95, 0x2, 0x0, 0x5, 0x50, 0x1, 0x5, 0x58, 0x1, 0x5, 0x60, 0x1, 0x5, 0x68, 0x1, 0xE5, 0x2, 0x0, 0xF6, 0x2, 0x0, 0x6, 0x3, 0x0, 0x16, 0x93, 0x1, 0x6, 0x98, 0x1, 0x46, 0xAB, 0x1, 0x6, 0xB0, 0x1, 0x76, 0x3, 0x0, 0x6, 0xC0, 0x1, 0x6, 0xC8, 0x1, 0xA7, 0x3, 0x0, 0xB7, 0x3, 0x0, 0x7, 0xE0, 0x1, 0x7, 0xE8, 0x1, 0xE7, 0x3, 0x0, 0x7, 0xF8, 0x1, 0x7, 0xC, 0x2, 0x27, 0x4, 0x0, 0x38, 0x24, 0x2, 0x8, 0x28, 0x2, 0x8, 0x30, 0x2, 0x8, 0x38, 0x2, 0x88, 0x4C, 0x2, 0x8, 0x50, 0x2, 0xB9, 0x4, 0x0, 0x9, 0x60, 0x2, 0x9, 0x68, 0x2, 0x9, 0x70, 0x2, 0xF9, 0x4, 0x0, 0x9, 0x80, 0x2, 0x1A, 0x5, 0x0, 0xA, 0x90, 0x2, 0xA, 0x98, 0x2, 0xA, 0xA0, 0x2, 0x5A, 0xB5, 0x2, 0x7B, 0x5, 0x0, 0xB, 0xC0, 0x2, 0x9B, 0x5, 0x0, 0xAB, 0xDD, 0x2, 0xCC, 0x5, 0x0, 0xC, 0xE8, 0x2, 0xEC, 0xFD, 0x2, 0xD, 0x6, 0x0, 0x1D, 0x16, 0x3, 0x3E, 0x6, 0x0, 0xE, 0x20, 0x3, 0x5F, 0x36, 0x3, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.Surrogate.</summary>
        private static ReadOnlySpan<byte> SerializedCategory16_Surrogate => new byte[] { 0x2, 0x4, 0x0, 0x3, 0x0, 0x1B, 0x0, 0x2C, 0x0, 0x8D, 0x1, 0x4E, 0x0, 0x5F, 0x0, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.PrivateUse.</summary>
        private static ReadOnlySpan<byte> SerializedCategory17_PrivateUse => new byte[] { 0x2, 0x4, 0x0, 0x4, 0x0, 0x8, 0x1, 0x9, 0x2, 0xA, 0x3, 0x4B, 0x1, 0x5C, 0x1, 0x6D, 0x0, 0x7E, 0x0, 0x8F, 0x0, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.ConnectorPunctuation.</summary>
        private static ReadOnlySpan<byte> SerializedCategory18_ConnectorPunctuation => new byte[] { 0x2, 0x4, 0x0, 0x6, 0x0, 0x10, 0x0, 0x0, 0x4, 0x21, 0x0, 0x11, 0x8, 0x1, 0xC, 0x21, 0x0, 0x42, 0x0, 0x52, 0x0, 0x62, 0x10, 0x62, 0x0, 0x2, 0x18, 0x72, 0x0, 0x83, 0x0, 0x93, 0x0, 0x3, 0x28, 0x3, 0x2C, 0x3, 0x30, 0xD3, 0x0, 0xE4, 0x0, 0x4, 0x3C, 0x4, 0x1, 0x14, 0x49, 0x34, 0x1, 0x45, 0x1, 0x5, 0x54, 0x65, 0x1, 0x5, 0x5C, 0x5, 0x60, 0x6, 0x64, 0xA6, 0x6D, 0xC6, 0x65, 0xD6, 0x1, 0x7, 0x78, 0x7, 0x7C, 0x7, 0x80, 0x7, 0x84, 0x28, 0x8E, 0x8, 0x90, 0x8, 0x94, 0x69, 0x2, 0x9, 0x9C, 0x9, 0xA0, 0x9A, 0x2, 0xA, 0xA8, 0xA, 0xAC, 0xCB, 0x2, 0xB, 0xB4, 0xB, 0xB8, 0xFC, 0x2, 0xC, 0xC0, 0xC, 0xC4, 0x2D, 0x3, 0x3D, 0xD3, 0x5E, 0x3, 0xE, 0xD8, 0x7F, 0xE3, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.DashPunctuation.</summary>
        private static ReadOnlySpan<byte> SerializedCategory19_DashPunctuation => new byte[] { 0x3, 0x4, 0x0, 0x0, 0x7, 0x0, 0x0, 0x10, 0x0, 0x0, 0x0, 0x8, 0x0, 0x1, 0x10, 0x0, 0x21, 0x0, 0x0, 0x1, 0x18, 0x0, 0x31, 0x10, 0x0, 0x11, 0x0, 0x0, 0x31, 0x0, 0x0, 0x1, 0x8, 0x0, 0x1, 0x10, 0x0, 0x42, 0x0, 0x0, 0x2, 0x28, 0x0, 0x2, 0x30, 0x0, 0x2, 0x38, 0x0, 0x62, 0x0, 0x0, 0x2, 0x40, 0x0, 0x2, 0x48, 0x0, 0x52, 0x0, 0x0, 0xA2, 0x8, 0x0, 0x92, 0x0, 0x0, 0xB2, 0x0, 0x0, 0xC3, 0x0, 0x0, 0x3, 0x68, 0x0, 0xE3, 0x0, 0x0, 0x3, 0x78, 0x0, 0x3, 0x70, 0x0, 0x3, 0x1, 0x0, 0x13, 0x1, 0x0, 0x23, 0x99, 0x0, 0x3, 0xA0, 0x0, 0x3, 0xA8, 0x0, 0x53, 0x1, 0x0, 0x23, 0x1, 0x0, 0x63, 0x1, 0x0, 0x4, 0xB8, 0x0, 0x4, 0xC0, 0x0, 0x94, 0x1, 0x0, 0xA4, 0x1, 0x0, 0x4, 0xD8, 0x0, 0xB4, 0x1, 0x0, 0xC4, 0x1, 0x0, 0x74, 0xD9, 0x0, 0xD4, 0x1, 0x0, 0xE4, 0x1, 0x0, 0xF4, 0x1, 0x0, 0x4, 0x0, 0x1, 0x14, 0x2, 0x0, 0x4, 0x10, 0x1, 0x4, 0x18, 0x1, 0x5, 0x20, 0x1, 0x55, 0x32, 0x1, 0x75, 0x2, 0x0, 0x85, 0x2, 0x0, 0x95, 0x52, 0x1, 0x5, 0x58, 0x1, 0xC5, 0x6A, 0x1, 0x5, 0x70, 0x1, 0x5, 0x78, 0x1, 0x5, 0x40, 0x1, 0x5, 0x8B, 0x1, 0x25, 0x3, 0x0, 0x6, 0x98, 0x1, 0x46, 0xAB, 0x1, 0x6, 0xB0, 0x1, 0x6, 0xB8, 0x1, 0x86, 0xCB, 0x1, 0x6, 0xD0, 0x1, 0x6, 0xD8, 0x1, 0x6, 0xE0, 0x1, 0x6, 0xE8, 0x1, 0x6, 0xF0, 0x1, 0x7, 0xF8, 0x1, 0x7, 0x0, 0x2, 0x17, 0x14, 0x2, 0x7, 0x18, 0x2, 0x7, 0x20, 0x2, 0x7, 0x28, 0x2, 0x7, 0x30, 0x2, 0x77, 0x4, 0x0, 0x7, 0x40, 0x2, 0x98, 0x54, 0x2, 0x8, 0x58, 0x2, 0x8, 0x60, 0x2, 0x8, 0x68, 0x2, 0x8, 0x70, 0x2, 0x8, 0x78, 0x2, 0x8, 0x5, 0x0, 0x8, 0x88, 0x2, 0x29, 0x5, 0x0, 0x9, 0x98, 0x2, 0x49, 0x5, 0x0, 0x9, 0xA8, 0x2, 0x9, 0xB0, 0x2, 0x9, 0xB8, 0x2, 0x9, 0xC0, 0x2, 0x9, 0xC8, 0x2, 0xAA, 0x5, 0x0, 0xA, 0xD8, 0x2, 0xCA, 0x5, 0x0, 0xA, 0xE8, 0x2, 0xA, 0xF0, 0x2, 0xFA, 0x5, 0x0, 0xA, 0xE, 0x3, 0x2B, 0x6, 0x0, 0xB, 0x18, 0x3, 0x4B, 0x2E, 0x3, 0x6B, 0x3E, 0x3, 0xB, 0x40, 0x3, 0x9C, 0x6, 0x0, 0xAC, 0x5E, 0x3, 0xCC, 0x6E, 0x3, 0xED, 0x6, 0x0, 0xFD, 0x86, 0x3, 0x1E, 0x7, 0x0, 0xE, 0x90, 0x3, 0x3F, 0xA7, 0x3, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.OpenPunctuation.</summary>
        private static ReadOnlySpan<byte> SerializedCategory20_OpenPunctuation => new byte[] { 0x3, 0x4, 0x0, 0x0, 0x8, 0x0, 0x0, 0x0, 0x10, 0x0, 0x10, 0x0, 0x0, 0x21, 0x0, 0x0, 0x31, 0x0, 0x0, 0x1, 0x20, 0x0, 0x1, 0x30, 0x0, 0x31, 0x0, 0x0, 0x31, 0x0, 0x0, 0x1, 0x20, 0x0, 0x2, 0x40, 0x0, 0x2, 0x50, 0x0, 0x2, 0x60, 0x0, 0x72, 0x30, 0x0, 0x52, 0x30, 0x0, 0x32, 0x0, 0x0, 0x52, 0x0, 0x0, 0x72, 0x20, 0x0, 0x22, 0x60, 0x0, 0x2, 0x30, 0x0, 0x22, 0x40, 0x0, 0x62, 0x0, 0x0, 0x2, 0x20, 0x0, 0x32, 0x50, 0x0, 0x42, 0x0, 0x0, 0x72, 0x0, 0x0, 0x62, 0x20, 0x0, 0x2, 0x70, 0x0, 0x62, 0x40, 0x0, 0x2, 0x80, 0x0, 0x2, 0x90, 0x0, 0x2, 0xA0, 0x0, 0x3, 0xB0, 0x0, 0x53, 0x0, 0x0, 0xC3, 0x0, 0x0, 0xD3, 0x0, 0x0, 0xE3, 0x0, 0x0, 0x3, 0xF0, 0x0, 0x33, 0x0, 0x1, 0x3, 0x10, 0x1, 0x13, 0x1, 0x0, 0x23, 0x31, 0x1, 0x23, 0x0, 0x0, 0x43, 0x1, 0x1, 0xD3, 0x50, 0x1, 0x63, 0x1, 0x0, 0x73, 0x1, 0x0, 0x3, 0x30, 0x0, 0x33, 0x80, 0x1, 0x23, 0x90, 0x1, 0x3, 0xA0, 0x1, 0x3, 0xB0, 0x1, 0xC3, 0x1, 0x0, 0xA3, 0x1, 0x0, 0x43, 0x0, 0x0, 0xD3, 0x1, 0x0, 0xE3, 0x1, 0x0, 0xF3, 0x1, 0x0, 0x3, 0x2, 0x0, 0x4, 0x10, 0x2, 0x24, 0x2, 0x0, 0x34, 0x2, 0x0, 0x4, 0x40, 0x2, 0x54, 0x62, 0x2, 0x74, 0x2, 0x0, 0x84, 0x2, 0x0, 0x94, 0x2, 0x0, 0xA4, 0xB2, 0x2, 0xC4, 0x12, 0x2, 0x4, 0xD0, 0x2, 0xE4, 0x2, 0x0, 0xF4, 0x2, 0x0, 0x4, 0x13, 0x3, 0x4, 0x20, 0x3, 0x4, 0x30, 0x3, 0x44, 0xB3, 0x2, 0x4, 0x50, 0x3, 0x4, 0xF0, 0x2, 0x4, 0x60, 0x3, 0x64, 0x3, 0x0, 0x74, 0x3, 0x0, 0x84, 0x3, 0x0, 0x94, 0x3, 0x0, 0xA4, 0x3, 0x0, 0x4, 0xB0, 0x3, 0xC5, 0xD3, 0x3, 0xE5, 0xF3, 0x3, 0x5, 0x0, 0x4, 0x15, 0x24, 0x4, 0x35, 0x4, 0x0, 0x5, 0x40, 0x4, 0x5, 0x50, 0x4, 0x65, 0x4, 0x0, 0x75, 0x84, 0x4, 0x5, 0x90, 0x4, 0xA5, 0xB4, 0x4, 0xC5, 0x4, 0x0, 0xD5, 0xE4, 0x4, 0x5, 0xF0, 0x4, 0x5, 0xB5, 0x4, 0x5, 0x10, 0x5, 0x5, 0xE0, 0x3, 0x25, 0x5, 0x0, 0x35, 0x45, 0x5, 0x55, 0x5, 0x0, 0x66, 0x75, 0x5, 0x86, 0x95, 0x5, 0x6, 0xA0, 0x5, 0x6, 0xB0, 0x5, 0xC6, 0xD5, 0x5, 0xE6, 0xF5, 0x5, 0x6, 0x6, 0x0, 0x16, 0x6, 0x0, 0x6, 0x20, 0x6, 0x6, 0x30, 0x6, 0x46, 0x56, 0x6, 0x6, 0x60, 0x6, 0x6, 0x70, 0x6, 0x86, 0x96, 0x6, 0x7, 0xA0, 0x6, 0x7, 0xB0, 0x6, 0x7, 0xC0, 0x6, 0x7, 0xD0, 0x6, 0x7, 0xE0, 0x6, 0xF7, 0x6, 0x0, 0x7, 0x17, 0x7, 0x7, 0x20, 0x7, 0x37, 0x47, 0x7, 0x57, 0x7, 0x0, 0x7, 0x60, 0x7, 0x7, 0x70, 0x7, 0x88, 0x97, 0x7, 0xA8, 0x7, 0x0, 0x8, 0xB0, 0x7, 0x8, 0xC0, 0x7, 0xD8, 0x7, 0x0, 0xE8, 0x7, 0x0, 0xF8, 0x7, 0x0, 0x8, 0x0, 0x8, 0x8, 0x10, 0x8, 0x28, 0x8, 0x0, 0x8, 0x30, 0x8, 0x49, 0x58, 0x8, 0x9, 0x60, 0x8, 0x79, 0x8, 0x0, 0x9, 0x80, 0x8, 0x99, 0x8, 0x0, 0xA9, 0xB8, 0x8, 0xC9, 0x8, 0x0, 0xD9, 0x8, 0x0, 0x9, 0xE0, 0x8, 0xFA, 0x8, 0x0, 0xA, 0x0, 0x9, 0x1A, 0x29, 0x9, 0x3A, 0x49, 0x9, 0x5A, 0x9, 0x0, 0x6A, 0x9, 0x0, 0xA, 0x70, 0x9, 0x8B, 0x9, 0x0, 0xB, 0x90, 0x9, 0xAB, 0xB9, 0x9, 0xB, 0xC0, 0x9, 0xDB, 0xE9, 0x9, 0xFC, 0x9, 0x0, 0xC, 0x1A, 0xA, 0x2C, 0x3A, 0xA, 0x4D, 0xA, 0x0, 0x5D, 0x6A, 0xA, 0x7E, 0xA, 0x0, 0xE, 0x80, 0xA, 0x9F, 0xAA, 0xA, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.ClosePunctuation.</summary>
        private static ReadOnlySpan<byte> SerializedCategory21_ClosePunctuation => new byte[] { 0x3, 0x4, 0x0, 0x0, 0x8, 0x0, 0x0, 0x10, 0x0, 0x0, 0x0, 0x10, 0x0, 0x21, 0x30, 0x0, 0x1, 0x20, 0x0, 0x31, 0x0, 0x0, 0x1, 0x30, 0x0, 0x11, 0x0, 0x0, 0x21, 0x0, 0x0, 0x1, 0x20, 0x0, 0x1, 0x20, 0x0, 0x1, 0x20, 0x0, 0x2, 0x40, 0x0, 0x52, 0x0, 0x0, 0x2, 0x50, 0x0, 0x32, 0x60, 0x0, 0x2, 0x70, 0x0, 0x72, 0x30, 0x0, 0x62, 0x0, 0x0, 0x82, 0x20, 0x0, 0x22, 0x50, 0x0, 0x22, 0x90, 0x0, 0x2, 0x20, 0x0, 0x32, 0x0, 0x0, 0x92, 0x0, 0x0, 0x52, 0x20, 0x0, 0x2, 0x60, 0x0, 0x72, 0x0, 0x0, 0x52, 0x90, 0x0, 0xA2, 0x0, 0x0, 0xB2, 0x0, 0x0, 0x2, 0xC0, 0x0, 0x3, 0xD0, 0x0, 0xE3, 0x0, 0x0, 0xF3, 0x0, 0x0, 0x3, 0x1, 0x0, 0x13, 0x21, 0x1, 0x33, 0x30, 0x1, 0x13, 0x1, 0x0, 0x33, 0x1, 0x0, 0x43, 0x51, 0x1, 0x23, 0x0, 0x0, 0x23, 0x31, 0x1, 0xF3, 0x60, 0x1, 0x73, 0x1, 0x0, 0x13, 0x31, 0x0, 0x33, 0x80, 0x1, 0x23, 0x90, 0x1, 0x3, 0x30, 0x1, 0x3, 0xA0, 0x1, 0xB3, 0x1, 0x0, 0xC3, 0x1, 0x0, 0xD3, 0x1, 0x0, 0xE3, 0x1, 0x0, 0xF3, 0x1, 0x0, 0x3, 0x2, 0x0, 0x4, 0x10, 0x2, 0x24, 0x2, 0x0, 0x4, 0x30, 0x2, 0x44, 0x52, 0x2, 0x64, 0x2, 0x0, 0x74, 0x2, 0x0, 0x84, 0x2, 0x0, 0x94, 0xA2, 0x2, 0xB4, 0x2, 0x0, 0x4, 0xC0, 0x2, 0xD4, 0x2, 0x0, 0xE4, 0xF2, 0x2, 0x4, 0x0, 0x3, 0x4, 0x10, 0x3, 0x24, 0xA3, 0x2, 0x4, 0x30, 0x3, 0x4, 0xD0, 0x2, 0x4, 0x80, 0x2, 0x44, 0x3, 0x0, 0x54, 0x3, 0x0, 0x64, 0x3, 0x0, 0x74, 0x3, 0x0, 0x4, 0x80, 0x3, 0x95, 0xA3, 0x3, 0xA5, 0xB3, 0x3, 0x5, 0xC0, 0x3, 0xD5, 0xE3, 0x3, 0xF5, 0x3, 0x0, 0x5, 0x0, 0x4, 0x5, 0x10, 0x4, 0x25, 0x4, 0x0, 0xA5, 0x33, 0x4, 0x5, 0x40, 0x4, 0x55, 0x64, 0x4, 0x75, 0x4, 0x0, 0x85, 0x94, 0x4, 0x5, 0xA0, 0x4, 0xF5, 0x63, 0x4, 0x5, 0xB0, 0x4, 0xC5, 0x4, 0x0, 0xD5, 0xE4, 0x4, 0xF5, 0x4, 0x0, 0x6, 0x15, 0x5, 0x26, 0x35, 0x5, 0x6, 0x40, 0x5, 0x6, 0x50, 0x5, 0x66, 0x75, 0x5, 0x86, 0x95, 0x5, 0xA6, 0x5, 0x0, 0xB6, 0x5, 0x0, 0x6, 0xC0, 0x5, 0x6, 0xD0, 0x5, 0xE6, 0x5, 0x0, 0x6, 0xF0, 0x5, 0x6, 0x0, 0x6, 0x16, 0x26, 0x6, 0x7, 0x30, 0x6, 0x7, 0x40, 0x6, 0x7, 0x50, 0x6, 0x7, 0x60, 0x6, 0x7, 0x70, 0x6, 0x87, 0x6, 0x0, 0x97, 0xA6, 0x6, 0x7, 0xB0, 0x6, 0xC7, 0xD6, 0x6, 0xE7, 0x6, 0x0, 0x7, 0xF0, 0x6, 0x7, 0x0, 0x7, 0x18, 0x27, 0x7, 0x38, 0x7, 0x0, 0x8, 0x40, 0x7, 0x8, 0x50, 0x7, 0x68, 0x7, 0x0, 0x78, 0x7, 0x0, 0x88, 0x7, 0x0, 0x8, 0x90, 0x7, 0x8, 0xA0, 0x7, 0xB8, 0x7, 0x0, 0x8, 0xC0, 0x7, 0xD9, 0xE7, 0x7, 0x9, 0xF0, 0x7, 0x9, 0x8, 0x0, 0x9, 0x10, 0x8, 0x29, 0x8, 0x0, 0x39, 0x48, 0x8, 0x59, 0x8, 0x0, 0x69, 0x8, 0x0, 0x9, 0x70, 0x8, 0x8A, 0x8, 0x0, 0xA, 0x90, 0x8, 0xAA, 0xB8, 0x8, 0xCA, 0xD8, 0x8, 0xEA, 0x8, 0x0, 0xFA, 0x8, 0x0, 0xA, 0x0, 0x9, 0x1B, 0x9, 0x0, 0xB, 0x20, 0x9, 0x3B, 0x49, 0x9, 0xB, 0x50, 0x9, 0x6B, 0x79, 0x9, 0x8C, 0x9, 0x0, 0x9C, 0xA9, 0x9, 0xBC, 0xC9, 0x9, 0xDD, 0x9, 0x0, 0xED, 0xF9, 0x9, 0xE, 0xA, 0x0, 0xE, 0x10, 0xA, 0x2F, 0x3A, 0xA, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.InitialQuotePunctuation.</summary>
        private static ReadOnlySpan<byte> SerializedCategory22_InitialQuotePunctuation => new byte[] { 0x2, 0x4, 0x0, 0x6, 0x0, 0x0, 0x4, 0x10, 0x0, 0x1, 0x8, 0x1, 0xC, 0x21, 0x0, 0x31, 0x8, 0x31, 0x0, 0x2, 0x10, 0x42, 0x0, 0x42, 0x14, 0x42, 0x18, 0x2, 0x14, 0x2, 0x20, 0x3, 0x24, 0xA3, 0x0, 0xB3, 0x30, 0xD3, 0x0, 0x73, 0x0, 0xE3, 0x0, 0x4, 0x3C, 0x4, 0x45, 0x24, 0x1, 0x34, 0x1, 0x4, 0x50, 0x55, 0x59, 0x75, 0x61, 0x95, 0x1, 0x6, 0x68, 0x6, 0x6C, 0x6, 0x70, 0x7, 0x74, 0x7, 0x78, 0xF7, 0x1, 0x8, 0x80, 0x8, 0x84, 0x8, 0x88, 0x39, 0x2, 0x9, 0x90, 0x9, 0x94, 0x6A, 0x2, 0xA, 0x9C, 0xA, 0xA0, 0x9B, 0xAA, 0xB, 0xAC, 0xC, 0xB0, 0xC, 0xB4, 0xED, 0xBE, 0xE, 0xC0, 0xF, 0xC4, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.FinalQuotePunctuation.</summary>
        private static ReadOnlySpan<byte> SerializedCategory23_FinalQuotePunctuation => new byte[] { 0x2, 0x4, 0x0, 0x6, 0x0, 0x10, 0x0, 0x0, 0x4, 0x1, 0x8, 0x31, 0x0, 0x21, 0x0, 0x2, 0x10, 0x42, 0x0, 0x42, 0x14, 0x42, 0x18, 0x2, 0x14, 0x2, 0x18, 0x3, 0x1C, 0x83, 0x0, 0x93, 0x28, 0xB3, 0x0, 0x43, 0x0, 0xC3, 0x0, 0x4, 0x34, 0xE4, 0x3C, 0x4, 0x1, 0x14, 0x1, 0x24, 0x1, 0x35, 0x51, 0x55, 0x59, 0x75, 0x1, 0x6, 0x60, 0x6, 0x64, 0x6, 0x68, 0x7, 0x6C, 0x7, 0x70, 0xD7, 0x1, 0x8, 0x78, 0x8, 0x7C, 0x8, 0x80, 0x19, 0x2, 0x9, 0x88, 0x9, 0x8C, 0x4A, 0x2, 0xA, 0x94, 0xA, 0x98, 0x7B, 0xA2, 0xB, 0xA4, 0xC, 0xA8, 0xC, 0xAC, 0xCD, 0xB6, 0xE, 0xB8, 0xF, 0xBC, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.OtherPunctuation.</summary>
        private static ReadOnlySpan<byte> SerializedCategory24_OtherPunctuation => new byte[] { 0x3, 0x4, 0x0, 0x0, 0x9, 0x0, 0x0, 0x10, 0x0, 0x0, 0x0, 0x20, 0x0, 0x1, 0x20, 0x0, 0x1, 0x40, 0x0, 0x1, 0x60, 0x0, 0x21, 0x0, 0x0, 0x11, 0x0, 0x0, 0x11, 0x60, 0x0, 0x31, 0x0, 0x0, 0x11, 0x40, 0x0, 0x31, 0x20, 0x0, 0x31, 0x40, 0x0, 0x21, 0x20, 0x0, 0x21, 0x60, 0x0, 0x1, 0x60, 0x0, 0x31, 0x0, 0x0, 0x42, 0xA0, 0x0, 0x62, 0x0, 0x0, 0x2, 0xC0, 0x0, 0x72, 0x0, 0x1, 0x92, 0x40, 0x1, 0x2, 0x20, 0x1, 0x2, 0x80, 0x0, 0x72, 0x0, 0x0, 0x12, 0x80, 0x1, 0x62, 0x60, 0x1, 0xD2, 0x0, 0x0, 0x2, 0xA0, 0x0, 0xC2, 0x20, 0x0, 0x2, 0xE0, 0x0, 0x82, 0x0, 0x0, 0x12, 0x0, 0x0, 0x42, 0x20, 0x0, 0x12, 0x60, 0x1, 0x62, 0x80, 0x1, 0x12, 0x0, 0x1, 0xA2, 0x0, 0x0, 0xB2, 0x0, 0x0, 0x52, 0x0, 0x0, 0x2, 0x60, 0x1, 0x62, 0x0, 0x1, 0x12, 0x40, 0x0, 0x12, 0x80, 0x0, 0xC2, 0x0, 0x1, 0x82, 0xC0, 0x1, 0x82, 0xE0, 0x1, 0x82, 0x80, 0x0, 0x92, 0x60, 0x1, 0xB2, 0xC0, 0x1, 0x72, 0x60, 0x1, 0xC2, 0xE0, 0x1, 0x12, 0xE0, 0x0, 0x42, 0x0, 0x0, 0x2, 0x80, 0x1, 0xE2, 0x20, 0x0, 0xC2, 0x0, 0x0, 0x42, 0xE0, 0x0, 0x2, 0xA0, 0x1, 0x62, 0x20, 0x0, 0x2, 0x0, 0x1, 0x42, 0x0, 0x1, 0xB2, 0xE0, 0x0, 0x42, 0xA0, 0x1, 0x62, 0xE0, 0x0, 0xA2, 0xE0, 0x1, 0x72, 0xA0, 0x0, 0x2, 0x1, 0x0, 0x92, 0x20, 0x2, 0x3, 0x40, 0x2, 0x33, 0x1, 0x0, 0x3, 0x80, 0x2, 0x53, 0x1, 0x0, 0x63, 0x61, 0x1, 0x73, 0x1, 0x3, 0x93, 0x41, 0x3, 0xB3, 0x81, 0x3, 0xD3, 0xC1, 0x3, 0xF3, 0x1, 0x0, 0x3, 0x0, 0x3, 0x3, 0x2, 0x0, 0x13, 0x2, 0x0, 0x23, 0x62, 0x4, 0x93, 0x1, 0x0, 0x43, 0x2, 0x0, 0x3, 0x20, 0x4, 0x3, 0xA0, 0x4, 0x63, 0xE2, 0x3, 0x73, 0x2, 0x0, 0x83, 0x2, 0x0, 0x3, 0x20, 0x5, 0x3, 0x40, 0x5, 0x13, 0x60, 0x5, 0xC3, 0x22, 0x0, 0xD3, 0x2, 0x0, 0xE3, 0xC2, 0x3, 0xF3, 0x2, 0x6, 0x13, 0x3, 0x0, 0xE3, 0x41, 0x6, 0x13, 0x60, 0x6, 0x43, 0x23, 0x0, 0x3, 0x20, 0x0, 0x3, 0x0, 0x4, 0x3, 0xE0, 0x3, 0x53, 0x3, 0x0, 0xC3, 0x1, 0x0, 0x53, 0x2, 0x0, 0x23, 0xC2, 0x3, 0x3, 0xC0, 0x6, 0x73, 0x3, 0x7, 0x73, 0x23, 0x7, 0x3, 0x80, 0x3, 0xA3, 0x3, 0x0, 0x63, 0x2, 0x0, 0x43, 0x21, 0x0, 0xB3, 0x83, 0x7, 0x3, 0x0, 0x5, 0x3, 0x80, 0x4, 0x13, 0x20, 0x4, 0xD3, 0x3, 0x0, 0x3, 0x60, 0x2, 0x3, 0x20, 0x3, 0x3, 0xC0, 0x4, 0xE3, 0x21, 0x0, 0x83, 0x21, 0x3, 0x23, 0x22, 0x0, 0xE3, 0x3, 0x0, 0xF3, 0x3, 0x0, 0x3, 0x4, 0x0, 0x3, 0x20, 0x8, 0x3, 0x40, 0x8, 0xD3, 0x1, 0x0, 0x93, 0x1, 0x4, 0x3, 0x60, 0x8, 0x43, 0x4, 0x0, 0x53, 0x64, 0x1, 0x4, 0xC0, 0x8, 0x74, 0x4, 0x9, 0x94, 0x44, 0x9, 0x4, 0x60, 0x9, 0xC4, 0xA4, 0x9, 0x84, 0x4, 0x0, 0xE4, 0x4, 0x0, 0x4, 0xE0, 0x9, 0x4, 0x5, 0x0, 0x14, 0x5, 0x0, 0x24, 0x5, 0x0, 0x14, 0x65, 0xA, 0x44, 0x5, 0x0, 0x4, 0x20, 0xA, 0x54, 0x5, 0x0, 0x64, 0x5, 0x0, 0x74, 0x5, 0x0, 0x84, 0x5, 0x0, 0x4, 0x20, 0xB, 0xF4, 0x4, 0x0, 0xA4, 0x5, 0x0, 0x4, 0x60, 0xB, 0xC4, 0xA5, 0xB, 0xE4, 0xE5, 0xB, 0x4, 0x26, 0xC, 0x24, 0x6, 0x0, 0x34, 0x86, 0xC, 0x54, 0xC6, 0xC, 0x74, 0x6, 0x0, 0x84, 0xC6, 0xC, 0x94, 0x6, 0x0, 0xA4, 0x6, 0x9, 0xB4, 0x6, 0x0, 0x4, 0x80, 0xD, 0x4, 0xA0, 0xD, 0x4, 0xC0, 0xD, 0xF4, 0x6, 0x0, 0x4, 0x7, 0x0, 0x4, 0x20, 0xE, 0x4, 0x40, 0xE, 0x4, 0x60, 0xE, 0x4, 0x60, 0xD, 0x44, 0x7, 0x0, 0x4, 0xA0, 0xE, 0x64, 0xE7, 0xE, 0x84, 0x87, 0xA, 0x94, 0x7, 0x0, 0x4, 0x20, 0xF, 0xA4, 0x7, 0x0, 0xB4, 0x7, 0x0, 0x84, 0xA4, 0xD, 0x24, 0x7, 0x0, 0xC4, 0x7, 0x0, 0xD4, 0x7, 0x0, 0x4, 0xC0, 0xF, 0x4, 0xE0, 0xF, 0x4, 0x28, 0x10, 0x24, 0x8, 0x0, 0x4, 0x60, 0x10, 0x4, 0x80, 0x10, 0x4, 0x40, 0xF, 0x54, 0xC8, 0x10, 0x74, 0x8, 0x9, 0x94, 0x4, 0x11, 0x95, 0x8, 0x0, 0xA5, 0x68, 0x11, 0xC5, 0xA8, 0x11, 0xE5, 0xE8, 0x11, 0x5, 0x9, 0x0, 0x15, 0x49, 0x12, 0x5, 0x60, 0x12, 0x5, 0x80, 0x12, 0x5, 0xA0, 0x12, 0x65, 0x9, 0x0, 0x75, 0xC9, 0x12, 0x85, 0x9, 0x0, 0x95, 0x9, 0x0, 0xA5, 0x9, 0x0, 0x5, 0x60, 0x13, 0x25, 0x9, 0x0, 0xC5, 0x9, 0x0, 0xD5, 0xC9, 0x13, 0x5, 0xE0, 0x13, 0x5, 0x2A, 0x14, 0xE5, 0x8, 0x0, 0x25, 0xA, 0x0, 0x5, 0x60, 0x14, 0x45, 0xAA, 0x14, 0x5, 0xC0, 0x14, 0x75, 0xA, 0x0, 0x35, 0x9, 0x0, 0x85, 0x2A, 0x15, 0xA5, 0xA, 0x0, 0x5, 0x40, 0x12, 0x5, 0x60, 0x15, 0x5, 0x80, 0x15, 0x5, 0xA0, 0x15, 0xE5, 0xA, 0x0, 0xF5, 0xA, 0x0, 0x5, 0xB, 0x0, 0x15, 0xB, 0x0, 0x5, 0x40, 0x16, 0x5, 0x60, 0x16, 0x5, 0x80, 0x16, 0x5, 0xA0, 0x16, 0x5, 0xC0, 0x16, 0x75, 0xB, 0x0, 0x5, 0x0, 0x17, 0x95, 0xB, 0x0, 0xA5, 0xB, 0x0, 0xD5, 0x9, 0x0, 0xB5, 0xB, 0x0, 0x5, 0x80, 0x17, 0xD5, 0xB, 0x0, 0xE5, 0xB, 0x0, 0x5, 0xE0, 0x17, 0x5, 0xE0, 0x16, 0x5, 0xC, 0x0, 0x5, 0x20, 0x18, 0x25, 0x6C, 0x18, 0x5, 0x80, 0x18, 0x5, 0x20, 0x15, 0x5, 0xA0, 0x18, 0xC5, 0xB, 0x0, 0x65, 0xC, 0x0, 0x5, 0xE0, 0x18, 0x85, 0xC, 0x0, 0x96, 0x4C, 0x19, 0xB6, 0x8C, 0x19, 0xD6, 0xC, 0x0, 0xE6, 0xC, 0x0, 0xF6, 0xC, 0x0, 0x6, 0xD, 0x0, 0x16, 0x4D, 0x1A, 0x36, 0xD, 0x0, 0x46, 0xD, 0x0, 0x56, 0xD, 0x0, 0x66, 0xED, 0x1A, 0x86, 0xD, 0x0, 0x96, 0xD, 0x0, 0x6, 0x40, 0x1B, 0xB6, 0x8D, 0x1B, 0xD6, 0xD, 0x0, 0xE6, 0xD, 0x0, 0xF6, 0xD, 0x1C, 0x16, 0xE, 0x0, 0x86, 0x4D, 0x1C, 0x36, 0xE, 0x0, 0x46, 0xE, 0x0, 0x6, 0xA0, 0x1C, 0x6, 0xC0, 0x1C, 0x76, 0xE, 0x0, 0x6, 0x0, 0x1D, 0x96, 0xE, 0x0, 0x6, 0x40, 0x1D, 0xB6, 0xE, 0x0, 0xC6, 0xE, 0x0, 0xD6, 0xE, 0x0, 0xE6, 0xE, 0x0, 0xF6, 0xE, 0x1E, 0x6, 0x20, 0x1E, 0x26, 0xF, 0x0, 0x36, 0xF, 0x0, 0x6, 0x80, 0x1E, 0x56, 0xF, 0x0, 0x66, 0xF, 0x0, 0x76, 0xF, 0x0, 0x86, 0xF, 0x0, 0x96, 0x4F, 0x1F, 0xB6, 0xF, 0x0, 0x6, 0x80, 0x1F, 0xD6, 0xF, 0x0, 0xE6, 0xEF, 0x1F, 0x6, 0x30, 0x20, 0x26, 0x10, 0x0, 0x6, 0x60, 0x20, 0x46, 0x10, 0x0, 0x6, 0xA0, 0x20, 0x66, 0xF0, 0x20, 0x7, 0x0, 0x21, 0x7, 0x20, 0x21, 0xA7, 0x10, 0x0, 0xB7, 0x90, 0x21, 0xD7, 0xD0, 0x21, 0xF7, 0x10, 0x22, 0x17, 0x51, 0x22, 0x37, 0x11, 0x0, 0x47, 0xB1, 0x22, 0x7, 0xC0, 0x22, 0x7, 0xE0, 0x22, 0x87, 0x11, 0x0, 0x7, 0x20, 0x23, 0xA7, 0x71, 0x23, 0xC7, 0xB1, 0x23, 0xE7, 0xF1, 0x23, 0x7, 0x0, 0x24, 0x7, 0x20, 0x24, 0x27, 0x72, 0x24, 0x47, 0xB2, 0x24, 0x7, 0xC0, 0x24, 0x47, 0xF1, 0x24, 0x87, 0x32, 0x25, 0x7, 0x40, 0x25, 0xB7, 0x12, 0x0, 0xC7, 0xB2, 0x25, 0x77, 0xD1, 0x25, 0xF7, 0x12, 0x26, 0x7, 0x20, 0x26, 0x27, 0x73, 0x26, 0x47, 0xB3, 0x26, 0x67, 0xF3, 0x26, 0x87, 0x33, 0x27, 0xA7, 0x73, 0x27, 0xC8, 0xB3, 0x27, 0xE8, 0xF3, 0x27, 0x8, 0x34, 0x28, 0x8, 0x40, 0x28, 0x8, 0x60, 0x28, 0x8, 0x80, 0x28, 0x8, 0xA0, 0x28, 0x68, 0xF4, 0x28, 0x8, 0x0, 0x29, 0x8, 0x20, 0x29, 0xA8, 0x74, 0x29, 0xC8, 0xB4, 0x29, 0xE8, 0xF4, 0x29, 0x8, 0x15, 0x0, 0x8, 0x20, 0x2A, 0x28, 0x75, 0x2A, 0x48, 0xB5, 0x2A, 0x8, 0xC0, 0x2A, 0x78, 0x15, 0x2B, 0x98, 0x55, 0x2B, 0xB8, 0x15, 0x0, 0xC8, 0x15, 0x0, 0x8, 0xA0, 0x2B, 0xE9, 0x15, 0x0, 0xF9, 0x15, 0x2C, 0x19, 0x56, 0x2C, 0x9, 0x60, 0x2C, 0x49, 0xB6, 0x2C, 0x9, 0xC0, 0x2C, 0x9, 0xE0, 0x2C, 0x89, 0x36, 0x2D, 0xA9, 0x16, 0x0, 0xB9, 0x96, 0x2D, 0xD9, 0xD6, 0x2D, 0xF9, 0x16, 0x2E, 0x19, 0x57, 0x2E, 0x39, 0x97, 0x2E, 0x5A, 0x17, 0x0, 0xA, 0xC0, 0x2E, 0x7A, 0x17, 0x0, 0xA, 0x0, 0x2F, 0x9A, 0x17, 0x0, 0xA, 0x40, 0x2F, 0xBA, 0x97, 0x2F, 0xDA, 0xD7, 0x2F, 0xFA, 0x17, 0x30, 0x1A, 0x58, 0x30, 0x3B, 0x18, 0x0, 0x4B, 0xB8, 0x30, 0xB, 0xC0, 0x30, 0x7B, 0x18, 0x31, 0x9B, 0x58, 0x31, 0xBB, 0x98, 0x31, 0xDC, 0x18, 0x0, 0xC, 0xC0, 0x31, 0xFC, 0x18, 0x32, 0x1C, 0x59, 0x32, 0x3D, 0x19, 0x0, 0x4D, 0x19, 0x0, 0x5D, 0xD9, 0x32, 0x7E, 0x19, 0x33, 0xE, 0x20, 0x33, 0xAF, 0x79, 0x33, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.MathSymbol.</summary>
        private static ReadOnlySpan<byte> SerializedCategory25_MathSymbol => new byte[] { 0x3, 0x4, 0x0, 0x0, 0x8, 0x0, 0x0, 0x0, 0x10, 0x0, 0x10, 0x0, 0x0, 0x1, 0x20, 0x0, 0x11, 0x30, 0x0, 0x21, 0x0, 0x0, 0x21, 0x10, 0x0, 0x31, 0x0, 0x0, 0x1, 0x30, 0x0, 0x31, 0x20, 0x0, 0x11, 0x0, 0x0, 0x1, 0x10, 0x0, 0x21, 0x20, 0x0, 0x31, 0x0, 0x0, 0x42, 0x50, 0x0, 0x2, 0x60, 0x0, 0x22, 0x0, 0x0, 0x72, 0x0, 0x0, 0x2, 0x80, 0x0, 0x72, 0x60, 0x0, 0x2, 0x90, 0x0, 0x42, 0x10, 0x0, 0xA2, 0x10, 0x0, 0xB2, 0x10, 0x0, 0x12, 0x0, 0x0, 0x12, 0x50, 0x0, 0x2, 0x70, 0x0, 0xC2, 0x10, 0x0, 0x82, 0x0, 0x0, 0x2, 0xC0, 0x0, 0x2, 0x10, 0x0, 0x12, 0x80, 0x0, 0x42, 0x0, 0x0, 0x42, 0x60, 0x0, 0xB2, 0x0, 0x0, 0x62, 0x0, 0x0, 0x62, 0xA0, 0x0, 0x2, 0xB0, 0x0, 0x2, 0x40, 0x0, 0x42, 0xB0, 0x0, 0xD2, 0x0, 0x0, 0x2, 0xE0, 0x0, 0xF3, 0x0, 0x1, 0x13, 0x1, 0x0, 0x23, 0x1, 0x0, 0x33, 0x1, 0x0, 0x3, 0x40, 0x1, 0x53, 0x1, 0x0, 0x63, 0x71, 0x1, 0x83, 0x11, 0x0, 0x93, 0x11, 0x0, 0xA3, 0x1, 0x0, 0x3, 0xB0, 0x1, 0x3, 0xC0, 0x1, 0x13, 0x70, 0x1, 0xD3, 0x1, 0x0, 0x13, 0x0, 0x0, 0x3, 0x50, 0x1, 0x3, 0xD0, 0x1, 0x3, 0xE0, 0x1, 0x93, 0x1, 0x0, 0x3, 0xF0, 0x1, 0x3, 0x2, 0x0, 0x13, 0x2, 0x0, 0x13, 0x90, 0x1, 0x3, 0x20, 0x2, 0x33, 0x2, 0x0, 0x43, 0x52, 0x2, 0x63, 0x62, 0x1, 0x33, 0x61, 0x1, 0x73, 0x2, 0x0, 0x83, 0x2, 0x0, 0x3, 0x0, 0x1, 0x3, 0x10, 0x2, 0x73, 0x32, 0x2, 0x3, 0x40, 0x2, 0x93, 0x2, 0x0, 0xA3, 0x2, 0x0, 0x4, 0xB0, 0x2, 0xC4, 0x2, 0x0, 0xD4, 0xE2, 0x2, 0x4, 0xF0, 0x2, 0x4, 0x0, 0x3, 0x4, 0x10, 0x3, 0x14, 0x0, 0x0, 0x24, 0x13, 0x0, 0x34, 0x13, 0x0, 0x44, 0x53, 0x3, 0x14, 0x60, 0x3, 0x14, 0x70, 0x3, 0x4, 0x80, 0x3, 0x94, 0x3, 0x0, 0x4, 0xA0, 0x3, 0xB4, 0x3, 0x0, 0x4, 0xC0, 0x3, 0xD4, 0x3, 0x0, 0xE4, 0x13, 0x0, 0xF4, 0x3, 0x0, 0x4, 0x4, 0x0, 0x14, 0x4, 0x0, 0x24, 0x34, 0x4, 0x4, 0x40, 0x4, 0x54, 0x4, 0x0, 0x4, 0x60, 0x4, 0x74, 0x4, 0x0, 0x4, 0x80, 0x4, 0x84, 0x4, 0x0, 0x94, 0xA4, 0x4, 0x4, 0xB0, 0x4, 0xC4, 0x4, 0x0, 0xA4, 0x3, 0x4, 0xD4, 0x4, 0x0, 0xD4, 0xE2, 0x4, 0xF5, 0x4, 0x0, 0x5, 0x0, 0x5, 0x5, 0x10, 0x5, 0x25, 0x5, 0x0, 0x35, 0x5, 0x0, 0x5, 0x40, 0x5, 0x55, 0x5, 0x0, 0x65, 0x75, 0x5, 0x15, 0x80, 0x5, 0x95, 0xA5, 0x5, 0xB5, 0x5, 0x0, 0xC5, 0xD5, 0x5, 0xE5, 0x5, 0x0, 0xF5, 0x5, 0x6, 0x15, 0x26, 0x6, 0x35, 0x6, 0x0, 0xF5, 0x5, 0x0, 0x45, 0x56, 0x6, 0x65, 0x76, 0x6, 0x5, 0x80, 0x6, 0x5, 0x90, 0x6, 0x5, 0xA0, 0x6, 0xB5, 0xC6, 0x6, 0x5, 0xD0, 0x6, 0xE5, 0x6, 0x0, 0xF5, 0x6, 0x0, 0x5, 0x7, 0x0, 0x15, 0x7, 0x0, 0x26, 0x7, 0x0, 0x36, 0x47, 0x7, 0x56, 0x7, 0x0, 0x6, 0x60, 0x7, 0x76, 0x87, 0x7, 0x96, 0xA7, 0x7, 0xB6, 0x7, 0x0, 0xC6, 0x7, 0x0, 0xD6, 0xE7, 0x7, 0xF6, 0x7, 0x8, 0x16, 0x28, 0x8, 0x36, 0x48, 0x8, 0x56, 0x68, 0x8, 0x6, 0x70, 0x8, 0x86, 0x8, 0x0, 0x6, 0x90, 0x8, 0xA6, 0x8, 0x0, 0xE6, 0xB5, 0x8, 0xC6, 0xD8, 0x8, 0xE7, 0xF8, 0x8, 0x7, 0x0, 0x9, 0x7, 0x10, 0x9, 0x7, 0x20, 0x9, 0x37, 0x19, 0x0, 0x47, 0x9, 0x0, 0x7, 0x50, 0x9, 0x67, 0x9, 0x0, 0x77, 0x89, 0x9, 0x97, 0xA9, 0x9, 0xB7, 0xC9, 0x9, 0x7, 0xD0, 0x9, 0xE7, 0x9, 0x0, 0xF7, 0x9, 0xA, 0x18, 0x2A, 0xA, 0x38, 0xA, 0x0, 0x48, 0x1A, 0x0, 0x58, 0xA, 0x0, 0x68, 0x7A, 0xA, 0x88, 0xA, 0x0, 0x98, 0x1A, 0x0, 0xA8, 0xBA, 0xA, 0x8, 0xC0, 0xA, 0xD8, 0xA, 0x0, 0x8, 0xE0, 0xA, 0xF9, 0xA, 0x0, 0x9, 0xB, 0x0, 0x19, 0x2B, 0xB, 0x39, 0x4B, 0xB, 0x59, 0x6B, 0xB, 0x79, 0xB, 0x0, 0x89, 0x9B, 0xB, 0xAA, 0xBB, 0xB, 0xA, 0xC0, 0xB, 0xDA, 0xEB, 0xB, 0xFA, 0xB, 0xC, 0x1B, 0xC, 0x0, 0x2B, 0x3C, 0xC, 0xB, 0x40, 0xC, 0x5C, 0xC, 0x0, 0xC, 0x60, 0xC, 0xC, 0x70, 0xC, 0x8D, 0xC, 0x0, 0x9D, 0xAC, 0xC, 0xBE, 0xC, 0x0, 0xE, 0xC0, 0xC, 0xDF, 0xEC, 0xC, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.CurrencySymbol.</summary>
        private static ReadOnlySpan<byte> SerializedCategory26_CurrencySymbol => new byte[] { 0x3, 0x4, 0x0, 0x0, 0x8, 0x0, 0x0, 0x0, 0x10, 0x0, 0x10, 0x0, 0x0, 0x21, 0x30, 0x0, 0x1, 0x10, 0x0, 0x1, 0x20, 0x0, 0x1, 0x30, 0x0, 0x31, 0x0, 0x0, 0x11, 0x0, 0x0, 0x1, 0x20, 0x0, 0x42, 0x50, 0x0, 0x62, 0x0, 0x0, 0x2, 0x70, 0x0, 0x2, 0x60, 0x0, 0x2, 0x80, 0x0, 0x82, 0x0, 0x0, 0x2, 0x90, 0x0, 0x92, 0x0, 0x0, 0x52, 0x90, 0x0, 0xA2, 0x0, 0x0, 0x3, 0xB0, 0x0, 0x3, 0xC0, 0x0, 0xD3, 0x0, 0x0, 0xC3, 0x0, 0x0, 0xE3, 0x0, 0x0, 0x3, 0xE0, 0x0, 0xF3, 0x0, 0x0, 0x3, 0x1, 0x0, 0x3, 0xD0, 0x0, 0xF3, 0x10, 0x1, 0x23, 0x1, 0x0, 0x3, 0x30, 0x1, 0x3, 0x40, 0x1, 0x4, 0x50, 0x1, 0x4, 0x60, 0x1, 0x4, 0x70, 0x1, 0x84, 0x1, 0x0, 0x94, 0x1, 0x0, 0x4, 0xA0, 0x1, 0xB4, 0x1, 0x0, 0xC4, 0x1, 0x0, 0x74, 0x1, 0x0, 0xD4, 0x1, 0x0, 0xE4, 0x1, 0x0, 0xF4, 0x1, 0x0, 0x4, 0xB0, 0x1, 0x4, 0xC0, 0x1, 0x4, 0x0, 0x2, 0x4, 0x10, 0x2, 0x25, 0x2, 0x0, 0x5, 0x30, 0x2, 0x45, 0x2, 0x0, 0x55, 0x2, 0x0, 0x65, 0x2, 0x0, 0x5, 0x70, 0x2, 0x15, 0x0, 0x0, 0x5, 0x80, 0x2, 0x95, 0x2, 0x0, 0xA5, 0x2, 0x0, 0xB5, 0x2, 0x0, 0xC5, 0x2, 0x0, 0xD5, 0x2, 0x0, 0x5, 0xE0, 0x2, 0x5, 0xF0, 0x2, 0x5, 0x3, 0x0, 0x15, 0x3, 0x0, 0x26, 0x3, 0x0, 0x6, 0x30, 0x3, 0x46, 0x3, 0x0, 0x56, 0x3, 0x0, 0x6, 0x60, 0x3, 0x76, 0x83, 0x3, 0x96, 0x3, 0x0, 0x6, 0xA0, 0x3, 0xB6, 0x3, 0x0, 0xC6, 0x3, 0x0, 0xD6, 0x3, 0x0, 0xE6, 0x3, 0x0, 0x6, 0xF0, 0x3, 0x6, 0x0, 0x4, 0x6, 0x10, 0x4, 0x6, 0x20, 0x4, 0x37, 0x44, 0x4, 0x7, 0x50, 0x4, 0x67, 0x4, 0x0, 0x7, 0x70, 0x4, 0x87, 0x4, 0x0, 0x97, 0x4, 0x0, 0x7, 0xA0, 0x4, 0xB7, 0x4, 0x0, 0xC7, 0x4, 0x0, 0xD7, 0x4, 0x0, 0xE7, 0x4, 0x0, 0x7, 0xF0, 0x4, 0x7, 0x5, 0x0, 0x17, 0x25, 0x5, 0x38, 0x45, 0x5, 0x58, 0x5, 0x0, 0x8, 0x60, 0x5, 0x8, 0x70, 0x5, 0x88, 0x5, 0x0, 0x8, 0x90, 0x5, 0xA8, 0xB5, 0x5, 0xC8, 0x5, 0x0, 0xD8, 0xE5, 0x5, 0xF8, 0x5, 0x0, 0x8, 0x0, 0x6, 0x19, 0x26, 0x6, 0x9, 0x30, 0x6, 0x9, 0x40, 0x6, 0x59, 0x6, 0x0, 0x69, 0x6, 0x0, 0x79, 0x86, 0x6, 0x99, 0xA6, 0x6, 0x9, 0xB0, 0x6, 0xCA, 0x6, 0x0, 0xA, 0xD0, 0x6, 0xA, 0xE0, 0x6, 0xFA, 0x6, 0x0, 0xA, 0x17, 0x7, 0x2A, 0x37, 0x7, 0x4B, 0x7, 0x0, 0x5B, 0x7, 0x0, 0xB, 0x60, 0x7, 0xB, 0x70, 0x7, 0x8B, 0x97, 0x7, 0xAC, 0x7, 0x0, 0xC, 0xB0, 0x7, 0xC, 0xC0, 0x7, 0xDC, 0xE7, 0x7, 0xFD, 0x7, 0x0, 0xD, 0x8, 0x0, 0x1D, 0x28, 0x8, 0x3E, 0x48, 0x8, 0xE, 0x50, 0x8, 0x6F, 0x78, 0x8, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.ModifierSymbol.</summary>
        private static ReadOnlySpan<byte> SerializedCategory27_ModifierSymbol => new byte[] { 0x3, 0x4, 0x0, 0x0, 0x8, 0x0, 0x0, 0x10, 0x0, 0x0, 0x0, 0x10, 0x0, 0x21, 0x0, 0x0, 0x1, 0x30, 0x0, 0x31, 0x0, 0x0, 0x31, 0x10, 0x0, 0x11, 0x0, 0x0, 0x31, 0x20, 0x0, 0x1, 0x10, 0x0, 0x11, 0x20, 0x0, 0x1, 0x20, 0x0, 0x31, 0x0, 0x0, 0x2, 0x40, 0x0, 0x2, 0x50, 0x0, 0x62, 0x0, 0x0, 0x2, 0x70, 0x0, 0x12, 0x80, 0x0, 0x2, 0x80, 0x0, 0x2, 0x90, 0x0, 0x2, 0xA0, 0x0, 0x72, 0x10, 0x0, 0x52, 0x40, 0x0, 0x92, 0x0, 0x0, 0xB2, 0x0, 0x0, 0x22, 0x0, 0x0, 0xA2, 0x0, 0x0, 0xC2, 0x0, 0x0, 0x22, 0x10, 0x0, 0xA2, 0x80, 0x0, 0x52, 0x0, 0x0, 0x42, 0x50, 0x0, 0xD2, 0x0, 0x0, 0x3, 0xE0, 0x0, 0x3, 0xF0, 0x0, 0x3, 0x1, 0x0, 0x3, 0x10, 0x1, 0x13, 0x20, 0x1, 0x33, 0x1, 0x0, 0xE3, 0x0, 0x0, 0x43, 0x1, 0x0, 0x3, 0x50, 0x1, 0x3, 0x60, 0x1, 0x73, 0x1, 0x0, 0x83, 0x1, 0x0, 0x93, 0x1, 0x0, 0x93, 0x51, 0x1, 0xA3, 0x1, 0x0, 0xF3, 0x0, 0x0, 0x3, 0xB0, 0x1, 0x3, 0xC0, 0x1, 0xD3, 0x91, 0x1, 0x3, 0xE0, 0x1, 0xF3, 0xF0, 0x1, 0x3, 0x2, 0x0, 0x13, 0x2, 0x0, 0x4, 0x20, 0x2, 0x4, 0x30, 0x2, 0x44, 0x2, 0x0, 0x4, 0x50, 0x2, 0x64, 0x2, 0x0, 0x4, 0x70, 0x2, 0x84, 0x2, 0x0, 0x4, 0x90, 0x2, 0x4, 0xA0, 0x2, 0xB4, 0x12, 0x0, 0xC4, 0x2, 0x0, 0xD4, 0xE2, 0x2, 0xE4, 0xF2, 0x2, 0x4, 0x3, 0x0, 0x4, 0x10, 0x3, 0x4, 0x20, 0x3, 0x34, 0x3, 0x0, 0x14, 0x40, 0x3, 0x64, 0x52, 0x3, 0x64, 0x73, 0x3, 0x84, 0x3, 0x0, 0x95, 0x3, 0x0, 0x5, 0xA0, 0x3, 0xB5, 0x3, 0x0, 0x5, 0xC0, 0x3, 0xD5, 0x3, 0x0, 0xE5, 0xF3, 0x3, 0x5, 0x0, 0x4, 0x15, 0x24, 0x4, 0x5, 0x30, 0x4, 0x45, 0x54, 0x4, 0x65, 0x4, 0x0, 0x5, 0x70, 0x4, 0x5, 0x80, 0x4, 0x95, 0x4, 0x0, 0xA5, 0xB4, 0x4, 0xC5, 0x4, 0x0, 0xA5, 0xD3, 0x4, 0xE6, 0x4, 0x0, 0xF6, 0x4, 0x5, 0x16, 0x25, 0x5, 0x36, 0x5, 0x0, 0x6, 0x40, 0x5, 0x6, 0x50, 0x5, 0x6, 0x60, 0x5, 0x76, 0x85, 0x5, 0x6, 0x90, 0x5, 0x6, 0xA0, 0x5, 0xB6, 0x5, 0x0, 0xC6, 0x5, 0x0, 0x6, 0xD0, 0x5, 0xE6, 0x5, 0x0, 0xF7, 0x5, 0x6, 0x17, 0x6, 0x0, 0x7, 0x20, 0x6, 0x37, 0x46, 0x6, 0x57, 0x6, 0x0, 0x67, 0x6, 0x0, 0x77, 0x6, 0x0, 0x87, 0x96, 0x6, 0xA7, 0x6, 0x0, 0xB7, 0xC6, 0x6, 0xD8, 0x6, 0x0, 0xE8, 0x6, 0x0, 0xF8, 0x6, 0x0, 0x8, 0x7, 0x0, 0x8, 0x10, 0x7, 0x28, 0x7, 0x0, 0x8, 0x30, 0x7, 0x48, 0x57, 0x7, 0x8, 0x60, 0x7, 0x79, 0x7, 0x0, 0x89, 0x7, 0x0, 0x99, 0x7, 0x0, 0xA9, 0x7, 0x0, 0x9, 0xB0, 0x7, 0xC9, 0x7, 0x0, 0x9, 0xD0, 0x7, 0xE9, 0xF7, 0x7, 0xA, 0x18, 0x8, 0xA, 0x20, 0x8, 0x3A, 0x8, 0x0, 0xA, 0x40, 0x8, 0x5A, 0x8, 0x0, 0xA, 0x60, 0x8, 0xA, 0x70, 0x8, 0x8B, 0x8, 0x0, 0x9B, 0xA8, 0x8, 0xB, 0xB0, 0x8, 0xCB, 0x8, 0x0, 0xDB, 0xE8, 0x8, 0xFC, 0x8, 0x0, 0xC, 0x0, 0x9, 0x1C, 0x9, 0x0, 0x2C, 0x39, 0x9, 0x4D, 0x9, 0x0, 0x5D, 0x9, 0x0, 0x6D, 0x79, 0x9, 0x8E, 0x99, 0x9, 0xE, 0xA0, 0x9, 0xBF, 0xC9, 0x9, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.OtherSymbol.</summary>
        private static ReadOnlySpan<byte> SerializedCategory28_OtherSymbol => new byte[] { 0x3, 0x4, 0x0, 0x0, 0x9, 0x0, 0x0, 0x0, 0x20, 0x0, 0x10, 0x0, 0x0, 0x1, 0x20, 0x0, 0x21, 0x60, 0x0, 0x1, 0x40, 0x0, 0x11, 0x60, 0x0, 0x31, 0x0, 0x0, 0x1, 0x60, 0x0, 0x11, 0x0, 0x0, 0x21, 0x20, 0x0, 0x31, 0x20, 0x0, 0x11, 0x40, 0x0, 0x31, 0x40, 0x0, 0x21, 0x0, 0x0, 0x42, 0x0, 0x0, 0x52, 0xC0, 0x0, 0x62, 0x0, 0x0, 0x72, 0x0, 0x0, 0x82, 0x0, 0x0, 0x2, 0x80, 0x0, 0x2, 0x20, 0x1, 0xA2, 0x0, 0x0, 0x2, 0x20, 0x0, 0xB2, 0x20, 0x0, 0x2, 0xC0, 0x0, 0x12, 0x40, 0x1, 0xA2, 0x80, 0x0, 0x2, 0x40, 0x1, 0x42, 0x20, 0x0, 0x12, 0x80, 0x1, 0x2, 0x60, 0x1, 0xC2, 0x20, 0x0, 0xA2, 0x20, 0x0, 0x52, 0x0, 0x0, 0x12, 0x0, 0x0, 0x12, 0xA0, 0x1, 0x72, 0x20, 0x0, 0x12, 0xC0, 0x1, 0x72, 0x80, 0x1, 0xC2, 0xA0, 0x0, 0x12, 0x80, 0x0, 0xC2, 0xE0, 0x1, 0xF2, 0x20, 0x1, 0x32, 0x20, 0x0, 0xD2, 0x0, 0x0, 0xB2, 0x80, 0x1, 0x62, 0x20, 0x0, 0x12, 0xE0, 0x0, 0x92, 0x0, 0x0, 0xD2, 0x20, 0x0, 0x22, 0x0, 0x0, 0x72, 0x0, 0x1, 0x2, 0xE0, 0x0, 0x2, 0x40, 0x0, 0x12, 0x0, 0x1, 0x2, 0xE0, 0x1, 0xF2, 0x0, 0x0, 0x3, 0x1, 0x0, 0x13, 0x41, 0x2, 0x33, 0x1, 0x0, 0x43, 0x1, 0x0, 0x53, 0x81, 0x2, 0x63, 0xE1, 0x2, 0x83, 0x1, 0x0, 0x3, 0x20, 0x3, 0x3, 0x40, 0x3, 0xB3, 0x1, 0x0, 0x3, 0x20, 0x0, 0x93, 0x21, 0x0, 0x3, 0x0, 0x3, 0x13, 0x80, 0x3, 0x3, 0xA0, 0x3, 0x3, 0x40, 0x2, 0x83, 0x21, 0x0, 0x3, 0xC0, 0x3, 0xF3, 0x21, 0x0, 0x3, 0xA0, 0x2, 0x3, 0x62, 0x2, 0x13, 0x20, 0x4, 0x13, 0x40, 0x4, 0x33, 0x61, 0x4, 0x13, 0x80, 0x4, 0x13, 0xA0, 0x4, 0x13, 0x20, 0x3, 0x53, 0x21, 0x0, 0x43, 0x2, 0x0, 0x3, 0x22, 0x0, 0x13, 0x60, 0x3, 0x63, 0x22, 0x0, 0x73, 0x62, 0x3, 0x43, 0x22, 0x0, 0x13, 0x0, 0x5, 0xE3, 0x21, 0x0, 0x13, 0x22, 0x5, 0xA3, 0x62, 0x2, 0xD3, 0x1, 0x0, 0xB3, 0x2, 0x0, 0xC3, 0xA2, 0x5, 0x73, 0xC1, 0x5, 0x53, 0xE1, 0x5, 0x3, 0x83, 0x4, 0x3, 0x22, 0x6, 0x73, 0x1, 0x0, 0x23, 0x3, 0x0, 0xA3, 0x61, 0x2, 0x33, 0x23, 0x4, 0xA3, 0x81, 0x6, 0xB3, 0xA1, 0x6, 0x3, 0xC0, 0x6, 0x63, 0x1, 0x0, 0x73, 0x3, 0x7, 0x93, 0x3, 0x0, 0x3, 0x40, 0x7, 0x33, 0x2, 0x0, 0xA3, 0x3, 0x0, 0x3, 0x20, 0x7, 0xC3, 0x42, 0x7, 0xB4, 0x83, 0x7, 0xD4, 0x3, 0x0, 0x4, 0xC0, 0x7, 0x4, 0x20, 0x0, 0xF4, 0x3, 0x0, 0x4, 0x24, 0x8, 0x4, 0x40, 0x8, 0x14, 0x0, 0x0, 0x34, 0x24, 0x0, 0x14, 0x80, 0x8, 0x34, 0xA4, 0x8, 0x64, 0x24, 0x0, 0x4, 0xE0, 0x8, 0x84, 0x4, 0x0, 0x74, 0x61, 0x8, 0x94, 0x44, 0x9, 0xB4, 0x4, 0x0, 0xC4, 0x24, 0x0, 0x74, 0x24, 0x0, 0xD4, 0x24, 0x0, 0xE4, 0x4, 0x0, 0x4, 0xE0, 0x9, 0x4, 0x25, 0x0, 0x14, 0x25, 0x0, 0x14, 0x40, 0xA, 0x34, 0x5, 0x0, 0x4, 0xA0, 0x8, 0x14, 0xC0, 0x8, 0x54, 0x24, 0x0, 0x14, 0x80, 0xA, 0x54, 0x25, 0x0, 0x4, 0xC0, 0xA, 0x74, 0x5, 0x0, 0x4, 0x0, 0xB, 0x14, 0x20, 0xB, 0xB4, 0x24, 0x0, 0x84, 0x25, 0x0, 0xA4, 0x25, 0x0, 0x14, 0x60, 0xB, 0x14, 0x80, 0xB, 0xD4, 0xC5, 0xB, 0x14, 0xE0, 0xB, 0x4, 0x26, 0xC, 0x4, 0x40, 0xC, 0x14, 0x66, 0xC, 0x44, 0xA6, 0xC, 0x64, 0xE6, 0xC, 0x84, 0x6, 0x0, 0x4, 0x60, 0x8, 0x4, 0x20, 0xD, 0x64, 0x5, 0x0, 0xA4, 0x66, 0xD, 0xC4, 0x6, 0x0, 0xD4, 0xC6, 0xD, 0xF4, 0x6, 0x0, 0xE4, 0x3, 0x0, 0x4, 0x7, 0x0, 0x34, 0x4, 0x0, 0x14, 0x7, 0x0, 0x24, 0x7, 0x0, 0x34, 0xE7, 0xD, 0x44, 0x7, 0x0, 0x4, 0x0, 0xD, 0x4, 0x60, 0xE, 0x4, 0xA0, 0xE, 0x34, 0xC4, 0xE, 0x75, 0x7, 0x0, 0x85, 0x27, 0xF, 0x5, 0x40, 0xF, 0xB5, 0x7, 0x0, 0xC5, 0x7, 0x0, 0x5, 0xA0, 0xF, 0x15, 0xC0, 0xF, 0xF5, 0x7, 0x10, 0x15, 0x20, 0x10, 0x5, 0x48, 0x10, 0x35, 0x28, 0x0, 0x5, 0x80, 0x10, 0x55, 0xC8, 0x10, 0x75, 0x8, 0x11, 0x95, 0x28, 0x0, 0x15, 0x40, 0x11, 0x5, 0x60, 0x11, 0xC5, 0x8, 0x0, 0x15, 0xA0, 0x11, 0xE5, 0xE8, 0x11, 0xA5, 0x27, 0x0, 0x15, 0x0, 0x12, 0x15, 0x29, 0x0, 0x25, 0x29, 0x0, 0x35, 0x89, 0x12, 0x55, 0x29, 0x0, 0x65, 0x29, 0x0, 0x15, 0xE0, 0x12, 0x5, 0x0, 0x13, 0xD5, 0x27, 0x0, 0x95, 0x49, 0x13, 0x5, 0x69, 0x13, 0xC5, 0x29, 0x0, 0xD5, 0xC9, 0x13, 0x95, 0xE8, 0x13, 0x5, 0x2A, 0x14, 0x5, 0x40, 0x14, 0x35, 0x8A, 0x14, 0x55, 0xA, 0x0, 0x15, 0xC0, 0x14, 0x5, 0xE0, 0x14, 0x85, 0xA, 0x0, 0x5, 0x20, 0x15, 0x5, 0xC0, 0x14, 0x5, 0x40, 0x15, 0x65, 0xA, 0x0, 0xB5, 0x8A, 0x15, 0xD5, 0x2A, 0xF, 0xE5, 0xA, 0x0, 0xF5, 0xA, 0x0, 0x5, 0xB, 0x0, 0x15, 0xB, 0x0, 0x25, 0xB, 0x0, 0x35, 0x8B, 0x16, 0x5, 0xA0, 0x16, 0x5, 0xC0, 0x16, 0x5, 0xE0, 0x16, 0x85, 0xB, 0x0, 0x96, 0xB, 0x0, 0xA6, 0xB, 0x0, 0xB6, 0xB, 0x0, 0xC6, 0xB, 0x0, 0x6, 0xA0, 0x17, 0xE6, 0xEB, 0x17, 0x16, 0x0, 0x0, 0x16, 0x0, 0x18, 0x16, 0x4C, 0x18, 0x36, 0x8C, 0x18, 0x6, 0xA0, 0x18, 0x66, 0x2C, 0x0, 0x76, 0xC, 0x19, 0x96, 0xC, 0x0, 0xA6, 0xC, 0x0, 0x16, 0x60, 0x19, 0xC6, 0xAC, 0x19, 0x6, 0xC0, 0x19, 0xF6, 0x2C, 0x0, 0x6, 0x2D, 0x0, 0x16, 0x4D, 0x1A, 0x36, 0x8D, 0x1A, 0x56, 0xCD, 0x1A, 0x76, 0xD, 0x1B, 0x96, 0x4D, 0x1B, 0xB6, 0x8D, 0x1B, 0xD6, 0xCD, 0x1B, 0xF6, 0xD, 0x0, 0x6, 0xE, 0x0, 0x16, 0xE, 0x0, 0x26, 0xE, 0x0, 0x6, 0x60, 0x1C, 0x6, 0x80, 0x1C, 0x56, 0xCE, 0x1C, 0x6, 0xE0, 0x1C, 0x86, 0xE, 0x0, 0x96, 0xE, 0x0, 0xA6, 0xE, 0x0, 0xB6, 0xE, 0x0, 0xC6, 0xE, 0x0, 0xD6, 0xE, 0x0, 0xE6, 0xE, 0x0, 0x6, 0xE0, 0x1D, 0x6, 0x0, 0x1E, 0x6, 0x20, 0x1E, 0x6, 0x40, 0x1E, 0x37, 0xF, 0x0, 0x47, 0xAF, 0x1E, 0x7, 0xC0, 0x1E, 0x7, 0xE0, 0x1E, 0x87, 0xF, 0x0, 0x97, 0xF, 0x0, 0xA7, 0x6F, 0x1F, 0xC7, 0xF, 0x0, 0x7, 0xA0, 0x1F, 0xE7, 0x2F, 0x0, 0xF7, 0xF, 0x20, 0x17, 0x10, 0x0, 0x27, 0x70, 0x20, 0x47, 0xB0, 0x20, 0x17, 0xC0, 0x20, 0x77, 0x30, 0x0, 0x87, 0x30, 0x21, 0xA7, 0x70, 0x21, 0xC7, 0xB0, 0x21, 0x7, 0xC0, 0x21, 0xF7, 0x10, 0x22, 0x7, 0x20, 0x22, 0x27, 0x11, 0x0, 0x37, 0x11, 0x0, 0x47, 0xB1, 0x22, 0x7, 0xC0, 0x22, 0x7, 0xE0, 0x22, 0x87, 0x31, 0x23, 0xA7, 0x11, 0x0, 0xB7, 0x11, 0x0, 0xC7, 0xB1, 0x23, 0xE7, 0x11, 0x0, 0xF7, 0x11, 0x0, 0x7, 0x12, 0x0, 0x18, 0x12, 0x0, 0x28, 0x12, 0x0, 0x8, 0x60, 0x24, 0x8, 0x80, 0x24, 0x8, 0xA0, 0x24, 0x68, 0x12, 0x0, 0x18, 0xE0, 0x24, 0x88, 0x32, 0x25, 0xA8, 0x72, 0x25, 0x8, 0x80, 0x25, 0xD8, 0x12, 0x0, 0x8, 0x20, 0x0, 0xE8, 0xF2, 0x25, 0x8, 0x33, 0x26, 0x28, 0x13, 0x0, 0x38, 0x13, 0x0, 0x48, 0x13, 0x0, 0x58, 0x13, 0x0, 0x8, 0xC0, 0x26, 0x78, 0x13, 0x0, 0x8, 0x0, 0x27, 0x98, 0x13, 0x0, 0xA8, 0x73, 0x27, 0xC8, 0x13, 0x0, 0xD8, 0x13, 0x0, 0xE8, 0xF3, 0x27, 0x8, 0x34, 0x28, 0x8, 0x40, 0x28, 0x39, 0x94, 0x28, 0x59, 0xD4, 0x28, 0x9, 0xE0, 0x28, 0x9, 0x0, 0x29, 0x99, 0x54, 0x29, 0xB9, 0x94, 0x29, 0xD9, 0xD4, 0x29, 0xF9, 0x14, 0x2A, 0x19, 0x55, 0x2A, 0x39, 0x95, 0x2A, 0x59, 0x15, 0x0, 0x69, 0xF5, 0x2A, 0x89, 0x35, 0x2B, 0xA9, 0x75, 0x2B, 0xC9, 0xB5, 0x2B, 0x9, 0xC0, 0x2B, 0xFA, 0x15, 0x0, 0xA, 0x0, 0x2C, 0x1A, 0x16, 0x0, 0x2A, 0x16, 0x0, 0xA, 0x60, 0x2C, 0x4A, 0xB6, 0x2C, 0x6A, 0xF6, 0x2C, 0xA, 0x0, 0x2D, 0x9A, 0x56, 0x2D, 0xBA, 0x96, 0x2D, 0xDA, 0xD6, 0x2D, 0xFB, 0x16, 0x0, 0xB, 0x37, 0x2E, 0x2B, 0x17, 0x0, 0xB, 0x60, 0x2E, 0x4B, 0xB7, 0x2E, 0x6B, 0xF7, 0x2E, 0x8B, 0x37, 0x2F, 0xAC, 0x17, 0x0, 0xC, 0x60, 0x2F, 0xC, 0x80, 0x2F, 0xDC, 0xD7, 0x2F, 0xFC, 0x17, 0x30, 0x1D, 0x18, 0x0, 0x2D, 0x18, 0x0, 0xD, 0x60, 0x30, 0x4D, 0xB8, 0x30, 0x6E, 0xF8, 0x30, 0x8E, 0x38, 0x31, 0xAF, 0x78, 0x31, };

        /// <summary>Serialized BDD representation of the set of all characters in UnicodeCategory.OtherNotAssigned.</summary>
        private static ReadOnlySpan<byte> SerializedCategory29_OtherNotAssigned => new byte[] { 0x3, 0x4, 0x0, 0x0, 0xA, 0x0, 0x0, 0x0, 0x40, 0x0, 0x10, 0x0, 0x0, 0x11, 0x0, 0x0, 0x1, 0x80, 0x0, 0x31, 0x0, 0x0, 0x11, 0xC0, 0x0, 0x1, 0x40, 0x0, 0x21, 0xC0, 0x0, 0x1, 0xC0, 0x0, 0x21, 0x40, 0x0, 0x21, 0x0, 0x0, 0x11, 0x80, 0x0, 0x31, 0x80, 0x0, 0x31, 0x40, 0x0, 0x42, 0x40, 0x1, 0x62, 0x0, 0x0, 0x72, 0x0, 0x2, 0x2, 0x0, 0x2, 0x2, 0x40, 0x1, 0x92, 0x0, 0x0, 0xA2, 0x0, 0x0, 0x12, 0x0, 0x0, 0x2, 0x80, 0x1, 0x12, 0x0, 0x1, 0xB2, 0x40, 0x0, 0x2, 0xC0, 0x2, 0x12, 0x80, 0x1, 0xA2, 0x0, 0x3, 0x32, 0x0, 0x0, 0x52, 0x40, 0x0, 0x42, 0x0, 0x0, 0x62, 0x40, 0x1, 0x2, 0x0, 0x1, 0x82, 0x0, 0x1, 0xC2, 0x0, 0x0, 0x72, 0x0, 0x0, 0x82, 0x40, 0x0, 0x52, 0x0, 0x3, 0x2, 0x0, 0x3, 0x12, 0xC0, 0x1, 0xD2, 0x40, 0x0, 0x82, 0x0, 0x0, 0xA2, 0x0, 0x2, 0x52, 0x0, 0x0, 0x52, 0x80, 0x1, 0x52, 0x0, 0x1, 0x2, 0x40, 0x0, 0x82, 0x80, 0x1, 0x2, 0xC0, 0x1, 0x12, 0x80, 0x2, 0x42, 0x80, 0x2, 0x22, 0x0, 0x0, 0xA2, 0x80, 0x3, 0xB2, 0x80, 0x1, 0xB2, 0xC0, 0x3, 0xD2, 0x0, 0x0, 0x2, 0x80, 0x2, 0x12, 0x40, 0x1, 0xE2, 0x40, 0x0, 0xD2, 0x80, 0x1, 0xB2, 0xC0, 0x1, 0x72, 0xC0, 0x2, 0x52, 0x0, 0x2, 0xC2, 0x40, 0x0, 0x42, 0x40, 0x2, 0x52, 0x80, 0x2, 0x2, 0x40, 0x2, 0x42, 0x0, 0x3, 0x32, 0x40, 0x1, 0x12, 0x40, 0x3, 0xA2, 0x0, 0x1, 0xE2, 0x80, 0x2, 0x62, 0x40, 0x2, 0x82, 0xC0, 0x0, 0xF2, 0x0, 0x0, 0xA2, 0x80, 0x1, 0x3, 0x41, 0x0, 0x23, 0xC1, 0x4, 0x13, 0x1, 0x0, 0x3, 0x0, 0x5, 0x53, 0x1, 0x0, 0x3, 0x80, 0x5, 0x73, 0x41, 0x4, 0x3, 0x0, 0x6, 0x93, 0x1, 0x0, 0xA3, 0x1, 0x0, 0x3, 0xC0, 0x4, 0x3, 0xC0, 0x6, 0x13, 0x0, 0x7, 0x3, 0x40, 0x7, 0xE3, 0x41, 0x4, 0xF3, 0xC1, 0x6, 0x13, 0x40, 0x4, 0x3, 0x2, 0x0, 0x73, 0x1, 0x0, 0xB3, 0x41, 0x4, 0x13, 0xC0, 0x5, 0x13, 0x40, 0x8, 0xB3, 0x41, 0x0, 0x23, 0x2, 0x0, 0x33, 0x2, 0x0, 0x43, 0x2, 0x0, 0x53, 0x2, 0x0, 0xA3, 0xC1, 0x5, 0x63, 0x2, 0x8, 0x13, 0x0, 0x0, 0x93, 0xC1, 0x9, 0xC3, 0x1, 0x0, 0x3, 0xC0, 0x7, 0x43, 0x41, 0x4, 0x13, 0x0, 0x8, 0x83, 0x2, 0x0, 0xA3, 0x41, 0xA, 0xA3, 0x2, 0x9, 0x43, 0xC1, 0x5, 0x3, 0x0, 0x9, 0x3, 0xC0, 0xA, 0x13, 0x40, 0xA, 0x3, 0x80, 0x8, 0x13, 0x1, 0xB, 0xD3, 0xC2, 0xA, 0x23, 0x0, 0x0, 0xE3, 0x2, 0x0, 0x93, 0x2, 0x0, 0xF3, 0x2, 0x0, 0xB3, 0x1, 0x0, 0x3, 0xC3, 0x5, 0xB3, 0x2, 0x0, 0x13, 0x3, 0x0, 0x13, 0x40, 0x9, 0x3, 0x80, 0xC, 0x13, 0xC0, 0xC, 0x63, 0x1, 0x0, 0xA3, 0x1, 0x8, 0x3, 0x0, 0xD, 0x3, 0x40, 0x4, 0x13, 0x1, 0xD, 0x43, 0x3, 0x0, 0x43, 0x43, 0x4, 0x43, 0x1, 0x0, 0x13, 0x81, 0x7, 0x3, 0x40, 0xD, 0x83, 0x81, 0xD, 0x73, 0x3, 0x0, 0x13, 0x80, 0x4, 0x3, 0x80, 0x9, 0x3, 0x80, 0x7, 0x83, 0x43, 0x4, 0x93, 0x3, 0xA, 0x33, 0x41, 0x4, 0x3, 0x40, 0x1, 0x3, 0x0, 0xC, 0xA3, 0x83, 0x5, 0x3, 0x80, 0xE, 0x13, 0xC0, 0xE, 0x43, 0x83, 0x5, 0x23, 0x42, 0xB, 0xA3, 0x3, 0x0, 0x3, 0x80, 0x6, 0xD3, 0x3, 0xF, 0x13, 0x80, 0xF, 0x43, 0x43, 0xC, 0x13, 0xC3, 0xF, 0x63, 0x3, 0xD, 0x13, 0x3, 0x10, 0x13, 0xC4, 0x7, 0x23, 0x44, 0x5, 0x23, 0xC2, 0x10, 0x3, 0x0, 0x11, 0x53, 0x41, 0x1, 0x43, 0x41, 0x6, 0x53, 0x4, 0x9, 0x3, 0x0, 0xA, 0x43, 0x42, 0x1, 0x63, 0xC4, 0x11, 0x23, 0xC4, 0xD, 0x83, 0x44, 0x12, 0x73, 0x43, 0x1, 0x13, 0x84, 0x6, 0xA3, 0x44, 0x5, 0x23, 0xC2, 0x12, 0x53, 0x41, 0xB, 0x3, 0x40, 0x6, 0xC3, 0x4, 0x0, 0x13, 0x40, 0x6, 0xD3, 0x4, 0xC, 0x33, 0x1, 0x0, 0xE4, 0x44, 0x4, 0xF4, 0xC4, 0x4, 0x4, 0x5, 0x0, 0x4, 0x40, 0x14, 0x24, 0x5, 0x0, 0x34, 0x5, 0x15, 0x54, 0x5, 0x0, 0x64, 0x5, 0x0, 0x4, 0x40, 0x0, 0x14, 0xC0, 0x15, 0x84, 0x5, 0x0, 0x94, 0x85, 0x16, 0x4, 0xC0, 0x16, 0xC4, 0x5, 0x0, 0xD4, 0x85, 0x17, 0x4, 0xC0, 0x17, 0x4, 0x6, 0x0, 0x4, 0x40, 0x18, 0x4, 0x80, 0x18, 0x64, 0xC5, 0x17, 0x4, 0x0, 0x18, 0x4, 0x40, 0x4, 0x34, 0x46, 0x8, 0xE4, 0x5, 0x0, 0x44, 0x86, 0x16, 0x54, 0xC6, 0x17, 0x64, 0xC6, 0x19, 0x84, 0x6, 0x0, 0x94, 0x6, 0x0, 0x64, 0x85, 0x1A, 0xB4, 0x6, 0x0, 0x64, 0x5, 0x1A, 0x84, 0x45, 0x0, 0xC4, 0x46, 0x1B, 0x14, 0x0, 0x18, 0x4, 0x80, 0x17, 0x4, 0x0, 0x1A, 0x4, 0x0, 0x14, 0x14, 0x5, 0x0, 0x4, 0x80, 0x1B, 0xF4, 0x6, 0x0, 0x4, 0x46, 0x0, 0x4, 0x7, 0x0, 0x24, 0x6, 0x0, 0x14, 0x7, 0x0, 0xF4, 0x5, 0x0, 0x24, 0xC7, 0x15, 0x4, 0xC0, 0x1C, 0x44, 0x7, 0x0, 0x54, 0x7, 0x0, 0x64, 0x7, 0x0, 0x14, 0x40, 0x1B, 0x14, 0x80, 0x17, 0x74, 0x7, 0x0, 0x4, 0xC0, 0x1D, 0x84, 0x6, 0x14, 0x84, 0xC7, 0x14, 0x94, 0x7, 0x0, 0xA4, 0xC7, 0x14, 0x34, 0x5, 0x0, 0xB4, 0x7, 0x8, 0x4, 0x2, 0x0, 0xD4, 0x6, 0x0, 0x4, 0xC0, 0x1A, 0xC4, 0x7, 0x0, 0x4, 0x40, 0x1F, 0x4, 0x80, 0x1F, 0xF4, 0x7, 0x0, 0x4, 0x8, 0x0, 0x14, 0x0, 0x14, 0x14, 0x8, 0x0, 0x24, 0x88, 0x15, 0x34, 0xC8, 0x17, 0x4, 0x0, 0x21, 0x4, 0x40, 0x1B, 0xD4, 0x7, 0x0, 0x54, 0x88, 0x21, 0x74, 0x8, 0x0, 0x84, 0x8, 0x0, 0x94, 0x8, 0x22, 0xA4, 0x8, 0x0, 0x4, 0xC0, 0x22, 0xC4, 0xC8, 0x22, 0xD4, 0x86, 0x21, 0x64, 0x8, 0x0, 0xD4, 0x8, 0x0, 0x14, 0x5, 0x1A, 0x4, 0x40, 0x23, 0x54, 0x86, 0x23, 0xF4, 0xC5, 0x23, 0x4, 0x0, 0x24, 0x14, 0x9, 0x0, 0x24, 0xC9, 0x24, 0x44, 0x49, 0x25, 0x64, 0x9, 0x0, 0x74, 0x9, 0x26, 0x4, 0x80, 0x1D, 0x94, 0x89, 0x26, 0xB4, 0x89, 0x21, 0xC4, 0x89, 0x1D, 0xC4, 0x43, 0x27, 0xE4, 0xC9, 0x27, 0x4, 0x8A, 0x1D, 0x14, 0x4A, 0x27, 0x54, 0xC6, 0x27, 0xD4, 0xC6, 0x24, 0x24, 0xCA, 0x28, 0x64, 0x6, 0x29, 0x54, 0x8A, 0x29, 0xB4, 0x86, 0x1D, 0x74, 0xA, 0x2A, 0x94, 0xCA, 0x27, 0xA4, 0xCA, 0x2A, 0xC4, 0x8A, 0x1D, 0x74, 0x47, 0x2B, 0xE4, 0xCA, 0x2B, 0xE4, 0xC5, 0x24, 0x4, 0x4B, 0x2C, 0x24, 0xCB, 0x27, 0xA4, 0xCA, 0x2C, 0x4, 0x85, 0x1D, 0x44, 0x4B, 0x2D, 0x64, 0xCB, 0x27, 0xA4, 0xCA, 0x2D, 0x84, 0xB, 0x14, 0x94, 0xB, 0x0, 0x4, 0xC5, 0x17, 0xA4, 0xB, 0x0, 0x4, 0x0, 0x1F, 0x4, 0xC0, 0x19, 0x34, 0x48, 0x24, 0x14, 0x5, 0x1F, 0x4, 0x80, 0x2B, 0x4, 0xC0, 0x2E, 0xC4, 0xB, 0x0, 0xD5, 0x8B, 0x2F, 0xF5, 0xB, 0x0, 0x5, 0x0, 0x30, 0x15, 0xC, 0x0, 0x25, 0xCC, 0x30, 0x5, 0x0, 0x31, 0x55, 0x8C, 0x31, 0x5, 0xC0, 0x31, 0x5, 0x0, 0x32, 0x5, 0x40, 0x32, 0xA5, 0xCC, 0x32, 0x15, 0x0, 0x31, 0xC5, 0xC, 0x0, 0xD5, 0x8C, 0x33, 0xF5, 0xC, 0x0, 0x5, 0xD, 0x0, 0x15, 0xD, 0x0, 0x25, 0xCD, 0x34, 0x45, 0x4D, 0x35, 0x5, 0x80, 0x35, 0x45, 0xD, 0x0, 0xF5, 0xCB, 0x35, 0x85, 0x4D, 0x36, 0x5, 0x80, 0x36, 0xB5, 0xD, 0x0, 0xC5, 0xD, 0x0, 0xD5, 0x8D, 0x37, 0xF5, 0xD, 0x0, 0x5, 0x0, 0x38, 0x5, 0x40, 0x38, 0x5, 0xC0, 0x2F, 0x5, 0x80, 0x38, 0x35, 0xE, 0x39, 0x5, 0x40, 0x39, 0x65, 0xCE, 0x39, 0x85, 0xE, 0x0, 0x5, 0x40, 0x3A, 0x15, 0x80, 0x3A, 0x5, 0x40, 0x4, 0x15, 0x1, 0x35, 0xB5, 0xE, 0x0, 0xC5, 0xE, 0x0, 0xD5, 0xE, 0x0, 0x5, 0x80, 0x3B, 0xF5, 0xE, 0x0, 0x5, 0x0, 0x3C, 0x15, 0xF, 0x0, 0x5, 0x40, 0x34, 0x25, 0xCF, 0x3C, 0x5, 0x0, 0x3D, 0x55, 0xF, 0x0, 0x65, 0xCF, 0x3D, 0x85, 0xF, 0x0, 0xA5, 0x4E, 0x3E, 0x5, 0x80, 0x3E, 0xB5, 0xF, 0x3F, 0xD5, 0x8F, 0x3F, 0x5, 0xC0, 0x3F, 0x5, 0x10, 0x0, 0x15, 0x10, 0x0, 0xF5, 0x4B, 0x38, 0x15, 0x80, 0x40, 0xC5, 0x8C, 0x15, 0x15, 0xCC, 0x2F, 0x5, 0xC0, 0x40, 0x5, 0x0, 0x41, 0x55, 0x90, 0x41, 0x5, 0xC6, 0x2F, 0x75, 0xE, 0x0, 0x75, 0x10, 0x0, 0x85, 0x10, 0x0, 0x65, 0x85, 0x3A, 0x95, 0x10, 0x3A, 0x45, 0x8D, 0x42, 0x5, 0x0, 0x36, 0xA5, 0xF, 0x0, 0x85, 0x4D, 0x3F, 0x5, 0xC0, 0x42, 0x5, 0x0, 0x43, 0xD5, 0x90, 0x43, 0x5, 0xC0, 0x43, 0x5, 0x0, 0x3B, 0x15, 0x0, 0x44, 0x15, 0x91, 0x44, 0x35, 0x11, 0x45, 0x15, 0x40, 0x45, 0x65, 0xD1, 0x45, 0x15, 0x40, 0x33, 0x85, 0x11, 0x30, 0x95, 0x91, 0x46, 0xB5, 0x11, 0x47, 0xD5, 0x91, 0x47, 0x5, 0xC0, 0x47, 0x5, 0x52, 0x48, 0x25, 0xD2, 0x47, 0x35, 0x12, 0x49, 0x55, 0xD2, 0x47, 0x65, 0xD2, 0x49, 0x85, 0x52, 0x4A, 0xA5, 0xD2, 0x4A, 0xC5, 0x52, 0x4B, 0xE5, 0xD2, 0x4B, 0xC5, 0x12, 0x4C, 0x15, 0x93, 0x4C, 0x35, 0x13, 0x4D, 0x55, 0x93, 0x4D, 0x75, 0x13, 0x4E, 0x5, 0x40, 0x4E, 0x75, 0x90, 0x4E, 0xB5, 0x13, 0x0, 0xD5, 0xF, 0x0, 0xC5, 0x13, 0x0, 0x5, 0x40, 0x4F, 0x5, 0x80, 0x4F, 0xF5, 0x13, 0x3F, 0x5, 0x0, 0x50, 0x35, 0xE, 0x0, 0x15, 0x94, 0x50, 0x35, 0x14, 0x0, 0x46, 0x54, 0x51, 0x6, 0x80, 0x51, 0x76, 0x14, 0x0, 0x86, 0x54, 0x52, 0xA6, 0xD4, 0x52, 0xC6, 0x14, 0x0, 0xD6, 0x94, 0x53, 0xF6, 0x14, 0x0, 0x6, 0x15, 0x0, 0x16, 0x95, 0x54, 0x36, 0x15, 0x0, 0x46, 0x55, 0x55, 0x66, 0x15, 0x0, 0x76, 0x15, 0x56, 0x96, 0x15, 0x0, 0xA6, 0x15, 0x0, 0xB6, 0x15, 0x0, 0xC6, 0x55, 0x57, 0xE6, 0x15, 0x0, 0xC6, 0x15, 0x0, 0x6, 0xC0, 0x57, 0x6, 0x56, 0x58, 0x6, 0x80, 0x58, 0x26, 0xD5, 0x58, 0x6, 0x0, 0x59, 0x6, 0x40, 0x59, 0x66, 0x14, 0x0, 0x66, 0x16, 0x0, 0x76, 0x16, 0x5A, 0x96, 0x16, 0x0, 0xA6, 0xD6, 0x5A, 0xC6, 0x56, 0x5B, 0xE6, 0x16, 0x0, 0x6, 0xC0, 0x5B, 0x6, 0x17, 0x0, 0x16, 0x97, 0x5C, 0x6, 0xC0, 0x5C, 0x46, 0x57, 0x5D, 0x66, 0x17, 0x0, 0x76, 0x17, 0x5E, 0x96, 0x97, 0x5E, 0xB6, 0x17, 0x5F, 0xD6, 0x97, 0x5F, 0xF6, 0x17, 0x0, 0x6, 0x18, 0x0, 0x16, 0x98, 0x60, 0x36, 0x18, 0x61, 0x56, 0x18, 0x55, 0x66, 0xD8, 0x61, 0x86, 0x58, 0x62, 0xA6, 0x58, 0x52, 0xB6, 0x18, 0x0, 0xC6, 0x58, 0x63, 0xA6, 0x98, 0x63, 0xF6, 0x58, 0x52, 0x6, 0x59, 0x64, 0x26, 0xD9, 0x64, 0x46, 0x19, 0x0, 0x56, 0x19, 0x0, 0x66, 0xD9, 0x65, 0x86, 0x19, 0x0, 0x96, 0x99, 0x66, 0xB6, 0x19, 0x67, 0xD6, 0x99, 0x67, 0xF6, 0x19, 0x68, 0x16, 0x9A, 0x68, 0x36, 0x1A, 0x69, 0x56, 0x9A, 0x69, 0x76, 0x1A, 0x6A, 0x96, 0x9A, 0x6A, 0xB6, 0x1A, 0x6B, 0xD6, 0x9A, 0x6B, 0x6, 0xC0, 0x6B, 0x6, 0x5B, 0x6C, 0x26, 0xDB, 0x6C, 0x46, 0x5B, 0x6D, 0x66, 0xDB, 0x6D, 0x56, 0x16, 0x6E, 0x6, 0x40, 0x6E, 0xA6, 0x1B, 0x0, 0xB7, 0x1B, 0x6F, 0xD7, 0x9B, 0x6F, 0xF7, 0x1B, 0x0, 0x7, 0x5C, 0x70, 0x27, 0xDC, 0x70, 0x47, 0x1C, 0x0, 0x57, 0x9C, 0x71, 0x77, 0x1C, 0x72, 0x97, 0x9C, 0x72, 0xB7, 0x1C, 0x73, 0xD7, 0x1C, 0x0, 0xE7, 0xDC, 0x73, 0x7, 0x1D, 0x0, 0x7, 0x40, 0x74, 0x27, 0xDD, 0x74, 0x47, 0x5D, 0x75, 0x67, 0x1D, 0x0, 0x77, 0x1D, 0x76, 0x97, 0x9D, 0x76, 0xB7, 0x1D, 0x0, 0xC7, 0x5D, 0x77, 0x7, 0x80, 0x77, 0xF7, 0x1D, 0x0, 0x7, 0x5E, 0x78, 0x27, 0xDE, 0x78, 0x47, 0x5E, 0x79, 0x67, 0xDE, 0x79, 0x87, 0x5E, 0x7A, 0xA7, 0xDE, 0x7A, 0xC7, 0x5E, 0x7B, 0xE7, 0xDE, 0x7B, 0x7, 0x1F, 0x0, 0x17, 0x9F, 0x7C, 0x37, 0x1F, 0x7D, 0x57, 0x1F, 0x0, 0x67, 0xDF, 0x7D, 0x87, 0x5F, 0x7E, 0xA7, 0xDF, 0x7E, 0xC7, 0x5F, 0x7F, 0xE7, 0xDF, 0x7F, 0x7, 0x60, 0x80, 0x27, 0x20, 0x0, 0x37, 0x20, 0x81, 0x57, 0xA0, 0x81, 0x77, 0x20, 0x82, 0x97, 0xA0, 0x82, 0xB8, 0x20, 0x83, 0xD8, 0x20, 0x0, 0xE8, 0xE0, 0x83, 0x8, 0x21, 0x0, 0x18, 0xA1, 0x84, 0x38, 0x21, 0x85, 0x58, 0xA1, 0x85, 0x8, 0xC0, 0x85, 0x8, 0x0, 0x86, 0x98, 0xA1, 0x86, 0xB8, 0x21, 0x87, 0xD8, 0xA1, 0x87, 0xF8, 0x21, 0x0, 0x8, 0x0, 0x88, 0x18, 0xA2, 0x88, 0x38, 0x22, 0x0, 0x8, 0x0, 0x89, 0x58, 0xA2, 0x89, 0x78, 0x22, 0x8A, 0x98, 0xA2, 0x8A, 0xB8, 0x22, 0x8B, 0x8, 0x40, 0x8B, 0xE8, 0xE2, 0x8B, 0x8, 0x63, 0x8C, 0x28, 0xE3, 0x8C, 0x48, 0x63, 0x8D, 0x68, 0x23, 0x0, 0x78, 0x23, 0x0, 0x88, 0x23, 0x0, 0x99, 0xA3, 0x8E, 0xB9, 0x23, 0x0, 0xC9, 0x23, 0x0, 0xD9, 0xA3, 0x8F, 0xF9, 0x23, 0x90, 0x19, 0xA4, 0x90, 0x39, 0x24, 0x91, 0x59, 0x24, 0x0, 0x9, 0x80, 0x91, 0x9, 0xC0, 0x91, 0x89, 0x64, 0x92, 0xA9, 0xE4, 0x92, 0xC9, 0x24, 0x0, 0xD9, 0xA4, 0x93, 0xF9, 0x24, 0x94, 0x19, 0xA5, 0x94, 0x39, 0x25, 0x95, 0x59, 0x25, 0x0, 0x6A, 0xE5, 0x95, 0x8A, 0x25, 0x0, 0xA, 0x40, 0x96, 0xAA, 0x25, 0x0, 0xA, 0xC0, 0x96, 0xCA, 0x65, 0x97, 0xEA, 0xE5, 0x97, 0xA, 0x66, 0x98, 0x2A, 0xE6, 0x98, 0x4A, 0x66, 0x99, 0x6A, 0xE6, 0x99, 0x8B, 0x26, 0x0, 0xB, 0x40, 0x9A, 0xAB, 0xE6, 0x9A, 0xB, 0x0, 0x9B, 0xDB, 0xA6, 0x9B, 0xFB, 0x26, 0x9C, 0x1B, 0xA7, 0x9C, 0x3C, 0x27, 0x0, 0x4C, 0x27, 0x0, 0xC, 0x40, 0x9D, 0x6C, 0xE7, 0x9D, 0x8C, 0x67, 0x9E, 0xAD, 0xE7, 0x9E, 0xCD, 0x27, 0x0, 0xDD, 0xA7, 0x9F, 0xFE, 0x27, 0xA0, 0xE, 0x40, 0xA0, 0x2F, 0xE8, 0xA0, };
    }
}
