#!/usr/bin/env bash

set -e

if [ -n "${EZ_VERBOSE}" ]; then
    set -x
fi

# Warning: This variable API is experimental so these variables may be subject
# to change in the future.
prefix=${prefix:=/usr}
initdir=${initdir:=/etc/init.d}
unitdir=${unitdir:=/usr/lib/systemd/system}
datadir=${datadir:=${prefix}/share}
real_name=${real_name:=puppetserver}
projdatadir=${projdatadir:=${datadir}/${real_name}}
confdir=${confdir:=/etc}
projconfdir=${projconfdir:=${confdir}/${real_name}}
rundir=${rundir:=/var/run}
bindir=${bindir:=/usr/bin}
localstatedir=${localstatedir:=/var}

##################
# EZBake Vars    #
##################

if [ -n "${EZ_VERBOSE}" ]; then
    set +x
    echo "#-------------------------------------------------#"
    echo "The following variables are set: "
    echo
    env | sort

    echo
    echo "End of variable print."
    echo "#-------------------------------------------------#"
    set -x
fi

##################
# Task functions #
##################

# The below functions are exposed to the user to be able to be called from
# the command line directly.

# Catch all, to install the lot, with osdetection included etc.
function task_all {
    task service
    task termini
}

# Run installer, and automatically choose correct tasks using os detection.
function task_service {
    osdetection

    if [ "$OSFAMILY" = "RedHat" ]; then
        defaultsdir="/etc/sysconfig"
        if [ $MAJREV -lt 7 ]; then
            task install_source_rpm_sysv
        else
            task install_source_rpm_systemd
        fi
    elif [ "$OSFAMILY" = "Debian" ]; then
        defaultsdir="/etc/default"
        task install_source_deb
    else
        echo "Unsupported platform, exiting ..."
        exit 1
    fi
}

# Source based install for Redhat based + sysv setups
function task_install_source_rpm_sysv {
    task preinst_redhat
    task install_redhat
    task sysv_init_redhat
    task logrotate_legacy
    task postinst_redhat
    task postinst_permissions
}

# Source based install for Redhat based + systemd setups
function task_install_source_rpm_systemd {
    task preinst_redhat
    task install_redhat
    task systemd_redhat
    task logrotate
    task postinst_redhat
    task postinst_permissions
}

# Source based install for Debian based setups
function task_install_source_deb {
    task preinst_deb
    task install_deb
    task sysv_init_deb
    task logrotate
    task postinst_deb
}

# Install the ezbake package software. This step is used during RPM &
# Debian packaging during the 'install' phases.
function task_install {
    install -d -m 0755 "${DESTDIR}${projdatadir}"
    install -m 0644 puppet-server-release.jar "${DESTDIR}${projdatadir}"
    install -m 0774 ext/ezbake-functions.sh "${DESTDIR}${projdatadir}"
    install -m 0644 ext/ezbake.manifest "${DESTDIR}${projdatadir}"
    install -d -m 0755 "${DESTDIR}${projconfdir}/conf.d"
    install -m 0644 ext/config/conf.d/ca.conf "${DESTDIR}${projconfdir}/conf.d/ca.conf"
    install -m 0644 ext/config/conf.d/puppetserver.conf "${DESTDIR}${projconfdir}/conf.d/puppetserver.conf"
    install -m 0644 ext/config/request-logging.xml "${DESTDIR}${projconfdir}/request-logging.xml"
    install -m 0644 ext/config/logback.xml "${DESTDIR}${projconfdir}/logback.xml"
    install -m 0644 ext/config/bootstrap.cfg "${DESTDIR}${projconfdir}/bootstrap.cfg"
    install -m 0644 ext/config/conf.d/global.conf "${DESTDIR}${projconfdir}/conf.d/global.conf"
    install -m 0644 ext/config/conf.d/web-routes.conf "${DESTDIR}${projconfdir}/conf.d/web-routes.conf"
    install -m 0644 ext/config/conf.d/webserver.conf "${DESTDIR}${projconfdir}/conf.d/webserver.conf"
    install -d -m 0755 "${DESTDIR}${projdatadir}/scripts"
    install -m 0755 install.sh "${DESTDIR}${projdatadir}/scripts"
    install -d -m 0755 "${DESTDIR}${projdatadir}/cli"
    install -d -m 0755 "${DESTDIR}${projdatadir}/cli/apps"
    install -d -m 0755 "${DESTDIR}${bindir}"
    install -m 0755 "ext/bin/${real_name}" "${DESTDIR}${bindir}/${real_name}"
    install -m 0755 ext/cli/irb "${DESTDIR}${projdatadir}/cli/apps/irb"
    install -m 0755 ext/cli/ruby "${DESTDIR}${projdatadir}/cli/apps/ruby"
    install -m 0755 ext/cli/foreground "${DESTDIR}${projdatadir}/cli/apps/foreground"
    install -m 0755 ext/cli/gem "${DESTDIR}${projdatadir}/cli/apps/gem"
    install -d -m 0755 "${DESTDIR}${rundir}"
    install -d -m 700 "${DESTDIR}${localstatedir}/log/puppetserver"
}

function task_install_redhat {
    task install
    echo "os-settings: {"                         > $DESTDIR/$projconfdir/conf.d/os-settings.conf
    echo "    ruby-load-path: [$rubylibdir]"      >> $DESTDIR/$projconfdir/conf.d/os-settings.conf
    echo "}"                                      >> $DESTDIR/$projconfdir/conf.d/os-settings.conf
}

function task_install_deb {
    task install
    echo "os-settings: {"                       > $DESTDIR/$projconfdir/conf.d/os-settings.conf
    echo "    ruby-load-path: [$rubylibdir]"    >> $DESTDIR/$projconfdir/conf.d/os-settings.conf
    echo "}"                                    >> $DESTDIR/$projconfdir/conf.d/os-settings.conf
}


function task_defaults {
    install -d -m 0755 "${DESTDIR}${defaultsdir}"
    install -m 0644 ext/default "${DESTDIR}${defaultsdir}/puppetserver"
}

# Install the sysv and defaults configuration for Redhat.
function task_sysv_init_redhat {
    task defaults
    install -d -m 0755 "${DESTDIR}${initdir}"
    install -m 0755 ext/redhat/init "${DESTDIR}${initdir}/puppetserver"
}

# Install the sysv and defaults configuration for SuSE.
function task_sysv_init_suse {
    task defaults
    install -d -m 0755 "${DESTDIR}${initdir}"
    install -m 0755 ext/redhat/init.suse "${DESTDIR}${initdir}/puppetserver"
}

# Install the systemd and defaults configuration for Redhat.
function task_systemd_redhat {
    task defaults
    install -d -m 0755 "${DESTDIR}${unitdir}"
    install -m 0755 ext/redhat/puppetserver.service "${DESTDIR}${unitdir}/puppetserver.service"
}

# Install the sysv and defaults configuration for Debian.
function task_sysv_init_deb {
    task defaults
    install -d -m 0755 "${DESTDIR}${initdir}"
    install -m 0755 ext/debian/${real_name}.init "${DESTDIR}${initdir}/${real_name}"
    install -d -m 0755 "${DESTDIR}${rundir}"
}

# RPM based pre-installation tasks.
# Note: Any changes to this section may require synchronisation with the
# packaging, due to the fact that we can't access this script from the pre
# section of an rpm/deb.
function task_preinst_redhat {
    # Add puppet group
    getent group puppet > /dev/null || \
           groupadd -r puppet || :
    # Add puppet user
    getent passwd puppet > /dev/null || \
           useradd -r -g puppet -d ${datadir}/puppetserver -s /sbin/nologin \
           -c "puppetserver daemon"  puppet || :
   install --group=puppet --owner=puppet -d /var/lib/puppet/jruby-gems
   mkdir -p /etc/puppet/manifests
}

# Debian based pre-installation tasks.
# Note: Any changes to this section may require synchronisation with the
# packaging, due to the fact that we can't access this script from the pre
# section of an rpm/deb.
function task_preinst_deb {
    # Create the "puppet" user
    if ! getent passwd puppet > /dev/null; then
        adduser --quiet --system --group --home ${datadir}/puppetserver  \
        --no-create-home                                 \
        --gecos "puppetserver daemon" \
        puppet ; \
    fi

    # Create the "puppet" group, if it is missing, and set the
    # primary group of the "puppet" user to this group.
    if ! getent group puppet > /dev/null; then
        addgroup --quiet --system puppet; \
        usermod -g puppet puppet; \
    fi
    install --group=puppet --owner=puppet -d /var/lib/puppet/jruby-gems
    mkdir -p /var/run/puppet
    mkdir -p /etc/puppet/manifests
}

# Debian based post-installation tasks.
function task_postinst_deb {
    task postinst_permissions
}

# RPM based post-installation tasks.
function task_postinst_redhat {
    : # Null command in case additional_postinst is empty
}

# Global post installation permissions setup. Not to be used by Redhat
# during package based installation, as this is done by the RPM itself
# by the %files definitions
function task_postinst_permissions {
    chown puppet:puppet /var/log/puppetserver
    chmod 700 /var/log/puppetserver
}

# Install logrotate (usually el7, fedora 16 and above)
function task_logrotate {
    install -d -m 0755 "${DESTDIR}${confdir}/logrotate.d"
    cp -pr ext/puppetserver.logrotate.conf "${DESTDIR}${confdir}/logrotate.d/puppetserver"
}

# Install legacy logrotate
function task_logrotate_legacy {
    install -d -m 0755 "${DESTDIR}${confdir}/logrotate.d"
    cp -pr ext/puppetserver.logrotate-legacy.conf "${DESTDIR}${confdir}/logrotate.d/puppetserver"
}

##################
# Misc functions #
##################

# Print output only if EZ_VERBOSE is set
function debug_echo {
    if [ -n "${EZ_VERBOSE}" ]; then
        echo $@
    fi
}

# Do basic OS detection using facter.
function osdetection {
    OSFAMILY=`facter osfamily`
    MAJREV=`facter operatingsystemmajrelease`

    debug_echo "OS Detection results"
    debug_echo
    debug_echo "OSFAMILY: ${OSFAMILY}"
    debug_echo "MAJREV: ${MAJREV}"
    debug_echo
}

# Run a task
# Accepts:
#   $1 = task to run
function task {
    local task=$1
    shift
    debug_echo "Running task ${task} ..."
    eval task_$task $@
}

# List available tasks
#
# Gathers a list of all functions starting with task_ so it can be displayed
# or used by other functions.
function available_tasks {
    declare -F | awk '{ print $3 }' | grep '^task_*' | cut -c 6-
}

# Dispatch a task from the CLI
# Accepts:
#   $1 = task to dispatch
function dispatch {
    local task=$1
    shift
    if [ -z "$task" ]; then
        echo "Starting full installation ..."
        echo
        task all
    elif [ "$1" = "-h" ]; then
        echo "Usage: $0 <task>"
        echo
        echo "Choose from one of the following tasks:"
        echo
        echo "$(available_tasks)"
        echo
        echo "Warning: this task system is still experimental and may be subject to change without notice"
        return 1
    else
        task $task $@
    fi
}

########
# Main #
########
dispatch $@
