<?php

/**
 * SPDX-FileCopyrightText: 2016-2024 Nextcloud GmbH and Nextcloud contributors
 * SPDX-FileCopyrightText: 2016 ownCloud, Inc.
 * SPDX-License-Identifier: AGPL-3.0-only
 */
namespace OC\Encryption;

use InvalidArgumentException;
use OC\Files\Filesystem;
use OC\Files\Mount;
use OC\Files\View;
use OCP\Encryption\Exceptions\GenericEncryptionException;
use Psr\Log\LoggerInterface;

/**
 * update encrypted files, e.g. because a file was shared
 */
class Update {
	/** @var View */
	protected $view;

	/** @var Util */
	protected $util;

	/** @var \OC\Files\Mount\Manager */
	protected $mountManager;

	/** @var Manager */
	protected $encryptionManager;

	/** @var string */
	protected $uid;

	/** @var File */
	protected $file;

	/** @var LoggerInterface */
	protected $logger;

	/**
	 * @param string $uid
	 */
	public function __construct(
		View $view,
		Util $util,
		Mount\Manager $mountManager,
		Manager $encryptionManager,
		File $file,
		LoggerInterface $logger,
		$uid
	) {
		$this->view = $view;
		$this->util = $util;
		$this->mountManager = $mountManager;
		$this->encryptionManager = $encryptionManager;
		$this->file = $file;
		$this->logger = $logger;
		$this->uid = $uid;
	}

	/**
	 * hook after file was shared
	 *
	 * @param array $params
	 */
	public function postShared($params) {
		if ($this->encryptionManager->isEnabled()) {
			if ($params['itemType'] === 'file' || $params['itemType'] === 'folder') {
				$path = Filesystem::getPath($params['fileSource']);
				[$owner, $ownerPath] = $this->getOwnerPath($path);
				$absPath = '/' . $owner . '/files/' . $ownerPath;
				$this->update($absPath);
			}
		}
	}

	/**
	 * hook after file was unshared
	 *
	 * @param array $params
	 */
	public function postUnshared($params) {
		if ($this->encryptionManager->isEnabled()) {
			if ($params['itemType'] === 'file' || $params['itemType'] === 'folder') {
				$path = Filesystem::getPath($params['fileSource']);
				[$owner, $ownerPath] = $this->getOwnerPath($path);
				$absPath = '/' . $owner . '/files/' . $ownerPath;
				$this->update($absPath);
			}
		}
	}

	/**
	 * inform encryption module that a file was restored from the trash bin,
	 * e.g. to update the encryption keys
	 *
	 * @param array $params
	 */
	public function postRestore($params) {
		if ($this->encryptionManager->isEnabled()) {
			$path = Filesystem::normalizePath('/' . $this->uid . '/files/' . $params['filePath']);
			$this->update($path);
		}
	}

	/**
	 * inform encryption module that a file was renamed,
	 * e.g. to update the encryption keys
	 *
	 * @param array $params
	 */
	public function postRename($params) {
		$source = $params['oldpath'];
		$target = $params['newpath'];
		if (
			$this->encryptionManager->isEnabled() &&
			dirname($source) !== dirname($target)
		) {
			[$owner, $ownerPath] = $this->getOwnerPath($target);
			$absPath = '/' . $owner . '/files/' . $ownerPath;
			$this->update($absPath);
		}
	}

	/**
	 * get owner and path relative to data/<owner>/files
	 *
	 * @param string $path path to file for current user
	 * @return array ['owner' => $owner, 'path' => $path]
	 * @throw \InvalidArgumentException
	 */
	protected function getOwnerPath($path) {
		$info = Filesystem::getFileInfo($path);
		$owner = Filesystem::getOwner($path);
		$view = new View('/' . $owner . '/files');
		$path = $view->getPath($info->getId());
		if ($path === null) {
			throw new InvalidArgumentException('No file found for ' . $info->getId());
		}

		return [$owner, $path];
	}

	/**
	 * notify encryption module about added/removed users from a file/folder
	 *
	 * @param string $path relative to data/
	 * @throws Exceptions\ModuleDoesNotExistsException
	 */
	public function update($path) {
		$encryptionModule = $this->encryptionManager->getEncryptionModule();

		// if the encryption module doesn't encrypt the files on a per-user basis
		// we have nothing to do here.
		if ($encryptionModule->needDetailedAccessList() === false) {
			return;
		}

		// if a folder was shared, get a list of all (sub-)folders
		if ($this->view->is_dir($path)) {
			$allFiles = $this->util->getAllFiles($path);
		} else {
			$allFiles = [$path];
		}



		foreach ($allFiles as $file) {
			$usersSharing = $this->file->getAccessList($file);
			try {
				$encryptionModule->update($file, $this->uid, $usersSharing);
			} catch (GenericEncryptionException $e) {
				// If the update of an individual file fails e.g. due to a corrupt key we should continue the operation and just log the failure
				$this->logger->error('Failed to update encryption module for ' . $this->uid . ' ' . $file, [ 'exception' => $e ]);
			}
		}
	}
}
