/*
 * Copyright 2000-2007 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.intellij.javaee.model.xml.web.converters;

import com.intellij.codeInsight.CodeInsightBundle;
import com.intellij.javaee.model.enums.WebAppVersion;
import com.intellij.javaee.model.xml.web.FilterMapping;
import com.intellij.javaee.model.xml.web.WebApp;
import com.intellij.javaee.web.CommonServlet;
import com.intellij.javaee.web.WebModel;
import com.intellij.javaee.web.WebUtil;
import com.intellij.javaee.web.facet.WebFacet;
import com.intellij.psi.xml.XmlFile;
import com.intellij.util.xml.ConvertContext;
import com.intellij.util.xml.ResolvingConverter;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.Set;

/**
 * @author peter
 */
public class ServletNameConverter extends ResolvingConverter<CommonServlet> {

  @NotNull
  @Override
  public Collection<? extends CommonServlet> getVariants(ConvertContext context) {
    WebModel webModel = getModel(context);
    return webModel == null ? Collections.emptyList() : webModel.getServlets();
  }

  @Nullable
  static WebModel getModel(ConvertContext context) {
    XmlFile file = context.getFile();
    WebFacet facet = WebUtil.getWebFacet(file);
    if (facet != null && !facet.getWebApps().isEmpty()) {
      return facet.getWebModel();
    }
    return WebUtil.getWebUtil().getLocalWebModel(file);
  }

  @Override
  @NotNull
  public Set<String> getAdditionalVariants(@NotNull final ConvertContext context) {
    if (context.getInvocationElement().getParent() instanceof FilterMapping) {
      WebApp webApp = context.getInvocationElement().getParentOfType(WebApp.class, false);
      assert webApp != null;
      WebAppVersion version = webApp.getVersion().getValue();
      if (version != null && WebAppVersion.WebAppVersion_2_5.compareTo(version) <= 0) {
        return Collections.singleton("*");
      }
    } else {
      return Collections.singleton("default");
    }
    return Collections.emptySet();
  }

  @Nullable
  @Override
  public CommonServlet fromString(@Nullable @NonNls String s, ConvertContext context) {
    if (s == null) return null;
    WebModel webModel = getModel(context);
    return webModel != null ? webModel.findServlet(s) : null;
  }

  @Nullable
  @Override
  public String toString(@Nullable CommonServlet servlet, ConvertContext context) {
    return servlet != null ? servlet.getServletName().getStringValue() : null;
  }

  @Override
  public String getErrorMessage(@Nullable String s, ConvertContext context) {
    return CodeInsightBundle.message("error.cannot.resolve.0.1", "Servlet", s);
  }
}
