// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.run.lifecycle;

import com.intellij.execution.process.ProcessHandler;
import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.Project;
import com.intellij.spring.boot.run.SpringBootApplicationRunConfigurationBase;
import com.intellij.spring.boot.run.lifecycle.tabs.EndpointTab;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 *
 * Enables retrieving and storing Spring Boot actuator endpoint data in {@link SpringBootApplicationInfo}
 * and provides a sub-tab for 'Endpoints' tab in Run/Debug/Run Dashboard tool window.
 *
 * @param <T> the type used to represent endpoint data
 * @since 2018.2
 */
@ApiStatus.Experimental
public abstract class Endpoint<T> {
  public static final ExtensionPointName<Endpoint> EP_NAME =
    ExtensionPointName.create("com.intellij.spring.boot.run.endpoint");

  private static final String ENDPOINT_BEAN_NAME_SUFFIX = "Endpoint";

  private final String myId;
  private final String myBeanName;
  private final String myOperationName;

  public Endpoint(@NotNull String id) {
    this(id, id + ENDPOINT_BEAN_NAME_SUFFIX);
  }

  public Endpoint(@NotNull String id, @NotNull String beanName) {
    this(id, beanName, id);
  }

  public Endpoint(@NotNull String id, @NotNull String beanName, @NotNull String operationName) {
    myId = id;
    myBeanName = beanName;
    myOperationName = operationName;
  }

  /**
   * @return the id of the endpoint
   */
  @NotNull
  public final String getId() {
    return myId;
  }

  /**
   * @return the name of the endpoint bean
   */
  @NotNull
  public final String getBeanName() {
    return myBeanName;
  }

  /**
   * @return the name of the operation invoked on an endpoint MBean to retrieve data (used for Spring Boot 2.x)
   */
  @NotNull
  public String getOperationName() {
    return myOperationName;
  }

  /**
   * Parses data object received from an endpoint MBean.
   */
  @Nullable
  public abstract T parseData(@Nullable Object data);

  /**
   * Creates an endpoint sub-tab for Run/Debug/Run Dashboard tool window.
   */
  @Nullable
  public EndpointTab<T> createEndpointTab(@NotNull SpringBootApplicationRunConfigurationBase runConfiguration,
                                          @NotNull ProcessHandler processHandler) {
    return null;
  }

  /**
   * The method is invoked when {@link SpringBootApplicationInfo} instance is created for running Spring Boot application.
   */
  public void infoCreated(@NotNull Project project, @NotNull SpringBootApplicationInfo info) {
  }

  /**
   * Checks that the endpoint is available for the given module. The method is invoked in a read action in smart mode.
   *
   * @param module the module to check
   * @return {@code true} if the endpoint is available for the given module, otherwise {@code false}
   */
  public boolean isAvailable(@NotNull Module module) {
    return true;
  }

  @Nullable
  public static Endpoint findEndpointById(@NotNull String id) {
    for (Endpoint endpoint : EP_NAME.getExtensions()) {
      if (endpoint.getId().equals(id)) {
        return endpoint;
      }
    }
    return null;
  }
}
