// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.metadata;

import com.intellij.openapi.util.text.StringUtil;
import org.jetbrains.annotations.NonNls;

import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Locale;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


/**
 * Generates relaxed name variations from a given source.
 *
 * @author Phillip Webb
 * @author Dave Syer
 */
public final class RelaxedNames implements Iterable<String> {

  private static final Pattern CAMEL_CASE_PATTERN = Pattern.compile("([^A-Z-])([A-Z])");

  private final Set<String> values = new LinkedHashSet<>();

  /**
   * Create a new {@link RelaxedNames} instance.
   *
   * @param name the source name. For the maximum number of variations specify the name
   *             using dashed notation (e.g. {@literal my-property-name}
   */
  public RelaxedNames(String name) {
    initialize(StringUtil.notNullize(name), this.values);
  }

  public Set<String> getValues() {
    return values;
  }

  @Override
  public Iterator<String> iterator() {
    return this.values.iterator();
  }

  private static void initialize(String name, Set<String> values) {
    if (values.contains(name)) {
      return;
    }
    for (Variation variation : Variation.values()) {
      for (Manipulation manipulation : Manipulation.values()) {
        String result = name;
        result = manipulation.apply(result);
        result = variation.apply(result);
        values.add(result);
        initialize(result, values);
      }
    }
  }

  /**
   * "spring.server.my-property" --> "myProperty"
   *
   * @param configKey Qualified Configuration key.
   * @return Property name.
   */
  static String dashedPropertyNameToCamelCase(String configKey) {
    String propertyKey = configKey.substring(configKey.lastIndexOf('.') + 1);
    if (!StringUtil.containsChar(propertyKey, '-')) {
      return propertyKey;
    }


    return RelaxedNames.Manipulation.SEPARATED_TO_CAMELCASE.apply(propertyKey);
  }

  public static String separatedPropertyNameToCamelCase(String propertyName) {
    return RelaxedNames.Manipulation.SEPARATED_TO_CAMELCASE.apply(propertyName);
  }

  public static String camelCaseToHyphen(String propertyName) {
    return Manipulation.CAMELCASE_TO_HYPHEN.apply(propertyName);
  }

  enum Variation {

    NONE {
      @Override
      public String apply(String value) {
        return value;
      }
    },

    LOWERCASE {
      @Override
      public String apply(String value) {
        return value.toLowerCase(Locale.US);
      }
    },

    UPPERCASE {
      @Override
      public String apply(String value) {
        return value.toUpperCase(Locale.US);
      }
    };

    public abstract String apply(String value);

  }

  enum Manipulation {

    NONE {
      @Override
      public String apply(String value) {
        return value;
      }
    },

    HYPHEN_TO_UNDERSCORE {
      @Override
      public String apply(String value) {
        return value.replace('-', '_');
      }
    },

    UNDERSCORE_TO_PERIOD {
      @Override
      public String apply(String value) {
        return value.replace('_', '.');
      }
    },

    PERIOD_TO_UNDERSCORE {
      @Override
      public String apply(String value) {
        return value.replace('.', '_');
      }
    },

    CAMELCASE_TO_UNDERSCORE {
      @Override
      public String apply(String value) {
        Matcher matcher = CAMEL_CASE_PATTERN.matcher(value);
        if (!matcher.find()) {
          return value;
        }
        matcher = matcher.reset();
        StringBuffer result = new StringBuffer();
        while (matcher.find()) {
          matcher.appendReplacement(result, matcher.group(1) + '_'
                                            + StringUtil.decapitalize(matcher.group(2)));
        }
        matcher.appendTail(result);
        return result.toString();
      }
    },

    CAMELCASE_TO_HYPHEN {
      @Override
      public String apply(String value) {
        Matcher matcher = CAMEL_CASE_PATTERN.matcher(value);
        if (!matcher.find()) {
          return value;
        }
        matcher = matcher.reset();
        StringBuffer result = new StringBuffer();
        while (matcher.find()) {
          matcher.appendReplacement(result, matcher.group(1) + '-'
                                            + StringUtil.decapitalize(matcher.group(2)));
        }
        matcher.appendTail(result);
        return result.toString();
      }
    },

    SEPARATED_TO_CAMELCASE {
      @Override
      public String apply(String value) {
        return separatedToCamelCase(value, false);
      }
    },

    CASE_INSENSITIVE_SEPARATED_TO_CAMELCASE {
      @Override
      public String apply(String value) {
        return separatedToCamelCase(value, true);
      }
    };

    public abstract String apply(String value);

    @NonNls private static final String SEPARATORS = "_-.";

    private static String separatedToCamelCase(String value, boolean caseInsensitive) {
      if (value.length() == 0) {
        return value;
      }
      StringBuilder builder = new StringBuilder();
      for (String field : StringUtil.tokenize(value, SEPARATORS)) {
        field = (caseInsensitive ? field.toLowerCase(Locale.US) : field);
        builder.append(builder.length() == 0 ? field : StringUtil.capitalize(field));
      }

      char lastChar = value.charAt(value.length() - 1);
      if (StringUtil.containsChar(SEPARATORS, lastChar)) {
        builder.append(lastChar);
      }
      return builder.toString();
    }
  }
}
