// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.config;

import com.intellij.codeInsight.completion.util.ParenthesesInsertHandler;
import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.codeInsight.lookup.LookupElementBuilder;
import com.intellij.lang.properties.IProperty;
import com.intellij.lang.properties.psi.PropertiesElementFactory;
import com.intellij.lang.properties.psi.PropertiesFile;
import com.intellij.lang.properties.psi.Property;
import com.intellij.lang.properties.psi.PropertyKeyIndex;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.util.TextRange;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.spring.boot.SpringBootClassesConstants;
import com.intellij.spring.boot.application.config.hints.HintReferenceBase;
import com.intellij.spring.boot.library.SpringBootLibraryUtil;
import com.intellij.spring.boot.library.SpringBootLibraryUtil.SpringBootVersion;
import com.intellij.spring.model.utils.PlaceholderTextRanges;
import com.intellij.spring.model.utils.SpringCommonUtils;
import com.intellij.util.ArrayUtil;
import com.intellij.util.ObjectUtils;
import com.intellij.util.PairFunction;
import com.intellij.util.SmartList;
import icons.SpringBootApiIcons;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import static com.intellij.spring.boot.application.config.SpringBootPlaceholderReferenceResolver.EP_NAME;

/**
 * Reference for {@code ${propertyKey}} placeholder expressions in value.
 *
 * @see SpringBootPlaceholderReferenceResolver
 * @see HintReferenceBase#PLACEHOLDER_PREFIX
 * @see HintReferenceBase#PLACEHOLDER_SUFFIX
 */
public abstract class SpringBootPlaceholderReference extends PsiReferenceBase.Poly<PsiElement> {

  private static final String RANDOM_KEY_PREFIX = "random.";

  protected SpringBootPlaceholderReference(PsiElement element, TextRange range) {
    super(element, range, false);
  }

  @Nullable
  protected abstract PsiElement resolveExistingKey(String key);

  protected abstract void addExistingKeyVariants(List<LookupElement> result);

  public static PsiReference[] createPlaceholderReferences(PsiElement element,
                                                           PairFunction<? super PsiElement, ? super TextRange, ? extends PsiReference> producer) {
    final String text = ElementManipulators.getValueText(element);
    final Set<TextRange> ranges = PlaceholderTextRanges.getPlaceholderRanges(text,
                                                                             HintReferenceBase.PLACEHOLDER_PREFIX,
                                                                             HintReferenceBase.PLACEHOLDER_SUFFIX);
    if (ranges.isEmpty()) {
      return EMPTY_ARRAY;
    }

    List<PsiReference> placeholderReferences = new ArrayList<>(ranges.size());
    int startOffset = ElementManipulators.getOffsetInElement(element);
    for (TextRange range : ranges) {
      int colonIdx = range.substring(text).indexOf(':');
      if (colonIdx != -1) {
        range = TextRange.from(range.getStartOffset(), colonIdx);
      }
      placeholderReferences.add(producer.fun(element, range.shiftRight(startOffset)));
    }
    return placeholderReferences.toArray(PsiReference.EMPTY_ARRAY);
  }

  @NotNull
  @Override
  public ResolveResult[] multiResolve(boolean incompleteCode) {
    final String key = getValue();

    if (StringUtil.startsWith(key, RANDOM_KEY_PREFIX)) {
      final String after = ObjectUtils.assertNotNull(StringUtil.substringAfter(key, RANDOM_KEY_PREFIX));

      for (Random random : Random.values()) {
        final String randomValue = random.getValue();
        if (random.isSupportsParameters() && StringUtil.startsWith(after, randomValue) ||
            after.equals(randomValue)) {
          if (SpringBootLibraryUtil.isBelowVersion(getModule(), random.getMinimumVersion())) {
            continue;
          }

          final PsiClass randomClass = findRandomClass();
          return randomClass != null
                 ? PsiElementResolveResult.createResults(randomClass)
                 : PsiElementResolveResult.createResults(getElement());
        }
      }
      return ResolveResult.EMPTY_ARRAY;
    }

    final IProperty systemProperty = getSystemProperties().findPropertyByKey(key);
    if (systemProperty != null) {
      return PsiElementResolveResult.createResults(systemProperty.getPsiElement());
    }

    PsiElement existingKey = resolveExistingKey(key);
    if (existingKey != null) {
      return PsiElementResolveResult.createResults(existingKey);
    }

    // fallback to key in any .properties file
    GlobalSearchScope contentScope = getModule().getModuleContentScope();
    Collection<Property> properties = PropertyKeyIndex.getInstance().get(key, getElement().getProject(),
                                                                         getElement().getResolveScope().uniteWith(contentScope));

    List<PsiElement> allResults = new ArrayList<>(properties);
    for (SpringBootPlaceholderReferenceResolver placeholderReferenceResolver : EP_NAME.getExtensions()) {
      allResults.addAll(placeholderReferenceResolver.resolve(this));
    }
    return PsiElementResolveResult.createResults(allResults);
  }

  @Nullable
  private PsiClass findRandomClass() {
    return SpringCommonUtils.findLibraryClass(getModule(), getRandomClassName());
  }

  @NotNull
  private String getRandomClassName() {
    if (SpringBootLibraryUtil.isAtLeastVersion(getModule(), SpringBootVersion.VERSION_2_0_0)) {
      return SpringBootClassesConstants.RANDOM_VALUE_PROPERTY_SOURCE_SB2;
    }

    return SpringBootClassesConstants.RANDOM_VALUE_PROPERTY_SOURCE;
  }

  private Module getModule() {
    return ModuleUtilCore.findModuleForPsiElement(getElement());
  }

  @NotNull
  private PropertiesFile getSystemProperties() {
    return PropertiesElementFactory.getSystemProperties(myElement.getProject());
  }

  @NotNull
  @Override
  public Object[] getVariants() {
    List<LookupElement> variants = new SmartList<>();
    addExistingKeyVariants(variants);

    for (IProperty property : getSystemProperties().getProperties()) {
      final String key = property.getKey();
      if (key == null) continue;
      variants.add(LookupElementBuilder.create(property, key).withIcon(SpringBootApiIcons.SpringBoot));
    }

    PsiClass randomClass = findRandomClass();
    if (randomClass == null) {
      return ArrayUtil.toObjectArray(variants);
    }

    final Module module = getModule();
    for (Random random : Random.values()) {
      if (SpringBootLibraryUtil.isBelowVersion(module, random.getMinimumVersion())) {
        continue;
      }

      final String randomText = random.getValue();
      final String insertString = RANDOM_KEY_PREFIX + randomText;

      variants.add(LookupElementBuilder.create(randomClass, insertString).withIcon(SpringBootApiIcons.SpringBoot));
      if (random.isSupportsParameters()) {
        variants.add(LookupElementBuilder.create(randomClass, insertString)
                       .withPresentableText(RANDOM_KEY_PREFIX + randomText + "(value,[max])")
                       .withIcon(SpringBootApiIcons.SpringBoot)
                       .withInsertHandler(ParenthesesInsertHandler.WITH_PARAMETERS));
      }
    }

    for (SpringBootPlaceholderReferenceResolver placeholderReferenceResolver : EP_NAME.getExtensions()) {
      variants.addAll(placeholderReferenceResolver.getVariants(this));
    }
    return ArrayUtil.toObjectArray(variants);
  }

  enum Random {
    INT("int", true, SpringBootVersion.ANY),
    LONG("long", true, SpringBootVersion.ANY),
    VALUE("value", false, SpringBootVersion.ANY),
    UUID("uuid", false, SpringBootVersion.VERSION_1_4_0);

    private final String myValue;
    private final boolean mySupportsParameters;
    private final SpringBootVersion myMinimumVersion;

    Random(String value, boolean supportsParameters, SpringBootVersion minimumVersion) {
      myValue = value;
      mySupportsParameters = supportsParameters;
      myMinimumVersion = minimumVersion;
    }

    SpringBootVersion getMinimumVersion() {
      return myMinimumVersion;
    }

    String getValue() {
      return myValue;
    }

    boolean isSupportsParameters() {
      return mySupportsParameters;
    }
  }
}
