// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.config;

import com.intellij.codeInsight.completion.*;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.JavaClassReference;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.JavaClassReferenceSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Delegates basic to smart completion for underlying JavaClassReference with "extends".
 */
public abstract class SpringBootConfigFileSmartClassReferenceCompletionContributor extends CompletionContributor {

  protected abstract boolean isMyFile(PsiFile file);

  protected abstract boolean isValueElement(PsiElement position);

  @Override
  public void duringCompletion(@NotNull CompletionInitializationContext context) {
    JavaClassReference reference =
      JavaClassReferenceCompletionContributor.findJavaClassReference(context.getFile(), context.getStartOffset());
    if (isRelevantReference(reference)) {
      JavaClassReferenceSet set = reference.getJavaClassReferenceSet();
      context.setReplacementOffset(set.getRangeInElement().getEndOffset() + set.getElement().getTextRange().getStartOffset());
    }
  }

  @Override
  public void fillCompletionVariants(@NotNull CompletionParameters parameters, @NotNull CompletionResultSet result) {
    if (parameters.getCompletionType() != CompletionType.BASIC) {
      return;
    }

    PsiElement position = parameters.getPosition();
    if (!isValueElement(position)) return;

    final PsiFile file = parameters.getOriginalFile();
    if (!isMyFile(file)) return;

    if (isJavaClassReferenceWithExtendClassNames(parameters)) {
      result.runRemainingContributors(parameters.withType(CompletionType.SMART), true);
    }
  }

  private static boolean isJavaClassReferenceWithExtendClassNames(CompletionParameters parameters) {
    final JavaClassReference reference =
      JavaClassReferenceCompletionContributor.findJavaClassReference(parameters.getPosition().getContainingFile(), parameters.getOffset());
    return isRelevantReference(reference);
  }


  private static boolean isRelevantReference(@Nullable JavaClassReference reference) {
    if (reference == null) {
      return false;
    }

    return !reference.getSuperClasses().isEmpty();
  }
}
