// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.profiles;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.TextRange;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiReferenceBase;
import org.jetbrains.annotations.NotNull;

import java.util.Set;

/**
 * @author Sergey Vasiliev
 * @see SpringProfilesFactory#getProfilesReferences(PsiElement, String, boolean)
 * @since 14.1
 */
public class SpringProfilePsiReference extends PsiReferenceBase<PsiElement> {
  private final String myProfileName;
  private final Module myModule;
  private final boolean myIsDefinition;

  public SpringProfilePsiReference(@NotNull PsiElement element,
                                   @NotNull TextRange range,
                                   @NotNull Module module,
                                   boolean definition) {
    super(element, range);
    myIsDefinition = definition;
    myProfileName = range.substring(element.getText());
    myModule = module;
  }

  @Override
  public PsiElement resolve() {
    if (StringUtil.isEmptyOrSpaces(myProfileName)) return myElement;

    final SpringProfilesFactory profilesFactory = SpringProfilesFactory.getInstance(myModule);
    if (myIsDefinition) {
      return profilesFactory.getOrCreateProfile(myProfileName, getElement());
    }

    return profilesFactory.findProfile(myProfileName);
  }

  @Override
  public boolean isSoft() {
    return true;
  }

  @Override
  @NotNull
  public Object[] getVariants() {
    Set<SpringProfile> profiles = SpringProfilesFactory.getInstance(myModule).getProfiles();
    return profiles.toArray(new SpringProfile[0]);
  }
}
