/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.values.converters;

import com.intellij.openapi.util.Condition;
import com.intellij.psi.PsiClass;
import com.intellij.psi.util.InheritanceUtil;
import com.intellij.psi.util.PsiTypesUtil;
import com.intellij.spring.model.converters.SpringConverterUtil;
import com.intellij.spring.model.xml.DomSpringBean;
import com.intellij.spring.model.xml.beans.ConstructorArg;
import com.intellij.spring.model.xml.beans.SpringProperty;
import com.intellij.util.xml.DomElement;
import com.intellij.util.xml.GenericDomValue;
import org.jetbrains.annotations.NotNull;

public class SpringValueConditionFactory {

  public static Condition<GenericDomValue> createPropertyNameCondition(@NotNull final String propertyName) {
    return genericDomValue -> checkPropertyName(genericDomValue, propertyName);
  }

  public static Condition<GenericDomValue> createBeanPropertyCondition(@NotNull final String beanClass,
                                                                       @NotNull final String... propertyNames) {
    return genericDomValue -> {
      for (String propertyName : propertyNames) {
        if (checkPropertyName(genericDomValue, propertyName)) {
          return checkBeanClass(genericDomValue, beanClass);
        }
      }
      return false;
    };
  }

  public static Condition<GenericDomValue> createBeanClassCondition(@NotNull final String beanClass) {
    return genericDomValue -> checkBeanClass(genericDomValue, beanClass);
  }

  public static Condition<GenericDomValue> createBeanClassConstructorArgCondition(@NotNull final String beanClass) {
    return genericDomValue -> isConstructorArg(genericDomValue) && checkBeanClass(genericDomValue, beanClass);
  }

  private static boolean isConstructorArg(@NotNull GenericDomValue genericDomValue) {
    return genericDomValue.getParentOfType(ConstructorArg.class, false) != null;
  }

  private static boolean checkBeanClass(@NotNull DomElement element, @NotNull String beanClassName) {
    DomSpringBean springBean = SpringConverterUtil.getCurrentBean(element);
    if (springBean == null) return false;

    final PsiClass beanClass = PsiTypesUtil.getPsiClass(springBean.getBeanType());
    return InheritanceUtil.isInheritor(beanClass, beanClassName);
  }

  private static boolean checkPropertyName(@NotNull DomElement element, @NotNull String propertyName) {
    final SpringProperty springProperty = element.getParentOfType(SpringProperty.class, false);
    return springProperty != null && propertyName.equals(springProperty.getName().getStringValue());
  }
}
