/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.intellij.spring.model.utils;

import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.openapi.module.Module;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import com.intellij.psi.util.CachedValueProvider;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.spring.SpringModificationTrackersManager;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.util.ArrayUtil;
import com.intellij.util.Function;
import com.intellij.util.containers.ConcurrentFactoryMap;
import org.jetbrains.annotations.NotNull;

import java.util.Collections;
import java.util.Map;
import java.util.Set;

/**
 * @since 15
 */
public class SpringCacheUtils {

  /**
   * Create and cache beans "created" by the given element.
   *
   * @param module          Module.
   * @param creationElement Element "creating" beans (e.g. component scan)
   * @param function        Provides resulting beans.
   * @return Cached beans.
   */
  public static Set<CommonSpringBean> getCreatedCachedBeans(@NotNull final Module module,
                                                            @NotNull final CommonModelElement creationElement,
                                                            @NotNull final Function<? super Module, ? extends Set<CommonSpringBean>> function) {
    if (module.isDisposed()) {
      return Collections.emptySet();
    }

    PsiElement identifyingPsiElement = creationElement.getIdentifyingPsiElement();
    if (identifyingPsiElement == null || !identifyingPsiElement.isValid()) {
      return Collections.emptySet();
    }

    final Map<Module, Set<CommonSpringBean>> cachedValue =
      CachedValuesManager.getCachedValue(identifyingPsiElement, () -> {
        Map<Module, Set<CommonSpringBean>> map = ConcurrentFactoryMap.createMap(key -> function.fun(module));
        Object[] dependencies = SpringModificationTrackersManager.getInstance(module.getProject()).getOuterModelsDependencies();
        PsiFile containingFile = identifyingPsiElement.getContainingFile();
        if (containingFile != null) {
          dependencies = ArrayUtil.append(dependencies, containingFile);
        }
        return CachedValueProvider.Result.create(map, dependencies);
      });

    return cachedValue.get(module);
  }
}
