// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.jam.stereotype;

import com.intellij.jam.reflect.*;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.PsiPackage;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.semantic.SemKey;
import com.intellij.spring.model.jam.converters.PackageJamConverter;
import com.intellij.spring.model.jam.utils.filters.*;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;
import java.util.Set;

public abstract class SpringJamComponentScanArchetype extends SpringComponentScan {
  public static final SemKey<JamAnnotationMeta> META_KEY = COMPONENT_SCAN_META_KEY.subKey("SpringJamComponentScan");

  protected static final JamStringAttributeMeta.Collection<Collection<PsiPackage>> VALUE_ATTRIBUTE_META =
    new JamStringAttributeMeta.Collection<>(VALUE_ATTR_NAME, new PackageJamConverter());

  protected static final JamStringAttributeMeta.Collection<Collection<PsiPackage>> BASE_PACKAGE_ATTR_META =
    new JamStringAttributeMeta.Collection<>(BASE_PACKAGES_ATTR_NAME, new PackageJamConverter());

  private static final JamClassAttributeMeta.Collection BASE_PACKAGE_CLASS_ATTR_META =
    JamAttributeMeta.classCollection(BASE_PACKAGE_CLASSES_ATTR_NAME);

  protected static final JamBooleanAttributeMeta USE_DEFAULT_FILTERS_META =
    JamAttributeMeta.singleBoolean("useDefaultFilters", true);

  protected static final JamBooleanAttributeMeta LAZY_INIT_ATTR_META =
    JamAttributeMeta.singleBoolean("lazyInit", false);

  protected static final JamStringAttributeMeta.Single<String> RESOURCE_PATTERN_META =
    JamAttributeMeta.singleString("resourcePattern");

  protected static final JamEnumAttributeMeta.Single<ScopedProxyMode> SCOPED_PROXY_META =
    JamAttributeMeta.singleEnum("scopedProxy", ScopedProxyMode.class);

  protected static final JamClassAttributeMeta.Single NAME_GENERATOR_META =
    JamAttributeMeta.singleClass("nameGenerator");

  protected static final JamClassAttributeMeta.Single SCOPE_RESOLVER_META =
    JamAttributeMeta.singleClass("scopeResolver");

  protected static final JamAttributeMeta<List<SpringComponentScanFilter>> INCLUDE_FILTERS_ATTR_META =
    JamAttributeMeta.annoCollection("includeFilters", SpringComponentScanFilter.ANNOTATION_META, SpringComponentScanFilter.class);

  protected static final JamAttributeMeta<List<SpringComponentScanFilter>> EXCLUDE_FILTERS_ATTR_META =
    JamAttributeMeta.annoCollection("excludeFilters", SpringComponentScanFilter.ANNOTATION_META, SpringComponentScanFilter.class);

  public static final JamAnnotationArchetype ARCHETYPE = new JamAnnotationArchetype()
    .addAttribute(VALUE_ATTRIBUTE_META)
    .addAttribute(BASE_PACKAGE_ATTR_META)
    .addAttribute(USE_DEFAULT_FILTERS_META)
    .addAttribute(LAZY_INIT_ATTR_META)
    .addAttribute(RESOURCE_PATTERN_META)
    .addAttribute(SCOPED_PROXY_META)
    .addAttribute(NAME_GENERATOR_META)
    .addAttribute(SCOPE_RESOLVER_META)
    .addAttribute(INCLUDE_FILTERS_ATTR_META)
    .addAttribute(EXCLUDE_FILTERS_ATTR_META);

  private final PsiElementRef<PsiAnnotation> myAnnotation;

  public SpringJamComponentScanArchetype(@NotNull PsiClass psiElement) {
    super(psiElement);
    myAnnotation = getAnnotationMeta().getAnnotationRef(psiElement);
  }

  public SpringJamComponentScanArchetype(PsiAnnotation annotation) {
    super(PsiTreeUtil.getParentOfType(annotation, PsiClass.class, true));
    myAnnotation = PsiElementRef.real(annotation);
  }


  @Override
  protected JamClassAttributeMeta.Collection getBasePackageClassMeta() {
    return BASE_PACKAGE_CLASS_ATTR_META;
  }

  /**
   * Returns all attributes containing package definitions to scan.
   */
  @Override
  @NotNull
  public List<JamStringAttributeMeta.Collection<Collection<PsiPackage>>> getPackageJamAttributes() {
    return ContainerUtil.immutableList(VALUE_ATTRIBUTE_META, BASE_PACKAGE_ATTR_META);
  }

  @Override
  public boolean useDefaultFilters() {
    return USE_DEFAULT_FILTERS_META.getJam(getAnnotationRef()).getValue();
  }

  public boolean isLazyInit() {
    return LAZY_INIT_ATTR_META.getJam(getAnnotationRef()).getValue();
  }

  @Nullable
  public String getResourcePattern() {
    return RESOURCE_PATTERN_META.getJam(getAnnotationRef()).getValue();
  }

  @Nullable
  public ScopedProxyMode getScopedProxy() {
    return SCOPED_PROXY_META.getJam(getAnnotationRef()).getValue();
  }

  @Nullable
  public PsiClass getNameGenerator() {
    return NAME_GENERATOR_META.getJam(getAnnotationRef()).getValue();
  }

  @Nullable
  public PsiClass getScopeResolver() {
    return SCOPE_RESOLVER_META.getJam(getAnnotationRef()).getValue();
  }

  @NotNull
  @Override
  public Set<SpringContextFilter.Exclude> getExcludeContextFilters() {
    Set<SpringContextFilter.Exclude> excludes = ContainerUtil.newLinkedHashSet();
    for (SpringComponentScanFilter filter : EXCLUDE_FILTERS_ATTR_META.getJam(getAnnotationRef())) {
      final FilterType value = filter.getFilterType();
      final Set<PsiClass> classes = filter.getFilteredClasses();
      if (value == FilterType.ASSIGNABLE_TYPE) {
        excludes.add(new ExcludeAssignableFilter(classes));
      }
      else if (value == FilterType.ANNOTATION) {
        excludes.add(new ExcludeAnnotationsFilter(classes));
      }
    }

    return excludes;
  }

  @NotNull
  @Override
  public Set<SpringContextFilter.Include> getIncludeContextFilters() {
    Set<SpringContextFilter.Include> includes = ContainerUtil.newLinkedHashSet();
    for (SpringComponentScanFilter filter : INCLUDE_FILTERS_ATTR_META.getJam(getAnnotationRef())) {
      final FilterType value = filter.getFilterType();
      final Set<PsiClass> classes = filter.getFilteredClasses();
      if (value == FilterType.ASSIGNABLE_TYPE) {
        includes.add(new IncludeAssignableFilter(classes));
      }
      else if (value == FilterType.ANNOTATION) {
        includes.add(new IncludeAnnotationsFilter(classes));
      }
      else if (value == FilterType.CUSTOM) {
        includes.add(new IncludeCustomFilter(classes));
      }
    }

    return includes;
  }


  @Override
  @NotNull
  public PsiElementRef<PsiAnnotation> getAnnotationRef() {
    return myAnnotation;
  }

  /**
   * {@link org.springframework.context.annotation.ScopedProxyMode}
   */
  public enum ScopedProxyMode {
    DEFAULT,
    NO,
    INTERFACES,
    TARGET_CLASS
  }
}
