// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.highlighting.jam;

import com.intellij.codeInspection.AbstractBaseJavaLocalInspectionTool;
import com.intellij.codeInspection.InspectionManager;
import com.intellij.codeInspection.ProblemDescriptor;
import com.intellij.codeInspection.ProblemsHolder;
import com.intellij.jam.model.util.JamCommonUtil;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiJavaFile;
import com.intellij.spring.model.utils.SpringCommonUtils;
import com.intellij.spring.model.utils.SpringModelUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

public abstract class AbstractSpringJavaConfigInspection extends AbstractBaseJavaLocalInspectionTool {

  @Override
  @Nullable
  public ProblemDescriptor[] checkFile(@NotNull PsiFile file, @NotNull InspectionManager manager, boolean isOnTheFly) {
    if (JamCommonUtil.isPlainJavaFile(file)) {
      final Module module = ModuleUtilCore.findModuleForPsiElement(file);
      if (SpringCommonUtils.hasSpringFacet(module) || SpringModelUtils.getInstance().hasAutoConfiguredModels(module)) {

        final ProblemsHolder holder = new ProblemsHolder(manager, file, isOnTheFly);

        checkJavaFile((PsiJavaFile)file, holder, isOnTheFly, module);

        final List<ProblemDescriptor> problemDescriptors = holder.getResults();
        return problemDescriptors.toArray(ProblemDescriptor.EMPTY_ARRAY);
      }
    }
    return null;
  }

  protected void checkJavaFile(@NotNull final PsiJavaFile javaFile,
                               @NotNull final ProblemsHolder holder,
                               final boolean isOnTheFly,
                               @NotNull Module module) {
    for (PsiClass psiClass : javaFile.getClasses()) {
      checkClassInternal(psiClass, holder, module);
    }
  }

  private void checkClassInternal(final PsiClass aClass, final ProblemsHolder holder, @NotNull Module module) {
    checkClass(aClass, holder, module);
    for (PsiClass psiClass : aClass.getInnerClasses()) {
      checkClass(psiClass, holder, module);
    }
  }

  protected abstract void checkClass(final PsiClass aClass, final ProblemsHolder holder, @NotNull Module module);
}
