// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.actions.patterns.frameworks;

import com.intellij.openapi.actionSystem.*;
import com.intellij.openapi.editor.Editor;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.project.Project;
import com.intellij.psi.JavaPsiFacade;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiDocumentManager;
import com.intellij.psi.PsiFile;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.xml.XmlFile;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.SpringManager;
import com.intellij.spring.dom.SpringDomUtils;
import com.intellij.spring.model.utils.SpringModelSearchers;
import com.intellij.util.ArrayUtil;
import com.intellij.util.ui.EmptyIcon;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;

public abstract class FrameworkIntegrationAction extends AnAction implements FrameworkSupportProvider {

  protected FrameworkIntegrationAction() {
    super();

    // TODO we cannot use getTemplatePresentation() and set icon/text once
    // TODO due to BasicDelegateFrameworkIntegrationAction, so reserve space for icon
    final Presentation template = getTemplatePresentation();
    template.setIcon(EmptyIcon.ICON_16);
  }

  @Override
  public void actionPerformed(@NotNull AnActionEvent e) {
    final DataContext dataContext = e.getDataContext();

    final Module module = getModule(dataContext);
    final Editor editor = getEditor(dataContext);
    final XmlFile xmlFile = getXmlFile(dataContext);

    if (module != null && editor != null && xmlFile != null) {
      generateSpringBeans(module, editor, xmlFile);
    }
  }

  @Override
  public void update(@NotNull final AnActionEvent event) {
    Presentation presentation = event.getPresentation();
    DataContext dataContext = event.getDataContext();

    XmlFile file = getXmlFile(dataContext);

    final boolean isSpringBeanFile = file != null && SpringDomUtils.isSpringXml(file);

    final boolean enabled = isSpringBeanFile && accept(file);

    presentation.setEnabledAndVisible(enabled);
    presentation.setText(getDescription());
    presentation.setIcon(getIcon());
  }

  @Nullable
  protected static XmlFile getXmlFile(final DataContext dataContext) {
    return getXmlFile(getProject(dataContext), getEditor(dataContext));
  }

  @Nullable
  protected static XmlFile getXmlFile(final Project project, final Editor editor) {
    if (project == null || editor == null) return null;

    final PsiFile psiFile = PsiDocumentManager.getInstance(project).getPsiFile(editor.getDocument());
    return psiFile instanceof XmlFile ? (XmlFile)psiFile : null;
  }

  @Nullable
  protected static Editor getEditor(final DataContext dataContext) {
    return CommonDataKeys.EDITOR.getData(dataContext);
  }

  @Nullable
  protected static Project getProject(final DataContext dataContext) {
    return CommonDataKeys.PROJECT.getData(dataContext);
  }

  @Nullable
  protected static Module getModule(final DataContext dataContext) {
    return LangDataKeys.MODULE.getData(dataContext);
  }

  protected boolean accept(@NotNull final XmlFile file) {
    return acceptBeansByClassNames(file, getBeansClassNames());
  }

  protected String[] getBeansClassNames() {
    return ArrayUtil.EMPTY_STRING_ARRAY;
  }

  private static boolean acceptBeansByClassNames(XmlFile file, String... classNames) {
    if (classNames.length == 0) return true;

    final Module module = ModuleUtilCore.findModuleForPsiElement(file);
    if (module == null) return false;

    final JavaPsiFacade javaPsiFacade = JavaPsiFacade.getInstance(module.getProject());
    final GlobalSearchScope searchScope = GlobalSearchScope.moduleWithDependenciesAndLibrariesScope(module);

    final CommonSpringModel model = SpringManager.getInstance(file.getProject()).getSpringModelByFile(file);
    if (model != null) {
      for (String className : classNames) {
        final PsiClass psiClass = javaPsiFacade.findClass(className, searchScope);
        if (psiClass == null) continue;

        if (SpringModelSearchers.doesBeanExist(model, psiClass)) return false;
      }
    }
    return true;
  }

  protected abstract void generateSpringBeans(final Module module, final Editor editor, final XmlFile xmlFile);

  @Nullable
  protected Icon getIcon() {
    return null;
  }
}
