// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.contexts.model.visitors;

import com.intellij.spring.CommonSpringModel;
import com.intellij.util.Processor;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;

import java.util.Set;

public class CommonSpringModelVisitorContext<P> {
  private final Set<CommonSpringModel> visited = ContainerUtil.newTroveSet();

  private final Exec<P> exec;
  private final VisitorAwareProcessor<P> vp;

  private CommonSpringModelVisitorContext(Processor<P> p, Exec<P> e) {
    exec = e;
    vp = new VisitorAwareProcessor<>(this, p);
  }

  public interface Exec<P> {
    boolean run(@NotNull CommonSpringModel model, @NotNull Processor<P> params);
  }

  public boolean visit(@NotNull CommonSpringModel model) {
    if (hasBeenVisited(model)) return true;

    visited.add(model);
    return exec.run(model, vp);
  }

  public final boolean hasBeenVisited(CommonSpringModel model) {
    return visited.contains(model);
  }

  private static class VisitorAwareProcessor<P> implements Processor<P> {
    private final CommonSpringModelVisitorContext<P> visitor;
    private final Processor<P> delegate;

    VisitorAwareProcessor(CommonSpringModelVisitorContext<P> visitor, Processor<P> delegate) {
      this.visitor = visitor;
      this.delegate = delegate;
    }

    @Override
    public boolean process(P p) {
      return delegate.process(p);
    }
  }

  public static <P> CommonSpringModelVisitorContext<P> context(@NotNull Processor<P> p, Exec<P> exec) {
      if (p instanceof VisitorAwareProcessor) return ((VisitorAwareProcessor)p).visitor;
      return new CommonSpringModelVisitorContext<>(p, exec);
  }
}
