// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring;

import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiFile;
import com.intellij.psi.xml.XmlTag;
import com.intellij.spring.contexts.model.AbstractSimpleSpringModel;
import com.intellij.spring.contexts.model.LocalModel;
import com.intellij.spring.contexts.model.LocalXmlModel;
import com.intellij.spring.contexts.model.visitors.CommonSpringModelVisitorContext;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringQualifier;
import com.intellij.spring.model.xml.context.SpringBeansPackagesScan;
import com.intellij.util.CommonProcessors;
import com.intellij.util.CommonProcessors.CollectProcessor;
import com.intellij.util.CommonProcessors.FindFirstProcessor;
import com.intellij.util.Processor;
import com.intellij.util.Processors;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.xml.util.PsiElementPointer;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import static com.intellij.spring.contexts.model.visitors.CommonSpringModelVisitorContext.context;
import static com.intellij.spring.contexts.model.visitors.SpringModelVisitors.visitRecursionAwareRelatedModels;
import static com.intellij.spring.contexts.model.visitors.SpringModelVisitors.visitRelatedModels;

public class SpringModelVisitorUtils {

  private static final CommonSpringModelVisitorContext.Exec<SpringBeanPointer>
    DOM_BEANS = (m, p) -> {
    if (m instanceof LocalXmlModel) {
      for (SpringBeanPointer pointer : ((LocalXmlModel)m).getLocalBeans()) {
        if (!p.process(pointer)) return false;
      }
    }
    return true;
  };

  private static final CommonSpringModelVisitorContext.Exec<SpringBeanPointer>
    PLACEHOLDERS = (m, p) -> {
    if (m instanceof LocalXmlModel) {
      for (SpringBeanPointer pointer : ((LocalXmlModel)m).getPlaceholderConfigurerBeans()) {
        if (!p.process(pointer)) return false;
      }
    }
    return true;
  };
  private static final CommonSpringModelVisitorContext.Exec<SpringBeansPackagesScan>
    SCANS = (m, p) -> {
    if (m instanceof LocalModel) {
      for (SpringBeansPackagesScan scan : ((LocalModel<?>)m).getPackagesScans()) {
        if (!p.process(scan)) return false;
      }
    }
    return true;
  };
  private static final CommonSpringModelVisitorContext.Exec<SpringBeanPointer>
    ANNO_CONFIGS = (m, p) -> {
    if (m instanceof LocalXmlModel) {
      for (SpringBeanPointer pointer : ((LocalXmlModel)m).getAnnotationConfigAppContexts()) {
        if (!p.process(pointer)) return false;
      }
    }
    return true;
  };
  private static final CommonSpringModelVisitorContext.Exec<String> PROFILES = (m, p) -> {
    if (m instanceof AbstractSimpleSpringModel) {
      for (String s : ((AbstractSimpleSpringModel)m).getProfiles()) {
        if (!p.process(s)) return false;
      }
    }
    return true;
  };

  @NotNull
  public static Set<String> getProfiles(@NotNull CommonSpringModel model) {
    CollectProcessor<String> processor = new CollectProcessor<>();
    visitRelatedModels(model, context(processor, PROFILES));
    return ContainerUtil.newHashSet(processor.getResults());
  }

  @NotNull
  public static Collection<SpringBeanPointer> getAllDomBeans(@NotNull CommonSpringModel model) {
    CollectProcessor<SpringBeanPointer> processor = new CollectProcessor<>();
    visitRelatedModels(model, context(processor, DOM_BEANS));
    return processor.getResults();
  }

  public static Collection<PsiElementPointer> getDuplicatedNames(@NotNull CommonSpringModel model, @NotNull final String beanName) {
    Collection<PsiElementPointer> pointers = ContainerUtil.newArrayList();
    visitRelatedModels(model, context(Processors.cancelableCollectProcessor(pointers),
                                      (m, p) -> {
                                        if (m instanceof LocalXmlModel) {
                                          for (PsiElementPointer pointer : ((LocalXmlModel)m)
                                            .getDuplicatedBeanNames(beanName)) {
                                            if (!p.process(pointer)) return false;
                                          }
                                        }
                                        return true;
                                      }));
    return pointers;
  }

  @NotNull
  public static List<SpringBeanPointer> getPlaceholderConfigurers(@NotNull CommonSpringModel model) {
    List<SpringBeanPointer> placeholders = ContainerUtil.newArrayList();
    visitRelatedModels(model, context(Processors.cancelableCollectProcessor(placeholders), PLACEHOLDERS));
    return placeholders;
  }

  @NotNull
  public static List<SpringBeansPackagesScan> getComponentScans(@NotNull CommonSpringModel model) {
    List<SpringBeansPackagesScan> scans = ContainerUtil.newArrayList();
    visitRecursionAwareRelatedModels(model, context(Processors.cancelableCollectProcessor(scans), SCANS));
    return scans;
  }

  public static boolean hasComponentScans(@NotNull CommonSpringModel model) {
    FindFirstProcessor<SpringBeansPackagesScan> processor = new FindFirstProcessor<>();
    visitRecursionAwareRelatedModels(model, context(processor, SCANS));
    return processor.isFound();
  }

  @NotNull
  public static List<SpringBeanPointer> getAnnotationConfigApplicationContexts(@NotNull CommonSpringModel model) {
    List<SpringBeanPointer> pointers = ContainerUtil.newLinkedList();
    visitRecursionAwareRelatedModels(model, context(Processors.cancelableCollectProcessor(pointers), ANNO_CONFIGS));
    return pointers;
  }

  public static Collection<XmlTag> getCustomBeanCandidates(@NotNull CommonSpringModel model, final String id) {
    Set<XmlTag> tags = ContainerUtil.newHashSet();
    visitRecursionAwareRelatedModels(model, context(Processors.cancelableCollectProcessor(tags), (m, p) -> {
      if (m instanceof LocalXmlModel) {
        for (XmlTag xmlTag : ((LocalXmlModel)m).getCustomBeans(id)) {
          if (!p.process(xmlTag)) return false;
        }
      }
      return true;
    }));
    return tags;
  }

  @NotNull
  public static List<SpringBeanPointer> getDescendants(@NotNull CommonSpringModel model, @NotNull final SpringBeanPointer context) {
    List<SpringBeanPointer> pointers = ContainerUtil.newLinkedList();

    visitRecursionAwareRelatedModels(model, context(Processors.cancelableCollectProcessor(pointers), (m, p) -> {
      if (m instanceof LocalXmlModel) {
        for (SpringBeanPointer pointer : ((LocalXmlModel)m).getDescendantBeans(context)) {
          if (!p.process(pointer)) return false;
        }
      }
      return true;
    }));

    return pointers;
  }

  @NotNull
  public static Set<String> getAllBeanNames(@NotNull CommonSpringModel model, @NotNull final SpringBeanPointer pointer) {
    String name = pointer.getName();
    if (StringUtil.isEmptyOrSpaces(name)) return Collections.emptySet();

    Set<String> results = ContainerUtil.newHashSet();

    visitRelatedModels(model, context(Processors.cancelableCollectProcessor(results), (m, p) -> {
      if (m instanceof AbstractSimpleSpringModel) {
        for (String s : ((AbstractSimpleSpringModel)m).getAllBeanNames(pointer)) {
          if (!p.process(s)) return false;
        }
      }
      return true;
    }));
    return results.size() > 0 ? ContainerUtil.newHashSet(results) : Collections.singleton(name);
  }

  @NotNull
  public static List<SpringBeanPointer> findQualifiedBeans(@NotNull CommonSpringModel model, @NotNull SpringQualifier qualifier) {
    List<SpringBeanPointer> pointers = ContainerUtil.newLinkedList();
    visitRecursionAwareRelatedModels(model, context(Processors.cancelableCollectProcessor(pointers), (m, p) -> {
      if (m instanceof AbstractSimpleSpringModel) {
        for (SpringBeanPointer pointer : ((AbstractSimpleSpringModel)m).findQualified(qualifier)) {
          if (!p.process(pointer)) return false;
        }
      }
      return true;
    }));

    return pointers;
  }

  /**
   * Collects config files from the given model and all related models.
   * @param model the model to traverse.
   * @return config files from the given model and all related models.
   */
  @NotNull
  public static Set<PsiFile> getConfigFiles(@NotNull CommonSpringModel model) {
    CommonProcessors.CollectProcessor<PsiFile> processor = new CommonProcessors.CollectProcessor<>(ContainerUtil.newHashSet());
    processConfigFiles(model, processor);
    return ContainerUtil.newHashSet(processor.getResults());
  }

  /**
   * Checks whether the given config file belongs to the model or one of the related models or not.
   * @param model the model to traverse.
   * @param configFile the configuration file to search.
   * @return {@code true} if the given model or one of the related models uses the given config file, otherwise {@code false}.
   */
  public static boolean hasConfigFile(@NotNull CommonSpringModel model, @NotNull PsiFile configFile) {
    CommonProcessors.FindProcessor<PsiFile> findProcessor = new CommonProcessors.FindFirstProcessor<PsiFile>() {
      @Override
      protected boolean accept(PsiFile file) {
        return configFile.equals(file);
      }
    };
    processConfigFiles(model, findProcessor);
    return findProcessor.isFound();
  }

  public static void processConfigFiles(@NotNull CommonSpringModel model, @NotNull Processor<? super PsiFile> processor) {
    visitRecursionAwareRelatedModels(model, context(processor, (m, p) -> {
      if (m instanceof LocalModel) {
        PsiFile file = ((LocalModel)m).getConfig().getContainingFile();
        if (file != null && !p.process(file)) return false;
      }
      return true;
    }), false);
  }
}
