/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring;

import com.intellij.openapi.module.Module;
import com.intellij.psi.PsiFile;
import com.intellij.psi.xml.XmlTag;
import com.intellij.spring.contexts.model.visitors.CommonSpringModelVisitorContext;
import com.intellij.spring.contexts.model.visitors.SpringModelVisitors;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.SpringQualifier;
import com.intellij.spring.model.xml.context.SpringBeansPackagesScan;
import com.intellij.util.Processor;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

public interface CommonSpringModel {

  /**
   * Spring models:
   * - have connections/links/associations to other models (via imports/component-scans/implicit fileset configurations/auto-configurations/etc.)
   * - contain other models {@link com.intellij.spring.contexts.model.SpringModel}
   *
   * @return related models
   * @see SpringModelVisitors#visitRelatedModels(CommonSpringModel, CommonSpringModelVisitorContext)
   * @since 2018.1
   */
  default Set<CommonSpringModel> getRelatedModels() {
    return Collections.emptySet();
  }

  /**
   * @param params    Search parameters.
   * @param processor Results processor.
   * @return {@code true} to continue processing or {@code false} to stop.
   * @see com.intellij.spring.model.utils.SpringModelSearchers
   * @see SpringModelSearchParameters#canSearch()
   * @since 14
   */
  boolean processByClass(@NotNull SpringModelSearchParameters.BeanClass params, @NotNull Processor<SpringBeanPointer> processor);

  /**
   * @param params    Search parameters.
   * @param processor Results processor.
   * @return {@code true} to continue processing or {@code false} to stop.
   * @see com.intellij.spring.model.utils.SpringModelSearchers
   * @see SpringModelSearchParameters#canSearch()
   * @since 14
   */
  boolean processByName(@NotNull SpringModelSearchParameters.BeanName params, @NotNull Processor<SpringBeanPointer> processor);

  /**
   * @param processor Results processor.
   * @return {@code true} to continue processing or {@code false} to stop.
   * @see com.intellij.spring.model.utils.SpringModelSearchers
   * @since 2017.3
   */
  default boolean processAllBeans(@NotNull Processor<SpringBeanPointer> processor) {return true;}

  /**
   * NOTE: Expensive operation. Consider using {@code process...()} methods instead.
   *
   * @return All beans.
   */
  @NotNull
  Collection<SpringBeanPointer> getAllCommonBeans();

  /**
   * Returns the associated module.
   *
   * @return Module containing this model or {@code null} if model is not bound to a specific module (e.g. Project-global).
   */
  @Nullable
  Module getModule();

  @Nullable
  Set<String> getActiveProfiles();

  /**
   * @return all configuration files.
   * @deprecated use {@link SpringModelVisitorUtils#getConfigFiles(CommonSpringModel)} instead.
   */
  @Deprecated
  @ApiStatus.ScheduledForRemoval(inVersion = "2019.2")
  @NotNull
  default Set<PsiFile> getConfigFiles() {
    return SpringModelVisitorUtils.getConfigFiles(this);
  }

  /**
   * Returns whether this model uses the given config file.
   *
   * @param configFile config file.
   * @return {@code true} if file belongs to model.
   * @since 14
   * @deprecated use {@link SpringModelVisitorUtils#hasConfigFile(CommonSpringModel, PsiFile)} instead.
   */
  @Deprecated
  @ApiStatus.ScheduledForRemoval(inVersion = "2019.2")
  default boolean hasConfigFile(@NotNull PsiFile configFile) {
    return SpringModelVisitorUtils.hasConfigFile(this, configFile);
  }

  @Deprecated // to remove in 2018.3, moved to AbstractSimpleSpringModel#getProfiles().
  @NotNull
  default Set<String> getAllProfiles() {return SpringModelVisitorUtils.getProfiles(this);}

  @Deprecated // to remove in 2018.3, moved to AbstractSimpleSpringModel#getAllBeanNames().
  @NotNull
  default Set<String> getAllBeanNames(@NotNull final String beanName) {
    return Collections.emptySet();
  }

  @Deprecated // to remove in 2018.3, moved to LocalXmlModel#getPlaceholderConfigurerBeans().
  @NotNull
  default List<SpringBeanPointer> getPlaceholderConfigurers() {
    return SpringModelVisitorUtils.getPlaceholderConfigurers(this);
  }

  @Deprecated // to remove in 2018.3, moved to LocalModel#getPackagesScans().
  @NotNull
  default List<? extends SpringBeansPackagesScan> getComponentScans() {
    return SpringModelVisitorUtils.getComponentScans(this);
  }

  @Deprecated // to remove in 2018.3, moved to LocalXmlModel#getAnnotationConfigAppContexts().
  @NotNull
  default List<SpringBeanPointer> getAnnotationConfigApplicationContexts() {
    return SpringModelVisitorUtils.getAnnotationConfigApplicationContexts(this);
  }

  @Deprecated // to remove in 2018.3, moved to LocalXmlModel#getCustomBeans().
  default Collection<XmlTag> getCustomBeanCandidates(String id) {
    return SpringModelVisitorUtils.getCustomBeanCandidates(this, id);
  }

  @Deprecated // to remove in 2018.3, moved to LocalModel#findQualified()
  @NotNull
  default List<SpringBeanPointer> findQualifiedBeans(@NotNull SpringQualifier qualifier) {
    return SpringModelVisitorUtils.findQualifiedBeans(this, qualifier);
  }

  @Deprecated // to remove in 2018.3. moved to LocalXmlModel#getDescendantBeans()
  @NotNull
  default List<SpringBeanPointer> getDescendants(final @NotNull SpringBeanPointer context) {
    return SpringModelVisitorUtils.getDescendants(this, context);
  }

  @Deprecated // to remove in 2018.3.
  @NotNull
  default Collection<SpringBeanPointer> getAllDomBeans() {
    return SpringModelVisitorUtils.getAllDomBeans(this);
  }
}
