package com.intellij.database;

import com.intellij.openapi.extensions.AbstractExtensionPointBean;
import com.intellij.openapi.extensions.ExtensionPoint;
import com.intellij.openapi.extensions.Extensions;
import com.intellij.openapi.util.KeyedExtensionCollector;
import com.intellij.openapi.util.NotNullLazyValue;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.util.KeyedLazyInstance;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xmlb.annotations.Attribute;
import org.jetbrains.annotations.NotNull;

import java.lang.reflect.Constructor;
import java.util.Collections;
import java.util.List;

/**
 * @author gregsh
 */
public class DbmsExtension<T> extends KeyedExtensionCollector<T, Dbms> {
  public DbmsExtension(@NotNull String epName) {
    super(epName);
  }

  @NotNull
  @Override
  protected String keyToString(@NotNull Dbms key) {
    return key.getName();
  }

  @NotNull
  public List<T> allForDbms(@NotNull Dbms dbms) {
    return forKey(dbms);
  }

  public T forDbms(@NotNull Dbms dbms) {
    return findSingle(dbms);
  }

  @NotNull
  public List<Bean<T>> allExtensions() {
    if (!Extensions.getRootArea().hasExtensionPoint(getName())) return Collections.emptyList();
    ExtensionPoint<Bean<T>> point = Extensions.getRootArea().getExtensionPoint(getName());
    return point.getExtensionList();
  }

  @NotNull
  @Override
  protected List<T> buildExtensions(@NotNull String stringKey, @NotNull Dbms key) {
    List<T> result = super.buildExtensions(stringKey, key);
    if (result.isEmpty() && key != Dbms.UNKNOWN &&
        !forKey(Dbms.UNKNOWN).isEmpty() &&
        Extensions.getRootArea().hasExtensionPoint(getName())) {
      ExtensionPoint<Bean<T>> point = Extensions.getRootArea().getExtensionPoint(getName());
      result = ContainerUtil.newArrayList();
      for (Bean<T> bean : point.getExtensions()) {
        if (!StringUtil.equals(bean.dbmsStr, Dbms.UNKNOWN.getName())) continue;
        try {
          result.add(bean.doGetInstance(bean.findClass(bean.implementationClass), key));
        }
        catch (ReflectiveOperationException e) {
          throw new AssertionError(e);
        }
      }
    }
    return result;
  }

  public static class Bean<T> extends AbstractExtensionPointBean implements KeyedLazyInstance<T> {

    @Attribute("dbms")
    public String dbmsStr;

    @Attribute("implementationClass")
    public String implementationClass;

    private final NotNullLazyValue<T> myHandler = NotNullLazyValue.createValue(() -> {
      try {
        return doGetInstance(findClass(implementationClass), getDbms());
      }
      catch (ReflectiveOperationException e) {
        throw new RuntimeException(e);
      }
    });

    @NotNull
    T doGetInstance(@NotNull Class<T> tClass, @NotNull Dbms dbms) throws ReflectiveOperationException {
      Constructor<T> constructor;
      try {
        constructor = tClass.getConstructor(Dbms.class);
        return constructor.newInstance(dbms);
      }
      catch (NoSuchMethodException ignored) {
        constructor = tClass.getConstructor();
        return constructor.newInstance();
      }
    }

    @NotNull
    public Dbms getDbms() {
      Dbms dbms = Dbms.byName(dbmsStr);
      if (dbms == null) {
        throw new IllegalArgumentException("Dbms not found: " + dbmsStr);
      }
      return dbms;
    }

    @NotNull
    @Override
    public T getInstance() {
      return myHandler.getValue();
    }

    @Override
    public String getKey() {
      return dbmsStr;
    }

    @Override
    public String toString() {
      return dbmsStr;
    }
  }

  public static class InstanceBean<T> extends Bean<T> {
    @NotNull
    @Override
    T doGetInstance(@NotNull Class<T> aClass, @NotNull Dbms dbms) throws ReflectiveOperationException {
      //noinspection unchecked
      return (T)aClass.getField("INSTANCE").get(null);
    }
  }
}