/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.database;

import com.intellij.database.model.DasDataSource;
import com.intellij.database.model.NameVersion;
import com.intellij.database.model.RawConnectionConfig;
import com.intellij.openapi.util.Comparing;
import com.intellij.openapi.util.Version;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.util.ConcurrencyUtil;
import com.intellij.util.containers.ContainerUtil;
import org.intellij.lang.annotations.RegExp;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.TestOnly;

import java.util.Collection;
import java.util.concurrent.ConcurrentMap;

/**
 * @author gregsh
 */
public final class Dbms implements Comparable<Dbms> {

  private final static ConcurrentMap<String, Dbms> ourValues = ContainerUtil.newConcurrentMap();

  @NotNull
  public static Dbms create(@NotNull String displayName) {
    return create(displayName, displayName);
  }

  @NotNull
  public static Dbms create(@NotNull String name, @NotNull String displayName) {
    name = StringUtil.toUpperCase(name); // force uppercase
    Dbms existing = ourValues.get(name);
    if (existing != null) return existing;
    return ConcurrencyUtil.cacheOrGet(ourValues, name, new Dbms(name, displayName));
  }


  // supported dialects
  public static final Dbms UNKNOWN = create("UNKNOWN");
  public static final Dbms ORACLE = create("Oracle");
  public static final Dbms MEMSQL = create("MemSQL");
  public static final Dbms MARIADB = create("MariaDB");
  public static final Dbms MYSQL = create("MySQL");
  public static final Dbms POSTGRES = create("POSTGRES", "PostgreSQL");
  public static final Dbms REDSHIFT = create("REDSHIFT", "Amazon Redshift");
  public static final Dbms MSSQL = create("MSSQL", "Microsoft SQL Server");
  public static final Dbms AZURE = create("AZURE", "Azure SQL Database");
  public static final Dbms SYBASE = create("SYBASE", "Sybase ASE");
  public static final Dbms DB2 = create("DB2", "IBM Db2 LUW");
  public static final Dbms SQLITE = create("SQLite");
  public static final Dbms HSQL = create("HSQLDB");
  public static final Dbms H2 = create("H2");
  public static final Dbms DERBY = create("DERBY", "Apache Derby");
  public static final Dbms EXASOL = create("Exasol");
  public static final Dbms CLICKHOUSE = create("ClickHouse");
  public static final Dbms CASSANDRA = create("CASSANDRA", "Apache Cassandra");

  // unsupported dialects
  public static final Dbms VERTICA = create("Vertica");
  public static final Dbms GREENPLUM = create("Greenplum");
  public static final Dbms HANA = create("HANA");
  public static final Dbms FIREBIRD = create("Firebird");
  public static final Dbms PRESTO = create("Presto");
  public static final Dbms INFORMIX = create("Informix");
  public static final Dbms IMPALA = create("Impala");
  public static final Dbms NETEZZA = create("Netezza");
  public static final Dbms PHOENIX = create("Phoenix");
  public static final Dbms HIVE = create("Hive");
  public static final Dbms SPARK = create("SPARK", "Spark SQL");
  public static final Dbms SNOWFLAKE = create("Snowflake");
  public static final Dbms INGRES = create("Ingres");
  public static final Dbms TERADATA = create("Teradata");
  public static final Dbms OPENEDGE = create("OpenEdge");
  public static final Dbms TIBERO = create("Tibero");
  public static final Dbms FILEMAKER = create("FileMaker");
  public static final Dbms FRONTBASE = create("FrontBase");
  public static final Dbms MONGO = create("Mongo", "MongoDB");

  private final String myName;
  private final String myDisplayName;

  private Dbms(@NotNull String name, @NotNull String displayName) {
    myName = name;
    myDisplayName = displayName;
  }

  @NotNull
  public String getName() {
    return myName;
  }

  @NotNull
  public String getDisplayName() {
    return myDisplayName;
  }

  @Override
  public String toString() {
    return getName();
  }

  public boolean isOracle() { return this == ORACLE; }
  public boolean isMysql() { return this == MYSQL || this == MARIADB || this == MEMSQL; }
  public boolean isPostgres() { return this == POSTGRES || this == REDSHIFT; }
  public boolean isRedshift() { return this == REDSHIFT; }
  public boolean isVertica() {return this == VERTICA; }
  public boolean isMicrosoft() { return this == MSSQL || this == AZURE; }
  public boolean isSybase() { return this == SYBASE; }
  public boolean isDb2() { return this == DB2; }
  public boolean isHsqldb() { return this == HSQL; }
  public boolean isH2() { return this == H2; }
  public boolean isDerby() { return this == DERBY; }
  public boolean isSqlite() { return this == SQLITE; }
  public boolean isTransactSql() { return isMicrosoft() || isSybase(); }
  public boolean isExasol() { return this == EXASOL; }
  public boolean isClickHouse() { return this == CLICKHOUSE; }
  public boolean isCassandra() { return this == CASSANDRA; }

  @NotNull
  public static Dbms forDataSource(@NotNull DasDataSource o) {
    NameVersion p = o.getDatabaseVersion();
    Dbms result = fromString(p.version);
    if (result == UNKNOWN) {
      result = fromString(p.name);
    }
    if (result == UNKNOWN && o instanceof RawConnectionConfig) {
      result = forConnection((RawConnectionConfig)o);
    }
    if (result == POSTGRES) {
      Version version = p.version != null && p.version.contains("8") ? Version.parseVersion(p.version) : null;
      if (version != null && version.major == 8 && version.bugfix == 2) {
        result = REDSHIFT;
      }
    }
    return result;
  }

  @NotNull
  public static Dbms forConnection(@Nullable RawConnectionConfig o) {
    if (o == null) return UNKNOWN;
    Dbms result = fromString(o.getUrl());
    if (result != UNKNOWN) return result;
    return fromString(o.getDriverClass());
  }

  @NotNull
  public static Dbms fromString(@Nullable String text) {
    if (text == null) return UNKNOWN;
    @RegExp
    String pattern = "(?i).*\\b(?:%s).*";
    for (Dbms family : ourValues.values()) {
      if (text.matches(String.format(pattern, family.getName()))) return family;
    }
    if (text.matches(String.format(pattern, "hsql"))) return HSQL;
    if (text.matches(String.format(pattern, "microsoft|sqlserver"))) return MSSQL;
    if (text.matches(String.format(pattern, "adaptive server")) || StringUtil.startsWithIgnoreCase(text, "ase")) return SYBASE;
    if (text.startsWith("ids")) return INFORMIX;
    if (text.matches(String.format(pattern, "exa"))) return EXASOL;
    return UNKNOWN;
  }

  @Nullable
  public static Dbms byName(@NotNull String name) {
    return ourValues.get(name);
  }

  @TestOnly
  public static Collection<Dbms> allValues() {
    return ourValues.values();
  }

  @Override
  public int compareTo(@NotNull Dbms o) {
    return Comparing.compare(getName(), o.getName());
  }
}
