/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.jam

import com.intellij.codeInsight.completion.CompletionUtil
import com.intellij.jam.reflect.JamStringAttributeMeta
import com.intellij.patterns.StandardPatterns
import com.intellij.patterns.uast.ULiteralExpressionPattern
import com.intellij.patterns.uast.capture
import com.intellij.patterns.uast.literalExpression
import com.intellij.psi.*
import com.intellij.semantic.SemService
import org.jetbrains.uast.*

class JamReferenceContributor : PsiReferenceContributor() {

  override fun registerReferenceProviders(registrar: PsiReferenceRegistrar) {

    registrar.registerUastReferenceProvider(
      { element, _ -> element is ULiteralExpression && getContainingAnnotationEntry(element) != null },
      uastLiteralReferenceProvider(
        fun(uElement: ULiteralExpression, physicalPsiHost: PsiLanguageInjectionHost): Array<PsiReference> {
          val (psiAnnotation, name) = getContainingAnnotationEntry(uElement) ?: return PsiReference.EMPTY_ARRAY
          val (originalPsiAnnotation, _) = getContainingAnnotationEntry(CompletionUtil.getOriginalOrSelf(physicalPsiHost).toUElement())
                                           ?: return PsiReference.EMPTY_ARRAY
          val metas = SemService.getSemService(originalPsiAnnotation.project).getSemElements(JamService.ANNO_META_KEY,
                                                                                             originalPsiAnnotation)
          for (annotationMeta in metas) {
            val meta = annotationMeta?.findAttribute(name) as? JamStringAttributeMeta<*, *> ?: continue
            val converter = meta.converter as JamConverter<in Any>
            val jam = meta.getJam(PsiElementRef.real(psiAnnotation))

            (jam  as? List<JamStringAttributeElement<*>>)?.let { list ->
              return list.filter { isJamElementFromHost(it, physicalPsiHost) }.flatMap {
                converter.createReferences(it).toList().map { wrapRef(it, physicalPsiHost) }
              }.toTypedArray()
            }
            (jam as? JamStringAttributeElement<*>)?.let {
              return converter.createReferences(it).map {
                wrapRef(it, physicalPsiHost)
              }.toTypedArray()
            }

          }

          return PsiReference.EMPTY_ARRAY
        }
      )
    )

  }

  companion object {
    @JvmField
    val STRING_IN_ANNO: ULiteralExpressionPattern =
      literalExpression().annotationParams(capture(UAnnotation::class.java), StandardPatterns.string())

  }
}

fun isJamElementFromHost(attributeElement: JamStringAttributeElement<*>, physicalPsiHost: PsiLanguageInjectionHost) =
  attributeElement.psiElement == physicalPsiHost ||
  (attributeElement.psiElement.toUElement() as? ULiteralExpression)?.psiLanguageInjectionHost == physicalPsiHost

fun unwrapReference(reference: PsiReference): PsiReference = when (reference) {
  is NonPhysicalReferenceWrapper -> reference.wrappedReference
  else -> reference
}

private fun wrapRef(it: PsiReference, physicalPsi: PsiElement): PsiReference =
  if (it.element === physicalPsi) it else NonPhysicalReferenceWrapper(physicalPsi, it)

class NonPhysicalReferenceWrapper(physicalElement: PsiElement, val wrappedReference: PsiReference) :
  PsiReferenceBase<PsiElement>(physicalElement, wrappedReference.rangeInElement) {
  override fun resolve(): PsiElement? = wrappedReference.resolve()

  override fun getVariants(): Array<Any> = wrappedReference.variants

  override fun isSoft(): Boolean = wrappedReference.isSoft
}
