;;; sisheng.el --- sisheng input method for Chinese pinyin transliteration -*- coding: ctext -*- 

;; Copyright (C) 2004, 2006  Free Software Foundation, Inc.

;; Author: Werner LEMBERG <wl@gnu.org>

;; Keywords: multilingual, input method, Chinese, pinyin, sisheng

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 51 Franklin Street, Fifth Floor,
;; Boston, MA 02110-1301, USA.

;;; Synched up with: FSF 22.0.50.6

;;; Commentary:

;;; XEmacs change: use ctext as coding system for compatibility with 21.4

;;; Code:

(require 'quail)

(defconst sisheng-regexp
  "[-D$(A(5(B]\\|-A-D")-A

;; First element is the key,
;; second element is the vowel used for the input sequence,
;; last four elements are the resulting tones.
;;
(defconst sisheng-vowel-table
  '(("-D" "a" "" "-A" "$(A(#(B" "")
    ("-D" "e" "" "-A" "-B" "-A")
    ("-D" "i" "" "-A" "$(A(+(B" "")
    ("-D" "o" "" "-A" "$(A(/(B" "")
    ("-D" "u" "" "-A" "$(A(3(B" "")
    ("$(A(5(B" "v" "$(A(5(B" "$(A(6(B" "$(A(7(B" "$(A(8(B")
    ("-D" "ve" "-A-D" "-A" "-B" "-A")))


;; All possible syllables in Mandarin Chinese, presented in the first
;; tone.  Note that make-sisheng-rules always constructs rules for all
;; four tones even if some of those tones aren't used in Mandarin.
;;
(defconst sisheng-syllable-table
  '("-D" "i" "n" "ng" "o"-A

    "b-D" "bi" "bn" "bng" "bo"-A
    "b-Di" "bn" "bng"-A
    "b-D" "bin" "bio" "bi" "bn" "bng"-A
    "b-D"-A
    "b-D"-A

    "c-D" "ci" "cn" "cng" "co"-A
    "c-D" "cn" "cng"-A
    "c-D"-A
    "c-Dng" "cu"-A
    "c-D" "cun" "cu" "cn" "cu"-A

    "ch-D" "chi" "chn" "chng" "cho"-A
    "ch-D" "chn" "chng"-A
    "ch-D"-A
    "ch-Dng" "chu"-A
    "ch-D" "chu" "chui" "chun" "chung" "chu" "chn" "chu"-A

    "d-D" "di" "dn" "dng" "do"-A
    "d-D" "di" "dn" "dng"-A
    "d-D" "din" "dio" "di" "dng" "di"-A
    "d-Dng" "du"-A
    "d-D" "dun" "du" "dn" "du"-A

    "-D" "i" "n" "ng" "r"-A

    "f-D" "fn" "fng"-A
    "f-Di" "fn" "fng"-A
    "fi-Do"-A
    "f-D" "fu"-A
    "f-D"-A

    "g-D" "gi" "gn" "gng" "go"-A
    "g-D" "gi" "gn" "gng"-A
    "g-Dng" "gu"-A
    "g-D" "gu" "gui" "gun" "gung" "gu" "gn" "gu"-A

    "h-D" "hi" "hn" "hng" "ho"-A
    "h-D" "hi" "hn" "hng"-A
    "h-Dng" "hu"-A
    "h-D" "hu" "hui" "hun" "hung" "hu" "hn" "hu"-A

    "j-D" "ji" "jin" "jing" "jio" "ji" "jn" "jng" "jing" "ji"-A
    "j-D" "jun" "ju" "jn"-A

    "k-D" "ki" "kn" "kng" "ko"-A
    "k-D" "ki" "kn" "kng"-A
    "k-Dng" "ku"-A
    "k-D" "ku" "kui" "kun" "kung" "ku" "kn" "ku"-A

    "l-D" "li" "ln" "lng" "lo"-A
    "l-D" "li" "lng"-A
    "l-D" "li" "lin" "ling" "lio" "li" "ln" "lng" "li"-A
    "l-Dng" "lu"-A
    "l-D" "lun" "ln" "lu"-A
    "l$(A(5(B" "l-D"-A

    "m-D" "mi" "mn" "mng" "mo"-A
    "m-D" "mi" "mn" "mng"-A
    "m-D" "min" "mio" "mi" "mn" "mng" "mi"-A
    "m-D" "mu"-A
    "m-D"-A

    "n-D" "ni" "nn" "nng" "no"-A
    "n-D" "ni" "nn" "nng"-A
    "n-D" "nin" "ning" "nio" "ni" "nn" "nng" "ni"-A
    "n-Dng" "nu"-A
    "n-D" "nun" "nu"-A
    "n$(A(5(B" "n-D"-A

    "-D" "u"-A

    "p-D" "pi" "pn" "png" "po"-A
    "p-Di" "pn" "png"-A
    "p-D" "pin" "pio" "pi" "pn" "png"-A
    "p-D" "pu"-A
    "p-D"-A

    "q-D" "qi" "qin" "qing" "qio" "qi" "qn" "qng" "qing" "qi"-A
    "q-D" "qun" "qu" "qn"-A

    "r-Dn" "rng" "ro"-A
    "r-D" "rn" "rng"-A
    "r-D"-A
    "r-Dng" "ru"-A
    "r-D" "ru" "run" "ru" "rn" "ru"-A

    "s-D" "si" "sn" "sng" "so"-A
    "s-D" "sn" "sng"-A
    "s-D"-A
    "s-Dng" "su"-A
    "s-D" "sun" "su" "sn" "su"-A

    "sh-D" "shi" "shn" "shng" "sho"-A
    "sh-D" "shi" "shn" "shng"-A
    "sh-D"-A
    "sh-Du"-A
    "sh-D" "shu" "shui" "shun" "shung" "shu" "shn" "shu"-A

    "t-D" "ti" "tn" "tng" "to"-A
    "t-D" "ti" "tng"-A
    "t-D" "tin" "tio" "ti" "tng"-A
    "t-Dng" "tu"-A
    "t-D" "tun" "tu" "tn" "tu"-A

    "w-D" "wi" "wn" "wng"-A
    "w-Di" "wn" "wng"-A
    "w-D"-A
    "w-D"-A

    "x-D" "xi" "xin" "xing" "xio" "xi" "xn" "xng" "xing" "xi"-A
    "x-D" "xun" "xu" "xn"-A

    "y-D" "yn" "yng" "yo"-A
    "y-D"-A
    "y-D" "yn" "yng"-A
    "y-D" "yng" "yu"-A
    "y-D" "yun" "yu" "yn"-A

    "z-D" "zi" "zn" "zng" "zo"-A
    "z-D" "zi" "zn" "zng"-A
    "z-D"-A
    "z-Dng" "zu"-A
    "z-D" "zun" "zu" "zn" "zu"-A

    "zh-D" "zhi" "zhn" "zhng" "zho"-A
    "zh-D" "zhi" "zhn" "zhng"-A
    "zh-D"-A
    "zh-Dng" "zhu"-A
    "zh-D" "zhu" "zhui" "zhun" "zhung" "zhu" "zhn" "zhu"))-A

;; This function converts e.g.
;;
;;   "zhu-D"-A
;;
;; into
;;
;;   (("zhuo4" ["zhu"])
;;    ("zhuo3" ["zhu$(A(/(B"])
;;    ("zhuo2" ["zhu"])
;;    ("zhuo1" ["zhu-D"]))-A
;;
(defun quail-make-sisheng-rules (syllable)
  (let ((case-fold-search t)
	vowel-match
	vowel-list
	input-vowel
	base-key
	key
	value
	key-value-list
	(i 1))
    (string-match sisheng-regexp syllable)
    (setq vowel-match (downcase (match-string 0 syllable)))
    (setq vowel-list
	  (cdr (assoc-string vowel-match sisheng-vowel-table)))
    (setq input-vowel (car vowel-list))
    (setq base-key (replace-match input-vowel nil nil syllable))
    (while (<= i 4)
      (setq key (concat base-key (number-to-string i)))
      (setq value (vector (replace-match (nth i vowel-list) nil nil syllable)))
      (push (list key value) key-value-list)
      (setq i (1+ i)))
    key-value-list))

;; Set up sisheng input method.
;;
(quail-define-package
 "chinese-sisheng"			; name
 "Chinese"				; language
 "$(A(7(B"					; title
 t					; guidance
 "Ssh-Dng input method for pnyn transliteration of Chinese.-A

Examples: shuang1 -> shu-Dng-A
          Lv3     -> L$(A(7(B
          AN4     -> N

Use the fifth (unstressed) tone for syllables containing `'
without a tone mark.

Example:  nve5    -> ne
"					; docstring
 nil					; translation-keys
 t					; forget-last-selection
 nil					; deterministic
 nil					; kbd-translate
 nil					; show-layout
 nil					; create-decode-map
 nil					; maximum-shortest
 nil					; overlay-plist
 nil					; update-translation-function
 nil					; conversion-keys
 t					; simple
 )

;; Call quail-make-sisheng-rules for all syllables in sisheng-syllable-table.
;;
(let ((case-table-save (current-case-table))
      sisheng-list)
  (set-case-table (standard-case-table))
  (dolist (syllable sisheng-syllable-table)
    (setq sisheng-list
	  (append (quail-make-sisheng-rules syllable)
		  sisheng-list)))

  (dolist (syllable sisheng-syllable-table)
    (setq sisheng-list
	  (append (quail-make-sisheng-rules (upcase-initials syllable))
		  sisheng-list)))

  (dolist (syllable sisheng-syllable-table)
    (setq sisheng-list
	  (append (quail-make-sisheng-rules (upcase syllable))
		  sisheng-list)))

  (eval `(quail-define-rules
	  ,@sisheng-list
	
	  ("lv5" ["l"])
	  ("lve5" ["le"])
	  ("nv5" ["n"])
	  ("nve5" ["ne"])

	  ("Lv5" ["L"])
	  ("Lve5" ["Le"])
	  ("Nv5" ["N"])
	  ("Nve5" ["Ne"])

	  ("LV5" ["L"])
	  ("LVE5" ["LE"])
	  ("NV5" ["N"])
	  ("NVE5" ["NE"])))
  (set-case-table case-table-save))

;;; arch-tag: 1fa6ba5f-6747-44bc-bf12-30628ad3e8ad
