<?php

declare(strict_types=1);

/*
 * This file is part of the TYPO3 CMS project.
 *
 * It is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, either version 2
 * of the License, or any later version.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 * The TYPO3 project - inspiring people to share!
 */

namespace TYPO3\CMS\Backend\Form\FormDataProvider;

use TYPO3\CMS\Backend\Form\FormDataProviderInterface;
use TYPO3\CMS\Core\Resource\Exception\ResourceDoesNotExistException;
use TYPO3\CMS\Core\Resource\Folder;
use TYPO3\CMS\Core\Resource\ResourceFactory;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Utility\MathUtility;

/**
 * Resolve databaseRow field content for type=folder
 */
readonly class TcaFolder implements FormDataProviderInterface
{
    public function __construct(
        private ResourceFactory $resourceFactory,
    ) {}

    /**
     * Initialize new row with default values from various sources
     *
     * @return array
     */
    public function addData(array $result)
    {
        foreach ($result['processedTca']['columns'] as $fieldName => $fieldConfig) {
            if (empty($fieldConfig['config']['type']) || $fieldConfig['config']['type'] !== 'folder') {
                continue;
            }

            // Sanitize max items, set to 99999 if not defined
            $result['processedTca']['columns'][$fieldName]['config']['maxitems'] = MathUtility::forceIntegerInRange(
                $fieldConfig['config']['maxitems'] ?? 0,
                0,
                99999
            );
            if ($result['processedTca']['columns'][$fieldName]['config']['maxitems'] === 0) {
                $result['processedTca']['columns'][$fieldName]['config']['maxitems'] = 99999;
            }

            $databaseRowFieldContent = '';
            if (!empty($result['databaseRow'][$fieldName])) {
                $databaseRowFieldContent = (string)$result['databaseRow'][$fieldName];
            }

            $items = [];
            // Simple list of folders
            $folderList = GeneralUtility::trimExplode(',', $databaseRowFieldContent, true);
            foreach ($folderList as $folder) {
                if (empty($folder)) {
                    continue;
                }
                try {
                    $folderObject = $this->resourceFactory->retrieveFileOrFolderObject($folder);
                    if ($folderObject instanceof Folder) {
                        $items[] = [
                            'folder' => $folder,
                        ];
                    }
                } catch (ResourceDoesNotExistException) {
                    continue;
                }
            }

            $result['databaseRow'][$fieldName] = $items;
        }

        return $result;
    }
}
