// Copyright (C) 2016 The Qt Company Ltd.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR BSD-3-Clause

import QtQuick 2.1
import QtQuick.Controls 1.1
import QtQuick.Controls.Private 1.0
import QtQuick.Controls.Styles 1.1

/*!
    \qmltype ScrollView
    \inqmlmodule QtQuick.Controls
    \since  5.1
    \ingroup views
    \brief Provides a scrolling view within another Item.

    A ScrollView can be used either to replace a \l Flickable or decorate an
    existing \l Flickable. Depending on the platform, it will add scroll bars and
    a content frame.

    Only one Item can be a direct child of the ScrollView and the child is implicitly anchored
    to fill the scroll view.

    Example:
    \code
    ScrollView {
        Image { imageSource: "largeImage.png" }
    }
    \endcode

    In the previous example the Image item will implicitly get scroll behavior as if it was
    used within a \l Flickable. The width and height of the child item will be used to
    define the size of the content area.

    Example:
    \code
    ScrollView {
        ListView {
            ...
        }
    }
    \endcode

    In this case the content size of the ScrollView will simply mirror that of its contained
    \l flickableItem.

    You can create a custom appearance for a ScrollView by
    assigning a \l {QtQuick.Controls.Styles::ScrollViewStyle}{ScrollViewStyle}.
*/

FocusScope {
    id: root

    implicitWidth: 240
    implicitHeight: 150

    /*!
        This property tells the ScrollView if it should render
        a frame around its content.

        The default value is \c false.
    */
    property bool frameVisible: false

    /*!
        This property controls if there should be a highlight
        around the frame when the ScrollView has input focus.

        The default value is \c false.

        \note This property is only applicable on some platforms, such
        as Mac OS.
    */
    property bool highlightOnFocus: false

    /*!
        \qmlproperty Item ScrollView::viewport

        The viewport determines the current "window" on the contentItem.
        In other words, it clips it and the size of the viewport tells you
        how much of the content area is visible.
    */
    property alias viewport: viewportItem

    /*!
        \qmlproperty Item ScrollView::flickableItem

        The flickableItem of the ScrollView. If the contentItem provided
        to the ScrollView is a Flickable, it will be the \l contentItem.
    */
    readonly property alias flickableItem: internal.flickableItem

    /*!
        The contentItem of the ScrollView. This is set by the user.

        Note that the definition of contentItem is somewhat different to that
        of a Flickable, where the contentItem is implicitly created.
    */
    default property Item contentItem

    /*! \internal */
    property Item __scroller: scroller
    /*! \internal */
    property alias __wheelAreaScrollSpeed: wheelArea.scrollSpeed
    /*! \internal */
    property int __scrollBarTopMargin: 0
    /*! \internal */
    property int __viewTopMargin: 0
    /*! \internal */
    property alias __horizontalScrollBar: scroller.horizontalScrollBar
    /*! \internal */
    property alias __verticalScrollBar: scroller.verticalScrollBar
    /*! \qmlproperty Component ScrollView::style

        The style Component for this control.
        \sa {Qt Quick Controls Styles QML Types}

    */
    property Component style: Qt.createComponent(Settings.style + "/ScrollViewStyle.qml", root)

    /*! \internal */
    property Style __style: styleLoader.item

    activeFocusOnTab: true

    onContentItemChanged: {

        if (contentItem.hasOwnProperty("contentY") && // Check if flickable
                contentItem.hasOwnProperty("contentHeight")) {
            internal.flickableItem = contentItem // "Use content if it is a flickable
            internal.flickableItem.parent = viewportItem
        } else {
            internal.flickableItem = flickableComponent.createObject(viewportItem)
            contentItem.parent = internal.flickableItem.contentItem
        }
        internal.flickableItem.anchors.fill = viewportItem
        if (!Settings.hasTouchScreen)
            internal.flickableItem.interactive = false
    }


    children: Item {
        id: internal

        property Flickable flickableItem

        Loader {
            id: styleLoader
            sourceComponent: style
            onStatusChanged: {
                if (status === Loader.Error)
                    console.error("Failed to load Style for", root)
            }
            property alias __control: root
        }

        Binding {
            target: flickableItem
            property: "contentHeight"
            when: contentItem !== flickableItem
            value: contentItem ? contentItem.height : 0
        }

        Binding {
            target: flickableItem
            when: contentItem !== flickableItem
            property: "contentWidth"
            value: contentItem ? contentItem.width : 0
        }

        Connections {
            target: flickableItem

            onContentYChanged:  {
                scroller.blockUpdates = true
                scroller.verticalScrollBar.value = flickableItem.contentY
                scroller.blockUpdates = false
            }

            onContentXChanged:  {
                scroller.blockUpdates = true
                scroller.horizontalScrollBar.value = flickableItem.contentX
                scroller.blockUpdates = false
            }

        }

        anchors.fill: parent

        Component {
            id: flickableComponent
            Flickable {}
        }

        WheelArea {
            id: wheelArea
            parent: flickableItem

            // ### Note this is needed due to broken mousewheel behavior in Flickable.

            anchors.fill: parent

            property int acceleration: 40
            property int flickThreshold: 20
            property real speedThreshold: 3
            property real ignored: 0.001 // ## flick() does not work with 0 yVelocity
            property int maxFlick: 400

            property bool horizontalRecursionGuard: false
            property bool verticalRecursionGuard: false

            horizontalMinimumValue: flickableItem ? flickableItem.originX : 0
            horizontalMaximumValue: flickableItem ? flickableItem.originX + flickableItem.contentWidth - viewport.width : 0

            verticalMinimumValue: flickableItem ? flickableItem.originY : 0
            verticalMaximumValue: flickableItem ? flickableItem.originY + flickableItem.contentHeight - viewport.height + __viewTopMargin : 0

            Connections {
                target: flickableItem

                onContentYChanged: {
                    wheelArea.verticalRecursionGuard = true
                    wheelArea.verticalValue = flickableItem.contentY
                    wheelArea.verticalRecursionGuard = false
                }
                onContentXChanged: {
                    wheelArea.horizontalRecursionGuard = true
                    wheelArea.horizontalValue = flickableItem.contentX
                    wheelArea.horizontalRecursionGuard = false
                }
            }

            onVerticalValueChanged: {
                if (!verticalRecursionGuard) {
                    if (flickableItem.contentY < flickThreshold && verticalDelta > speedThreshold) {
                        flickableItem.flick(ignored, Math.min(maxFlick, acceleration * verticalDelta))
                    } else if (flickableItem.contentY > flickableItem.contentHeight
                               - flickThreshold - viewport.height && verticalDelta < -speedThreshold) {
                        flickableItem.flick(ignored, Math.max(-maxFlick, acceleration * verticalDelta))
                    } else {
                        flickableItem.contentY = verticalValue
                    }
                }
            }

            onHorizontalValueChanged: {
                if (!horizontalRecursionGuard)
                    flickableItem.contentX = horizontalValue
            }
        }

        ScrollViewHelper {
            id: scroller
            anchors.fill: parent
            active: wheelArea.active
            property bool outerFrame: !frameVisible || !(__style ? __style.__externalScrollBars : 0)
            property int scrollBarSpacing: outerFrame ? 0 : (__style ? __style.__scrollBarSpacing : 0)
            property int verticalScrollbarOffset: verticalScrollBar.visible && !verticalScrollBar.isTransient ?
                                                      verticalScrollBar.width + scrollBarSpacing : 0
            property int horizontalScrollbarOffset: horizontalScrollBar.visible && !horizontalScrollBar.isTransient ?
                                                        horizontalScrollBar.height + scrollBarSpacing : 0
            Loader {
                id: frameLoader
                sourceComponent: __style ? __style.frame : null
                anchors.fill: parent
                anchors.rightMargin: scroller.outerFrame ? 0 : scroller.verticalScrollbarOffset
                anchors.bottomMargin: scroller.outerFrame ? 0 : scroller.horizontalScrollbarOffset
            }

            Item {
                id: viewportItem
                anchors.fill: frameLoader
                anchors.topMargin: frameVisible ? __style.padding.top : 0
                anchors.leftMargin: frameVisible ? __style.padding.left : 0
                anchors.rightMargin:  (frameVisible ? __style.padding.right : 0) +  (scroller.outerFrame ? scroller.verticalScrollbarOffset : 0)
                anchors.bottomMargin: (frameVisible ? __style.padding.bottom : 0) + (scroller.outerFrame ? scroller.horizontalScrollbarOffset : 0)
                clip: true
            }
        }
        FocusFrame { visible: highlightOnFocus && root.activeFocus }
    }
}
