"""
pyubx2 Performance benchmarking utility

Usage (kwargs optional): python3 benchmark.py cycles=10000

Created on 5 Nov 2021

:author: semuadmin (Steve Smith)
:copyright: semuadmin © 2021
:license: BSD 3-Clause
"""

# pylint: disable=line-too-long

from sys import argv
from time import process_time_ns
from platform import version as osver, python_version
from pyubx2.ubxreader import UBXReader
from pyubx2._version import __version__ as ubxver

UBXMESSAGES = (
    b"\xb5b\x01\x03\x10\x00\x08\xb5b!\x03\xdf\x00\x088[\x00\x00+\xba\x9e\x00T\xe5",
    b'\xb5b\x01\x01\x14\x00\x08\xb5b!"\xe7\xab\x16\x19\xf3\x1c\xff\x16\xf8f\x1e\xff\x00\x00\x00\xd8F',
    b"\xb5b\x01\x02\x1c\x00\x08\xb5b!\xd9(\xaa\xfe\xdc\xee\xdb\x1f0d\x01\x00\xcd\xa6\x00\x00\xd3\x05\x00\x00\x11\x08\x00\x00\xc5:",
    b"\xb5b\x01\x04\x12\x00\x08\xb5b!r\x00e\x005\x00S\x00:\x00.\x00#\x00Ar",
    b"\xb5b\x01\x11\x14\x00\x08\xb5b!\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xff\xff\x1a\x00\x00\x00|\xc9",
    b"\xb5b\x01\x12$\x00\x08\xb5b!\xff\xff\xff\xff\x00\x00\x00\x00\x01\x00\x00\x00\x01\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x1a\x00\x00\x00\x80\xa8\x12\x01\xcb\x11",
    b"\xb5b\x01 \x10\x00\x08\xb5b!<{\xfa\xff\x8a\x08\x12\x07\x06\x00\x00\x00\xd2\xbb",
    b"\xb5b\x01#\x14\x00\x08\xb5b!\x13\xcd\x00\x00.{\xfa\xff\xc0\x02\x07\x03\x05\x00\x00\x00\xcb\xd3",
    b"\xb5b\x01$\x14\x00\x08\xb5b!\xe7\x8b\x08\x00#{\xfa\xff>\x03\x04\x07\x05\x00\x00\x00\xdb1",
    b"\xb5b\x01%\x14\x00\x08\xb5b!\xf5\x8b\x08\x00I{\xfa\xff\x8a\x04\x12\x07\x04\x00\x00\x00j\xa7",
    b'\xb5b\x01\x07\\\x00\x08\xb5b!\xe5\x07\x0c\x04\x0b";7\x1a\x00\x00\x00<{\xfa\xff\x03\x03\xea\x1a\xd9(\xaa\xfe\xdc\xee\xdb\x1f0d\x01\x00\xcd\xa6\x00\x00\xd3\x05\x00\x00\x11\x08\x00\x00\xf9\xff\xff\xff\x04\x00\x00\x00\t\x00\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00\x03\x01\x00\x00\x80\xa8\x12\x01e\x00\x00\x00\xee\x13O/\x00\x00\x00\x00\x00\x00\x00\x00\xa7D',
    b'\xb5b\x014\x8a\x02\x08\xb5b!\x01k\x00\x00\x00\x01\x05\x001\x00\x00\x02\r*1G\x00\x03\x05\x001\x00\x00\x04\x05*1G\x00\x05\r*1G\x00\x06\x05*1G\x00\x07\r*1G\x00\x08\x05\x001\x00\x00\t\r*1G\x00\n\x05\x001\x00\x00\x0b\x0e*1\x00\x00\x0c\x05\x001\x00\x00\r\r*1G\x00\x0e\r\x001\x00\x00\x0f\r*1G\x00\x10\t\x001\x00\x00\x11\x05\x001\x00\x00\x12\r*1G\x00\x13\x05\x001\x00\x00\x14\r*1G\x00\x15\x05\x001\x00\x00\x16\x06\x001\x00\x00\x17\x05\x001\x00\x00\x18\x05\x001\x00\x00\x19\x05\x001\x00\x00\x1a\x05\x001\x00\x00\x1b\t\x001\x00\x00\x1d\x05*1G\x00\x1e\r*1G\x00\x1f\x05\x001\x00\x00 \x05\x001\x00\x01{\r\x00?\x00\x01\x88\r!?\x00\x02\x01\x05\x00.\x00\x02\x02\r\x00.\x00\x02\x03\r-.G\x02\x04\x05\x00.\x00\x02\x05\x05).G\x02\x07\r/.G\x02\x08\r-.G\x02\t\x05\x00.\x00\x02\x0b\x05\x00.\x00\x02\x0c\x05\x00.\x00\x02\r\r-.G\x02\x0f\x05).D\x02\x13\x05\x00.\x00\x02\x15\x05\x00.\x00\x02\x18\r+.G\x02\x19\r/.G\x02\x1a\r/.G\x02\x1b\x05\x00.\x00\x02\x1e\t\x00.\x00\x02\x1f\x05\x00.\x00\x02!\r,.G\x02$\x05\x00.\x00\x03\x01\x05\x00(\x00\x03\x02\x05\x00(\x00\x03\x03\x05\x00(\x00\x03\x04\x05\x00(\x00\x03\x05\r\x00(\x00\x03\x06\x05\x00(\x00\x03\x07\r*(G\x03\x08\x05\x00(\x00\x03\t\x05\x00(\x00\x03\n\r*(G\x03\x0b\r*(G\x03\x0c\x05&(G\x03\r\x05\x00(\x00\x03\x0e\r*(G\x03\x10\x05\x00(\x00\x03\x13\x05"(G\x03\x14\x05\x00(\x00\x03\x15\r*(G\x03\x16\r*(G\x03\x17\x05\x00(\x00\x03\x18\x05\x00(\x00\x03\x19\x05\x00(\x00\x03\x1a\r*(G\x03\x1b\x05\x00(\x00\x03\x1c\r*(G\x03\x1d\x05\x00(\x00\x03\x1e\x05\x00(\x00\x03"\x05&\x00D\x06\x01\r&5G\x06\x02\r&5G\x06\x03\r&5G\x06\x04\x05\x005\x00\x06\x05\x05\x005\x00\x06\x06\x05\x005\x00\x06\x07\x15 5G\x06\x08\x15$5G\x06\t\x15 5G\x06\n\x1d"5D\x06\x0b\x0e&5\x00\x06\x0c\r&5G\x06\r\x05\x005\x00\x06\x0e\x05\x005\x00\x06\x0f\x05\x005\x00\x06\x10\x05\x005\x00\x06\x11\r&5G\x06\x12\r&5G\x06\x13\r\x005\x00\x06\x14\x05\x005\x00\x06\x15\x05\x005\x00\x06\x16\x05\x005\x00\x06\x17\x15 5G\x06\x18\x15"5G&e',
    b"\xb5b\x015\x0c\x02\x08\xb5b!\x01+\x00\x00\x00\x02\x1f\x12\xdd\x00\x11\x00_YQ\x00\x00\x05\x11=\x1b\x01\t\x00\\YQ\x00\x00\x07(3C\x00\x03\x00_YQ\x00\x00\t\x00\x0fY\x00\x00\x00\x10Y\x00\x00\x00\x0b\x00\x11\xd0\x00\x00\x00 \x1a\x00\x00\x00\r )\t\x01\xaf\xff_YQ\x00\x00\x0e\x00\x0e\x92\x00\x00\x00\x10\x12\x00\x00\x00\x0f)\x0c\x13\x01\x01\x00_YQ\x00\x00\x10\x00\x01\x0c\x00\x00\x00\x10\x12\x00\x00\x00\x12\x1e\x0cK\x01\xfb\xff^YQ\x00\x00\x14\x13A\xcd\x00\xe4\xff\\YQ\x00\x00\x1b\x00\x02 \x00\x00\x00\x10\x12\x00\x00\x00\x1e)G\x80\x00\xfe\xff_YQ\x00\x01{\x00\x16\x8c\x00\x00\x00\x11\x12\x00\x00\x01\x88\x1f\x1d\xab\x00\x00\x00\x17\x1a\x00\x00\x02\x02\x00!\x87\x00\x00\x00\x10\x12\x00\x00\x02\x03\x1f'_\x001\x00\x1fY\x00\x00\x02\x07!\x1e\x15\x01\x04\x00\x1fY\x00\x00\x02\x08%T&\x01\xf9\xff\x1fY\x00\x00\x02\r\"\x17\xed\x00\n\x00\x1fY\x00\x00\x02\x18\x00\x08\x17\x00\x00\x00\x10Y\x00\x00\x02\x19\x1d&H\x003\x00\x1fY\x00\x00\x02\x1a\x1d\x1d)\x01\"\x00\x1fY\x00\x00\x02\x1e\x00\x02\xae\x00\x00\x00\x10\x12\x00\x00\x02!\x1a\tV\x01\xd8\xff\x1fY\x00\x00\x03\x05\x00\x07s\x00\x00\x00\x11\x12\x00\x00\x03\x07\x1b\x169\x00z\xff\x1fY\x00\x00\x03\n\x00\x18H\x00\x00\x00\x10Y\x00\x00\x03\x0b\x11\x05\x02\x01\xa7\xff\x14Y\x00\x00\x03\x0e'\x13)\x01\x07\x00\x1fY\x00\x00\x03\x15\x1dO\x91\x00\"\x00\x1fY\x00\x00\x03\x16\x1a\x1az\x00\xbd\xff\x1cY\x00\x00\x03\x1a\x00\x16\xbf\x00\x00\x00\x10Y\x00\x00\x03\x1c\x1c\nL\x01\xe5\xff\x1fY\x00\x00\x06\x01\x1d*i\x00!\x00\x1fY\x00\x00\x06\x02\"DC\x01\x18\x00\x1fY\x00\x00\x06\x03\x1f\x11*\x01\xc5\xff\x1fY\x00\x00\x06\n\r\x0e\x1a\x00\xe3\xfe\x14Y\x00\x00\x06\x0b\x1e8G\x00\x00\x00'\x1a\x00\x00\x06\x0c\x1a'\xa3\x00\xfb\xff\x1fY\x00\x00\x06\x11\x1d\r\xfd\x00\x10\x00\x1fY\x00\x00\x06\x12'\x1a.\x01\xed\xff\x1fY\x00\x00\x06\x13\x00\x10]\x01\x00\x00\x10\x12\x00\x00\xba\xd3",
    b'\xb5b\x01C\x08\x02\x08\xb5b!\x00 \x00\x00\x00\x02\x00\x00\x11\x00\x1f\x07\x01\x02i\x01\x00\x00\x00\x00\x00\x05\x00\x00\t\x00\x11\x04\x01\x02i\x01\x00\x00\x00\x00\x00\x07\x00\x00\x03\x00(\x07\x01\x02i\x01\x00\x00\x00\x00\x00\r\x00\x00\xaf\xff \x07\x01\x02i\x01\x00\x00\x00\x00\x00\x0f\x00\x00\x01\x00)\x07\x01\x02i\x01\x00\x00\x00\x00\x00\x12\x00\x00\xfb\xff\x1e\x06\x01\x02i\x01\x00\x00\x00\x00\x00\x14\x00\x00\xe4\xff\x13\x04\x01\x02i\x01\x00\x00\x00\x00\x00\x1e\x00\x00\xfe\xff)\x07\x01\x02i\x01\x00\x00\x00\x00\x01{\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x01\x88\x00\x00\x00\x00\x1f\x07\x00\x00\x00\x00\x00\x00\x00\x00\x02\x03\x00\x001\x00\x1f\x07\x00\x02)\x00\x00\x00\x00\x00\x02\x07\x00\x00\x04\x00!\x07\x00\x02)\x00\x00\x00\x00\x00\x02\x08\x00\x00\xf9\xff%\x07\x00\x02)\x00\x00\x00\x00\x00\x02\r\x00\x00\n\x00"\x07\x00\x02)\x00\x00\x00\x00\x00\x02\x19\x00\x003\x00\x1d\x07\x00\x02)\x00\x00\x00\x00\x00\x02\x1a\x00\x00"\x00\x1d\x07\x00\x02)\x00\x00\x00\x00\x00\x02!\x00\x00\xd8\xff\x1a\x07\x00\x02)\x00\x00\x00\x00\x00\x03\x05\x01\x00\x00\x00\x00\x01\x00\x00\x01\x00\x00\x00\x00\x00\x03\x07\x00\x00z\xff\x1b\x07\x00\x02)\x00\x00\x00\x00\x00\x03\x0b\x00\x00\xa7\xff\x11\x04\x00\x00\x01\x00\x00\x00\x00\x00\x03\x0e\x00\x00\x07\x00\'\x07\x00\x02)\x00\x00\x00\x00\x00\x03\x15\x00\x00"\x00\x1d\x07\x00\x02)\x00\x00\x00\x00\x00\x03\x16\x00\x00\xbd\xff\x1a\x04\x00\x02)\x00\x00\x00\x00\x00\x03\x1c\x00\x00\xe5\xff\x1c\x07\x00\x02)\x00\x00\x00\x00\x00\x06\x01\x00\x08!\x00\x1d\x07\x00\x02)\x00\x00\x00\x00\x00\x06\x02\x00\x03\x18\x00"\x07\x00\x02)\x00\x00\x00\x00\x00\x06\x03\x00\x0c\xc5\xff\x1f\x07\x00\x02)\x00\x00\x00\x00\x00\x06\n\x00\x00\xe3\xfe\r\x04\x00\x00\x01\x00\x00\x00\x00\x00\x06\x0b\x00\x07\x00\x00\x1e\x07\x00\x00\x00\x00\x00\x00\x00\x00\x06\x0c\x00\x06\xfb\xff\x1a\x07\x00\x02)\x00\x00\x00\x00\x00\x06\x11\x00\x0b\x10\x00\x1d\x07\x00\x02)\x00\x00\x00\x00\x00\x06\x12\x00\x04\xed\xff\'\x07\x00\x02)\x00\x00\x00\x00\x00MS',
    b"\xb5b\x01\x03\x10\x00\x08\xb5b!\x03\xdf\x00\x088[\x00\x00+\xba\x9e\x00T\xe5",
    b'\xb5b\x01\x01\x14\x00\x08\xb5b!"\xe7\xab\x16\x19\xf3\x1c\xff\x16\xf8f\x1e\xff\x00\x00\x00\xd8F',
    b"\xb5b\x01\x02\x1c\x00\x08\xb5b!\xd9(\xaa\xfe\xdc\xee\xdb\x1f0d\x01\x00\xcd\xa6\x00\x00\xd3\x05\x00\x00\x11\x08\x00\x00\xc5:",
    b"\xb5b\x01\x04\x12\x00\x08\xb5b!r\x00e\x005\x00S\x00:\x00.\x00#\x00Ar",
    b"\xb5b\x01\x11\x14\x00\x08\xb5b!\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xff\xff\x1a\x00\x00\x00|\xc9",
    b"\xb5b\x01\x12$\x00\x08\xb5b!\xff\xff\xff\xff\x00\x00\x00\x00\x01\x00\x00\x00\x01\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x1a\x00\x00\x00\x80\xa8\x12\x01\xcb\x11",
    b"\xb5b\x01 \x10\x00\x08\xb5b!<{\xfa\xff\x8a\x08\x12\x07\x06\x00\x00\x00\xd2\xbb",
    b"\xb5b\x01#\x14\x00\x08\xb5b!\x13\xcd\x00\x00.{\xfa\xff\xc0\x02\x07\x03\x05\x00\x00\x00\xcb\xd3",
    b"\xb5b\x01$\x14\x00\x08\xb5b!\xe7\x8b\x08\x00#{\xfa\xff>\x03\x04\x07\x05\x00\x00\x00\xdb1",
    b"\xb5b\x01%\x14\x00\x08\xb5b!\xf5\x8b\x08\x00I{\xfa\xff\x8a\x04\x12\x07\x04\x00\x00\x00j\xa7",
    b'\xb5b\x01\x07\\\x00\x08\xb5b!\xe5\x07\x0c\x04\x0b";7\x1a\x00\x00\x00<{\xfa\xff\x03\x03\xea\x1a\xd9(\xaa\xfe\xdc\xee\xdb\x1f0d\x01\x00\xcd\xa6\x00\x00\xd3\x05\x00\x00\x11\x08\x00\x00\xf9\xff\xff\xff\x04\x00\x00\x00\t\x00\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00\x03\x01\x00\x00\x80\xa8\x12\x01e\x00\x00\x00\xee\x13O/\x00\x00\x00\x00\x00\x00\x00\x00\xa7D',
    b'\xb5b\x014\x8a\x02\x08\xb5b!\x01k\x00\x00\x00\x01\x05\x001\x00\x00\x02\r*1G\x00\x03\x05\x001\x00\x00\x04\x05*1G\x00\x05\r*1G\x00\x06\x05*1G\x00\x07\r*1G\x00\x08\x05\x001\x00\x00\t\r*1G\x00\n\x05\x001\x00\x00\x0b\x0e*1\x00\x00\x0c\x05\x001\x00\x00\r\r*1G\x00\x0e\r\x001\x00\x00\x0f\r*1G\x00\x10\t\x001\x00\x00\x11\x05\x001\x00\x00\x12\r*1G\x00\x13\x05\x001\x00\x00\x14\r*1G\x00\x15\x05\x001\x00\x00\x16\x06\x001\x00\x00\x17\x05\x001\x00\x00\x18\x05\x001\x00\x00\x19\x05\x001\x00\x00\x1a\x05\x001\x00\x00\x1b\t\x001\x00\x00\x1d\x05*1G\x00\x1e\r*1G\x00\x1f\x05\x001\x00\x00 \x05\x001\x00\x01{\r\x00?\x00\x01\x88\r!?\x00\x02\x01\x05\x00.\x00\x02\x02\r\x00.\x00\x02\x03\r-.G\x02\x04\x05\x00.\x00\x02\x05\x05).G\x02\x07\r/.G\x02\x08\r-.G\x02\t\x05\x00.\x00\x02\x0b\x05\x00.\x00\x02\x0c\x05\x00.\x00\x02\r\r-.G\x02\x0f\x05).D\x02\x13\x05\x00.\x00\x02\x15\x05\x00.\x00\x02\x18\r+.G\x02\x19\r/.G\x02\x1a\r/.G\x02\x1b\x05\x00.\x00\x02\x1e\t\x00.\x00\x02\x1f\x05\x00.\x00\x02!\r,.G\x02$\x05\x00.\x00\x03\x01\x05\x00(\x00\x03\x02\x05\x00(\x00\x03\x03\x05\x00(\x00\x03\x04\x05\x00(\x00\x03\x05\r\x00(\x00\x03\x06\x05\x00(\x00\x03\x07\r*(G\x03\x08\x05\x00(\x00\x03\t\x05\x00(\x00\x03\n\r*(G\x03\x0b\r*(G\x03\x0c\x05&(G\x03\r\x05\x00(\x00\x03\x0e\r*(G\x03\x10\x05\x00(\x00\x03\x13\x05"(G\x03\x14\x05\x00(\x00\x03\x15\r*(G\x03\x16\r*(G\x03\x17\x05\x00(\x00\x03\x18\x05\x00(\x00\x03\x19\x05\x00(\x00\x03\x1a\r*(G\x03\x1b\x05\x00(\x00\x03\x1c\r*(G\x03\x1d\x05\x00(\x00\x03\x1e\x05\x00(\x00\x03"\x05&\x00D\x06\x01\r&5G\x06\x02\r&5G\x06\x03\r&5G\x06\x04\x05\x005\x00\x06\x05\x05\x005\x00\x06\x06\x05\x005\x00\x06\x07\x15 5G\x06\x08\x15$5G\x06\t\x15 5G\x06\n\x1d"5D\x06\x0b\x0e&5\x00\x06\x0c\r&5G\x06\r\x05\x005\x00\x06\x0e\x05\x005\x00\x06\x0f\x05\x005\x00\x06\x10\x05\x005\x00\x06\x11\r&5G\x06\x12\r&5G\x06\x13\r\x005\x00\x06\x14\x05\x005\x00\x06\x15\x05\x005\x00\x06\x16\x05\x005\x00\x06\x17\x15 5G\x06\x18\x15"5G&e',
    b"\xb5b\x015\x0c\x02\x08\xb5b!\x01+\x00\x00\x00\x02\x1f\x12\xdd\x00\x11\x00_YQ\x00\x00\x05\x11=\x1b\x01\t\x00\\YQ\x00\x00\x07(3C\x00\x03\x00_YQ\x00\x00\t\x00\x0fY\x00\x00\x00\x10Y\x00\x00\x00\x0b\x00\x11\xd0\x00\x00\x00 \x1a\x00\x00\x00\r )\t\x01\xaf\xff_YQ\x00\x00\x0e\x00\x0e\x92\x00\x00\x00\x10\x12\x00\x00\x00\x0f)\x0c\x13\x01\x01\x00_YQ\x00\x00\x10\x00\x01\x0c\x00\x00\x00\x10\x12\x00\x00\x00\x12\x1e\x0cK\x01\xfb\xff^YQ\x00\x00\x14\x13A\xcd\x00\xe4\xff\\YQ\x00\x00\x1b\x00\x02 \x00\x00\x00\x10\x12\x00\x00\x00\x1e)G\x80\x00\xfe\xff_YQ\x00\x01{\x00\x16\x8c\x00\x00\x00\x11\x12\x00\x00\x01\x88\x1f\x1d\xab\x00\x00\x00\x17\x1a\x00\x00\x02\x02\x00!\x87\x00\x00\x00\x10\x12\x00\x00\x02\x03\x1f'_\x001\x00\x1fY\x00\x00\x02\x07!\x1e\x15\x01\x04\x00\x1fY\x00\x00\x02\x08%T&\x01\xf9\xff\x1fY\x00\x00\x02\r\"\x17\xed\x00\n\x00\x1fY\x00\x00\x02\x18\x00\x08\x17\x00\x00\x00\x10Y\x00\x00\x02\x19\x1d&H\x003\x00\x1fY\x00\x00\x02\x1a\x1d\x1d)\x01\"\x00\x1fY\x00\x00\x02\x1e\x00\x02\xae\x00\x00\x00\x10\x12\x00\x00\x02!\x1a\tV\x01\xd8\xff\x1fY\x00\x00\x03\x05\x00\x07s\x00\x00\x00\x11\x12\x00\x00\x03\x07\x1b\x169\x00z\xff\x1fY\x00\x00\x03\n\x00\x18H\x00\x00\x00\x10Y\x00\x00\x03\x0b\x11\x05\x02\x01\xa7\xff\x14Y\x00\x00\x03\x0e'\x13)\x01\x07\x00\x1fY\x00\x00\x03\x15\x1dO\x91\x00\"\x00\x1fY\x00\x00\x03\x16\x1a\x1az\x00\xbd\xff\x1cY\x00\x00\x03\x1a\x00\x16\xbf\x00\x00\x00\x10Y\x00\x00\x03\x1c\x1c\nL\x01\xe5\xff\x1fY\x00\x00\x06\x01\x1d*i\x00!\x00\x1fY\x00\x00\x06\x02\"DC\x01\x18\x00\x1fY\x00\x00\x06\x03\x1f\x11*\x01\xc5\xff\x1fY\x00\x00\x06\n\r\x0e\x1a\x00\xe3\xfe\x14Y\x00\x00\x06\x0b\x1e8G\x00\x00\x00'\x1a\x00\x00\x06\x0c\x1a'\xa3\x00\xfb\xff\x1fY\x00\x00\x06\x11\x1d\r\xfd\x00\x10\x00\x1fY\x00\x00\x06\x12'\x1a.\x01\xed\xff\x1fY\x00\x00\x06\x13\x00\x10]\x01\x00\x00\x10\x12\x00\x00\xba\xd3",
    b'\xb5b\x01C\x08\x02\x08\xb5b!\x00 \x00\x00\x00\x02\x00\x00\x11\x00\x1f\x07\x01\x02i\x01\x00\x00\x00\x00\x00\x05\x00\x00\t\x00\x11\x04\x01\x02i\x01\x00\x00\x00\x00\x00\x07\x00\x00\x03\x00(\x07\x01\x02i\x01\x00\x00\x00\x00\x00\r\x00\x00\xaf\xff \x07\x01\x02i\x01\x00\x00\x00\x00\x00\x0f\x00\x00\x01\x00)\x07\x01\x02i\x01\x00\x00\x00\x00\x00\x12\x00\x00\xfb\xff\x1e\x06\x01\x02i\x01\x00\x00\x00\x00\x00\x14\x00\x00\xe4\xff\x13\x04\x01\x02i\x01\x00\x00\x00\x00\x00\x1e\x00\x00\xfe\xff)\x07\x01\x02i\x01\x00\x00\x00\x00\x01{\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x01\x88\x00\x00\x00\x00\x1f\x07\x00\x00\x00\x00\x00\x00\x00\x00\x02\x03\x00\x001\x00\x1f\x07\x00\x02)\x00\x00\x00\x00\x00\x02\x07\x00\x00\x04\x00!\x07\x00\x02)\x00\x00\x00\x00\x00\x02\x08\x00\x00\xf9\xff%\x07\x00\x02)\x00\x00\x00\x00\x00\x02\r\x00\x00\n\x00"\x07\x00\x02)\x00\x00\x00\x00\x00\x02\x19\x00\x003\x00\x1d\x07\x00\x02)\x00\x00\x00\x00\x00\x02\x1a\x00\x00"\x00\x1d\x07\x00\x02)\x00\x00\x00\x00\x00\x02!\x00\x00\xd8\xff\x1a\x07\x00\x02)\x00\x00\x00\x00\x00\x03\x05\x01\x00\x00\x00\x00\x01\x00\x00\x01\x00\x00\x00\x00\x00\x03\x07\x00\x00z\xff\x1b\x07\x00\x02)\x00\x00\x00\x00\x00\x03\x0b\x00\x00\xa7\xff\x11\x04\x00\x00\x01\x00\x00\x00\x00\x00\x03\x0e\x00\x00\x07\x00\'\x07\x00\x02)\x00\x00\x00\x00\x00\x03\x15\x00\x00"\x00\x1d\x07\x00\x02)\x00\x00\x00\x00\x00\x03\x16\x00\x00\xbd\xff\x1a\x04\x00\x02)\x00\x00\x00\x00\x00\x03\x1c\x00\x00\xe5\xff\x1c\x07\x00\x02)\x00\x00\x00\x00\x00\x06\x01\x00\x08!\x00\x1d\x07\x00\x02)\x00\x00\x00\x00\x00\x06\x02\x00\x03\x18\x00"\x07\x00\x02)\x00\x00\x00\x00\x00\x06\x03\x00\x0c\xc5\xff\x1f\x07\x00\x02)\x00\x00\x00\x00\x00\x06\n\x00\x00\xe3\xfe\r\x04\x00\x00\x01\x00\x00\x00\x00\x00\x06\x0b\x00\x07\x00\x00\x1e\x07\x00\x00\x00\x00\x00\x00\x00\x00\x06\x0c\x00\x06\xfb\xff\x1a\x07\x00\x02)\x00\x00\x00\x00\x00\x06\x11\x00\x0b\x10\x00\x1d\x07\x00\x02)\x00\x00\x00\x00\x00\x06\x12\x00\x04\xed\xff\'\x07\x00\x02)\x00\x00\x00\x00\x00MS',
    b"\xb5b\x01\x03\x10\x00\x08\xb5b!\x03\xdf\x00\x088[\x00\x00+\xba\x9e\x00T\xe5",
    b'\xb5b\x01\x01\x14\x00\x08\xb5b!"\xe7\xab\x16\x19\xf3\x1c\xff\x16\xf8f\x1e\xff\x00\x00\x00\xd8F',
    b"\xb5b\x01\x02\x1c\x00\x08\xb5b!\xd9(\xaa\xfe\xdc\xee\xdb\x1f0d\x01\x00\xcd\xa6\x00\x00\xd3\x05\x00\x00\x11\x08\x00\x00\xc5:",
    b"\xb5b\x01\x04\x12\x00\x08\xb5b!r\x00e\x005\x00S\x00:\x00.\x00#\x00Ar",
    b"\xb5b\x01\x11\x14\x00\x08\xb5b!\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xff\xff\x1a\x00\x00\x00|\xc9",
    b"\xb5b\x01\x12$\x00\x08\xb5b!\xff\xff\xff\xff\x00\x00\x00\x00\x01\x00\x00\x00\x01\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x1a\x00\x00\x00\x80\xa8\x12\x01\xcb\x11",
    b"\xb5b\x01 \x10\x00\x08\xb5b!<{\xfa\xff\x8a\x08\x12\x07\x06\x00\x00\x00\xd2\xbb",
    b"\xb5b\x01#\x14\x00\x08\xb5b!\x13\xcd\x00\x00.{\xfa\xff\xc0\x02\x07\x03\x05\x00\x00\x00\xcb\xd3",
    b"\xb5b\x01$\x14\x00\x08\xb5b!\xe7\x8b\x08\x00#{\xfa\xff>\x03\x04\x07\x05\x00\x00\x00\xdb1",
    b"\xb5b\x01%\x14\x00\x08\xb5b!\xf5\x8b\x08\x00I{\xfa\xff\x8a\x04\x12\x07\x04\x00\x00\x00j\xa7",
    b"\xb5b\x015\x0c\x02\x08\xb5b!\x01+\x00\x00\x00\x02\x1f\x12\xdd\x00\x11\x00_YQ\x00\x00\x05\x11=\x1b\x01\t\x00\\YQ\x00\x00\x07(3C\x00\x03\x00_YQ\x00\x00\t\x00\x0fY\x00\x00\x00\x10Y\x00\x00\x00\x0b\x00\x11\xd0\x00\x00\x00 \x1a\x00\x00\x00\r )\t\x01\xaf\xff_YQ\x00\x00\x0e\x00\x0e\x92\x00\x00\x00\x10\x12\x00\x00\x00\x0f)\x0c\x13\x01\x01\x00_YQ\x00\x00\x10\x00\x01\x0c\x00\x00\x00\x10\x12\x00\x00\x00\x12\x1e\x0cK\x01\xfb\xff^YQ\x00\x00\x14\x13A\xcd\x00\xe4\xff\\YQ\x00\x00\x1b\x00\x02 \x00\x00\x00\x10\x12\x00\x00\x00\x1e)G\x80\x00\xfe\xff_YQ\x00\x01{\x00\x16\x8c\x00\x00\x00\x11\x12\x00\x00\x01\x88\x1f\x1d\xab\x00\x00\x00\x17\x1a\x00\x00\x02\x02\x00!\x87\x00\x00\x00\x10\x12\x00\x00\x02\x03\x1f'_\x001\x00\x1fY\x00\x00\x02\x07!\x1e\x15\x01\x04\x00\x1fY\x00\x00\x02\x08%T&\x01\xf9\xff\x1fY\x00\x00\x02\r\"\x17\xed\x00\n\x00\x1fY\x00\x00\x02\x18\x00\x08\x17\x00\x00\x00\x10Y\x00\x00\x02\x19\x1d&H\x003\x00\x1fY\x00\x00\x02\x1a\x1d\x1d)\x01\"\x00\x1fY\x00\x00\x02\x1e\x00\x02\xae\x00\x00\x00\x10\x12\x00\x00\x02!\x1a\tV\x01\xd8\xff\x1fY\x00\x00\x03\x05\x00\x07s\x00\x00\x00\x11\x12\x00\x00\x03\x07\x1b\x169\x00z\xff\x1fY\x00\x00\x03\n\x00\x18H\x00\x00\x00\x10Y\x00\x00\x03\x0b\x11\x05\x02\x01\xa7\xff\x14Y\x00\x00\x03\x0e'\x13)\x01\x07\x00\x1fY\x00\x00\x03\x15\x1dO\x91\x00\"\x00\x1fY\x00\x00\x03\x16\x1a\x1az\x00\xbd\xff\x1cY\x00\x00\x03\x1a\x00\x16\xbf\x00\x00\x00\x10Y\x00\x00\x03\x1c\x1c\nL\x01\xe5\xff\x1fY\x00\x00\x06\x01\x1d*i\x00!\x00\x1fY\x00\x00\x06\x02\"DC\x01\x18\x00\x1fY\x00\x00\x06\x03\x1f\x11*\x01\xc5\xff\x1fY\x00\x00\x06\n\r\x0e\x1a\x00\xe3\xfe\x14Y\x00\x00\x06\x0b\x1e8G\x00\x00\x00'\x1a\x00\x00\x06\x0c\x1a'\xa3\x00\xfb\xff\x1fY\x00\x00\x06\x11\x1d\r\xfd\x00\x10\x00\x1fY\x00\x00\x06\x12'\x1a.\x01\xed\xff\x1fY\x00\x00\x06\x13\x00\x10]\x01\x00\x00\x10\x12\x00\x00\xba\xd3",
    b'\xb5b\x01C\x08\x02\x08\xb5b!\x00 \x00\x00\x00\x02\x00\x00\x11\x00\x1f\x07\x01\x02i\x01\x00\x00\x00\x00\x00\x05\x00\x00\t\x00\x11\x04\x01\x02i\x01\x00\x00\x00\x00\x00\x07\x00\x00\x03\x00(\x07\x01\x02i\x01\x00\x00\x00\x00\x00\r\x00\x00\xaf\xff \x07\x01\x02i\x01\x00\x00\x00\x00\x00\x0f\x00\x00\x01\x00)\x07\x01\x02i\x01\x00\x00\x00\x00\x00\x12\x00\x00\xfb\xff\x1e\x06\x01\x02i\x01\x00\x00\x00\x00\x00\x14\x00\x00\xe4\xff\x13\x04\x01\x02i\x01\x00\x00\x00\x00\x00\x1e\x00\x00\xfe\xff)\x07\x01\x02i\x01\x00\x00\x00\x00\x01{\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x01\x88\x00\x00\x00\x00\x1f\x07\x00\x00\x00\x00\x00\x00\x00\x00\x02\x03\x00\x001\x00\x1f\x07\x00\x02)\x00\x00\x00\x00\x00\x02\x07\x00\x00\x04\x00!\x07\x00\x02)\x00\x00\x00\x00\x00\x02\x08\x00\x00\xf9\xff%\x07\x00\x02)\x00\x00\x00\x00\x00\x02\r\x00\x00\n\x00"\x07\x00\x02)\x00\x00\x00\x00\x00\x02\x19\x00\x003\x00\x1d\x07\x00\x02)\x00\x00\x00\x00\x00\x02\x1a\x00\x00"\x00\x1d\x07\x00\x02)\x00\x00\x00\x00\x00\x02!\x00\x00\xd8\xff\x1a\x07\x00\x02)\x00\x00\x00\x00\x00\x03\x05\x01\x00\x00\x00\x00\x01\x00\x00\x01\x00\x00\x00\x00\x00\x03\x07\x00\x00z\xff\x1b\x07\x00\x02)\x00\x00\x00\x00\x00\x03\x0b\x00\x00\xa7\xff\x11\x04\x00\x00\x01\x00\x00\x00\x00\x00\x03\x0e\x00\x00\x07\x00\'\x07\x00\x02)\x00\x00\x00\x00\x00\x03\x15\x00\x00"\x00\x1d\x07\x00\x02)\x00\x00\x00\x00\x00\x03\x16\x00\x00\xbd\xff\x1a\x04\x00\x02)\x00\x00\x00\x00\x00\x03\x1c\x00\x00\xe5\xff\x1c\x07\x00\x02)\x00\x00\x00\x00\x00\x06\x01\x00\x08!\x00\x1d\x07\x00\x02)\x00\x00\x00\x00\x00\x06\x02\x00\x03\x18\x00"\x07\x00\x02)\x00\x00\x00\x00\x00\x06\x03\x00\x0c\xc5\xff\x1f\x07\x00\x02)\x00\x00\x00\x00\x00\x06\n\x00\x00\xe3\xfe\r\x04\x00\x00\x01\x00\x00\x00\x00\x00\x06\x0b\x00\x07\x00\x00\x1e\x07\x00\x00\x00\x00\x00\x00\x00\x00\x06\x0c\x00\x06\xfb\xff\x1a\x07\x00\x02)\x00\x00\x00\x00\x00\x06\x11\x00\x0b\x10\x00\x1d\x07\x00\x02)\x00\x00\x00\x00\x00\x06\x12\x00\x04\xed\xff\'\x07\x00\x02)\x00\x00\x00\x00\x00MS',
    b"\xb5b\x01\x03\x10\x00\x08\xb5b!\x03\xdf\x00\x088[\x00\x00+\xba\x9e\x00T\xe5",
    b'\xb5b\x01\x01\x14\x00\x08\xb5b!"\xe7\xab\x16\x19\xf3\x1c\xff\x16\xf8f\x1e\xff\x00\x00\x00\xd8F',
    b"\xb5b\x01\x02\x1c\x00\x08\xb5b!\xd9(\xaa\xfe\xdc\xee\xdb\x1f0d\x01\x00\xcd\xa6\x00\x00\xd3\x05\x00\x00\x11\x08\x00\x00\xc5:",
    b"\xb5b\x01\x04\x12\x00\x08\xb5b!r\x00e\x005\x00S\x00:\x00.\x00#\x00Ar",
    b"\xb5b\x01\x11\x14\x00\x08\xb5b!\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xff\xff\x1a\x00\x00\x00|\xc9",
    b"\xb5b\x01\x12$\x00\x08\xb5b!\xff\xff\xff\xff\x00\x00\x00\x00\x01\x00\x00\x00\x01\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x1a\x00\x00\x00\x80\xa8\x12\x01\xcb\x11",
    b"\xb5b\x01 \x10\x00\x08\xb5b!<{\xfa\xff\x8a\x08\x12\x07\x06\x00\x00\x00\xd2\xbb",
    b"\xb5b\x01#\x14\x00\x08\xb5b!\x13\xcd\x00\x00.{\xfa\xff\xc0\x02\x07\x03\x05\x00\x00\x00\xcb\xd3",
    b"\xb5b\x01$\x14\x00\x08\xb5b!\xe7\x8b\x08\x00#{\xfa\xff>\x03\x04\x07\x05\x00\x00\x00\xdb1",
    b"\xb5b\x01%\x14\x00\x08\xb5b!\xf5\x8b\x08\x00I{\xfa\xff\x8a\x04\x12\x07\x04\x00\x00\x00j\xa7",
)


def progbar(i: int, lim: int, inc: int = 20):
    """
    Display progress bar on console.

    :param int i: iteration
    :param int lim: max iterations
    :param int inc: bar increments (20)
    """

    i = min(i, lim)
    pct = int(i * inc / lim)
    if not i % int(lim / inc):
        print(
            f"{int(pct*100/inc):02}% " + "\u2593" * pct + "\u2591" * (inc - pct),
            end="\r",
        )


def benchmark(**kwargs) -> float:
    """
    pyrtcm Performance benchmark test.

    :param int cycles: (kwarg) number of test cycles (10,000)
    :returns: benchmark as transactions/second
    :rtype: float
    :raises: UBXStreamError
    """

    cyc = int(kwargs.get("cycles", 10000))
    txnc = len(UBXMESSAGES)
    txnt = txnc * cyc

    print(
        f"\nOperating system: {osver()}",
        f"\nPython version: {python_version()}",
        f"\npyubx2 version: {ubxver}",
        f"\nTest cycles: {cyc:,}",
        f"\nTxn per cycle: {txnc:,}",
    )

    msglen = 0
    start = process_time_ns()
    print(f"\nBenchmark test started at {start}")
    for i in range(cyc):
        progbar(i, cyc)
        for msg in UBXMESSAGES:
            msglen += len(msg)
            _ = UBXReader.parse(msg)
    end = process_time_ns()
    print(f"Benchmark test ended at {end}.")
    duration = end - start
    txs = round(txnt * 1e9 / duration, 2)
    kbs = round(msglen * 1e9 / duration / 2**10, 2)

    print(
        f"\n{txnt:,} messages processed in {duration/1e9:,.3f} seconds = {txs:,.2f} txns/second, {kbs:,.2f} kB/second.\n"
    )

    return txs, kbs


def main():
    """
    CLI Entry point.

    args as benchmark() method
    """

    benchmark(**dict(arg.split("=") for arg in argv[1:]))


if __name__ == "__main__":
    main()
