"""
Stream method tests for pysbf2

Created on 19 May 2025

*** NB: must be saved in UTF-8 format ***

@author: semuadmin
"""

# pylint: disable=line-too-long, invalid-name, missing-docstring, no-member

import os
import sys
import logging
from io import BufferedReader
import unittest

from pyrtcm import RTCMReader
from pysbf2 import (
    ERR_RAISE,
    ERR_LOG,
    ERR_IGNORE,
    SBFReader,
    SBF_PROTOCOL,
    NMEA_PROTOCOL,
    RTCM3_PROTOCOL,
    SBFStreamError,
    SBFParseError,
)

DIRNAME = os.path.dirname(__file__)


class StreamTest(unittest.TestCase):
    def setUp(self):
        self.maxDiff = None
        self.logger = logging.getLogger()
        self.logger.level = logging.DEBUG

    def tearDown(self):
        pass

    def testmixedall(
        self,
    ):  # test mixed stream of NMEA, SBF & RTCM messages with protfilter = all
        EXPECTED_RESULTS = (
            "<NMEA(GNGLL, lat=32.0658325, NS=N, lon=34.773819, EW=E, time=08:41:58, status=A, posMode=D)>",
            "<RTCM(1005, DF002=1005, DF003=0, DF021=0, DF022=1, DF023=1, DF024=1, DF141=0, DF025=4444030.802800001, DF142=1, DF001_1=0, DF026=3085671.2349, DF364=0, DF027=3366658.256)>",
            "<RTCM(1230, DF002=1230, DF003=0, DF421=1, DF001_3=0, DF422_1=0, DF422_2=0, DF422_3=0, DF422_4=0)>",
            "<NMEA(GNRMC, time=08:41:59, status=A, lat=32.0658325, NS=N, lon=34.773819, EW=E, spd=0.0, cog=, date=2022-02-08, mv=, mvEW=, posMode=D, navStatus=V)>",
            "<SBF(PVTGeodetic, TOW=14:07:09, WNc=2367, Type=6, Reserved1=0, AutoSet=0, 2D=0, Error=0, Latitude=0.9310293523340808, Longitude=-0.03921206770879602, Height=131.18596542546626, Undulation=48.477840423583984, Vn=0.0012262271484360099, Ve=-0.0007957226480357349, Vu=0.0004817131848540157, COG=-20000000000.0, RxClkBias=0.0693948459476198, RxClkDrift=0.1977977603673935, TimeSystem=0, Datum=0, NrSV=36, Corr_OrbClkUsed=1, Corr_RngUsed=1, Corr_IonoUsed=1, Corr_OrbAccUsed=0, Corr_DO229Active=0, Corr_RTKType=0, Reserved2=0, ReferenceID=123, MeanCorrAge=278, SignalInfo=1344405761, RAIMIntegrity=1, GalHPCAFail=0, GalIonStorm=0, Reserved3=0, NrBases=1, PPPSeedAge=0, Reserved4=0, PPPSeedType=0, Latency=55, HAccuracy=102, VAccuracy=138, BaseARP=0, PhaseCtrOffset=0, Reserved5=8, ARPOffset=1)>",
            "<SBF(PosLocal, TOW=14:07:09, WNc=2367, Type=6, Reserved1=0, AutoSet=0, 2D=0, Error=17, Lat=-20000000000.0, Lon=-20000000000.0, Alt=-20000000000.0, Datum=255)>",
        )
        i = 0
        with open(os.path.join(DIRNAME, "pygpsdata_mixed.log"), "rb") as stream:
            sbr = SBFReader(
                stream,
                protfilter=SBF_PROTOCOL | NMEA_PROTOCOL | RTCM3_PROTOCOL,
                quitonerror=ERR_RAISE,
            )
            for raw, parsed in sbr:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
            self.assertEqual(i, len(EXPECTED_RESULTS))

    def testmixednmea(
        self,
    ):  # test mixed stream of NMEA, SBF & RTCM messages with protfilter = nmea only
        EXPECTED_RESULTS = (
            "<NMEA(GNGLL, lat=32.0658325, NS=N, lon=34.773819, EW=E, time=08:41:58, status=A, posMode=D)>",
            "<NMEA(GNRMC, time=08:41:59, status=A, lat=32.0658325, NS=N, lon=34.773819, EW=E, spd=0.0, cog=, date=2022-02-08, mv=, mvEW=, posMode=D, navStatus=V)>",
        )
        i = 0
        with open(os.path.join(DIRNAME, "pygpsdata_mixed.log"), "rb") as stream:
            sbr = SBFReader(
                stream,
                protfilter=NMEA_PROTOCOL,
                quitonerror=ERR_RAISE,
            )
            for raw, parsed in sbr:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
            self.assertEqual(i, len(EXPECTED_RESULTS))

    def testmixedrtcm(
        self,
    ):  # test mixed stream of NMEA, SBF & RTCM messages with protfilter = rtcm3 only
        EXPECTED_RESULTS = (
            "<RTCM(1005, DF002=1005, DF003=0, DF021=0, DF022=1, DF023=1, DF024=1, DF141=0, DF025=4444030.802800001, DF142=1, DF001_1=0, DF026=3085671.2349, DF364=0, DF027=3366658.256)>",
            "<RTCM(1230, DF002=1230, DF003=0, DF421=1, DF001_3=0, DF422_1=0, DF422_2=0, DF422_3=0, DF422_4=0)>",
        )
        i = 0
        with open(os.path.join(DIRNAME, "pygpsdata_mixed.log"), "rb") as stream:
            sbr = SBFReader(
                stream,
                protfilter=RTCM3_PROTOCOL,
                quitonerror=ERR_RAISE,
            )
            for raw, parsed in sbr:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
            self.assertEqual(i, len(EXPECTED_RESULTS))

    def testmixedsbfrtcm(
        self,
    ):  # test mixed stream of NMEA, SBF & RTCM messages with protfilter = 7
        EXPECTED_RESULTS = (
            "<RTCM(1005, DF002=1005, DF003=0, DF021=0, DF022=1, DF023=1, DF024=1, DF141=0, DF025=4444030.802800001, DF142=1, DF001_1=0, DF026=3085671.2349, DF364=0, DF027=3366658.256)>",
            "<RTCM(1230, DF002=1230, DF003=0, DF421=1, DF001_3=0, DF422_1=0, DF422_2=0, DF422_3=0, DF422_4=0)>",
            "<SBF(PVTGeodetic, TOW=14:07:09, WNc=2367, Type=6, Reserved1=0, AutoSet=0, 2D=0, Error=0, Latitude=0.9310293523340808, Longitude=-0.03921206770879602, Height=131.18596542546626, Undulation=48.477840423583984, Vn=0.0012262271484360099, Ve=-0.0007957226480357349, Vu=0.0004817131848540157, COG=-20000000000.0, RxClkBias=0.0693948459476198, RxClkDrift=0.1977977603673935, TimeSystem=0, Datum=0, NrSV=36, Corr_OrbClkUsed=1, Corr_RngUsed=1, Corr_IonoUsed=1, Corr_OrbAccUsed=0, Corr_DO229Active=0, Corr_RTKType=0, Reserved2=0, ReferenceID=123, MeanCorrAge=278, SignalInfo=1344405761, RAIMIntegrity=1, GalHPCAFail=0, GalIonStorm=0, Reserved3=0, NrBases=1, PPPSeedAge=0, Reserved4=0, PPPSeedType=0, Latency=55, HAccuracy=102, VAccuracy=138, BaseARP=0, PhaseCtrOffset=0, Reserved5=8, ARPOffset=1)>",
            "<SBF(PosLocal, TOW=14:07:09, WNc=2367, Type=6, Reserved1=0, AutoSet=0, 2D=0, Error=17, Lat=-20000000000.0, Lon=-20000000000.0, Alt=-20000000000.0, Datum=255)>",
        )
        i = 0
        with open(os.path.join(DIRNAME, "pygpsdata_mixed.log"), "rb") as stream:
            sbr = SBFReader(
                stream,
                protfilter=SBF_PROTOCOL | RTCM3_PROTOCOL,
                quitonerror=ERR_RAISE,
            )
            for raw, parsed in sbr:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
            self.assertEqual(i, len(EXPECTED_RESULTS))

    def testmixedsbfrtcmraw(
        self,
    ):  # test mixed stream of NMEA, SBF & RTCM messages with protfilter = SBF + RTCM3 only
        EXPECTED_RESULTS = (
            b"\xd3\x00\x13>\xd0\x00\x03\x8aX\xd9I<\x87/4\x10\x9d\x07\xd6\xafH Z\xd7\xf7",
            b"\xd3\x00\x04L\xe0\x00\x80\xed\xed\xd6",
            b'$@q\x0e\xa7O`\x00\x18\xa9\xc7\x1c?\t\x06\x00\x91\x82|\x11\xfe\xca\xed?\r\xec\xd5\xa8\x9a\x13\xa4\xbf\x0f\x92\xc3m\xf3e`@O\xe9AB[\xb9\xa0:\x0b\x98P\xbas\x8e\xfc9\xf9\x02\x95\xd0\xd8!\xdbQ\xdc\xc3\xb1?\x7f\x8bJ>\x00\x00$\x07{\x00\x16\x01\x01\x01"P\x01\x01\x00\x007\x00f\x00\x8a\x00`\x00',
            b"$@b\x81\xd4\x0f,\x00\x18\xa9\xc7\x1c?\t\x06\x11\x00\x00\x00 _\xa0\x12\xc2\x00\x00\x00 _\xa0\x12\xc2\x00\x00\x00 _\xa0\x12\xc2\xff\x00\x00\x00",
        )
        i = 0
        with open(os.path.join(DIRNAME, "pygpsdata_mixed.log"), "rb") as stream:
            sbr = SBFReader(
                stream,
                protfilter=SBF_PROTOCOL | RTCM3_PROTOCOL,
                quitonerror=ERR_RAISE,
                parsing=False,
            )
            for raw, parsed in sbr:
                # print(f"{raw},")
                self.assertEqual(parsed, None)
                self.assertEqual(raw, EXPECTED_RESULTS[i])
                i += 1
            self.assertEqual(i, len(EXPECTED_RESULTS))

    def testmeasurements(self):  # test measurement message stream
        EXPECTED_RESULTS = (
            "<SBF(MeasEpoch, TOW=13:58:23, WNc=2367, N1=44, SB1Length=20, SB2Length=12, MultipathMitigation=1, CodeSmoothing=0, Reserved=1, ClockSteering=0, Reserved1=0, HighDynamics=0, E6BUsed=0, Scrambling=0, CumClkJumps=12, Reserved2=0, RxChannel_01=1, SigIdxLo_01=0, AntennaID_01=0, SVID_01=17, CodeMSB_01=5, Reserved3_01=0, CodeLSB_01=976531514, Doppler_01=20771658, CarrierLSB_01=2039, CarrierMSB_01=0, CN0_01=144, LockTime_01=513, PRSmoothed_01=0, Reserved4_01=0, HalfCycleAmbiguity_01=0, SigIdxHi_01=0, N2_01=2, SigIdxLo_01_01=2, AntennaID_01_01=0, LockTime_01_01=254, CN0_01_01=177, CodeOffsetMSB_01_01=7, DopplerOffsetMSB_01_01=0, CarrierMSB_01_01=0, PRSmoothed_01_01=0, Reserved6_01_01=0, HalfCycleAmbiguity_01_01=0, SigIdxHi_01_01=0, CodeOffsetLSB_01_01=63565, CarrierLSB_01_01=5503, DopplerOffsetLSB_01_01=4, SigIdxLo_01_02=3, AntennaID_01_02=0, LockTime_01_02=254, CN0_01_02=128, CodeOffsetMSB_01_02=7, DopplerOffsetMSB_01_02=31, CarrierMSB_01_02=0, PRSmoothed_01_02=0, Reserved6_01_02=0, HalfCycleAmbiguity_01_02=0, SigIdxHi_01_02=0, CodeOffsetLSB_01_02=63431, CarrierLSB_01_02=6060, DopplerOffsetLSB_01_02=64703, RxChannel_02=2, SigIdxLo_02=0, AntennaID_02=0, SVID_02=14, CodeMSB_02=5, Reserved3_02=0, CodeLSB_02=1524925917, Doppler_02=-18827264, CarrierLSB_02=62615, CarrierMSB_02=-1, CN0_02=123, LockTime_02=508, PRSmoothed_02=0, Reserved4_02=0, HalfCycleAmbiguity_02=0, SigIdxHi_02=0, N2_02=2, SigIdxLo_02_01=2, AntennaID_02_01=0, LockTime_02_01=254, CN0_02_01=117, CodeOffsetMSB_02_01=0, DopplerOffsetMSB_02_01=0, CarrierMSB_02_01=-1, PRSmoothed_02_01=0, Reserved6_02_01=0, HalfCycleAmbiguity_02_01=0, SigIdxHi_02_01=0, CodeOffsetLSB_02_01=789, CarrierLSB_02_01=61424, DopplerOffsetLSB_02_01=48, SigIdxLo_02_02=3, AntennaID_02_02=0, LockTime_02_02=254, CN0_02_02=131, CodeOffsetMSB_02_02=0, DopplerOffsetMSB_02_02=31, CarrierMSB_02_02=-1, PRSmoothed_02_02=0, Reserved6_02_02=0, HalfCycleAmbiguity_02_02=0, SigIdxHi_02_02=0, CodeOffsetLSB_02_02=648, CarrierLSB_02_02=52002, DopplerOffsetLSB_02_02=64736, RxChannel_03=3, SigIdxLo_03=20, AntennaID_03=0, SVID_03=97, CodeMSB_03=6, Reserved3_03=0, CodeLSB_03=2288695338, Doppler_03=11778827, CarrierLSB_03=61554, CarrierMSB_03=-1, CN0_03=119, LockTime_03=449, PRSmoothed_03=0, Reserved4_03=0, HalfCycleAmbiguity_03=0, SigIdxHi_03=0, N2_03=2, SigIdxLo_03_01=17, AntennaID_03_01=0, LockTime_03_01=254, CN0_03_01=108, CodeOffsetMSB_03_01=7, DopplerOffsetMSB_03_01=0, CarrierMSB_03_01=0, PRSmoothed_03_01=0, Reserved6_03_01=0, HalfCycleAmbiguity_03_01=0, SigIdxHi_03_01=0, CodeOffsetLSB_03_01=60033, CarrierLSB_03_01=5592, DopplerOffsetLSB_03_01=1216, SigIdxLo_03_02=21, AntennaID_03_02=0, LockTime_03_02=254, CN0_03_02=122, CodeOffsetMSB_03_02=7, DopplerOffsetMSB_03_02=31, CarrierMSB_03_02=0, PRSmoothed_03_02=0, Reserved6_03_02=0, HalfCycleAmbiguity_03_02=0, SigIdxHi_03_02=0, CodeOffsetLSB_03_02=63757, CarrierLSB_03_02=881, DopplerOffsetLSB_03_02=65301, RxChannel_04=4, SigIdxLo_04=21, AntennaID_04=0, SVID_04=80, CodeMSB_04=6, Reserved3_04=0, CodeLSB_04=2423207221, Doppler_04=-22449326, CarrierLSB_04=0, CarrierMSB_04=-128, CN0_04=43, LockTime_04=65535, PRSmoothed_04=0, Reserved4_04=0, HalfCycleAmbiguity_04=0, SigIdxHi_04=0, N2_04=0, RxChannel_05=5, SigIdxLo_05=8, AntennaID_05=0, SVID_05=48, CodeMSB_05=5, Reserved3_05=0, CodeLSB_05=1361802492, Doppler_05=4614561, CarrierLSB_05=5637, CarrierMSB_05=0, CN0_05=141, LockTime_05=509, PRSmoothed_05=0, Reserved4_05=0, HalfCycleAmbiguity_05=0, SigIdxHi_05=8, N2_05=1, SigIdxLo_05_01=11, AntennaID_05_01=0, LockTime_05_01=254, CN0_05_01=135, CodeOffsetMSB_05_01=0, DopplerOffsetMSB_05_01=31, CarrierMSB_05_01=0, PRSmoothed_05_01=0, Reserved6_05_01=0, HalfCycleAmbiguity_05_01=0, SigIdxHi_05_01=0, CodeOffsetLSB_05_01=4229, CarrierLSB_05_01=5253, DopplerOffsetLSB_05_01=64580, RxChannel_06=6, SigIdxLo_06=0, AntennaID_06=0, SVID_06=22, CodeMSB_06=5, Reserved3_06=0, CodeLSB_06=1956035146, Doppler_06=-7155988, CarrierLSB_06=65271, CarrierMSB_06=-1, CN0_06=134, LockTime_06=513, PRSmoothed_06=0, Reserved4_06=0, HalfCycleAmbiguity_06=0, SigIdxHi_06=0, N2_06=1, SigIdxLo_06_01=2, AntennaID_06_01=0, LockTime_06_01=254, CN0_06_01=116, CodeOffsetMSB_06_01=7, DopplerOffsetMSB_06_01=31, CarrierMSB_06_01=0, PRSmoothed_06_01=0, Reserved6_06_01=0, HalfCycleAmbiguity_06_01=0, SigIdxHi_06_01=0, CodeOffsetLSB_06_01=64004, CarrierLSB_06_01=1526, DopplerOffsetLSB_06_01=65493, RxChannel_07=7, SigIdxLo_07=8, AntennaID_07=0, SVID_07=42, CodeMSB_07=5, Reserved3_07=0, CodeLSB_07=1949387238, Doppler_07=43550547, CarrierLSB_07=27523, CarrierMSB_07=0, CN0_07=128, LockTime_07=509, PRSmoothed_07=0, Reserved4_07=0, HalfCycleAmbiguity_07=0, SigIdxHi_07=9, N2_07=1, SigIdxLo_07_01=11, AntennaID_07_01=0, LockTime_07_01=254, CN0_07_01=112, CodeOffsetMSB_07_01=0, DopplerOffsetMSB_07_01=31, CarrierMSB_07_01=0, PRSmoothed_07_01=0, Reserved6_07_01=0, HalfCycleAmbiguity_07_01=0, SigIdxHi_07_01=0, CodeOffsetLSB_07_01=12491, CarrierLSB_07_01=34035, DopplerOffsetLSB_07_01=64574, RxChannel_08=8, SigIdxLo_08=17, AntennaID_08=0, SVID_08=72, CodeMSB_08=6, Reserved3_08=0, CodeLSB_08=795891476, Doppler_08=-30585217, CarrierLSB_08=54019, CarrierMSB_08=-1, CN0_08=126, LockTime_08=504, PRSmoothed_08=0, Reserved4_08=0, HalfCycleAmbiguity_08=0, SigIdxHi_08=0, N2_08=2, SigIdxLo_08_01=20, AntennaID_08_01=0, LockTime_08_01=254, CN0_08_01=133, CodeOffsetMSB_08_01=0, DopplerOffsetMSB_08_01=0, CarrierMSB_08_01=-1, PRSmoothed_08_01=0, Reserved6_08_01=0, HalfCycleAmbiguity_08_01=0, SigIdxHi_08_01=0, CodeOffsetLSB_08_01=5608, CarrierLSB_08_01=57243, DopplerOffsetLSB_08_01=596, SigIdxLo_08_02=21, AntennaID_08_02=0, LockTime_08_02=254, CN0_08_02=135, CodeOffsetMSB_08_02=0, DopplerOffsetMSB_08_02=0, CarrierMSB_08_02=-1, PRSmoothed_08_02=0, Reserved6_08_02=0, HalfCycleAmbiguity_08_02=0, SigIdxHi_08_02=0, CodeOffsetLSB_08_02=4694, CarrierLSB_08_02=56654, DopplerOffsetLSB_08_02=648, RxChannel_09=9, SigIdxLo_09=20, AntennaID_09=0, SVID_09=75, CodeMSB_09=6, Reserved3_09=0, CodeLSB_09=1162161216, Doppler_09=22267000, CarrierLSB_09=6995, CarrierMSB_09=0, CN0_09=88, LockTime_09=238, PRSmoothed_09=0, Reserved4_09=0, HalfCycleAmbiguity_09=0, SigIdxHi_09=0, N2_09=2, SigIdxLo_09_01=17, AntennaID_09_01=0, LockTime_09_01=86, CN0_09_01=82, CodeOffsetMSB_09_01=7, DopplerOffsetMSB_09_01=0, CarrierMSB_09_01=0, PRSmoothed_09_01=0, Reserved6_09_01=0, HalfCycleAmbiguity_09_01=0, SigIdxHi_09_01=0, CodeOffsetLSB_09_01=51331, CarrierLSB_09_01=13812, DopplerOffsetLSB_09_01=1233, SigIdxLo_09_02=21, AntennaID_09_02=0, LockTime_09_02=201, CN0_09_02=94, CodeOffsetMSB_09_02=7, DopplerOffsetMSB_09_02=31, CarrierMSB_09_02=0, PRSmoothed_09_02=0, Reserved6_09_02=0, HalfCycleAmbiguity_09_02=0, SigIdxHi_09_02=0, CodeOffsetLSB_09_02=63742, CarrierLSB_09_02=5860, DopplerOffsetLSB_09_02=64151, RxChannel_10=10, SigIdxLo_10=8, AntennaID_10=0, SVID_10=47, CodeMSB_10=5, Reserved3_10=0, CodeLSB_10=1545017690, Doppler_10=-29059602, CarrierLSB_10=63833, CarrierMSB_10=-1, CN0_10=116, LockTime_10=509, PRSmoothed_10=0, Reserved4_10=0, HalfCycleAmbiguity_10=0, SigIdxHi_10=1, N2_10=0, RxChannel_11=11, SigIdxLo_11=28, AntennaID_11=0, SVID_11=173, CodeMSB_11=5, Reserved3_11=0, CodeLSB_11=2918059934, Doppler_11=29240168, CarrierLSB_11=4058, CarrierMSB_11=0, CN0_11=143, LockTime_11=509, PRSmoothed_11=0, Reserved4_11=0, HalfCycleAmbiguity_11=0, SigIdxHi_11=0, N2_11=1, SigIdxLo_11_01=30, AntennaID_11_01=0, LockTime_11_01=254, CN0_11_01=139, CodeOffsetMSB_11_01=0, DopplerOffsetMSB_11_01=0, CarrierMSB_11_01=0, PRSmoothed_11_01=0, Reserved6_11_01=0, HalfCycleAmbiguity_11_01=0, SigIdxHi_11_01=0, CodeOffsetLSB_11_01=15193, CarrierLSB_11_01=14124, DopplerOffsetLSB_11_01=40, RxChannel_12=12, SigIdxLo_12=21, AntennaID_12=0, SVID_12=104, CodeMSB_12=5, Reserved3_12=0, CodeLSB_12=3930828179, Doppler_12=14448691, CarrierLSB_12=2678, CarrierMSB_12=0, CN0_12=153, LockTime_12=509, PRSmoothed_12=0, Reserved4_12=0, HalfCycleAmbiguity_12=0, SigIdxHi_12=0, N2_12=2, SigIdxLo_12_01=17, AntennaID_12_01=0, LockTime_12_01=254, CN0_12_01=136, CodeOffsetMSB_12_01=7, DopplerOffsetMSB_12_01=31, CarrierMSB_12_01=0, PRSmoothed_12_01=0, Reserved6_12_01=0, HalfCycleAmbiguity_12_01=0, SigIdxHi_12_01=0, CodeOffsetLSB_12_01=62028, CarrierLSB_12_01=2497, DopplerOffsetLSB_12_01=64471, SigIdxLo_12_02=20, AntennaID_12_02=0, LockTime_12_02=254, CN0_12_02=149, CodeOffsetMSB_12_02=0, DopplerOffsetMSB_12_02=0, CarrierMSB_12_02=0, PRSmoothed_12_02=0, Reserved6_12_02=0, HalfCycleAmbiguity_12_02=0, SigIdxHi_12_02=0, CodeOffsetLSB_12_02=319, CarrierLSB_12_02=3653, DopplerOffsetLSB_12_02=101, RxChannel_13=13, SigIdxLo_13=8, AntennaID_13=0, SVID_13=56, CodeMSB_13=4, Reserved3_13=0, CodeLSB_13=2000457461, Doppler_13=-1655104, CarrierLSB_13=5039, CarrierMSB_13=0, CN0_13=153, LockTime_13=509, PRSmoothed_13=0, Reserved4_13=0, HalfCycleAmbiguity_13=0, SigIdxHi_13=11, N2_13=1, SigIdxLo_13_01=11, AntennaID_13_01=0, LockTime_13_01=254, CN0_13_01=156, CodeOffsetMSB_13_01=0, DopplerOffsetMSB_13_01=31, CarrierMSB_13_01=0, PRSmoothed_13_01=0, Reserved6_13_01=0, HalfCycleAmbiguity_13_01=0, SigIdxHi_13_01=0, CodeOffsetLSB_13_01=2445, CarrierLSB_13_01=370, DopplerOffsetLSB_13_01=65496, RxChannel_14=14, SigIdxLo_14=21, AntennaID_14=0, SVID_14=106, CodeMSB_14=5, Reserved3_14=0, CodeLSB_14=1891954691, Doppler_14=-2810916, CarrierLSB_14=65478, CarrierMSB_14=-1, CN0_14=163, LockTime_14=509, PRSmoothed_14=0, Reserved4_14=0, HalfCycleAmbiguity_14=0, SigIdxHi_14=0, N2_14=2, SigIdxLo_14_01=17, AntennaID_14_01=0, LockTime_14_01=254, CN0_14_01=147, CodeOffsetMSB_14_01=7, DopplerOffsetMSB_14_01=31, CarrierMSB_14_01=-1, PRSmoothed_14_01=0, Reserved6_14_01=0, HalfCycleAmbiguity_14_01=0, SigIdxHi_14_01=0, CodeOffsetLSB_14_01=63130, CarrierLSB_14_01=64518, DopplerOffsetLSB_14_01=64389, SigIdxLo_14_02=20, AntennaID_14_02=0, LockTime_14_02=254, CN0_14_02=158, CodeOffsetMSB_14_02=0, DopplerOffsetMSB_14_02=31, CarrierMSB_14_02=-1, PRSmoothed_14_02=0, Reserved6_14_02=0, HalfCycleAmbiguity_14_02=0, SigIdxHi_14_02=0, CodeOffsetLSB_14_02=113, CarrierLSB_14_02=65254, DopplerOffsetLSB_14_02=64764, RxChannel_15=15, SigIdxLo_15=28, AntennaID_15=0, SVID_15=174, CodeMSB_15=5, Reserved3_15=0, CodeLSB_15=3413429397, Doppler_15=-34157146, CarrierLSB_15=52449, CarrierMSB_15=-1, CN0_15=128, LockTime_15=509, PRSmoothed_15=0, Reserved4_15=0, HalfCycleAmbiguity_15=0, SigIdxHi_15=0, N2_15=1, SigIdxLo_15_01=30, AntennaID_15_01=0, LockTime_15_01=254, CN0_15_01=139, CodeOffsetMSB_15_01=0, DopplerOffsetMSB_15_01=31, CarrierMSB_15_01=-1, PRSmoothed_15_01=0, Reserved6_15_01=0, HalfCycleAmbiguity_15_01=0, SigIdxHi_15_01=0, CodeOffsetLSB_15_01=5590, CarrierLSB_15_01=54400, DopplerOffsetLSB_15_01=65482, RxChannel_16=16, SigIdxLo_16=0, AntennaID_16=0, SVID_16=32, CodeMSB_16=5, Reserved3_16=0, CodeLSB_16=1577058413, Doppler_16=-16276393, CarrierLSB_16=1867, CarrierMSB_16=0, CN0_16=140, LockTime_16=512, PRSmoothed_16=0, Reserved4_16=0, HalfCycleAmbiguity_16=0, SigIdxHi_16=0, N2_16=2, SigIdxLo_16_01=2, AntennaID_16_01=0, LockTime_16_01=254, CN0_16_01=174, CodeOffsetMSB_16_01=0, DopplerOffsetMSB_16_01=0, CarrierMSB_16_01=0, PRSmoothed_16_01=0, Reserved6_16_01=0, HalfCycleAmbiguity_16_01=0, SigIdxHi_16_01=0, CodeOffsetLSB_16_01=1522, CarrierLSB_16_01=851, DopplerOffsetLSB_16_01=0, SigIdxLo_16_02=3, AntennaID_16_02=0, LockTime_16_02=254, CN0_16_02=131, CodeOffsetMSB_16_02=0, DopplerOffsetMSB_16_02=31, CarrierMSB_16_02=-1, PRSmoothed_16_02=0, Reserved6_16_02=0, HalfCycleAmbiguity_16_02=0, SigIdxHi_16_02=0, CodeOffsetLSB_16_02=2093, CarrierLSB_16_02=60064, DopplerOffsetLSB_16_02=64624, RxChannel_17=17, SigIdxLo_17=17, AntennaID_17=0, SVID_17=79, CodeMSB_17=5, Reserved3_17=0, CodeLSB_17=4275925425, Doppler_17=20425073, CarrierLSB_17=3482, CarrierMSB_17=0, CN0_17=140, LockTime_17=512, PRSmoothed_17=0, Reserved4_17=0, HalfCycleAmbiguity_17=0, SigIdxHi_17=0, N2_17=2, SigIdxLo_17_01=20, AntennaID_17_01=0, LockTime_17_01=254, CN0_17_01=137, CodeOffsetMSB_17_01=0, DopplerOffsetMSB_17_01=31, CarrierMSB_17_01=0, PRSmoothed_17_01=0, Reserved6_17_01=0, HalfCycleAmbiguity_17_01=0, SigIdxHi_17_01=0, CodeOffsetLSB_17_01=4299, CarrierLSB_17_01=4704, DopplerOffsetLSB_17_01=65499, SigIdxLo_17_02=21, AntennaID_17_02=0, LockTime_17_02=254, CN0_17_02=147, CodeOffsetMSB_17_02=0, DopplerOffsetMSB_17_02=31, CarrierMSB_17_02=0, PRSmoothed_17_02=0, Reserved6_17_02=0, HalfCycleAmbiguity_17_02=0, SigIdxHi_17_02=0, CodeOffsetLSB_17_02=3570, CarrierLSB_17_02=6378, DopplerOffsetLSB_17_02=65168, RxChannel_18=18, SigIdxLo_18=24, AntennaID_18=0, SVID_18=123, CodeMSB_18=9, Reserved3_18=0, CodeLSB_18=427010248, Doppler_18=-2805431, CarrierLSB_18=423, CarrierMSB_18=0, CN0_18=129, LockTime_18=510, PRSmoothed_18=0, Reserved4_18=0, HalfCycleAmbiguity_18=0, SigIdxHi_18=0, N2_18=0, RxChannel_19=19, SigIdxLo_19=17, AntennaID_19=0, SVID_19=81, CodeMSB_19=6, Reserved3_19=0, CodeLSB_19=116621685, Doppler_19=-24798572, CarrierLSB_19=62401, CarrierMSB_19=-1, CN0_19=121, LockTime_19=512, PRSmoothed_19=0, Reserved4_19=0, HalfCycleAmbiguity_19=0, SigIdxHi_19=0, N2_19=2, SigIdxLo_19_01=20, AntennaID_19_01=0, LockTime_19_01=254, CN0_19_01=124, CodeOffsetMSB_19_01=0, DopplerOffsetMSB_19_01=0, CarrierMSB_19_01=-1, PRSmoothed_19_01=0, Reserved6_19_01=0, HalfCycleAmbiguity_19_01=0, SigIdxHi_19_01=0, CodeOffsetLSB_19_01=140, CarrierLSB_19_01=60582, DopplerOffsetLSB_19_01=765, SigIdxLo_19_02=21, AntennaID_19_02=0, LockTime_19_02=254, CN0_19_02=127, CodeOffsetMSB_19_02=7, DopplerOffsetMSB_19_02=0, CarrierMSB_19_02=-1, PRSmoothed_19_02=0, Reserved6_19_02=0, HalfCycleAmbiguity_19_02=0, SigIdxHi_19_02=0, CodeOffsetLSB_19_02=65267, CarrierLSB_19_02=61976, DopplerOffsetLSB_19_02=1458, RxChannel_20=20, SigIdxLo_20=8, AntennaID_20=0, SVID_20=40, CodeMSB_20=4, Reserved3_20=0, CodeLSB_20=2952387166, Doppler_20=-19070425, CarrierLSB_20=159, CarrierMSB_20=0, CN0_20=160, LockTime_20=509, PRSmoothed_20=0, Reserved4_20=0, HalfCycleAmbiguity_20=0, SigIdxHi_20=13, N2_20=1, SigIdxLo_20_01=11, AntennaID_20_01=0, LockTime_20_01=254, CN0_20_01=150, CodeOffsetMSB_20_01=0, DopplerOffsetMSB_20_01=0, CarrierMSB_20_01=-1, PRSmoothed_20_01=0, Reserved6_20_01=0, HalfCycleAmbiguity_20_01=0, SigIdxHi_20_01=0, CodeOffsetLSB_20_01=3252, CarrierLSB_20_01=63786, DopplerOffsetLSB_20_01=175, RxChannel_21=21, SigIdxLo_21=28, AntennaID_21=0, SVID_21=151, CodeMSB_21=5, Reserved3_21=0, CodeLSB_21=1739416015, Doppler_21=-26095005, CarrierLSB_21=63871, CarrierMSB_21=-1, CN0_21=147, LockTime_21=509, PRSmoothed_21=0, Reserved4_21=0, HalfCycleAmbiguity_21=0, SigIdxHi_21=0, N2_21=2, SigIdxLo_21_01=29, AntennaID_21_01=0, LockTime_21_01=254, CN0_21_01=158, CodeOffsetMSB_21_01=0, DopplerOffsetMSB_21_01=0, CarrierMSB_21_01=-1, PRSmoothed_21_01=0, Reserved6_21_01=0, HalfCycleAmbiguity_21_01=0, SigIdxHi_21_01=0, CodeOffsetLSB_21_01=921, CarrierLSB_21_01=59128, DopplerOffsetLSB_21_01=432, SigIdxLo_21_02=30, AntennaID_21_02=0, LockTime_21_02=254, CN0_21_02=155, CodeOffsetMSB_21_02=7, DopplerOffsetMSB_21_02=0, CarrierMSB_21_02=-1, PRSmoothed_21_02=0, Reserved6_21_02=0, HalfCycleAmbiguity_21_02=0, SigIdxHi_21_02=0, CodeOffsetLSB_21_02=65175, CarrierLSB_21_02=62012, DopplerOffsetLSB_21_02=201, RxChannel_22=23, SigIdxLo_22=21, AntennaID_22=0, SVID_22=100, CodeMSB_22=6, Reserved3_22=0, CodeLSB_22=625727838, Doppler_22=-6472268, CarrierLSB_22=65507, CarrierMSB_22=-1, CN0_22=145, LockTime_22=509, PRSmoothed_22=0, Reserved4_22=0, HalfCycleAmbiguity_22=0, SigIdxHi_22=0, N2_22=2, SigIdxLo_22_01=17, AntennaID_22_01=0, LockTime_22_01=254, CN0_22_01=134, CodeOffsetMSB_22_01=7, DopplerOffsetMSB_22_01=31, CarrierMSB_22_01=-1, PRSmoothed_22_01=0, Reserved6_22_01=0, HalfCycleAmbiguity_22_01=0, SigIdxHi_22_01=0, CodeOffsetLSB_22_01=62895, CarrierLSB_22_01=62833, DopplerOffsetLSB_22_01=65151, SigIdxLo_22_02=20, AntennaID_22_02=0, LockTime_22_02=254, CN0_22_02=143, CodeOffsetMSB_22_02=0, DopplerOffsetMSB_22_02=31, CarrierMSB_22_02=0, PRSmoothed_22_02=0, Reserved6_22_02=0, HalfCycleAmbiguity_22_02=0, SigIdxHi_22_02=0, CodeOffsetLSB_22_02=678, CarrierLSB_22_02=491, DopplerOffsetLSB_22_02=65159, RxChannel_23=24, SigIdxLo_23=8, AntennaID_23=0, SVID_23=41, CodeMSB_23=4, Reserved3_23=0, CodeLSB_23=2794514306, Doppler_23=22554895, CarrierLSB_23=4553, CarrierMSB_23=0, CN0_23=154, LockTime_23=509, PRSmoothed_23=0, Reserved4_23=0, HalfCycleAmbiguity_23=0, SigIdxHi_23=14, N2_23=1, SigIdxLo_23_01=11, AntennaID_23_01=0, LockTime_23_01=254, CN0_23_01=148, CodeOffsetMSB_23_01=0, DopplerOffsetMSB_23_01=0, CarrierMSB_23_01=0, PRSmoothed_23_01=0, Reserved6_23_01=0, HalfCycleAmbiguity_23_01=0, SigIdxHi_23_01=0, CodeOffsetLSB_23_01=4236, CarrierLSB_23_01=3532, DopplerOffsetLSB_23_01=89, RxChannel_24=25, SigIdxLo_24=28, AntennaID_24=0, SVID_24=224, CodeMSB_24=5, Reserved3_24=0, CodeLSB_24=750350471, Doppler_24=898594, CarrierLSB_24=2879, CarrierMSB_24=0, CN0_24=161, LockTime_24=494, PRSmoothed_24=0, Reserved4_24=0, HalfCycleAmbiguity_24=0, SigIdxHi_24=0, N2_24=1, SigIdxLo_24_01=30, AntennaID_24_01=0, LockTime_24_01=254, CN0_24_01=167, CodeOffsetMSB_24_01=0, DopplerOffsetMSB_24_01=31, CarrierMSB_24_01=0, PRSmoothed_24_01=0, Reserved6_24_01=0, HalfCycleAmbiguity_24_01=0, SigIdxHi_24_01=0, CodeOffsetLSB_24_01=10410, CarrierLSB_24_01=1445, DopplerOffsetLSB_24_01=65332, RxChannel_25=26, SigIdxLo_25=28, AntennaID_25=0, SVID_25=225, CodeMSB_25=5, Reserved3_25=0, CodeLSB_25=344717727, Doppler_25=-14633867, CarrierLSB_25=63152, CarrierMSB_25=-1, CN0_25=160, LockTime_25=494, PRSmoothed_25=0, Reserved4_25=0, HalfCycleAmbiguity_25=0, SigIdxHi_25=0, N2_25=1, SigIdxLo_25_01=30, AntennaID_25_01=0, LockTime_25_01=254, CN0_25_01=159, CodeOffsetMSB_25_01=0, DopplerOffsetMSB_25_01=0, CarrierMSB_25_01=-1, PRSmoothed_25_01=0, Reserved6_25_01=0, HalfCycleAmbiguity_25_01=0, SigIdxHi_25_01=0, CodeOffsetLSB_25_01=1884, CarrierLSB_25_01=64340, DopplerOffsetLSB_25_01=363, RxChannel_26=28, SigIdxLo_26=28, AntennaID_26=0, SVID_26=168, CodeMSB_26=5, Reserved3_26=0, CodeLSB_26=1622429935, Doppler_26=20351702, CarrierLSB_26=1940, CarrierMSB_26=0, CN0_26=154, LockTime_26=509, PRSmoothed_26=0, Reserved4_26=0, HalfCycleAmbiguity_26=0, SigIdxHi_26=0, N2_26=1, SigIdxLo_26_01=30, AntennaID_26_01=0, LockTime_26_01=254, CN0_26_01=154, CodeOffsetMSB_26_01=0, DopplerOffsetMSB_26_01=0, CarrierMSB_26_01=0, PRSmoothed_26_01=0, Reserved6_26_01=0, HalfCycleAmbiguity_26_01=0, SigIdxHi_26_01=0, CodeOffsetLSB_26_01=1964, CarrierLSB_26_01=1446, DopplerOffsetLSB_26_01=349, RxChannel_27=29, SigIdxLo_27=24, AntennaID_27=0, SVID_27=121, CodeMSB_27=9, Reserved3_27=0, CodeLSB_27=9186875, Doppler_27=-2781512, CarrierLSB_27=44098, CarrierMSB_27=-1, CN0_27=127, LockTime_27=510, PRSmoothed_27=0, Reserved4_27=0, HalfCycleAmbiguity_27=0, SigIdxHi_27=0, N2_27=0, RxChannel_28=30, SigIdxLo_28=24, AntennaID_28=0, SVID_28=136, CodeMSB_28=9, Reserved3_28=0, CodeLSB_28=46071492, Doppler_28=-2795735, CarrierLSB_28=48941, CarrierMSB_28=-1, CN0_28=129, LockTime_28=508, PRSmoothed_28=0, Reserved4_28=0, HalfCycleAmbiguity_28=0, SigIdxHi_28=0, N2_28=0, RxChannel_29=31, SigIdxLo_29=0, AntennaID_29=0, SVID_29=8, CodeMSB_29=5, Reserved3_29=0, CodeLSB_29=1805213230, Doppler_29=-38914431, CarrierLSB_29=54173, CarrierMSB_29=-1, CN0_29=124, LockTime_29=508, PRSmoothed_29=0, Reserved4_29=0, HalfCycleAmbiguity_29=0, SigIdxHi_29=0, N2_29=2, SigIdxLo_29_01=2, AntennaID_29_01=0, LockTime_29_01=254, CN0_29_01=152, CodeOffsetMSB_29_01=0, DopplerOffsetMSB_29_01=0, CarrierMSB_29_01=-1, PRSmoothed_29_01=0, Reserved6_29_01=0, HalfCycleAmbiguity_29_01=0, SigIdxHi_29_01=0, CodeOffsetLSB_29_01=3938, CarrierLSB_29_01=53326, DopplerOffsetLSB_29_01=44, SigIdxLo_29_02=3, AntennaID_29_02=0, LockTime_29_02=254, CN0_29_02=121, CodeOffsetMSB_29_02=0, DopplerOffsetMSB_29_02=31, CarrierMSB_29_02=-1, PRSmoothed_29_02=0, Reserved6_29_02=0, HalfCycleAmbiguity_29_02=0, SigIdxHi_29_02=0, CodeOffsetLSB_29_02=4466, CarrierLSB_29_02=53174, DopplerOffsetLSB_29_02=64674, RxChannel_30=34, SigIdxLo_30=28, AntennaID_30=0, SVID_30=179, CodeMSB_30=9, Reserved3_30=0, CodeLSB_30=1797091129, Doppler_30=11219513, CarrierLSB_30=7313, CarrierMSB_30=0, CN0_30=112, LockTime_30=490, PRSmoothed_30=0, Reserved4_30=0, HalfCycleAmbiguity_30=0, SigIdxHi_30=0, N2_30=1, SigIdxLo_30_01=30, AntennaID_30_01=0, LockTime_30_01=254, CN0_30_01=125, CodeOffsetMSB_30_01=0, DopplerOffsetMSB_30_01=0, CarrierMSB_30_01=0, PRSmoothed_30_01=0, Reserved6_30_01=0, HalfCycleAmbiguity_30_01=0, SigIdxHi_30_01=0, CodeOffsetLSB_30_01=1654, CarrierLSB_30_01=4131, DopplerOffsetLSB_30_01=617, RxChannel_31=35, SigIdxLo_31=8, AntennaID_31=0, SVID_31=55, CodeMSB_31=4, Reserved3_31=0, CodeLSB_31=4208607095, Doppler_31=-34261814, CarrierLSB_31=59383, CarrierMSB_31=-1, CN0_31=151, LockTime_31=509, PRSmoothed_31=0, Reserved4_31=0, HalfCycleAmbiguity_31=0, SigIdxHi_31=5, N2_31=1, SigIdxLo_31_01=11, AntennaID_31_01=0, LockTime_31_01=254, CN0_31_01=148, CodeOffsetMSB_31_01=0, DopplerOffsetMSB_31_01=0, CarrierMSB_31_01=-1, PRSmoothed_31_01=0, Reserved6_31_01=0, HalfCycleAmbiguity_31_01=0, SigIdxHi_31_01=0, CodeOffsetLSB_31_01=5411, CarrierLSB_31_01=55280, DopplerOffsetLSB_31_01=10, RxChannel_32=36, SigIdxLo_32=17, AntennaID_32=0, SVID_32=74, CodeMSB_32=5, Reserved3_32=0, CodeLSB_32=4105010252, Doppler_32=-5617780, CarrierLSB_32=895, CarrierMSB_32=0, CN0_32=128, LockTime_32=507, PRSmoothed_32=0, Reserved4_32=0, HalfCycleAmbiguity_32=0, SigIdxHi_32=0, N2_32=2, SigIdxLo_32_01=20, AntennaID_32_01=0, LockTime_32_01=254, CN0_32_01=144, CodeOffsetMSB_32_01=0, DopplerOffsetMSB_32_01=31, CarrierMSB_32_01=-1, PRSmoothed_32_01=0, Reserved6_32_01=0, HalfCycleAmbiguity_32_01=0, SigIdxHi_32_01=0, CodeOffsetLSB_32_01=2245, CarrierLSB_32_01=64354, DopplerOffsetLSB_32_01=65533, SigIdxLo_32_02=21, AntennaID_32_02=0, LockTime_32_02=254, CN0_32_02=149, CodeOffsetMSB_32_02=0, DopplerOffsetMSB_32_02=0, CarrierMSB_32_02=0, PRSmoothed_32_02=0, Reserved6_32_02=0, HalfCycleAmbiguity_32_02=0, SigIdxHi_32_02=0, CodeOffsetLSB_32_02=1560, CarrierLSB_32_02=1321, DopplerOffsetLSB_32_02=850, RxChannel_33=37, SigIdxLo_33=28, AntennaID_33=0, SVID_33=154, CodeMSB_33=5, Reserved3_33=0, CodeLSB_33=1167521539, Doppler_33=18168234, CarrierLSB_33=2956, CarrierMSB_33=0, CN0_33=146, LockTime_33=509, PRSmoothed_33=0, Reserved4_33=0, HalfCycleAmbiguity_33=0, SigIdxHi_33=0, N2_33=2, SigIdxLo_33_01=29, AntennaID_33_01=0, LockTime_33_01=254, CN0_33_01=157, CodeOffsetMSB_33_01=0, DopplerOffsetMSB_33_01=0, CarrierMSB_33_01=0, PRSmoothed_33_01=0, Reserved6_33_01=0, HalfCycleAmbiguity_33_01=0, SigIdxHi_33_01=0, CodeOffsetLSB_33_01=100, CarrierLSB_33_01=2679, DopplerOffsetLSB_33_01=277, SigIdxLo_33_02=30, AntennaID_33_02=0, LockTime_33_02=254, CN0_33_02=157, CodeOffsetMSB_33_02=7, DopplerOffsetMSB_33_02=31, CarrierMSB_33_02=0, PRSmoothed_33_02=0, Reserved6_33_02=0, HalfCycleAmbiguity_33_02=0, SigIdxHi_33_02=0, CodeOffsetLSB_33_02=64049, CarrierLSB_33_02=4553, DopplerOffsetLSB_33_02=65091, RxChannel_34=38, SigIdxLo_34=8, AntennaID_34=0, SVID_34=57, CodeMSB_34=5, Reserved3_34=0, CodeLSB_34=228467676, Doppler_34=31112725, CarrierLSB_34=12576, CarrierMSB_34=0, CN0_34=105, LockTime_34=505, PRSmoothed_34=0, Reserved4_34=0, HalfCycleAmbiguity_34=0, SigIdxHi_34=10, N2_34=1, SigIdxLo_34_01=11, AntennaID_34_01=0, LockTime_34_01=254, CN0_34_01=145, CodeOffsetMSB_34_01=0, DopplerOffsetMSB_34_01=31, CarrierMSB_34_01=0, PRSmoothed_34_01=0, Reserved6_34_01=0, HalfCycleAmbiguity_34_01=0, SigIdxHi_34_01=0, CodeOffsetLSB_34_01=7125, CarrierLSB_34_01=5944, DopplerOffsetLSB_34_01=63947, RxChannel_35=39, SigIdxLo_35=24, AntennaID_35=0, SVID_35=205, CodeMSB_35=1, Reserved3_35=0, CodeLSB_35=3875060563, Doppler_35=-2811301, CarrierLSB_35=17570, CarrierMSB_35=11, CN0_35=111, LockTime_35=502, PRSmoothed_35=0, Reserved4_35=0, HalfCycleAmbiguity_35=0, SigIdxHi_35=0, N2_35=0, RxChannel_36=40, SigIdxLo_36=28, AntennaID_36=0, SVID_36=161, CodeMSB_36=5, Reserved3_36=0, CodeLSB_36=2909746097, Doppler_36=-27868662, CarrierLSB_36=63560, CarrierMSB_36=-1, CN0_36=142, LockTime_36=509, PRSmoothed_36=0, Reserved4_36=0, HalfCycleAmbiguity_36=0, SigIdxHi_36=0, N2_36=1, SigIdxLo_36_01=30, AntennaID_36_01=0, LockTime_36_01=254, CN0_36_01=151, CodeOffsetMSB_36_01=7, DopplerOffsetMSB_36_01=0, CarrierMSB_36_01=-1, PRSmoothed_36_01=0, Reserved6_36_01=0, HalfCycleAmbiguity_36_01=0, SigIdxHi_36_01=0, CodeOffsetLSB_36_01=63777, CarrierLSB_36_01=61471, DopplerOffsetLSB_36_01=80, RxChannel_37=41, SigIdxLo_37=0, AntennaID_37=0, SVID_37=19, CodeMSB_37=5, Reserved3_37=0, CodeLSB_37=3016572300, Doppler_37=34812312, CarrierLSB_37=1840, CarrierMSB_37=0, CN0_37=119, LockTime_37=283, PRSmoothed_37=0, Reserved4_37=0, HalfCycleAmbiguity_37=0, SigIdxHi_37=0, N2_37=1, SigIdxLo_37_01=2, AntennaID_37_01=0, LockTime_37_01=254, CN0_37_01=99, CodeOffsetMSB_37_01=0, DopplerOffsetMSB_37_01=0, CarrierMSB_37_01=-1, PRSmoothed_37_01=0, Reserved6_37_01=0, HalfCycleAmbiguity_37_01=0, SigIdxHi_37_01=0, CodeOffsetLSB_37_01=267, CarrierLSB_37_01=62073, DopplerOffsetLSB_37_01=100, RxChannel_38=42, SigIdxLo_38=17, AntennaID_38=0, SVID_38=76, CodeMSB_38=5, Reserved3_38=0, CodeLSB_38=3676664041, Doppler_38=8727350, CarrierLSB_38=2657, CarrierMSB_38=0, CN0_38=135, LockTime_38=508, PRSmoothed_38=0, Reserved4_38=0, HalfCycleAmbiguity_38=0, SigIdxHi_38=0, N2_38=2, SigIdxLo_38_01=20, AntennaID_38_01=0, LockTime_38_01=254, CN0_38_01=142, CodeOffsetMSB_38_01=0, DopplerOffsetMSB_38_01=0, CarrierMSB_38_01=0, PRSmoothed_38_01=0, Reserved6_38_01=0, HalfCycleAmbiguity_38_01=0, SigIdxHi_38_01=0, CodeOffsetLSB_38_01=3897, CarrierLSB_38_01=4928, DopplerOffsetLSB_38_01=273, SigIdxLo_38_02=21, AntennaID_38_02=0, LockTime_38_02=254, CN0_38_02=150, CodeOffsetMSB_38_02=0, DopplerOffsetMSB_38_02=31, CarrierMSB_38_02=0, PRSmoothed_38_02=0, Reserved6_38_02=0, HalfCycleAmbiguity_38_02=0, SigIdxHi_38_02=0, CodeOffsetLSB_38_02=3372, CarrierLSB_38_02=2779, DopplerOffsetLSB_38_02=65085, RxChannel_39=47, SigIdxLo_39=0, AntennaID_39=0, SVID_39=1, CodeMSB_39=4, Reserved3_39=0, CodeLSB_39=2944577227, Doppler_39=137795, CarrierLSB_39=763, CarrierMSB_39=0, CN0_39=155, LockTime_39=509, PRSmoothed_39=0, Reserved4_39=0, HalfCycleAmbiguity_39=0, SigIdxHi_39=0, N2_39=2, SigIdxLo_39_01=2, AntennaID_39_01=0, LockTime_39_01=254, CN0_39_01=171, CodeOffsetMSB_39_01=7, DopplerOffsetMSB_39_01=31, CarrierMSB_39_01=0, PRSmoothed_39_01=0, Reserved6_39_01=0, HalfCycleAmbiguity_39_01=0, SigIdxHi_39_01=0, CodeOffsetLSB_39_01=63616, CarrierLSB_39_01=814, DopplerOffsetLSB_39_01=65527, SigIdxLo_39_02=3, AntennaID_39_02=0, LockTime_39_02=254, CN0_39_02=158, CodeOffsetMSB_39_02=7, DopplerOffsetMSB_39_02=0, CarrierMSB_39_02=0, PRSmoothed_39_02=0, Reserved6_39_02=0, HalfCycleAmbiguity_39_02=0, SigIdxHi_39_02=0, CodeOffsetLSB_39_02=63545, CarrierLSB_39_02=118, DopplerOffsetLSB_39_02=126, RxChannel_40=48, SigIdxLo_40=0, AntennaID_40=0, SVID_40=2, CodeMSB_40=4, Reserved3_40=0, CodeLSB_40=4038225577, Doppler_40=-17911301, CarrierLSB_40=63809, CarrierMSB_40=-1, CN0_40=153, LockTime_40=509, PRSmoothed_40=0, Reserved4_40=0, HalfCycleAmbiguity_40=0, SigIdxHi_40=0, N2_40=1, SigIdxLo_40_01=2, AntennaID_40_01=0, LockTime_40_01=254, CN0_40_01=161, CodeOffsetMSB_40_01=7, DopplerOffsetMSB_40_01=0, CarrierMSB_40_01=-1, PRSmoothed_40_01=0, Reserved6_40_01=0, HalfCycleAmbiguity_40_01=0, SigIdxHi_40_01=0, CodeOffsetLSB_40_01=61397, CarrierLSB_40_01=63685, DopplerOffsetLSB_40_01=7, RxChannel_41=49, SigIdxLo_41=0, AntennaID_41=0, SVID_41=3, CodeMSB_41=4, Reserved3_41=0, CodeLSB_41=3528210460, Doppler_41=19744695, CarrierLSB_41=3857, CarrierMSB_41=0, CN0_41=150, LockTime_41=509, PRSmoothed_41=0, Reserved4_41=0, HalfCycleAmbiguity_41=0, SigIdxHi_41=0, N2_41=2, SigIdxLo_41_01=2, AntennaID_41_01=0, LockTime_41_01=254, CN0_41_01=195, CodeOffsetMSB_41_01=0, DopplerOffsetMSB_41_01=31, CarrierMSB_41_01=0, PRSmoothed_41_01=0, Reserved6_41_01=0, HalfCycleAmbiguity_41_01=0, SigIdxHi_41_01=0, CodeOffsetLSB_41_01=1094, CarrierLSB_41_01=2617, DopplerOffsetLSB_41_01=65527, SigIdxLo_41_02=3, AntennaID_41_02=0, LockTime_41_02=254, CN0_41_02=136, CodeOffsetMSB_41_02=0, DopplerOffsetMSB_41_02=0, CarrierMSB_41_02=0, PRSmoothed_41_02=0, Reserved6_41_02=0, HalfCycleAmbiguity_41_02=0, SigIdxHi_41_02=0, CodeOffsetLSB_41_02=1742, CarrierLSB_41_02=3970, DopplerOffsetLSB_41_02=302, RxChannel_42=52, SigIdxLo_42=15, AntennaID_42=0, SVID_42=217, CodeMSB_42=8, Reserved3_42=0, CodeLSB_42=3744493272, Doppler_42=-69968, CarrierLSB_42=2143, CarrierMSB_42=0, CN0_42=105, LockTime_42=503, PRSmoothed_42=0, Reserved4_42=0, HalfCycleAmbiguity_42=0, SigIdxHi_42=0, N2_42=0, RxChannel_43=54, SigIdxLo_43=11, AntennaID_43=0, SVID_43=39, CodeMSB_43=5, Reserved3_43=0, CodeLSB_43=2574732075, Doppler_43=-35412292, CarrierLSB_43=61283, CarrierMSB_43=-1, CN0_43=116, LockTime_43=378, PRSmoothed_43=0, Reserved4_43=0, HalfCycleAmbiguity_43=0, SigIdxHi_43=4, N2_43=1, SigIdxLo_43_01=8, AntennaID_43_01=0, LockTime_43_01=255, CN0_43_01=73, CodeOffsetMSB_43_01=7, DopplerOffsetMSB_43_01=0, CarrierMSB_43_01=-128, PRSmoothed_43_01=0, Reserved6_43_01=0, HalfCycleAmbiguity_43_01=1, SigIdxHi_43_01=0, CodeOffsetLSB_43_01=59698, CarrierLSB_43_01=0, DopplerOffsetLSB_43_01=9452, RxChannel_44=55, SigIdxLo_44=30, AntennaID_44=0, SVID_44=145, CodeMSB_44=9, Reserved3_44=0, CodeLSB_44=1792866417, Doppler_44=-2074647, CarrierLSB_44=321, CarrierMSB_44=0, CN0_44=115, LockTime_44=158, PRSmoothed_44=0, Reserved4_44=0, HalfCycleAmbiguity_44=0, SigIdxHi_44=0, N2_44=2, SigIdxLo_44_01=28, AntennaID_44_01=0, LockTime_44_01=156, CN0_44_01=104, CodeOffsetMSB_44_01=7, DopplerOffsetMSB_44_01=31, CarrierMSB_44_01=0, PRSmoothed_44_01=0, Reserved6_44_01=0, HalfCycleAmbiguity_44_01=0, SigIdxHi_44_01=0, CodeOffsetLSB_44_01=61432, CarrierLSB_44_01=1415, DopplerOffsetLSB_44_01=64882, SigIdxLo_44_02=29, AntennaID_44_02=0, LockTime_44_02=156, CN0_44_02=108, CodeOffsetMSB_44_02=7, DopplerOffsetMSB_44_02=31, CarrierMSB_44_02=-1, PRSmoothed_44_02=0, Reserved6_44_02=0, HalfCycleAmbiguity_44_02=0, SigIdxHi_44_02=0, CodeOffsetLSB_44_02=63683, CarrierLSB_44_02=64055, DopplerOffsetLSB_44_02=64428)>",
            "<SBF(MeasExtra, TOW=13:58:23, WNc=2367, N=100, SBLength=16, DopplerVarFactor=163.0, RxChannel_01=1, SigIdxLo_01=0, AntennaID_01=0, MPCorrection_01=335, SmoothingCorr_01=0, CodeVar_01=97, CarrierVar_01=11, LockTime_01=513, CumLossCont_01=1, CarMPCorr_01=3, Info_01=0, CN0HighRes_01=5, SigIdxHi_01=0, RxChannel_02=1, SigIdxLo_02=2, AntennaID_02=0, MPCorrection_02=0, SmoothingCorr_02=0, CodeVar_02=2, CarrierVar_02=1, LockTime_02=504, CumLossCont_02=1, CarMPCorr_02=0, Info_02=0, CN0HighRes_02=6, SigIdxHi_02=0, RxChannel_03=1, SigIdxLo_03=3, AntennaID_03=0, MPCorrection_03=44, SmoothingCorr_03=0, CodeVar_03=250, CarrierVar_03=27, LockTime_03=506, CumLossCont_03=1, CarMPCorr_03=-4, Info_03=0, CN0HighRes_03=4, SigIdxHi_03=0, RxChannel_04=2, SigIdxLo_04=0, AntennaID_04=0, MPCorrection_04=37, SmoothingCorr_04=0, CodeVar_04=292, CarrierVar_04=31, LockTime_04=508, CumLossCont_04=1, CarMPCorr_04=2, Info_04=0, CN0HighRes_04=2, SigIdxHi_04=0, RxChannel_05=2, SigIdxLo_05=2, AntennaID_05=0, MPCorrection_05=0, SmoothingCorr_05=0, CodeVar_05=35, CarrierVar_05=7, LockTime_05=498, CumLossCont_05=1, CarMPCorr_05=0, Info_05=0, CN0HighRes_05=4, SigIdxHi_05=0, RxChannel_06=2, SigIdxLo_06=3, AntennaID_06=0, MPCorrection_06=320, SmoothingCorr_06=0, CodeVar_06=191, CarrierVar_06=21, LockTime_06=500, CumLossCont_06=1, CarMPCorr_06=-1, Info_06=0, CN0HighRes_06=1, SigIdxHi_06=0, RxChannel_07=3, SigIdxLo_07=17, AntennaID_07=0, MPCorrection_07=766, SmoothingCorr_07=0, CodeVar_07=222, CarrierVar_07=71, LockTime_07=401, CumLossCont_07=1, CarMPCorr_07=1, Info_07=0, CN0HighRes_07=5, SigIdxHi_07=0, RxChannel_08=3, SigIdxLo_08=20, AntennaID_08=0, MPCorrection_08=-499, SmoothingCorr_08=0, CodeVar_08=38, CarrierVar_08=40, LockTime_08=449, CumLossCont_08=1, CarMPCorr_08=-1, Info_08=0, CN0HighRes_08=2, SigIdxHi_08=0, RxChannel_09=3, SigIdxLo_09=21, AntennaID_09=0, MPCorrection_09=735, SmoothingCorr_09=0, CodeVar_09=32, CarrierVar_09=34, LockTime_09=443, CumLossCont_09=1, CarMPCorr_09=-3, Info_09=0, CN0HighRes_09=3, SigIdxHi_09=0, RxChannel_10=4, SigIdxLo_10=21, AntennaID_10=0, MPCorrection_10=0, SmoothingCorr_10=0, CodeVar_10=25790, CarrierVar_10=65534, LockTime_10=65535, CumLossCont_10=1, CarMPCorr_10=0, Info_10=1, CN0HighRes_10=4, SigIdxHi_10=0, RxChannel_11=5, SigIdxLo_11=8, AntennaID_11=0, MPCorrection_11=-279, SmoothingCorr_11=0, CodeVar_11=218, CarrierVar_11=12, LockTime_11=509, CumLossCont_11=1, CarMPCorr_11=2, Info_11=0, CN0HighRes_11=2, SigIdxHi_11=0, RxChannel_12=5, SigIdxLo_12=11, AntennaID_12=0, MPCorrection_12=544, SmoothingCorr_12=0, CodeVar_12=311, CarrierVar_12=17, LockTime_12=508, CumLossCont_12=1, CarMPCorr_12=-4, Info_12=0, CN0HighRes_12=1, SigIdxHi_12=0, RxChannel_13=6, SigIdxLo_13=0, AntennaID_13=0, MPCorrection_13=-254, SmoothingCorr_13=0, CodeVar_13=147, CarrierVar_13=16, LockTime_13=513, CumLossCont_13=1, CarMPCorr_13=-1, Info_13=0, CN0HighRes_13=7, SigIdxHi_13=0, RxChannel_14=6, SigIdxLo_14=2, AntennaID_14=0, MPCorrection_14=0, SmoothingCorr_14=0, CodeVar_14=36, CarrierVar_14=7, LockTime_14=504, CumLossCont_14=1, CarMPCorr_14=0, Info_14=0, CN0HighRes_14=5, SigIdxHi_14=0, RxChannel_15=7, SigIdxLo_15=8, AntennaID_15=0, MPCorrection_15=42, SmoothingCorr_15=0, CodeVar_15=427, CarrierVar_15=23, LockTime_15=509, CumLossCont_15=1, CarMPCorr_15=-5, Info_15=0, CN0HighRes_15=3, SigIdxHi_15=0, RxChannel_16=7, SigIdxLo_16=11, AntennaID_16=0, MPCorrection_16=-374, SmoothingCorr_16=0, CodeVar_16=1147, CarrierVar_16=61, LockTime_16=479, CumLossCont_16=1, CarMPCorr_16=8, Info_16=0, CN0HighRes_16=6, SigIdxHi_16=0, RxChannel_17=8, SigIdxLo_17=17, AntennaID_17=0, MPCorrection_17=-126, SmoothingCorr_17=0, CodeVar_17=83, CarrierVar_17=27, LockTime_17=504, CumLossCont_17=1, CarMPCorr_17=1, Info_17=0, CN0HighRes_17=6, SigIdxHi_17=0, RxChannel_18=8, SigIdxLo_18=20, AntennaID_18=0, MPCorrection_18=-394, SmoothingCorr_18=0, CodeVar_18=19, CarrierVar_18=20, LockTime_18=504, CumLossCont_18=1, CarMPCorr_18=-1, Info_18=0, CN0HighRes_18=3, SigIdxHi_18=0, RxChannel_19=8, SigIdxLo_19=21, AntennaID_19=0, MPCorrection_19=52, SmoothingCorr_19=0, CodeVar_19=16, CarrierVar_19=17, LockTime_19=504, CumLossCont_19=1, CarMPCorr_19=1, Info_19=0, CN0HighRes_19=0, SigIdxHi_19=0, RxChannel_20=9, SigIdxLo_20=17, AntennaID_20=0, MPCorrection_20=-269, SmoothingCorr_20=0, CodeVar_20=1001, CarrierVar_20=325, LockTime_20=86, CumLossCont_20=1, CarMPCorr_20=5, Info_20=0, CN0HighRes_20=0, SigIdxHi_20=0, RxChannel_21=9, SigIdxLo_21=20, AntennaID_21=0, MPCorrection_21=404, SmoothingCorr_21=0, CodeVar_21=228, CarrierVar_21=244, LockTime_21=238, CumLossCont_21=1, CarMPCorr_21=0, Info_21=0, CN0HighRes_21=4, SigIdxHi_21=0, RxChannel_22=9, SigIdxLo_22=21, AntennaID_22=0, MPCorrection_22=504, SmoothingCorr_22=0, CodeVar_22=243, CarrierVar_22=260, LockTime_22=201, CumLossCont_22=1, CarMPCorr_22=0, Info_22=0, CN0HighRes_22=6, SigIdxHi_22=0, RxChannel_23=10, SigIdxLo_23=8, AntennaID_23=0, MPCorrection_23=135, SmoothingCorr_23=0, CodeVar_23=854, CarrierVar_23=45, LockTime_23=509, CumLossCont_23=1, CarMPCorr_23=6, Info_23=0, CN0HighRes_23=2, SigIdxHi_23=0, RxChannel_24=11, SigIdxLo_24=28, AntennaID_24=0, MPCorrection_24=-652, SmoothingCorr_24=0, CodeVar_24=52, CarrierVar_24=12, LockTime_24=509, CumLossCont_24=1, CarMPCorr_24=-6, Info_24=0, CN0HighRes_24=1, SigIdxHi_24=0, RxChannel_25=11, SigIdxLo_25=30, AntennaID_25=0, MPCorrection_25=-62, SmoothingCorr_25=0, CodeVar_25=12, CarrierVar_25=12, LockTime_25=508, CumLossCont_25=1, CarMPCorr_25=-1, Info_25=0, CN0HighRes_25=6, SigIdxHi_25=0, RxChannel_26=12, SigIdxLo_26=17, AntennaID_26=0, MPCorrection_26=0, SmoothingCorr_26=0, CodeVar_26=47, CarrierVar_26=15, LockTime_26=509, CumLossCont_26=1, CarMPCorr_26=-2, Info_26=0, CN0HighRes_26=0, SigIdxHi_26=0, RxChannel_27=12, SigIdxLo_27=20, AntennaID_27=0, MPCorrection_27=-239, SmoothingCorr_27=0, CodeVar_27=8, CarrierVar_27=8, LockTime_27=509, CumLossCont_27=1, CarMPCorr_27=0, Info_27=0, CN0HighRes_27=0, SigIdxHi_27=0, RxChannel_28=12, SigIdxLo_28=21, AntennaID_28=0, MPCorrection_28=62, SmoothingCorr_28=0, CodeVar_28=6, CarrierVar_28=7, LockTime_28=509, CumLossCont_28=1, CarMPCorr_28=0, Info_28=0, CN0HighRes_28=3, SigIdxHi_28=0, RxChannel_29=13, SigIdxLo_29=8, AntennaID_29=0, MPCorrection_29=-108, SmoothingCorr_29=0, CodeVar_29=109, CarrierVar_29=7, LockTime_29=509, CumLossCont_29=1, CarMPCorr_29=0, Info_29=0, CN0HighRes_29=4, SigIdxHi_29=0, RxChannel_30=13, SigIdxLo_30=11, AntennaID_30=0, MPCorrection_30=-32, SmoothingCorr_30=0, CodeVar_30=93, CarrierVar_30=6, LockTime_30=508, CumLossCont_30=1, CarMPCorr_30=2, Info_30=0, CN0HighRes_30=0, SigIdxHi_30=0, RxChannel_31=14, SigIdxLo_31=17, AntennaID_31=0, MPCorrection_31=31, SmoothingCorr_31=0, CodeVar_31=27, CarrierVar_31=9, LockTime_31=509, CumLossCont_31=1, CarMPCorr_31=0, Info_31=0, CN0HighRes_31=7, SigIdxHi_31=0, RxChannel_32=14, SigIdxLo_32=20, AntennaID_32=0, MPCorrection_32=241, SmoothingCorr_32=0, CodeVar_32=5, CarrierVar_32=5, LockTime_32=509, CumLossCont_32=1, CarMPCorr_32=1, Info_32=0, CN0HighRes_32=7, SigIdxHi_32=0, RxChannel_33=14, SigIdxLo_33=21, AntennaID_33=0, MPCorrection_33=69, SmoothingCorr_33=0, CodeVar_33=4, CarrierVar_33=4, LockTime_33=509, CumLossCont_33=1, CarMPCorr_33=0, Info_33=0, CN0HighRes_33=5, SigIdxHi_33=0, RxChannel_34=15, SigIdxLo_34=28, AntennaID_34=0, MPCorrection_34=-391, SmoothingCorr_34=0, CodeVar_34=111, CarrierVar_34=24, LockTime_34=509, CumLossCont_34=1, CarMPCorr_34=-1, Info_34=0, CN0HighRes_34=0, SigIdxHi_34=0, RxChannel_35=15, SigIdxLo_35=30, AntennaID_35=0, MPCorrection_35=403, SmoothingCorr_35=0, CodeVar_35=12, CarrierVar_35=12, LockTime_35=508, CumLossCont_35=1, CarMPCorr_35=0, Info_35=0, CN0HighRes_35=7, SigIdxHi_35=0, RxChannel_36=16, SigIdxLo_36=0, AntennaID_36=0, MPCorrection_36=524, SmoothingCorr_36=0, CodeVar_36=123, CarrierVar_36=14, LockTime_36=512, CumLossCont_36=1, CarMPCorr_36=1, Info_36=0, CN0HighRes_36=1, SigIdxHi_36=0, RxChannel_37=16, SigIdxLo_37=2, AntennaID_37=0, MPCorrection_37=0, SmoothingCorr_37=0, CodeVar_37=2, CarrierVar_37=1, LockTime_37=506, CumLossCont_37=1, CarMPCorr_37=0, Info_37=0, CN0HighRes_37=5, SigIdxHi_37=0, RxChannel_38=16, SigIdxLo_38=3, AntennaID_38=0, MPCorrection_38=77, SmoothingCorr_38=0, CodeVar_38=176, CarrierVar_38=19, LockTime_38=506, CumLossCont_38=1, CarMPCorr_38=1, Info_38=0, CN0HighRes_38=1, SigIdxHi_38=0, RxChannel_39=17, SigIdxLo_39=17, AntennaID_39=0, MPCorrection_39=-64, SmoothingCorr_39=0, CodeVar_39=37, CarrierVar_39=12, LockTime_39=512, CumLossCont_39=1, CarMPCorr_39=-4, Info_39=0, CN0HighRes_39=6, SigIdxHi_39=0, RxChannel_40=17, SigIdxLo_40=20, AntennaID_40=0, MPCorrection_40=-87, SmoothingCorr_40=0, CodeVar_40=14, CarrierVar_40=14, LockTime_40=511, CumLossCont_40=1, CarMPCorr_40=-2, Info_40=0, CN0HighRes_40=7, SigIdxHi_40=0, RxChannel_41=17, SigIdxLo_41=21, AntennaID_41=0, MPCorrection_41=115, SmoothingCorr_41=0, CodeVar_41=9, CarrierVar_41=9, LockTime_41=511, CumLossCont_41=1, CarMPCorr_41=0, Info_41=0, CN0HighRes_41=1, SigIdxHi_41=0, RxChannel_42=18, SigIdxLo_42=24, AntennaID_42=0, MPCorrection_42=0, SmoothingCorr_42=0, CodeVar_42=6674, CarrierVar_42=25, LockTime_42=510, CumLossCont_42=1, CarMPCorr_42=0, Info_42=0, CN0HighRes_42=3, SigIdxHi_42=0, RxChannel_43=19, SigIdxLo_43=17, AntennaID_43=0, MPCorrection_43=-297, SmoothingCorr_43=0, CodeVar_43=94, CarrierVar_43=30, LockTime_43=512, CumLossCont_43=1, CarMPCorr_43=1, Info_43=0, CN0HighRes_43=4, SigIdxHi_43=0, RxChannel_44=19, SigIdxLo_44=20, AntennaID_44=0, MPCorrection_44=485, SmoothingCorr_44=0, CodeVar_44=30, CarrierVar_44=32, LockTime_44=511, CumLossCont_44=1, CarMPCorr_44=0, Info_44=0, CN0HighRes_44=0, SigIdxHi_44=0, RxChannel_45=19, SigIdxLo_45=21, AntennaID_45=0, MPCorrection_45=130, SmoothingCorr_45=0, CodeVar_45=23, CarrierVar_45=24, LockTime_45=509, CumLossCont_45=1, CarMPCorr_45=0, Info_45=0, CN0HighRes_45=6, SigIdxHi_45=0, RxChannel_46=20, SigIdxLo_46=8, AntennaID_46=0, MPCorrection_46=98, SmoothingCorr_46=0, CodeVar_46=71, CarrierVar_46=5, LockTime_46=509, CumLossCont_46=1, CarMPCorr_46=0, Info_46=0, CN0HighRes_46=3, SigIdxHi_46=0, RxChannel_47=20, SigIdxLo_47=11, AntennaID_47=0, MPCorrection_47=-64, SmoothingCorr_47=0, CodeVar_47=120, CarrierVar_47=7, LockTime_47=508, CumLossCont_47=1, CarMPCorr_47=0, Info_47=0, CN0HighRes_47=6, SigIdxHi_47=0, RxChannel_48=21, SigIdxLo_48=28, AntennaID_48=0, MPCorrection_48=16, SmoothingCorr_48=0, CodeVar_48=36, CarrierVar_48=8, LockTime_48=509, CumLossCont_48=1, CarMPCorr_48=3, Info_48=0, CN0HighRes_48=4, SigIdxHi_48=0, RxChannel_49=21, SigIdxLo_49=29, AntennaID_49=0, MPCorrection_49=398, SmoothingCorr_49=0, CodeVar_49=20, CarrierVar_49=5, LockTime_49=508, CumLossCont_49=1, CarMPCorr_49=-9, Info_49=0, CN0HighRes_49=0, SigIdxHi_49=0, RxChannel_50=21, SigIdxLo_50=30, AntennaID_50=0, MPCorrection_50=104, SmoothingCorr_50=0, CodeVar_50=6, CarrierVar_50=6, LockTime_50=508, CumLossCont_50=1, CarMPCorr_50=2, Info_50=0, CN0HighRes_50=5, SigIdxHi_50=0, RxChannel_51=23, SigIdxLo_51=17, AntennaID_51=0, MPCorrection_51=315, SmoothingCorr_51=0, CodeVar_51=54, CarrierVar_51=17, LockTime_51=509, CumLossCont_51=1, CarMPCorr_51=0, Info_51=0, CN0HighRes_51=4, SigIdxHi_51=0, RxChannel_52=23, SigIdxLo_52=20, AntennaID_52=0, MPCorrection_52=-248, SmoothingCorr_52=0, CodeVar_52=11, CarrierVar_52=11, LockTime_52=509, CumLossCont_52=1, CarMPCorr_52=0, Info_52=0, CN0HighRes_52=0, SigIdxHi_52=0, RxChannel_53=23, SigIdxLo_53=21, AntennaID_53=0, MPCorrection_53=-158, SmoothingCorr_53=0, CodeVar_53=9, CarrierVar_53=10, LockTime_53=509, CumLossCont_53=1, CarMPCorr_53=1, Info_53=0, CN0HighRes_53=4, SigIdxHi_53=0, RxChannel_54=24, SigIdxLo_54=8, AntennaID_54=0, MPCorrection_54=287, SmoothingCorr_54=0, CodeVar_54=97, CarrierVar_54=6, LockTime_54=509, CumLossCont_54=1, CarMPCorr_54=-1, Info_54=0, CN0HighRes_54=3, SigIdxHi_54=0, RxChannel_55=24, SigIdxLo_55=11, AntennaID_55=0, MPCorrection_55=221, SmoothingCorr_55=0, CodeVar_55=136, CarrierVar_55=8, LockTime_55=508, CumLossCont_55=1, CarMPCorr_55=-2, Info_55=0, CN0HighRes_55=6, SigIdxHi_55=0, RxChannel_56=25, SigIdxLo_56=28, AntennaID_56=0, MPCorrection_56=20, SmoothingCorr_56=0, CodeVar_56=18, CarrierVar_56=4, LockTime_56=494, CumLossCont_56=1, CarMPCorr_56=1, Info_56=0, CN0HighRes_56=2, SigIdxHi_56=0, RxChannel_57=25, SigIdxLo_57=30, AntennaID_57=0, MPCorrection_57=-2, SmoothingCorr_57=0, CodeVar_57=3, CarrierVar_57=3, LockTime_57=494, CumLossCont_57=1, CarMPCorr_57=0, Info_57=0, CN0HighRes_57=7, SigIdxHi_57=0, RxChannel_58=26, SigIdxLo_58=28, AntennaID_58=0, MPCorrection_58=-75, SmoothingCorr_58=0, CodeVar_58=19, CarrierVar_58=5, LockTime_58=494, CumLossCont_58=1, CarMPCorr_58=0, Info_58=0, CN0HighRes_58=7, SigIdxHi_58=0, RxChannel_59=26, SigIdxLo_59=30, AntennaID_59=0, MPCorrection_59=109, SmoothingCorr_59=0, CodeVar_59=5, CarrierVar_59=5, LockTime_59=494, CumLossCont_59=1, CarMPCorr_59=0, Info_59=0, CN0HighRes_59=3, SigIdxHi_59=0, RxChannel_60=28, SigIdxLo_60=28, AntennaID_60=0, MPCorrection_60=55, SmoothingCorr_60=0, CodeVar_60=24, CarrierVar_60=6, LockTime_60=509, CumLossCont_60=1, CarMPCorr_60=1, Info_60=0, CN0HighRes_60=6, SigIdxHi_60=0, RxChannel_61=28, SigIdxLo_61=30, AntennaID_61=0, MPCorrection_61=-22, SmoothingCorr_61=0, CodeVar_61=6, CarrierVar_61=6, LockTime_61=508, CumLossCont_61=1, CarMPCorr_61=0, Info_61=0, CN0HighRes_61=0, SigIdxHi_61=0, RxChannel_62=29, SigIdxLo_62=24, AntennaID_62=0, MPCorrection_62=0, SmoothingCorr_62=0, CodeVar_62=6684, CarrierVar_62=25, LockTime_62=510, CumLossCont_62=1, CarMPCorr_62=0, Info_62=0, CN0HighRes_62=6, SigIdxHi_62=0, RxChannel_63=30, SigIdxLo_63=24, AntennaID_63=0, MPCorrection_63=0, SmoothingCorr_63=0, CodeVar_63=5996, CarrierVar_63=22, LockTime_63=508, CumLossCont_63=1, CarMPCorr_63=0, Info_63=0, CN0HighRes_63=0, SigIdxHi_63=0, RxChannel_64=31, SigIdxLo_64=0, AntennaID_64=0, MPCorrection_64=140, SmoothingCorr_64=0, CodeVar_64=322, CarrierVar_64=34, LockTime_64=508, CumLossCont_64=1, CarMPCorr_64=0, Info_64=0, CN0HighRes_64=0, SigIdxHi_64=0, RxChannel_65=31, SigIdxLo_65=2, AntennaID_65=0, MPCorrection_65=0, SmoothingCorr_65=0, CodeVar_65=5, CarrierVar_65=2, LockTime_65=498, CumLossCont_65=1, CarMPCorr_65=0, Info_65=0, CN0HighRes_65=3, SigIdxHi_65=0, RxChannel_66=31, SigIdxLo_66=3, AntennaID_66=0, MPCorrection_66=12, SmoothingCorr_66=0, CodeVar_66=314, CarrierVar_66=34, LockTime_66=500, CumLossCont_66=1, CarMPCorr_66=-1, Info_66=0, CN0HighRes_66=4, SigIdxHi_66=0, RxChannel_67=34, SigIdxLo_67=28, AntennaID_67=0, MPCorrection_67=-560, SmoothingCorr_67=0, CodeVar_67=270, CarrierVar_67=57, LockTime_67=490, CumLossCont_67=1, CarMPCorr_67=0, Info_67=0, CN0HighRes_67=7, SigIdxHi_67=0, RxChannel_68=34, SigIdxLo_68=30, AntennaID_68=0, MPCorrection_68=-381, SmoothingCorr_68=0, CodeVar_68=26, CarrierVar_68=27, LockTime_68=490, CumLossCont_68=1, CarMPCorr_68=1, Info_68=0, CN0HighRes_68=6, SigIdxHi_68=0, RxChannel_69=35, SigIdxLo_69=8, AntennaID_69=0, MPCorrection_69=-49, SmoothingCorr_69=0, CodeVar_69=114, CarrierVar_69=7, LockTime_69=509, CumLossCont_69=1, CarMPCorr_69=0, Info_69=0, CN0HighRes_69=2, SigIdxHi_69=0, RxChannel_70=35, SigIdxLo_70=11, AntennaID_70=0, MPCorrection_70=551, SmoothingCorr_70=0, CodeVar_70=138, CarrierVar_70=8, LockTime_70=509, CumLossCont_70=1, CarMPCorr_70=1, Info_70=0, CN0HighRes_70=7, SigIdxHi_70=0, RxChannel_71=36, SigIdxLo_71=17, AntennaID_71=0, MPCorrection_71=-188, SmoothingCorr_71=0, CodeVar_71=81, CarrierVar_71=26, LockTime_71=507, CumLossCont_71=1, CarMPCorr_71=-1, Info_71=0, CN0HighRes_71=3, SigIdxHi_71=0, RxChannel_72=36, SigIdxLo_72=20, AntennaID_72=0, MPCorrection_72=143, SmoothingCorr_72=0, CodeVar_72=10, CarrierVar_72=11, LockTime_72=506, CumLossCont_72=1, CarMPCorr_72=0, Info_72=0, CN0HighRes_72=2, SigIdxHi_72=0, RxChannel_73=36, SigIdxLo_73=21, AntennaID_73=0, MPCorrection_73=-189, SmoothingCorr_73=0, CodeVar_73=8, CarrierVar_73=8, LockTime_73=506, CumLossCont_73=1, CarMPCorr_73=0, Info_73=0, CN0HighRes_73=2, SigIdxHi_73=0, RxChannel_74=37, SigIdxLo_74=28, AntennaID_74=0, MPCorrection_74=-106, SmoothingCorr_74=0, CodeVar_74=39, CarrierVar_74=9, LockTime_74=509, CumLossCont_74=1, CarMPCorr_74=1, Info_74=0, CN0HighRes_74=0, SigIdxHi_74=0, RxChannel_75=37, SigIdxLo_75=29, AntennaID_75=0, MPCorrection_75=-231, SmoothingCorr_75=0, CodeVar_75=22, CarrierVar_75=5, LockTime_75=508, CumLossCont_75=1, CarMPCorr_75=5, Info_75=0, CN0HighRes_75=6, SigIdxHi_75=0, RxChannel_76=37, SigIdxLo_76=30, AntennaID_76=0, MPCorrection_76=-29, SmoothingCorr_76=0, CodeVar_76=5, CarrierVar_76=5, LockTime_76=508, CumLossCont_76=1, CarMPCorr_76=0, Info_76=0, CN0HighRes_76=7, SigIdxHi_76=0, RxChannel_77=38, SigIdxLo_77=8, AntennaID_77=0, MPCorrection_77=84, SmoothingCorr_77=0, CodeVar_77=1549, CarrierVar_77=82, LockTime_77=505, CumLossCont_77=1, CarMPCorr_77=-6, Info_77=0, CN0HighRes_77=1, SigIdxHi_77=0, RxChannel_78=38, SigIdxLo_78=11, AntennaID_78=0, MPCorrection_78=378, SmoothingCorr_78=0, CodeVar_78=161, CarrierVar_78=9, LockTime_78=505, CumLossCont_78=1, CarMPCorr_78=4, Info_78=0, CN0HighRes_78=0, SigIdxHi_78=0, RxChannel_79=39, SigIdxLo_79=24, AntennaID_79=0, MPCorrection_79=0, SmoothingCorr_79=0, CodeVar_79=17312, CarrierVar_79=63, LockTime_79=502, CumLossCont_79=3, CarMPCorr_79=0, Info_79=0, CN0HighRes_79=1, SigIdxHi_79=0, RxChannel_80=40, SigIdxLo_80=28, AntennaID_80=0, MPCorrection_80=46, SmoothingCorr_80=0, CodeVar_80=53, CarrierVar_80=12, LockTime_80=509, CumLossCont_80=1, CarMPCorr_80=6, Info_80=0, CN0HighRes_80=2, SigIdxHi_80=0, RxChannel_81=40, SigIdxLo_81=30, AntennaID_81=0, MPCorrection_81=-180, SmoothingCorr_81=0, CodeVar_81=7, CarrierVar_81=7, LockTime_81=508, CumLossCont_81=1, CarMPCorr_81=-2, Info_81=0, CN0HighRes_81=0, SigIdxHi_81=0, RxChannel_82=41, SigIdxLo_82=0, AntennaID_82=0, MPCorrection_82=583, SmoothingCorr_82=0, CodeVar_82=389, CarrierVar_82=41, LockTime_82=283, CumLossCont_82=1, CarMPCorr_82=3, Info_82=0, CN0HighRes_82=5, SigIdxHi_82=0, RxChannel_83=41, SigIdxLo_83=2, AntennaID_83=0, MPCorrection_83=0, SmoothingCorr_83=0, CodeVar_83=104, CarrierVar_83=19, LockTime_83=274, CumLossCont_83=1, CarMPCorr_83=0, Info_83=0, CN0HighRes_83=1, SigIdxHi_83=0, RxChannel_84=42, SigIdxLo_84=17, AntennaID_84=0, MPCorrection_84=190, SmoothingCorr_84=0, CodeVar_84=50, CarrierVar_84=16, LockTime_84=508, CumLossCont_84=1, CarMPCorr_84=2, Info_84=0, CN0HighRes_84=4, SigIdxHi_84=0, RxChannel_85=42, SigIdxLo_85=20, AntennaID_85=0, MPCorrection_85=-180, SmoothingCorr_85=0, CodeVar_85=11, CarrierVar_85=12, LockTime_85=507, CumLossCont_85=1, CarMPCorr_85=0, Info_85=0, CN0HighRes_85=4, SigIdxHi_85=0, RxChannel_86=42, SigIdxLo_86=21, AntennaID_86=0, MPCorrection_86=-4, SmoothingCorr_86=0, CodeVar_86=7, CarrierVar_86=7, LockTime_86=507, CumLossCont_86=1, CarMPCorr_86=0, Info_86=0, CN0HighRes_86=7, SigIdxHi_86=0, RxChannel_87=47, SigIdxLo_87=0, AntennaID_87=0, MPCorrection_87=1, SmoothingCorr_87=0, CodeVar_87=46, CarrierVar_87=6, LockTime_87=509, CumLossCont_87=1, CarMPCorr_87=-2, Info_87=0, CN0HighRes_87=3, SigIdxHi_87=0, RxChannel_88=47, SigIdxLo_88=2, AntennaID_88=0, MPCorrection_88=0, SmoothingCorr_88=0, CodeVar_88=2, CarrierVar_88=1, LockTime_88=506, CumLossCont_88=1, CarMPCorr_88=0, Info_88=0, CN0HighRes_88=6, SigIdxHi_88=0, RxChannel_89=47, SigIdxLo_89=3, AntennaID_89=0, MPCorrection_89=0, SmoothingCorr_89=0, CodeVar_89=40, CarrierVar_89=5, LockTime_89=506, CumLossCont_89=1, CarMPCorr_89=-3, Info_89=0, CN0HighRes_89=5, SigIdxHi_89=0, RxChannel_90=48, SigIdxLo_90=0, AntennaID_90=0, MPCorrection_90=228, SmoothingCorr_90=0, CodeVar_90=53, CarrierVar_90=6, LockTime_90=509, CumLossCont_90=1, CarMPCorr_90=-1, Info_90=0, CN0HighRes_90=1, SigIdxHi_90=0, RxChannel_91=48, SigIdxLo_91=2, AntennaID_91=0, MPCorrection_91=0, SmoothingCorr_91=0, CodeVar_91=4, CarrierVar_91=1, LockTime_91=506, CumLossCont_91=1, CarMPCorr_91=0, Info_91=0, CN0HighRes_91=6, SigIdxHi_91=0, RxChannel_92=49, SigIdxLo_92=0, AntennaID_92=0, MPCorrection_92=-230, SmoothingCorr_92=0, CodeVar_92=65, CarrierVar_92=8, LockTime_92=509, CumLossCont_92=1, CarMPCorr_92=1, Info_92=0, CN0HighRes_92=1, SigIdxHi_92=0, RxChannel_93=49, SigIdxLo_93=2, AntennaID_93=0, MPCorrection_93=0, SmoothingCorr_93=0, CodeVar_93=1, CarrierVar_93=1, LockTime_93=506, CumLossCont_93=1, CarMPCorr_93=0, Info_93=0, CN0HighRes_93=1, SigIdxHi_93=0, RxChannel_94=49, SigIdxLo_94=3, AntennaID_94=0, MPCorrection_94=103, SmoothingCorr_94=0, CodeVar_94=140, CarrierVar_94=15, LockTime_94=506, CumLossCont_94=1, CarMPCorr_94=-2, Info_94=0, CN0HighRes_94=7, SigIdxHi_94=0, RxChannel_95=52, SigIdxLo_95=15, AntennaID_95=0, MPCorrection_95=0, SmoothingCorr_95=0, CodeVar_95=966, CarrierVar_95=103, LockTime_95=503, CumLossCont_95=1, CarMPCorr_95=-3, Info_95=0, CN0HighRes_95=0, SigIdxHi_95=0, RxChannel_96=54, SigIdxLo_96=8, AntennaID_96=0, MPCorrection_96=0, SmoothingCorr_96=0, CodeVar_96=8391, CarrierVar_96=2301, LockTime_96=65535, CumLossCont_96=25, CarMPCorr_96=0, Info_96=1, CN0HighRes_96=4, SigIdxHi_96=0, RxChannel_97=54, SigIdxLo_97=11, AntennaID_97=0, MPCorrection_97=110, SmoothingCorr_97=0, CodeVar_97=991, CarrierVar_97=53, LockTime_97=378, CumLossCont_97=23, CarMPCorr_97=1, Info_97=0, CN0HighRes_97=7, SigIdxHi_97=0, RxChannel_98=55, SigIdxLo_98=28, AntennaID_98=0, MPCorrection_98=0, SmoothingCorr_98=0, CodeVar_98=436, CarrierVar_98=92, LockTime_98=156, CumLossCont_98=1, CarMPCorr_98=0, Info_98=0, CN0HighRes_98=0, SigIdxHi_98=0, RxChannel_99=55, SigIdxLo_99=29, AntennaID_99=0, MPCorrection_99=0, SmoothingCorr_99=0, CodeVar_99=480, CarrierVar_99=102, LockTime_99=156, CumLossCont_99=1, CarMPCorr_99=0, Info_99=0, CN0HighRes_99=0, SigIdxHi_99=0, RxChannel_100=55, SigIdxLo_100=30, AntennaID_100=0, MPCorrection_100=0, SmoothingCorr_100=0, CodeVar_100=50, CarrierVar_100=52, LockTime_100=158, CumLossCont_100=1, CarMPCorr_100=0, Info_100=0, CN0HighRes_100=4, SigIdxHi_100=0)>",
            "<SBF(EndOfMeas, TOW=13:58:23, WNc=2367)>",
        )
        i = 0
        with open(
            os.path.join(DIRNAME, "pygpsdata_x5_measurements.log"), "rb"
        ) as stream:
            ubr = SBFReader(stream, protfilter=7, quitonerror=ERR_RAISE)
            for raw, parsed in ubr:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
            self.assertEqual(i, len(EXPECTED_RESULTS))
            # print(ubr.datastream)
            self.assertIsInstance(ubr.datastream, BufferedReader)

    def testconstellations(self):  # test constellation message stream
        EXPECTED_RESULTS = (
            "<SBF(GEONetworkTime, TOW=14:01:50, WNc=2367, PRN=136, Reserved=0, A_1=9.769962616701378e-15, A_0=2.60770320892334e-08, t_ot=483328, WN_t=63, DEL_t_LS=18, WN_LSF=137, DN=7, DEL_t_LSF=18, UTC_std=3, GPS_WN=319, GPS_TOW=482527, GlonassID=0)>",
            "<SBF(GEOFastCorr, TOW=14:01:50, WNc=2367, PRN=121, MT=3, IODP=1, IODF=2, N=13, SBLength=8, PRNMaskNo_01=13, UDREI_01=5, Reserved_01=0, PRC_01=0.625, PRNMaskNo_02=14, UDREI_02=14, Reserved_02=0, PRC_02=0.0, PRNMaskNo_03=15, UDREI_03=14, Reserved_03=0, PRC_03=0.0, PRNMaskNo_04=16, UDREI_04=5, Reserved_04=0, PRC_04=-1.5, PRNMaskNo_05=17, UDREI_05=14, Reserved_05=0, PRC_05=0.0, PRNMaskNo_06=18, UDREI_06=5, Reserved_06=0, PRC_06=-0.375, PRNMaskNo_07=19, UDREI_07=14, Reserved_07=0, PRC_07=0.0, PRNMaskNo_08=20, UDREI_08=6, Reserved_08=0, PRC_08=-2.25, PRNMaskNo_09=21, UDREI_09=14, Reserved_09=0, PRC_09=0.0, PRNMaskNo_10=22, UDREI_10=5, Reserved_10=0, PRC_10=-1.0, PRNMaskNo_11=23, UDREI_11=14, Reserved_11=0, PRC_11=0.0, PRNMaskNo_12=24, UDREI_12=14, Reserved_12=0, PRC_12=0.0, PRNMaskNo_13=25, UDREI_13=5, Reserved_13=0, PRC_13=0.5)>",
            "<SBF(GEOLongTermCorr, TOW=14:01:51, WNc=2367, PRN=121, N=2, SBLength=40, Reserved=0, VelocityCode_01=0, PRNMaskNo_01=26, IODP_01=1, IODE_01=15, dx_01=-0.375, dy_01=0.625, dz_01=-0.25, dxRate_01=0.0, dyRate_01=0.0, dzRate_01=0.0, da_f0_01=-2.7939677238464355e-09, da_f1_01=0.0, t_oe_01=0, VelocityCode_02=0, PRNMaskNo_02=14, IODP_02=1, IODE_02=16, dx_02=-0.375, dy_02=-2.125, dz_02=1.125, dxRate_02=0.0, dyRate_02=0.0, dzRate_02=0.0, da_f0_02=5.587935447692871e-09, da_f1_02=0.0, t_oe_02=0)>",
            "<SBF(GALAlm, TOW=14:01:50, WNc=2367, SVID=106, Source=2, e=0.0004425048828125, t_oa=481800, delta_i=-0.00421142578125, OMEGADOT=-1.862645149230957e-09, SQRT_A=0.01171875, OMEGA_0=-0.68890380859375, omega=-0.176910400390625, M_0=-0.723724365234375, a_f1=-3.637978807091713e-12, a_f0=-0.000186920166015625, WN_a=3, SVID_A=77, L1-BHealth=1, L1-Bhs=0, E5bHealth=1, E5bhs=0, E5aHealth=0, E5ahs=0, Reserved=0, IODa=3)>",
            "<SBF(GEOMT00, TOW=14:01:52, WNc=2367, PRN=121)>",
            "<SBF(GEOIGPMask, TOW=14:01:53, WNc=2367, PRN=121, NbrBands=5, BandNbr=5, IODI=0, NbrIGPs=70, IGPMask_01=17, IGPMask_02=18, IGPMask_03=19, IGPMask_04=20, IGPMask_05=21, IGPMask_06=22, IGPMask_07=23, IGPMask_08=24, IGPMask_09=25, IGPMask_10=42, IGPMask_11=43, IGPMask_12=44, IGPMask_13=45, IGPMask_14=46, IGPMask_15=47, IGPMask_16=48, IGPMask_17=49, IGPMask_18=50, IGPMask_19=67, IGPMask_20=68, IGPMask_21=69, IGPMask_22=70, IGPMask_23=71, IGPMask_24=72, IGPMask_25=73, IGPMask_26=74, IGPMask_27=75, IGPMask_28=92, IGPMask_29=93, IGPMask_30=94, IGPMask_31=95, IGPMask_32=96, IGPMask_33=97, IGPMask_34=98, IGPMask_35=99, IGPMask_36=100, IGPMask_37=118, IGPMask_38=119, IGPMask_39=120, IGPMask_40=121, IGPMask_41=122, IGPMask_42=123, IGPMask_43=124, IGPMask_44=125, IGPMask_45=126, IGPMask_46=143, IGPMask_47=144, IGPMask_48=145, IGPMask_49=146, IGPMask_50=147, IGPMask_51=148, IGPMask_52=149, IGPMask_53=150, IGPMask_54=151, IGPMask_55=169, IGPMask_56=170, IGPMask_57=171, IGPMask_58=172, IGPMask_59=173, IGPMask_60=174, IGPMask_61=175, IGPMask_62=176, IGPMask_63=194, IGPMask_64=195, IGPMask_65=196, IGPMask_66=197, IGPMask_67=198, IGPMask_68=199, IGPMask_69=200, IGPMask_70=201)>",
            "<SBF(GEOPRNMask, TOW=14:01:54, WNc=2367, PRN=136, IODP=3, NbrPRNs=33, PRNMask_01=1, PRNMask_02=2, PRNMask_03=3, PRNMask_04=4, PRNMask_05=5, PRNMask_06=6, PRNMask_07=7, PRNMask_08=8, PRNMask_09=9, PRNMask_10=10, PRNMask_11=11, PRNMask_12=12, PRNMask_13=13, PRNMask_14=14, PRNMask_15=15, PRNMask_16=16, PRNMask_17=17, PRNMask_18=18, PRNMask_19=19, PRNMask_20=20, PRNMask_21=22, PRNMask_22=23, PRNMask_23=24, PRNMask_24=25, PRNMask_25=26, PRNMask_26=27, PRNMask_27=28, PRNMask_28=29, PRNMask_29=30, PRNMask_30=31, PRNMask_31=32, PRNMask_32=123, PRNMask_33=136)>",
            "<SBF(GEOIonoDelay, TOW=14:01:54, WNc=2367, PRN=123, BandNbr=9, IODI=1, N=15, SBLength=8, Reserved=0, IGPMaskNo_01=46, GIVEI_01=6, Reserved1_01=0, VerticalDelay_01=3.0, IGPMaskNo_02=47, GIVEI_02=6, Reserved1_02=0, VerticalDelay_02=3.125, IGPMaskNo_03=48, GIVEI_03=15, Reserved1_03=0, VerticalDelay_03=63.75, IGPMaskNo_04=49, GIVEI_04=15, Reserved1_04=0, VerticalDelay_04=63.75, IGPMaskNo_05=50, GIVEI_05=14, Reserved1_05=0, VerticalDelay_05=2.375, IGPMaskNo_06=51, GIVEI_06=11, Reserved1_06=0, VerticalDelay_06=2.375, IGPMaskNo_07=52, GIVEI_07=9, Reserved1_07=0, VerticalDelay_07=2.5, IGPMaskNo_08=53, GIVEI_08=7, Reserved1_08=0, VerticalDelay_08=2.625, IGPMaskNo_09=54, GIVEI_09=6, Reserved1_09=0, VerticalDelay_09=2.625, IGPMaskNo_10=55, GIVEI_10=6, Reserved1_10=0, VerticalDelay_10=2.625, IGPMaskNo_11=56, GIVEI_11=6, Reserved1_11=0, VerticalDelay_11=2.5, IGPMaskNo_12=57, GIVEI_12=6, Reserved1_12=0, VerticalDelay_12=2.5, IGPMaskNo_13=58, GIVEI_13=6, Reserved1_13=0, VerticalDelay_13=2.625, IGPMaskNo_14=59, GIVEI_14=6, Reserved1_14=0, VerticalDelay_14=2.875, IGPMaskNo_15=60, GIVEI_15=6, Reserved1_15=0, VerticalDelay_15=3.125)>",
            "<SBF(BDSAlm, TOW=14:01:56, WNc=2367, PRN=168, WN_a=243, t_oa=151552, SQRT_A=5282.57763671875, e=3.910064697265625e-05, omega=-0.717388391494751, M_0=0.8846232891082764, OMEGA_0=0.444791316986084, OMEGADOT=-2.3101165425032377e-09, delta_i=0.00360107421875, a_f0=0.00037860870361328125, a_f1=3.637978807091713e-12, Health=88, Reserved=0)>",
            "<SBF(GEONav, TOW=14:01:58, WNc=2367, PRN=123, Reserved=255, IODN=184, URA=15, t0=50528, Xg=38667964.0, Yg=16813308.0, Zg=0.0, Xgd=0.0, Ygd=0.0, Zgd=0.0, Xgdd=0.0, Ygdd=0.0, Zgdd=0.0, aGf0=0.0, aGf1=0.0)>",
            "<SBF(BDSIon, TOW=14:01:59.600000, WNc=2367, PRN=145, Reserved=0, alpha_0=2.7939677238464355e-08, alpha_1=6.705522537231445e-08, alpha_2=-8.940696716308594e-07, alpha_3=1.430511474609375e-06, beta_0=137216.0, beta_1=-262144.0, beta_2=2424832.0, beta_3=-2031616.0)>",
            "<SBF(GEOFastCorrDegr, TOW=14:02:00, WNc=2367, PRN=205, IODP=0, t_lat=1, ai_01=15, ai_02=15, ai_03=15, ai_04=15, ai_05=15, ai_06=15, ai_07=15, ai_08=15, ai_09=15, ai_10=15, ai_11=15, ai_12=15, ai_13=15, ai_14=15, ai_15=15, ai_16=15, ai_17=15, ai_18=15, ai_19=15, ai_20=15, ai_21=15, ai_22=15, ai_23=15, ai_24=15, ai_25=15, ai_26=15, ai_27=15, ai_28=15, ai_29=15, ai_30=15, ai_31=15, ai_32=15, ai_33=0, ai_34=0, ai_35=0, ai_36=0, ai_37=0, ai_38=0, ai_39=0, ai_40=0, ai_41=0, ai_42=0, ai_43=0, ai_44=0, ai_45=0, ai_46=0, ai_47=0, ai_48=0, ai_49=0, ai_50=0, ai_51=0)>",
            "<SBF(GEODegrFactors, TOW=14:02:02, WNc=2367, PRN=136, Reserved=0, Brrc=0.0, Cltc_lsb=0.358, Cltc_v1=0.0078000000000000005, Iltc_v1=120, Cltc_v0=0.9580000000000001, Iltc_v0=120, Cgeo_lsb=0.177, Cgeo_v=0.02, Igeo=120, Cer=0.0, Ciono_step=0.657, Iiono=300, Ciono_ramp=0.0, RSSudre=0, RSSiono=0, Reserved2=0, Ccovariance=0.0)>",
            "<SBF(GALNav, TOW=14:02:02, WNc=2367, SVID=106, Source=16, SQRT_A=5440.6060581207275, M_0=0.4124071183614433, e=0.0002392324386164546, i_0=0.318121699616313, omega=0.09772101603448391, OMEGA_0=0.6469861310906708, OMEGADOT=-1.7097363524953835e-09, IDOT=-1.1482370609883219e-10, DEL_N=8.282086128019728e-10, C_uc=-5.245208740234375e-06, C_us=1.002289354801178e-05, C_rc=145.15625, C_rs=-112.125, C_ic=-1.30385160446167e-08, C_is=8.381903171539307e-08, t_oe=481800, t_oc=481800, a_f2=0.0, a_f1=-5.4143356464919634e-12, a_f0=-0.0003986151423305273, WNt_oe=2367, WNt_oc=2367, IODnav=41, L1-BHealth=0, L1-Bdvs=0, L1-Bhs=0, E5bHealth=0, E5bdvs=0, E5bhs=0, E5aHealth=1, E5advs=0, E5ahs=0, Reserved=0, Health_PRS=0, SISA_L1E5a=107, SISA_L1E5b=255, SISA_L1AE6A=255, BGD_L1E5a=4.423782229423523e-09, BGD_L1E5b=-20000000000.0, BGD_L1AE6A=-20000000000.0, CNAVenc=255)>",
            "<SBF(GALUtc, TOW=14:02:02, WNc=2367, SVID=106, Source=16, A_1=0.0, A_0=-9.313225746154785e-10, t_ot=432000, WN_ot=63, DEL_t_LS=18, WN_LSF=137, DN=7, DEL_t_LSF=18)>",
            "<SBF(GALGstGps, TOW=14:02:02, WNc=2367, SVID=106, Source=16, A_1G=3.9968028886505635e-15, A_0G=1.2514647096395493e-09, t_oG=518400, WN_oG=63)>",
            "<SBF(GPSAlm, TOW=14:02:06, WNc=2367, PRN=32, Reserved=0, e=0.008572578430175781, t_oa=589824, delta_i=0.0071258544921875, OMEGADOT=-2.5247572921216488e-09, SQRT_A=5153.54931640625, OMEGA_0=-0.7277947664260864, omega=-0.6548347473144531, M_0=0.07362782955169678, a_f1=1.0913936421275139e-11, a_f0=-0.0004425048828125, WN_a=63, config=11, health8=0, health6=0)>",
            "<SBF(GEOAlm, TOW=14:02:08, WNc=2367, PRN=121, Reserved0=0, DataID=0, Reserved1=0, Health=17, t_oa=1432044160, Xg=42005600.0, Yg=-3673800.0, Zg=0.0, Xgd=0.0, Ygd=0.0, Zgd=0.0)>",
            "<SBF(GALIon, TOW=14:02:09, WNc=2367, SVID=106, Source=2, a_i0=124.75, a_i1=0.53125, a_i2=0.010009765625, StormSF5=0, StormSF4=0, StormSF3=0, StormSF2=0, StormSF1=0, Reserved=0)>",
            "<SBF(GALSARRLM, TOW=14:02:15, WNc=2367, SVID=76, Source=2, RLMLength=80, Reserved=0, RLMBits_01=2617408813, RLMBits_02=1661929999, RLMBits_03=1828913152)>",
            "<SBF(GLOTime, TOW=14:02:10, WNc=2367, SVID=56, FreqNr=11, N_4=8, KP=0, N=508, tau_GPS=1.862645149230957e-09, tau_c=-2.7939677238464355e-09, B1=0.029296875, B2=-0.0005950927734375)>",
        )
        i = 0
        with open(
            os.path.join(DIRNAME, "pygpsdata_x5_constellations.log"), "rb"
        ) as stream:
            ubr = SBFReader(stream, protfilter=7, quitonerror=ERR_RAISE)
            for raw, parsed in ubr:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
            self.assertEqual(i, len(EXPECTED_RESULTS))

    def testrawnav(self):  # test RAW navigation message stream
        EXPECTED_RESULTS = (
            "<SBF(GALRawINAV, TOW=13:59:57, WNc=2367, SVID=106, CRCPassed=1, ViterbiCnt=0, SigIdx=17, L1BCFlag=0, Reserved1=0, FreqNr=0, RxChannel=14, NavBits_01=326764596, NavBits_02=2205928546, NavBits_03=3297251616, NavBits_04=198602576, NavBits_05=3672834480, NavBits_06=2968864772, NavBits_07=1577778144, NavBits_08=1073741824)>",
            "<SBF(GEORawL1, TOW=13:59:57, WNc=2367, SVID=136, CRCPassed=1, ViterbiCnt=0, SigIdx=24, Reserved1=0, FreqNr=0, RxChannel=30, NavBits_01=3323133892, NavBits_02=1023, NavBits_03=4293394432, NavBits_04=0, NavBits_05=0, NavBits_06=244653, NavBits_07=2071986176, NavBits_08=231025024)>",
            "<SBF(BDSRaw, TOW=13:59:57.200000, WNc=2367, SVID=145, CRCPassed=1, ViterbiCnt=0, SigIdx=28, Reserved=0, RxChannel=55, NavBits_01=3795885419, NavBits_02=402815806, NavBits_03=3565563104, NavBits_04=36864, NavBits_05=265982, NavBits_06=458752, NavBits_07=0, NavBits_08=29, NavBits_09=3693724160, NavBits_10=1888485377)>",
            "<SBF(GLORawCA, TOW=13:59:58, WNc=2367, SVID=56, CRCPassed=1, ViterbiCnt=0, SigIdx=8, Reserved1=0, FreqNr=11, RxChannel=13, NavBits_01=1881049536, NavBits_02=0, NavBits_03=307200)>",
            "<SBF(GPSRawCA, TOW=14:00:00, WNc=2367, SVID=1, CRCPassed=1, ViterbiCnt=0, SigIdx=0, Reserved1=0, FreqNr=0, RxChannel=47, NavBits_01=583089936, NavBits_02=658664388, NavBits_03=1073337409, NavBits_04=155673825, NavBits_05=346565, NavBits_06=92672818, NavBits_07=109806032, NavBits_08=813744081, NavBits_09=1072285882, NavBits_10=16567612)>",
            "<SBF(GALRawFNAV, TOW=14:00:02, WNc=2367, SVID=106, CRCPassed=1, ViterbiCnt=0, SigIdx=20, Reserved1=0, FreqNr=0, RxChannel=14, NavBits_01=136851903, NavBits_02=2674196421, NavBits_03=822091610, NavBits_04=1772750041, NavBits_05=1901252729, NavBits_06=3622827343, NavBits_07=3715213574, NavBits_08=3844341760)>",
            "<SBF(NAVICRaw, TOW=14:00:06, WNc=2367, SVID=217, CRCPassed=1, ViterbiCnt=0, SigIdx=15, Reserved=0, RxChannel=52, NavBits_01=2337178890, NavBits_02=3090840421, NavBits_03=3608675289, NavBits_04=3322358700, NavBits_05=4114498885, NavBits_06=726795260, NavBits_07=811597650, NavBits_08=3779787917, NavBits_09=1910243032, NavBits_10=0)>",
            "<SBF(GPSRawL2C, TOW=14:00:06, WNc=2367, SVID=1, CRCPassed=1, ViterbiCnt=0, SigIdx=3, Reserved1=0, FreqNr=0, RxChannel=47, NavBits_01=2332342736, NavBits_02=2737645714, NavBits_03=2725153221, NavBits_04=2254684168, NavBits_05=4280991752, NavBits_06=3757826038, NavBits_07=597695079, NavBits_08=3223592959, NavBits_09=1983906341, NavBits_10=1911554048)>",
        )
        i = 0
        with open(os.path.join(DIRNAME, "pygpsdata_x5_rawnav.log"), "rb") as stream:
            ubr = SBFReader(stream, protfilter=7, quitonerror=ERR_RAISE)
            for raw, parsed in ubr:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
            self.assertEqual(i, len(EXPECTED_RESULTS))

    def testpvtcart(self):  # test PVT cartesian message stream
        EXPECTED_RESULTS = (
            "<SBF(PVTCartesian, TOW=14:03:23, WNc=2367, Type=6, Reserved1=0, AutoSet=0, 2D=0, Error=0, X=3813176.3935078024, Y=-149599.1631026641, Z=5093600.012439784, Undulation=48.477840423583984, Vx=-0.00013817181752528995, Vy=0.0006453946698457003, Vz=-0.0025749732740223408, COG=-20000000000.0, RxClkBias=0.02527270579840079, RxClkDrift=0.19080539047718048, TimeSystem=0, Datum=0, NrSV=36, Corr_OrbClkUsed=1, Corr_RngUsed=1, Corr_IonoUsed=1, Corr_OrbAccUsed=0, Corr_DO229Active=0, Corr_RTKType=0, Reserved2=0, ReferenceID=123, MeanCorrAge=300, SignalInfo=1344405761, RAIMIntegrity=1, GalHPCAFail=0, GalIonStorm=0, Reserved3=0, NrBases=1, PPPSeedAge=0, Reserved4=0, PPPSeedType=0, Latency=61, HAccuracy=102, VAccuracy=143, BaseARP=0, PhaseCtrOffset=0, Reserved5=8, ARPOffset=1)>",
            "<SBF(PosCovCartesian, TOW=14:03:23, WNc=2367, Type=6, Reserved1=0, AutoSet=0, 2D=0, Error=0, Cov_xx=0.22255674004554749, Cov_yy=0.10354391485452652, Cov_zz=0.4468877613544464, Cov_bb=0.2734336853027344, Cov_xy=-0.05616738647222519, Cov_xz=0.1480366438627243, Cov_xb=0.17288260161876678, Cov_yz=-0.04950747638940811, Cov_yb=-0.06711780279874802, Cov_zb=0.3099549114704132)>",
            "<SBF(VelCovCartesian, TOW=14:03:23, WNc=2367, Type=6, Reserved1=0, AutoSet=0, 2D=0, Error=0, Cov_VxVx=3.1918079912429675e-05, Cov_VyVy=1.866238562797662e-05, Cov_VzVz=4.0098846511682495e-05, Cov_DtDt=3.695358464028686e-05, Cov_VxVy=-4.2343403947597835e-06, Cov_VxVz=9.274422154703643e-06, Cov_VxDt=1.858825453382451e-05, Cov_VyVz=-1.4702366115670884e-06, Cov_VyDt=-2.182556045227102e-06, Cov_VzDt=2.621921339596156e-05)>",
            "<SBF(BaseVectorCart, TOW=14:03:23, WNc=2367, N=0, SBLength=52)>",
        )
        i = 0
        with open(os.path.join(DIRNAME, "pygpsdata_x5_pvtcart.log"), "rb") as stream:
            ubr = SBFReader(stream, protfilter=7, quitonerror=ERR_RAISE)
            for raw, parsed in ubr:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
            self.assertEqual(i, len(EXPECTED_RESULTS))

    def testpvtgeod(self):  # test PVT geodetic message stream
        EXPECTED_RESULTS = (
            "<SBF(PVTGeodetic, TOW=14:07:09, WNc=2367, Type=6, Reserved1=0, AutoSet=0, 2D=0, Error=0, Latitude=0.9310293523340808, Longitude=-0.03921206770879602, Height=131.18596542546626, Undulation=48.477840423583984, Vn=0.0012262271484360099, Ve=-0.0007957226480357349, Vu=0.0004817131848540157, COG=-20000000000.0, RxClkBias=0.0693948459476198, RxClkDrift=0.1977977603673935, TimeSystem=0, Datum=0, NrSV=36, Corr_OrbClkUsed=1, Corr_RngUsed=1, Corr_IonoUsed=1, Corr_OrbAccUsed=0, Corr_DO229Active=0, Corr_RTKType=0, Reserved2=0, ReferenceID=123, MeanCorrAge=278, SignalInfo=1344405761, RAIMIntegrity=1, GalHPCAFail=0, GalIonStorm=0, Reserved3=0, NrBases=1, PPPSeedAge=0, Reserved4=0, PPPSeedType=0, Latency=55, HAccuracy=102, VAccuracy=138, BaseARP=0, PhaseCtrOffset=0, Reserved5=8, ARPOffset=1)>",
            "<SBF(PosCovGeodetic, TOW=14:07:09, WNc=2367, Type=6, Reserved1=0, AutoSet=0, 2D=0, Error=0, Cov_latlat=0.16179977357387543, Cov_lonlon=0.09827368706464767, Cov_hgthgt=0.47691863775253296, Cov_bb=0.2628936767578125, Cov_latlon=0.014659467153251171, Cov_lathgt=0.052513487637043, Cov_latb=0.03550632670521736, Cov_lonhgt=-0.0588376559317112, Cov_lonb=-0.053778938949108124, Cov_hb=0.3346599340438843)>",
            "<SBF(VelCovGeodetic, TOW=14:07:09, WNc=2367, Type=6, Reserved1=0, AutoSet=0, 2D=0, Error=0, Cov_VnVn=2.607271926535759e-05, Cov_VeVe=1.854929178080056e-05, Cov_VuVu=4.582880865200423e-05, Cov_DtDt=3.6877550883218646e-05, Cov_VnVe=2.313254071850679e-06, Cov_VnVu=1.2074295909769717e-06, Cov_VnDt=7.145455924728594e-07, Cov_VeVu=-2.9310365334822563e-06, Cov_VeDt=-1.1441543392720632e-06, Cov_VuDt=3.1944739021128044e-05)>",
            "<SBF(BaseVectorGeod, TOW=14:07:09, WNc=2367, N=0, SBLength=52)>",
            "<SBF(PosLocal, TOW=14:07:09, WNc=2367, Type=6, Reserved1=0, AutoSet=0, 2D=0, Error=17, Lat=-20000000000.0, Lon=-20000000000.0, Alt=-20000000000.0, Datum=255)>",
        )
        i = 0
        with open(os.path.join(DIRNAME, "pygpsdata_x5_pvtgeod.log"), "rb") as stream:
            ubr = SBFReader(stream, protfilter=7, quitonerror=ERR_RAISE)
            for raw, parsed in ubr:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
            self.assertEqual(i, len(EXPECTED_RESULTS))

    def testpvtgeodnoparse(self):  # test PVT geodetic message stream
        EXPECTED_RESULTS = (
            "None",
            "None",
            "None",
            "None",
            "None",
        )
        i = 0
        with open(os.path.join(DIRNAME, "pygpsdata_x5_pvtgeod.log"), "rb") as stream:
            ubr = SBFReader(stream, protfilter=7, quitonerror=ERR_RAISE, parsing=False)
            for raw, parsed in ubr:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
            self.assertEqual(i, len(EXPECTED_RESULTS))

    def testpvtextra(self):  # test PVT extra message stream
        EXPECTED_RESULTS = (
            "<SBF(DOP, TOW=14:08:58, WNc=2367, NrSV=37, Reserved=0, PDOP=79, TDOP=40, HDOP=44, VDOP=66, HPL=3.88915753364563, VPL=5.674680709838867)>",
            "<SBF(EndOfPVT, TOW=14:08:58, WNc=2367)>",
            "<SBF(PVTSupport, NOT YET IMPLEMENTED)>",
            "<SBF(PVTSupportA, NOT YET IMPLEMENTED)>",
        )
        i = 0
        with open(os.path.join(DIRNAME, "pygpsdata_x5_pvtextra.log"), "rb") as stream:
            ubr = SBFReader(stream, protfilter=7, quitonerror=ERR_RAISE)
            for raw, parsed in ubr:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
            self.assertEqual(i, len(EXPECTED_RESULTS))

    def teststatus(self):  # test status message stream
        EXPECTED_RESULTS = (
            "<SBF(ChannelStatus, TOW=14:12:32, WNc=2367, N=58, SB1Length=12, SB2Length=8, Reserved=0, SVID_01=17, FreqNr_01=0, Reserved1_01=0, Azimuth_01=300, Reserved5_01=0, RiseSet_01=1, HealthStatus_01=81, Elevation_01=42, N2_01=1, RxChannel_01=1, Reserved2_01=0, Antenna_01_01=0, Reserved4_01_01=0, TrackingStatus_01_01=243, PVTStatus_01_01=2, PVTInfo_01_01=0, SVID_02=14, FreqNr_02=0, Reserved1_02=0, Azimuth_02=264, Reserved5_02=0, RiseSet_02=0, HealthStatus_02=81, Elevation_02=21, N2_02=1, RxChannel_02=2, Reserved2_02=0, Antenna_02_01=0, Reserved4_02_01=0, TrackingStatus_02_01=243, PVTStatus_02_01=2, PVTInfo_02_01=0, SVID_03=97, FreqNr_03=0, Reserved1_03=0, Azimuth_03=334, Reserved5_03=0, RiseSet_03=1, HealthStatus_03=1284, Elevation_03=12, N2_03=1, RxChannel_03=3, Reserved2_03=0, Antenna_03_01=0, Reserved4_03_01=0, TrackingStatus_03_01=3852, PVTStatus_03_01=2056, PVTInfo_03_01=0, SVID_04=80, FreqNr_04=0, Reserved1_04=0, Azimuth_04=97, Reserved5_04=0, RiseSet_04=0, HealthStatus_04=1284, Elevation_04=4, N2_04=1, RxChannel_04=4, Reserved2_04=0, Antenna_04_01=0, Reserved4_04_01=0, TrackingStatus_04_01=3844, PVTStatus_04_01=0, PVTInfo_04_01=0, SVID_05=48, FreqNr_05=8, Reserved1_05=0, Azimuth_05=332, Reserved5_05=0, RiseSet_05=0, HealthStatus_05=65, Elevation_05=18, N2_05=1, RxChannel_05=5, Reserved2_05=0, Antenna_05_01=0, Reserved4_05_01=0, TrackingStatus_05_01=195, PVTStatus_05_01=2, PVTInfo_05_01=0, SVID_06=22, FreqNr_06=0, Reserved1_06=0, Azimuth_06=285, Reserved5_06=0, RiseSet_06=0, HealthStatus_06=17, Elevation_06=26, N2_06=1, RxChannel_06=6, Reserved2_06=0, Antenna_06_01=0, Reserved4_06_01=0, TrackingStatus_06_01=51, PVTStatus_06_01=2, PVTInfo_06_01=0, SVID_07=42, FreqNr_07=9, Reserved1_07=0, Azimuth_07=189, Reserved5_07=0, RiseSet_07=1, HealthStatus_07=65, Elevation_07=19, N2_07=1, RxChannel_07=7, Reserved2_07=0, Antenna_07_01=0, Reserved4_07_01=0, TrackingStatus_07_01=195, PVTStatus_07_01=2, PVTInfo_07_01=0, SVID_08=72, FreqNr_08=0, Reserved1_08=0, Azimuth_08=229, Reserved5_08=0, RiseSet_08=0, HealthStatus_08=1284, Elevation_08=17, N2_08=1, RxChannel_08=8, Reserved2_08=0, Antenna_08_01=0, Reserved4_08_01=0, TrackingStatus_08_01=3852, PVTStatus_08_01=8, PVTInfo_08_01=0, SVID_09=75, FreqNr_09=0, Reserved1_09=0, Azimuth_09=171, Reserved5_09=0, RiseSet_09=1, HealthStatus_09=1284, Elevation_09=9, N2_09=1, RxChannel_09=9, Reserved2_09=0, Antenna_09_01=0, Reserved4_09_01=0, TrackingStatus_09_01=3852, PVTStatus_09_01=520, PVTInfo_09_01=0, SVID_10=47, FreqNr_10=1, Reserved1_10=0, Azimuth_10=282, Reserved5_10=0, RiseSet_10=0, HealthStatus_10=65, Elevation_10=13, N2_10=1, RxChannel_10=10, Reserved2_10=0, Antenna_10_01=0, Reserved4_10_01=0, TrackingStatus_10_01=67, PVTStatus_10_01=2, PVTInfo_10_01=0, SVID_11=173, FreqNr_11=0, Reserved1_11=0, Azimuth_11=193, Reserved5_11=0, RiseSet_11=1, HealthStatus_11=17, Elevation_11=37, N2_11=1, RxChannel_11=11, Reserved2_11=0, Antenna_11_01=0, Reserved4_11_01=0, TrackingStatus_11_01=51, PVTStatus_11_01=2, PVTInfo_11_01=0, SVID_12=104, FreqNr_12=0, Reserved1_12=0, Azimuth_12=283, Reserved5_12=0, RiseSet_12=1, HealthStatus_12=1284, Elevation_12=41, N2_12=1, RxChannel_12=12, Reserved2_12=0, Antenna_12_01=0, Reserved4_12_01=0, TrackingStatus_12_01=3852, PVTStatus_12_01=8, PVTInfo_12_01=0, SVID_13=56, FreqNr_13=11, Reserved1_13=0, Azimuth_13=76, Reserved5_13=0, RiseSet_13=0, HealthStatus_13=65, Elevation_13=79, N2_13=1, RxChannel_13=13, Reserved2_13=0, Antenna_13_01=0, Reserved4_13_01=0, TrackingStatus_13_01=195, PVTStatus_13_01=2, PVTInfo_13_01=0, SVID_14=106, FreqNr_14=0, Reserved1_14=0, Azimuth_14=66, Reserved5_14=0, RiseSet_14=0, HealthStatus_14=1284, Elevation_14=82, N2_14=1, RxChannel_14=14, Reserved2_14=0, Antenna_14_01=0, Reserved4_14_01=0, TrackingStatus_14_01=3852, PVTStatus_14_01=8, PVTInfo_14_01=0, SVID_15=174, FreqNr_15=0, Reserved1_15=0, Azimuth_15=132, Reserved5_15=0, RiseSet_15=0, HealthStatus_15=17, Elevation_15=14, N2_15=1, RxChannel_15=15, Reserved2_15=0, Antenna_15_01=0, Reserved4_15_01=0, TrackingStatus_15_01=51, PVTStatus_15_01=2, PVTInfo_15_01=0, SVID_16=32, FreqNr_16=0, Reserved1_16=0, Azimuth_16=49, Reserved5_16=0, RiseSet_16=0, HealthStatus_16=81, Elevation_16=28, N2_16=1, RxChannel_16=16, Reserved2_16=0, Antenna_16_01=0, Reserved4_16_01=0, TrackingStatus_16_01=243, PVTStatus_16_01=2, PVTInfo_16_01=0, SVID_17=79, FreqNr_17=0, Reserved1_17=0, Azimuth_17=124, Reserved5_17=0, RiseSet_17=1, HealthStatus_17=1284, Elevation_17=39, N2_17=1, RxChannel_17=17, Reserved2_17=0, Antenna_17_01=0, Reserved4_17_01=0, TrackingStatus_17_01=3852, PVTStatus_17_01=8, PVTInfo_17_01=0, SVID_18=123, FreqNr_18=0, Reserved1_18=0, Azimuth_18=149, Reserved5_18=0, RiseSet_18=0, HealthStatus_18=0, Elevation_18=25, N2_18=1, RxChannel_18=18, Reserved2_18=0, Antenna_18_01=0, Reserved4_18_01=0, TrackingStatus_18_01=3, PVTStatus_18_01=0, PVTInfo_18_01=0, SVID_19=81, FreqNr_19=0, Reserved1_19=0, Azimuth_19=95, Reserved5_19=0, RiseSet_19=0, HealthStatus_19=1284, Elevation_19=30, N2_19=1, RxChannel_19=19, Reserved2_19=0, Antenna_19_01=0, Reserved4_19_01=0, TrackingStatus_19_01=3852, PVTStatus_19_01=8, PVTInfo_19_01=0, SVID_20=40, FreqNr_20=13, Reserved1_20=0, Azimuth_20=47, Reserved5_20=0, RiseSet_20=0, HealthStatus_20=65, Elevation_20=48, N2_20=1, RxChannel_20=20, Reserved2_20=0, Antenna_20_01=0, Reserved4_20_01=0, TrackingStatus_20_01=195, PVTStatus_20_01=2, PVTInfo_20_01=0, SVID_21=151, FreqNr_21=0, Reserved1_21=0, Azimuth_21=132, Reserved5_21=0, RiseSet_21=0, HealthStatus_21=21, Elevation_21=39, N2_21=1, RxChannel_21=21, Reserved2_21=0, Antenna_21_01=0, Reserved4_21_01=0, TrackingStatus_21_01=63, PVTStatus_21_01=2, PVTInfo_21_01=0, SVID_22=28, FreqNr_22=0, Reserved1_22=0, Azimuth_22=85, Reserved5_22=0, RiseSet_22=1, HealthStatus_22=17, Elevation_22=3, N2_22=1, RxChannel_22=22, Reserved2_22=0, Antenna_22_01=0, Reserved4_22_01=0, TrackingStatus_22_01=17, PVTStatus_22_01=0, PVTInfo_22_01=0, SVID_23=100, FreqNr_23=0, Reserved1_23=0, Azimuth_23=284, Reserved5_23=0, RiseSet_23=0, HealthStatus_23=1284, Elevation_23=28, N2_23=1, RxChannel_23=23, Reserved2_23=0, Antenna_23_01=0, Reserved4_23_01=0, TrackingStatus_23_01=3852, PVTStatus_23_01=8, PVTInfo_23_01=0, SVID_24=41, FreqNr_24=14, Reserved1_24=0, Azimuth_24=150, Reserved5_24=0, RiseSet_24=1, HealthStatus_24=65, Elevation_24=62, N2_24=1, RxChannel_24=24, Reserved2_24=0, Antenna_24_01=0, Reserved4_24_01=0, TrackingStatus_24_01=195, PVTStatus_24_01=2, PVTInfo_24_01=0, SVID_25=224, FreqNr_25=0, Reserved1_25=0, Azimuth_25=101, Reserved5_25=0, RiseSet_25=0, HealthStatus_25=17, Elevation_25=66, N2_25=1, RxChannel_25=25, Reserved2_25=0, Antenna_25_01=0, Reserved4_25_01=0, TrackingStatus_25_01=51, PVTStatus_25_01=2, PVTInfo_25_01=0, SVID_26=225, FreqNr_26=0, Reserved1_26=0, Azimuth_26=151, Reserved5_26=0, RiseSet_26=0, HealthStatus_26=17, Elevation_26=63, N2_26=1, RxChannel_26=26, Reserved2_26=0, Antenna_26_01=0, Reserved4_26_01=0, TrackingStatus_26_01=51, PVTStatus_26_01=2, PVTInfo_26_01=0, SVID_27=4, FreqNr_27=0, Reserved1_27=0, Azimuth_27=178, Reserved5_27=0, RiseSet_27=1, HealthStatus_27=17, Elevation_27=5, N2_27=1, RxChannel_27=27, Reserved2_27=0, Antenna_27_01=0, Reserved4_27_01=0, TrackingStatus_27_01=17, PVTStatus_27_01=0, PVTInfo_27_01=0, SVID_28=168, FreqNr_28=0, Reserved1_28=0, Azimuth_28=295, Reserved5_28=0, RiseSet_28=1, HealthStatus_28=17, Elevation_28=48, N2_28=1, RxChannel_28=28, Reserved2_28=0, Antenna_28_01=0, Reserved4_28_01=0, TrackingStatus_28_01=51, PVTStatus_28_01=2, PVTInfo_28_01=0, SVID_29=121, FreqNr_29=0, Reserved1_29=0, Azimuth_29=183, Reserved5_29=0, RiseSet_29=0, HealthStatus_29=0, Elevation_29=29, N2_29=1, RxChannel_29=29, Reserved2_29=0, Antenna_29_01=0, Reserved4_29_01=0, TrackingStatus_29_01=3, PVTStatus_29_01=0, PVTInfo_29_01=0, SVID_30=136, FreqNr_30=0, Reserved1_30=0, Azimuth_30=171, Reserved5_30=0, RiseSet_30=0, HealthStatus_30=0, Elevation_30=29, N2_30=1, RxChannel_30=30, Reserved2_30=0, Antenna_30_01=0, Reserved4_30_01=0, TrackingStatus_30_01=3, PVTStatus_30_01=0, PVTInfo_30_01=0, SVID_31=8, FreqNr_31=0, Reserved1_31=0, Azimuth_31=164, Reserved5_31=0, RiseSet_31=0, HealthStatus_31=81, Elevation_31=19, N2_31=1, RxChannel_31=31, Reserved2_31=0, Antenna_31_01=0, Reserved4_31_01=0, TrackingStatus_31_01=243, PVTStatus_31_01=2, PVTInfo_31_01=0, SVID_32=49, FreqNr_32=7, Reserved1_32=0, Azimuth_32=24, Reserved5_32=0, RiseSet_32=1, HealthStatus_32=1, Elevation_32=1, N2_32=1, RxChannel_32=32, Reserved2_32=0, Antenna_32_01=0, Reserved4_32_01=0, TrackingStatus_32_01=1, PVTStatus_32_01=0, PVTInfo_32_01=0, SVID_33=240, FreqNr_33=0, Reserved1_33=0, Azimuth_33=511, Reserved5_33=0, RiseSet_33=3, HealthStatus_33=0, Elevation_33=-128, N2_33=1, RxChannel_33=33, Reserved2_33=0, Antenna_33_01=0, Reserved4_33_01=0, TrackingStatus_33_01=51, PVTStatus_33_01=0, PVTInfo_33_01=0, SVID_34=179, FreqNr_34=0, Reserved1_34=0, Azimuth_34=53, Reserved5_34=0, RiseSet_34=1, HealthStatus_34=17, Elevation_34=15, N2_34=1, RxChannel_34=34, Reserved2_34=0, Antenna_34_01=0, Reserved4_34_01=0, TrackingStatus_34_01=51, PVTStatus_34_01=2, PVTInfo_34_01=0, SVID_35=55, FreqNr_35=5, Reserved1_35=0, Azimuth_35=116, Reserved5_35=0, RiseSet_35=0, HealthStatus_35=65, Elevation_35=27, N2_35=1, RxChannel_35=35, Reserved2_35=0, Antenna_35_01=0, Reserved4_35_01=0, TrackingStatus_35_01=195, PVTStatus_35_01=2, PVTInfo_35_01=0, SVID_36=74, FreqNr_36=0, Reserved1_36=0, Azimuth_36=59, Reserved5_36=0, RiseSet_36=0, HealthStatus_36=1284, Elevation_36=34, N2_36=1, RxChannel_36=36, Reserved2_36=0, Antenna_36_01=0, Reserved4_36_01=0, TrackingStatus_36_01=3852, PVTStatus_36_01=8, PVTInfo_36_01=0, SVID_37=154, FreqNr_37=0, Reserved1_37=0, Azimuth_37=168, Reserved5_37=0, RiseSet_37=1, HealthStatus_37=21, Elevation_37=59, N2_37=1, RxChannel_37=37, Reserved2_37=0, Antenna_37_01=0, Reserved4_37_01=0, TrackingStatus_37_01=63, PVTStatus_37_01=2, PVTInfo_37_01=0, SVID_38=57, FreqNr_38=10, Reserved1_38=0, Azimuth_38=309, Reserved5_38=0, RiseSet_38=1, HealthStatus_38=65, Elevation_38=37, N2_38=1, RxChannel_38=38, Reserved2_38=0, Antenna_38_01=0, Reserved4_38_01=0, TrackingStatus_38_01=195, PVTStatus_38_01=0, PVTInfo_38_01=0, SVID_39=205, FreqNr_39=0, Reserved1_39=0, Azimuth_39=511, Reserved5_39=0, RiseSet_39=3, HealthStatus_39=0, Elevation_39=-128, N2_39=1, RxChannel_39=39, Reserved2_39=0, Antenna_39_01=0, Reserved4_39_01=0, TrackingStatus_39_01=3, PVTStatus_39_01=0, PVTInfo_39_01=0, SVID_40=161, FreqNr_40=0, Reserved1_40=0, Azimuth_40=47, Reserved5_40=0, RiseSet_40=0, HealthStatus_40=17, Elevation_40=26, N2_40=1, RxChannel_40=40, Reserved2_40=0, Antenna_40_01=0, Reserved4_40_01=0, TrackingStatus_40_01=51, PVTStatus_40_01=2, PVTInfo_40_01=0, SVID_41=19, FreqNr_41=0, Reserved1_41=0, Azimuth_41=314, Reserved5_41=0, RiseSet_41=1, HealthStatus_41=17, Elevation_41=16, N2_41=1, RxChannel_41=41, Reserved2_41=0, Antenna_41_01=0, Reserved4_41_01=0, TrackingStatus_41_01=51, PVTStatus_41_01=2, PVTInfo_41_01=0, SVID_42=76, FreqNr_42=0, Reserved1_42=0, Azimuth_42=91, Reserved5_42=0, RiseSet_42=1, HealthStatus_42=1284, Elevation_42=42, N2_42=1, RxChannel_42=42, Reserved2_42=0, Antenna_42_01=0, Reserved4_42_01=0, TrackingStatus_42_01=3852, PVTStatus_42_01=8, PVTInfo_42_01=0, SVID_43=84, FreqNr_43=0, Reserved1_43=0, Azimuth_43=301, Reserved5_43=0, RiseSet_43=1, HealthStatus_43=3852, Elevation_43=29, N2_43=1, RxChannel_43=43, Reserved2_43=0, Antenna_43_01=0, Reserved4_43_01=0, TrackingStatus_43_01=3852, PVTStatus_43_01=0, PVTInfo_43_01=0, SVID_44=149, FreqNr_44=0, Reserved1_44=0, Azimuth_44=81, Reserved5_44=0, RiseSet_44=1, HealthStatus_44=0, Elevation_44=5, N2_44=1, RxChannel_44=44, Reserved2_44=0, Antenna_44_01=0, Reserved4_44_01=0, TrackingStatus_44_01=1, PVTStatus_44_01=0, PVTInfo_44_01=0, SVID_45=146, FreqNr_45=0, Reserved1_45=0, Azimuth_45=68, Reserved5_45=0, RiseSet_45=1, HealthStatus_45=21, Elevation_45=10, N2_45=1, RxChannel_45=45, Reserved2_45=0, Antenna_45_01=0, Reserved4_45_01=0, TrackingStatus_45_01=61, PVTStatus_45_01=8, PVTInfo_45_01=0, SVID_46=156, FreqNr_46=0, Reserved1_46=0, Azimuth_46=62, Reserved5_46=0, RiseSet_46=1, HealthStatus_46=21, Elevation_46=11, N2_46=1, RxChannel_46=46, Reserved2_46=0, Antenna_46_01=0, Reserved4_46_01=0, TrackingStatus_46_01=63, PVTStatus_46_01=10, PVTInfo_46_01=0, SVID_47=1, FreqNr_47=0, Reserved1_47=0, Azimuth_47=61, Reserved5_47=0, RiseSet_47=0, HealthStatus_47=81, Elevation_47=88, N2_47=1, RxChannel_47=47, Reserved2_47=0, Antenna_47_01=0, Reserved4_47_01=0, TrackingStatus_47_01=243, PVTStatus_47_01=2, PVTInfo_47_01=0, SVID_48=2, FreqNr_48=0, Reserved1_48=0, Azimuth_48=108, Reserved5_48=0, RiseSet_48=0, HealthStatus_48=17, Elevation_48=57, N2_48=1, RxChannel_48=48, Reserved2_48=0, Antenna_48_01=0, Reserved4_48_01=0, TrackingStatus_48_01=51, PVTStatus_48_01=2, PVTInfo_48_01=0, SVID_49=3, FreqNr_49=0, Reserved1_49=0, Azimuth_49=223, Reserved5_49=0, RiseSet_49=1, HealthStatus_49=81, Elevation_49=62, N2_49=1, RxChannel_49=49, Reserved2_49=0, Antenna_49_01=0, Reserved4_49_01=0, TrackingStatus_49_01=243, PVTStatus_49_01=2, PVTInfo_49_01=0, SVID_50=231, FreqNr_50=0, Reserved1_50=0, Azimuth_50=256, Reserved5_50=0, RiseSet_50=1, HealthStatus_50=51, Elevation_50=78, N2_50=1, RxChannel_50=50, Reserved2_50=0, Antenna_50_01=0, Reserved4_50_01=0, TrackingStatus_50_01=51, PVTStatus_50_01=0, PVTInfo_50_01=0, SVID_51=232, FreqNr_51=0, Reserved1_51=0, Azimuth_51=62, Reserved5_51=0, RiseSet_51=0, HealthStatus_51=51, Elevation_51=46, N2_51=1, RxChannel_51=51, Reserved2_51=0, Antenna_51_01=0, Reserved4_51_01=0, TrackingStatus_51_01=51, PVTStatus_51_01=0, PVTInfo_51_01=0, SVID_52=217, FreqNr_52=0, Reserved1_52=0, Azimuth_52=511, Reserved5_52=0, RiseSet_52=3, HealthStatus_52=0, Elevation_52=-128, N2_52=1, RxChannel_52=52, Reserved2_52=0, Antenna_52_01=0, Reserved4_52_01=0, TrackingStatus_52_01=3, PVTStatus_52_01=0, PVTInfo_52_01=0, SVID_53=167, FreqNr_53=0, Reserved1_53=0, Azimuth_53=304, Reserved5_53=0, RiseSet_53=1, HealthStatus_53=0, Elevation_53=6, N2_53=1, RxChannel_53=53, Reserved2_53=0, Antenna_53_01=0, Reserved4_53_01=0, TrackingStatus_53_01=1, PVTStatus_53_01=0, PVTInfo_53_01=0, SVID_54=153, FreqNr_54=0, Reserved1_54=0, Azimuth_54=57, Reserved5_54=0, RiseSet_54=0, HealthStatus_54=0, Elevation_54=3, N2_54=1, RxChannel_54=54, Reserved2_54=0, Antenna_54_01=0, Reserved4_54_01=0, TrackingStatus_54_01=1, PVTStatus_54_01=0, PVTInfo_54_01=0, SVID_55=145, FreqNr_55=0, Reserved1_55=0, Azimuth_55=113, Reserved5_55=0, RiseSet_55=1, HealthStatus_55=21, Elevation_55=10, N2_55=1, RxChannel_55=55, Reserved2_55=0, Antenna_55_01=0, Reserved4_55_01=0, TrackingStatus_55_01=63, PVTStatus_55_01=34, PVTInfo_55_01=0, SVID_56=139, FreqNr_56=0, Reserved1_56=0, Azimuth_56=511, Reserved5_56=0, RiseSet_56=3, HealthStatus_56=0, Elevation_56=-128, N2_56=1, RxChannel_56=56, Reserved2_56=0, Antenna_56_01=0, Reserved4_56_01=0, TrackingStatus_56_01=1, PVTStatus_56_01=0, PVTInfo_56_01=0, SVID_57=140, FreqNr_57=0, Reserved1_57=0, Azimuth_57=511, Reserved5_57=0, RiseSet_57=3, HealthStatus_57=0, Elevation_57=-128, N2_57=1, RxChannel_57=59, Reserved2_57=0, Antenna_57_01=0, Reserved4_57_01=0, TrackingStatus_57_01=1, PVTStatus_57_01=0, PVTInfo_57_01=0, SVID_58=191, FreqNr_58=0, Reserved1_58=0, Azimuth_58=511, Reserved5_58=0, RiseSet_58=3, HealthStatus_58=0, Elevation_58=-128, N2_58=1, RxChannel_58=60, Reserved2_58=0, Antenna_58_01=0, Reserved4_58_01=0, TrackingStatus_58_01=3, PVTStatus_58_01=0, PVTInfo_58_01=0)>",
            "<SBF(SatVisibility, TOW=14:12:32, WNc=2367, N=52, SBLength=8, SVID_01=1, FreqNr_01=0, Azimuth_01=6137, Elevation_01=8786, RiseSet_01=0, SatelliteInfo_01=2, SVID_02=8, FreqNr_02=0, Azimuth_02=16422, Elevation_02=1905, RiseSet_02=0, SatelliteInfo_02=2, SVID_03=19, FreqNr_03=0, Azimuth_03=31424, Elevation_03=1614, RiseSet_03=1, SatelliteInfo_03=2, SVID_04=14, FreqNr_04=0, Azimuth_04=26422, Elevation_04=2101, RiseSet_04=0, SatelliteInfo_04=2, SVID_05=32, FreqNr_05=0, Azimuth_05=4903, Elevation_05=2814, RiseSet_05=0, SatelliteInfo_05=2, SVID_06=22, FreqNr_06=0, Azimuth_06=28470, Elevation_06=2577, RiseSet_06=0, SatelliteInfo_06=2, SVID_07=17, FreqNr_07=0, Azimuth_07=30023, Elevation_07=4245, RiseSet_07=1, SatelliteInfo_07=2, SVID_08=3, FreqNr_08=0, Azimuth_08=22288, Elevation_08=6238, RiseSet_08=1, SatelliteInfo_08=2, SVID_09=2, FreqNr_09=0, Azimuth_09=10771, Elevation_09=5669, RiseSet_09=0, SatelliteInfo_09=2, SVID_10=28, FreqNr_10=0, Azimuth_10=8543, Elevation_10=311, RiseSet_10=1, SatelliteInfo_10=1, SVID_11=4, FreqNr_11=0, Azimuth_11=17755, Elevation_11=502, RiseSet_11=1, SatelliteInfo_11=1, SVID_12=106, FreqNr_12=0, Azimuth_12=6636, Elevation_12=8159, RiseSet_12=0, SatelliteInfo_12=2, SVID_13=97, FreqNr_13=0, Azimuth_13=33387, Elevation_13=1236, RiseSet_13=1, SatelliteInfo_13=2, SVID_14=72, FreqNr_14=0, Azimuth_14=22950, Elevation_14=1698, RiseSet_14=0, SatelliteInfo_14=2, SVID_15=84, FreqNr_15=0, Azimuth_15=30063, Elevation_15=2869, RiseSet_15=1, SatelliteInfo_15=2, SVID_16=100, FreqNr_16=0, Azimuth_16=28392, Elevation_16=2822, RiseSet_16=0, SatelliteInfo_16=2, SVID_17=74, FreqNr_17=0, Azimuth_17=5898, Elevation_17=3380, RiseSet_17=0, SatelliteInfo_17=2, SVID_18=79, FreqNr_18=0, Azimuth_18=12406, Elevation_18=3928, RiseSet_18=1, SatelliteInfo_18=2, SVID_19=81, FreqNr_19=0, Azimuth_19=9546, Elevation_19=3020, RiseSet_19=0, SatelliteInfo_19=2, SVID_20=76, FreqNr_20=0, Azimuth_20=9113, Elevation_20=4176, RiseSet_20=1, SatelliteInfo_20=2, SVID_21=104, FreqNr_21=0, Azimuth_21=28277, Elevation_21=4114, RiseSet_21=1, SatelliteInfo_21=2, SVID_22=80, FreqNr_22=0, Azimuth_22=9743, Elevation_22=440, RiseSet_22=0, SatelliteInfo_22=2, SVID_23=75, FreqNr_23=0, Azimuth_23=17055, Elevation_23=931, RiseSet_23=1, SatelliteInfo_23=2, SVID_24=121, FreqNr_24=0, Azimuth_24=18343, Elevation_24=2904, RiseSet_24=0, SatelliteInfo_24=1, SVID_25=123, FreqNr_25=0, Azimuth_25=14897, Elevation_25=2465, RiseSet_25=0, SatelliteInfo_25=1, SVID_26=136, FreqNr_26=0, Azimuth_26=17099, Elevation_26=2872, RiseSet_26=0, SatelliteInfo_26=1, SVID_27=56, FreqNr_27=11, Azimuth_27=7615, Elevation_27=7897, RiseSet_27=0, SatelliteInfo_27=2, SVID_28=48, FreqNr_28=8, Azimuth_28=33246, Elevation_28=1846, RiseSet_28=0, SatelliteInfo_28=2, SVID_29=42, FreqNr_29=9, Azimuth_29=18864, Elevation_29=1850, RiseSet_29=1, SatelliteInfo_29=2, SVID_30=47, FreqNr_30=1, Azimuth_30=28170, Elevation_30=1264, RiseSet_30=0, SatelliteInfo_30=2, SVID_31=55, FreqNr_31=5, Azimuth_31=11638, Elevation_31=2722, RiseSet_31=0, SatelliteInfo_31=2, SVID_32=40, FreqNr_32=13, Azimuth_32=4723, Elevation_32=4782, RiseSet_32=0, SatelliteInfo_32=2, SVID_33=57, FreqNr_33=10, Azimuth_33=30862, Elevation_33=3723, RiseSet_33=1, SatelliteInfo_33=2, SVID_34=41, FreqNr_34=14, Azimuth_34=15022, Elevation_34=6214, RiseSet_34=1, SatelliteInfo_34=2, SVID_35=49, FreqNr_35=7, Azimuth_35=2403, Elevation_35=59, RiseSet_35=1, SatelliteInfo_35=1, SVID_36=231, FreqNr_36=0, Azimuth_36=25584, Elevation_36=7792, RiseSet_36=1, SatelliteInfo_36=2, SVID_37=174, FreqNr_37=0, Azimuth_37=13248, Elevation_37=1442, RiseSet_37=0, SatelliteInfo_37=2, SVID_38=179, FreqNr_38=0, Azimuth_38=5284, Elevation_38=1472, RiseSet_38=1, SatelliteInfo_38=2, SVID_39=145, FreqNr_39=0, Azimuth_39=11252, Elevation_39=1047, RiseSet_39=1, SatelliteInfo_39=2, SVID_40=173, FreqNr_40=0, Azimuth_40=19346, Elevation_40=3655, RiseSet_40=1, SatelliteInfo_40=2, SVID_41=156, FreqNr_41=0, Azimuth_41=6153, Elevation_41=1145, RiseSet_41=1, SatelliteInfo_41=2, SVID_42=161, FreqNr_42=0, Azimuth_42=4736, Elevation_42=2597, RiseSet_42=0, SatelliteInfo_42=2, SVID_43=151, FreqNr_43=0, Azimuth_43=13175, Elevation_43=3851, RiseSet_43=0, SatelliteInfo_43=2, SVID_44=168, FreqNr_44=0, Azimuth_44=29529, Elevation_44=4784, RiseSet_44=1, SatelliteInfo_44=2, SVID_45=154, FreqNr_45=0, Azimuth_45=16803, Elevation_45=5897, RiseSet_45=1, SatelliteInfo_45=2, SVID_46=232, FreqNr_46=0, Azimuth_46=6244, Elevation_46=4636, RiseSet_46=0, SatelliteInfo_46=2, SVID_47=225, FreqNr_47=0, Azimuth_47=15109, Elevation_47=6279, RiseSet_47=0, SatelliteInfo_47=2, SVID_48=224, FreqNr_48=0, Azimuth_48=10129, Elevation_48=6563, RiseSet_48=0, SatelliteInfo_48=2, SVID_49=149, FreqNr_49=0, Azimuth_49=8051, Elevation_49=526, RiseSet_49=1, SatelliteInfo_49=1, SVID_50=146, FreqNr_50=0, Azimuth_50=6781, Elevation_50=981, RiseSet_50=1, SatelliteInfo_50=2, SVID_51=167, FreqNr_51=0, Azimuth_51=30405, Elevation_51=610, RiseSet_51=1, SatelliteInfo_51=1, SVID_52=153, FreqNr_52=0, Azimuth_52=5666, Elevation_52=336, RiseSet_52=0, SatelliteInfo_52=1)>",
            "<SBF(InputLink, TOW=14:12:32, WNc=2367, N=2, SBLength=20, CD_01=33, Type_01=32, AgeOfLastMessage_01=65535, NrBytesReceived_01=18, NrBytesAccepted_01=18, NrMsgReceived_01=6, NrMsgAccepted_01=0, CD_02=34, Type_02=32, AgeOfLastMessage_02=65535, NrBytesReceived_02=18, NrBytesAccepted_02=18, NrMsgReceived_02=6, NrMsgAccepted_02=0)>",
            "<SBF(OutputLink, TOW=14:12:32, WNc=2367, N1=3, SB1Length=16, SB2Length=4, Reserved=0, CD_01=1, N2_01=2, AllowedRate_01=11, NrBytesProduced_01=2713104, NrBytesSent_01=2713104, NrClients_01=1, Reserved2_01=0, Type_01_01=33, Percentage_01_01=61, Type_01_02=64, Percentage_01_02=38, CD_02=33, N2_02=0, AllowedRate_02=0, NrBytesProduced_02=186, NrBytesSent_02=186, NrClients_02=1, Reserved2_02=0, CD_03=34, N2_03=0, AllowedRate_03=0, NrBytesProduced_03=186, NrBytesSent_03=186, NrClients_03=1, Reserved2_03=0)>",
            "<SBF(ReceiverStatus, TOW=14:12:32, WNc=2367, CPULoad=28, SISError=0, DiffCorrError=0, ExtSensorError=0, SetupError=0, Reserved1=0, UpTime=1365, Reserved2=0, ActiveAntenna=1, ExtFreq=0, ExtTime=0, WnSer=1, TowSet=1, FineTime=1, InternalDiskActivity=0, InternalDiskFull=0, InternalDiskMounted=0, IntAnt=0, RefoutLocked=0, Reserved3=0, ExternalDiskActivity=0, ExternalDiskFull=0, ExternalDiskMounted=0, PpsInCal=0, DiffCorrIn=0, Internet=0, Reserved4=0, Reserved5=0, SoftwareError=0, WatchdogError=0, AntennaError=0, CongestionError=0, Reserved6=0, MissEventError=0, CPUOverloadError=0, InvalidConfigError=0, OutOfGeofenceError=0, Reserved7=0, N=8, SBLength=4, CmdCount=16, Temperature=154, FrontEndID_01=3, Gain_01=41, SampleVar_01=98, BlankingStat_01=0, FrontEndID_02=4, Gain_02=40, SampleVar_02=93, BlankingStat_02=0, FrontEndID_03=12, Gain_03=39, SampleVar_03=98, BlankingStat_03=0, FrontEndID_04=5, Gain_04=39, SampleVar_04=102, BlankingStat_04=0, FrontEndID_05=6, Gain_05=40, SampleVar_05=104, BlankingStat_05=0, FrontEndID_06=0, Gain_06=41, SampleVar_06=100, BlankingStat_06=0, FrontEndID_07=1, Gain_07=43, SampleVar_07=98, BlankingStat_07=0, FrontEndID_08=11, Gain_08=43, SampleVar_08=100, BlankingStat_08=0)>",
            "<SBF(QualityInd, TOW=14:12:32, WNc=2367, N=5, Reserved=0, QualityType_01=11, QualityValue_01=10, Reserved1_01=0, QualityType_02=1, QualityValue_02=10, Reserved1_02=0, QualityType_03=21, QualityValue_03=10, Reserved1_03=0, QualityType_04=31, QualityValue_04=10, Reserved1_04=0, QualityType_05=0, QualityValue_05=10, Reserved1_05=0)>",
            "<SBF(NTRIPClientStatus, TOW=14:12:32, WNc=2367, N=1, SBLength=4, CDIndex_01=1, Status_01=3, ErrorCode_01=8, Info_01=0)>",
            "<SBF(NTRIPServerStatus, TOW=14:12:32, WNc=2367, N=0, SBLength=4)>",
            "<SBF(DiskStatus, TOW=14:12:32, WNc=2367, N=1, SBLength=16, Reserved=525057, DiskID_01=1, Mounted_01=0, Full_01=0, Activity_01=0, Logging_01=0, Mounting_01=0, Formatting_01=0, Reserved1_01=0, DiskUsageMSB_01=65535, DiskUsageLSB_01=4294967295, DiskSize_01=0, CreateDeleteCount_01=0, Error_01=255)>",
            "<SBF(RFStatus, TOW=14:12:32, WNc=2367, N=0, SBLength=8, Misleading=0, Inauthentic=0, Reserved2=0, Reserved=0)>",
            "<SBF(DynDNSStatus, TOW=14:12:32, WNc=2367, Status=0, ErrorCode=0, IPAddress=0)>",
            "<SBF(P2PPStatus, TOW=14:12:32, WNc=2367, N=0, SBLength=4)>",
            "<SBF(GALAuthStatus, TOW=14:12:32, WNc=2367, Status=0, OSNMAProgress=0, TimeSource=7, MerkleInProgress=0, Reserved=0, TrustedTimeDelta=-20000000000.0, GalActiveMask=0, GalAuthenticMask=0, GpsActiveMask=0, GpsAuthenticMask=0)>",
        )
        i = 0
        with open(os.path.join(DIRNAME, "pygpsdata_x5_status.log"), "rb") as stream:
            ubr = SBFReader(stream, protfilter=7, quitonerror=ERR_RAISE)
            for raw, parsed in ubr:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
            self.assertEqual(i, len(EXPECTED_RESULTS))

    def testtime(self):  # test time message stream
        EXPECTED_RESULTS = (
            "<SBF(xPPSOffset, TOW=14:11:00, WNc=2367, SyncAge=0, TimeScale=1, Offset=5.4056501388549805)>",
            "<SBF(ReceiverTime, TOW=14:11:00, WNc=2367, UTCYear=25, UTCMonth=5, UTCDay=23, UTCHour=14, UTCMin=11, UTCSec=0, DeltaLS=18, SyncLevel=7)>",
        )
        i = 0
        with open(os.path.join(DIRNAME, "pygpsdata_x5_time.log"), "rb") as stream:
            ubr = SBFReader(stream, protfilter=7, quitonerror=ERR_RAISE)
            for raw, parsed in ubr:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
            self.assertEqual(i, len(EXPECTED_RESULTS))

    def testattitude(self):  # test attitude message stream
        EXPECTED_RESULTS = (
            "<SBF(AttEuler, TOW=14:09:55, WNc=2367, NrSV=255, Error=128, Mode=0, Reserved=0, Heading=-20000000000.0, Pitch=-20000000000.0, Roll=-20000000000.0, PitchDot=-20000000000.0, RollDot=-20000000000.0, HeadingDot=-20000000000.0)>",
            "<SBF(AttCovEuler, TOW=14:09:55, WNc=2367, Reserved=0, Error=128, Cov_HeadHead=-20000000000.0, Cov_PitchPitch=-20000000000.0, Cov_RollRoll=-20000000000.0, Cov_HeadPitch=-20000000000.0, Cov_HeadRoll=-20000000000.0, Cov_PitchRoll=-20000000000.0)>",
            "<SBF(EndOfAtt, TOW=14:09:55, WNc=2367)>",
        )
        i = 0
        with open(os.path.join(DIRNAME, "pygpsdata_x5_attitude.log"), "rb") as stream:
            ubr = SBFReader(stream, protfilter=7, quitonerror=ERR_RAISE)
            for raw, parsed in ubr:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
            self.assertEqual(i, len(EXPECTED_RESULTS))

    def testdiffcorn1(self):  # test differential correction message stream
        EXPECTED_RESULTS = (
            "<SBF(BaseStation, TOW=14:20:18.700000, WNc=2367, BaseStationID=25, BaseType=255, Source=8, Datum=19, Reserved=0, X=3811965.6701, Y=-175800.07080000002, Z=5093615.5606)>",
        )
        i = 0
        with open(os.path.join(DIRNAME, "pygpsdata_x5_diffcorn.log"), "rb") as stream:
            ubr = SBFReader(stream, protfilter=7, quitonerror=ERR_RAISE)
            for raw, parsed in ubr:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
            self.assertEqual(i, len(EXPECTED_RESULTS))

    def testdiffcorn2(
        self,
    ):  # test RTCM3 content of differential correction message stream
        EXPECTED_RESULTS = (
            "<RTCM(1127, DF002=1127, DF003=25, DF427=555014000, DF393=1, DF409=0, DF001_7=0, DF411=1, DF412=0, DF417=0, DF418=0, DF394=655328480845627392, NSat=10, DF395=1090781569, NSig=6, DF396=732013670080413175, NCell=38, PRN_01=005, PRN_02=008, PRN_03=012, PRN_04=013, PRN_05=019, PRN_06=020, PRN_07=024, PRN_08=025, PRN_09=026, PRN_10=029, DF397_01=135, DF397_02=130, DF397_03=78, DF397_04=127, DF397_05=83, DF397_06=84, DF397_07=76, DF397_08=82, DF397_09=84, DF397_10=88, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, ExtSatInfo_04=0, ExtSatInfo_05=0, ExtSatInfo_06=0, ExtSatInfo_07=0, ExtSatInfo_08=0, ExtSatInfo_09=0, ExtSatInfo_10=0, DF398_01=0.439453125, DF398_02=0.5654296875, DF398_03=0.431640625, DF398_04=0.7861328125, DF398_05=0.140625, DF398_06=0.2333984375, DF398_07=0.873046875, DF398_08=0.04296875, DF398_09=0.05859375, DF398_10=0.07421875, DF399_01=-14, DF399_02=206, DF399_03=-460, DF399_04=112, DF399_05=-398, DF399_06=91, DF399_07=246, DF399_08=-342, DF399_09=635, DF399_10=586, CELLPRN_01=005, CELLSIG_01=2I, CELLPRN_02=005, CELLSIG_02=7I, CELLPRN_03=008, CELLSIG_03=2I, CELLPRN_04=008, CELLSIG_04=7I, CELLPRN_05=012, CELLSIG_05=2I, CELLPRN_06=012, CELLSIG_06=7I, CELLPRN_07=013, CELLSIG_07=2I, CELLPRN_08=013, CELLSIG_08=7I, CELLPRN_09=019, CELLSIG_09=2I, CELLPRN_10=019, CELLSIG_10=6I, CELLPRN_11=019, CELLSIG_11=5X, CELLPRN_12=019, CELLSIG_12=7D, CELLPRN_13=019, CELLSIG_13=1X, CELLPRN_14=020, CELLSIG_14=2I, CELLPRN_15=020, CELLSIG_15=6I, CELLPRN_16=020, CELLSIG_16=5X, CELLPRN_17=020, CELLSIG_17=7D, CELLPRN_18=020, CELLSIG_18=1X, CELLPRN_19=024, CELLSIG_19=2I, CELLPRN_20=024, CELLSIG_20=6I, CELLPRN_21=024, CELLSIG_21=5X, CELLPRN_22=024, CELLSIG_22=7D, CELLPRN_23=024, CELLSIG_23=1X, CELLPRN_24=025, CELLSIG_24=2I, CELLPRN_25=025, CELLSIG_25=6I, CELLPRN_26=025, CELLSIG_26=5X, CELLPRN_27=025, CELLSIG_27=7D, CELLPRN_28=025, CELLSIG_28=1X, CELLPRN_29=026, CELLSIG_29=2I, CELLPRN_30=026, CELLSIG_30=6I, CELLPRN_31=026, CELLSIG_31=5X, CELLPRN_32=026, CELLSIG_32=7D, CELLPRN_33=026, CELLSIG_33=1X, CELLPRN_34=029, CELLSIG_34=2I, CELLPRN_35=029, CELLSIG_35=6I, CELLPRN_36=029, CELLSIG_36=5X, CELLPRN_37=029, CELLSIG_37=7D, CELLPRN_38=029, CELLSIG_38=1X, DF405_01=-3.666430711746216e-05, DF405_02=-4.6096742153167725e-05, DF405_03=2.7956441044807434e-05, DF405_04=-2.8442591428756714e-06, DF405_05=0.00026414357125759125, DF405_06=0.00024555064737796783, DF405_07=-0.0004072412848472595, DF405_08=-0.00041072070598602295, DF405_09=-6.874464452266693e-05, DF405_10=-0.00011236779391765594, DF405_11=-8.387118577957153e-05, DF405_12=-8.547492325305939e-05, DF405_13=-8.581392467021942e-05, DF405_14=0.00018525868654251099, DF405_15=0.00013836286962032318, DF405_16=0.00016413629055023193, DF405_17=0.00016956962645053864, DF405_18=0.000176137313246727, DF405_19=0.00014740601181983948, DF405_20=0.0001056324690580368, DF405_21=0.00011808983981609344, DF405_22=0.00011541880667209625, DF405_23=0.0001378953456878662, DF405_24=0.00013836100697517395, DF405_25=0.0001079365611076355, DF405_26=0.0001192726194858551, DF405_27=0.00011988542973995209, DF405_28=0.00012893974781036377, DF405_29=-8.977018296718597e-05, DF405_30=-0.00011324882507324219, DF405_31=-9.636208415031433e-05, DF405_32=-9.831786155700684e-05, DF405_33=-9.577721357345581e-05, DF405_34=0.00022168457508087158, DF405_35=0.00019222497940063477, DF405_36=0.00021829642355442047, DF405_37=0.00020895525813102722, DF405_38=0.00020852498710155487, DF406_01=-1.220405101776123e-05, DF406_02=-5.4743606597185135e-05, DF406_03=3.692135214805603e-05, DF406_04=1.7884187400341034e-05, DF406_05=0.0002951743081212044, DF406_06=0.0002912776544690132, DF406_07=-0.00040158769115805626, DF406_08=-0.00041068345308303833, DF406_09=-3.0709896236658096e-05, DF406_10=-3.0716415494680405e-05, DF406_11=-2.088025212287903e-06, DF406_12=2.780463546514511e-06, DF406_13=-3.906805068254471e-05, DF406_14=0.00021843379363417625, DF406_15=0.00017860252410173416, DF406_16=0.00021539395675063133, DF406_17=0.00021125935018062592, DF406_18=0.00019877031445503235, DF406_19=0.00016383826732635498, DF406_20=0.00012186681851744652, DF406_21=0.00014182506129145622, DF406_22=0.00013411324471235275, DF406_23=0.00014624232426285744, DF406_24=0.0001712823286652565, DF406_25=0.0001286468468606472, DF406_26=0.00014832010492682457, DF406_27=0.00014305952936410904, DF406_28=-1.3154931366443634e-06, DF406_29=-9.679747745394707e-05, DF406_30=-0.00015483424067497253, DF406_31=-0.0001339237205684185, DF406_32=-0.00013503432273864746, DF406_33=-0.00011349283158779144, DF406_34=0.00017101923003792763, DF406_35=0.0001382250338792801, DF406_36=0.000147323589771986, DF406_37=0.0001535569317638874, DF406_38=0.00017748400568962097, DF407_01=594, DF407_02=594, DF407_03=483, DF407_04=483, DF407_05=483, DF407_06=483, DF407_07=483, DF407_08=483, DF407_09=569, DF407_10=569, DF407_11=569, DF407_12=569, DF407_13=569, DF407_14=483, DF407_15=483, DF407_16=483, DF407_17=483, DF407_18=483, DF407_19=620, DF407_20=620, DF407_21=620, DF407_22=620, DF407_23=620, DF407_24=483, DF407_25=483, DF407_26=483, DF407_27=483, DF407_28=483, DF407_29=644, DF407_30=644, DF407_31=644, DF407_32=644, DF407_33=644, DF407_34=633, DF407_35=633, DF407_36=633, DF407_37=633, DF407_38=340, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0, DF420_15=0, DF420_16=0, DF420_17=0, DF420_18=0, DF420_19=0, DF420_20=0, DF420_21=0, DF420_22=0, DF420_23=0, DF420_24=0, DF420_25=0, DF420_26=0, DF420_27=0, DF420_28=0, DF420_29=0, DF420_30=0, DF420_31=0, DF420_32=0, DF420_33=0, DF420_34=0, DF420_35=0, DF420_36=0, DF420_37=0, DF420_38=0, DF408_01=38.375, DF408_02=42.1875, DF408_03=40.6875, DF408_04=43.375, DF408_05=43.6875, DF408_06=49.875, DF408_07=44.625, DF408_08=44.625, DF408_09=43.875, DF408_10=47.625, DF408_11=49.5, DF408_12=46.875, DF408_13=43.5, DF408_14=44.1875, DF408_15=46.375, DF408_16=48.1875, DF408_17=46.6875, DF408_18=46.375, DF408_19=49.625, DF408_20=51.5, DF408_21=51.1875, DF408_22=46.6875, DF408_23=49.125, DF408_24=43.625, DF408_25=45.1875, DF408_26=45.625, DF408_27=42.1875, DF408_28=43.0, DF408_29=42.625, DF408_30=44.625, DF408_31=46.8125, DF408_32=44.5, DF408_33=42.3125, DF408_34=34.875, DF408_35=34.625, DF408_36=42.625, DF408_37=31.125, DF408_38=35.0, DF404_01=0.32170000000000004, DF404_02=0.3214, DF404_03=0.4852, DF404_04=0.48500000000000004, DF404_05=0.25520000000000004, DF404_06=0.2549, DF404_07=-0.08750000000000001, DF404_08=-0.0876, DF404_09=0.48950000000000005, DF404_10=0.4897, DF404_11=0.49010000000000004, DF404_12=0.4897, DF404_13=0.4899, DF404_14=0.4708, DF404_15=0.4708, DF404_16=0.4713, DF404_17=0.4711, DF404_18=0.47100000000000003, DF404_19=0.11810000000000001, DF404_20=0.11810000000000001, DF404_21=0.11810000000000001, DF404_22=0.11810000000000001, DF404_23=0.11810000000000001, DF404_24=-0.4897, DF404_25=-0.48910000000000003, DF404_26=-0.4889, DF404_27=-0.4893, DF404_28=-0.4892, DF404_29=0.34390000000000004, DF404_30=0.3438, DF404_31=0.3441, DF404_32=0.3437, DF404_33=0.3437, DF404_34=-0.35910000000000003, DF404_35=-0.3589, DF404_36=-0.3594, DF404_37=-0.3592, DF404_38=-0.3593)>",
        )
        i = 0
        with open(os.path.join(DIRNAME, "pygpsdata_x5_diffcorn2.log"), "rb") as stream:
            ubr = SBFReader(stream, protfilter=7, quitonerror=ERR_RAISE)
            for raw, parsed in ubr:
                rtcm = RTCMReader.parse(parsed.RTCM3Message)
                # print(f'"{rtcm}",')
                self.assertEqual(parsed.identity, "DiffCorrIn")
                self.assertEqual(str(rtcm), EXPECTED_RESULTS[i])
                i += 1
            self.assertEqual(i, len(EXPECTED_RESULTS))

    def testtruncatedraise(
        self,
    ):  # test truncated data stream, raise error
        with self.assertRaisesRegex(
            SBFStreamError,
            "Serial stream terminated unexpectedly. 8 bytes requested, 6 bytes returned.",
        ):
            with open(
                os.path.join(DIRNAME, "pygpsdata_x5_pvttruncated.log"), "rb"
            ) as stream:
                sbr = SBFReader(
                    stream,
                    quitonerror=ERR_RAISE,
                )
                for raw, parsed in sbr:
                    pass

    def testtruncatedlog(
        self,
    ):  # test truncated data stream, log error
        STDERR = os.path.join(DIRNAME, "stderr.log")
        stderr_saved = sys.stderr
        sys.stderr = open(STDERR, "w")
        stream_handler = logging.StreamHandler(sys.stderr)
        self.logger.addHandler(stream_handler)
        with open(
            os.path.join(DIRNAME, "pygpsdata_x5_pvttruncated.log"), "rb"
        ) as stream:
            sbr = SBFReader(
                stream,
                quitonerror=ERR_LOG,
            )
            for raw, parsed in sbr:
                pass
        self.logger.removeHandler(stream_handler)
        sys.stderr = stderr_saved
        with open(STDERR, "r") as errlog:
            el = errlog.readline()
            self.assertEqual(
                el,
                "Serial stream terminated unexpectedly. 8 bytes requested, 6 bytes returned.\n",
            )

    def testtruncatedignore(
        self,
    ):  # test truncated data stream, ignore error
        STDERR = os.path.join(DIRNAME, "stderr.log")
        stderr_saved = sys.stderr
        sys.stderr = open(STDERR, "w")
        stream_handler = logging.StreamHandler(sys.stderr)
        self.logger.addHandler(stream_handler)
        with open(
            os.path.join(DIRNAME, "pygpsdata_x5_pvttruncated.log"), "rb"
        ) as stream:
            sbr = SBFReader(
                stream,
                quitonerror=ERR_IGNORE,
            )
            for raw, parsed in sbr:
                pass
        self.logger.removeHandler(stream_handler)
        sys.stderr = stderr_saved
        with open(STDERR, "r") as errlog:
            el = errlog.readline()
            self.assertEqual(el, "")

    def testnorelevantdata(
        self,
    ):  # test data stream with no SBF, NMEA or RTCM3 content
        with self.assertRaisesRegex(
            SBFParseError, "Unknown protocol header b'\\\\xd3\\\\x05'."
        ):
            with open(os.path.join(DIRNAME, "pygpsdata_ubx.log"), "rb") as stream:
                sbr = SBFReader(
                    stream,
                    protfilter=SBF_PROTOCOL | NMEA_PROTOCOL | RTCM3_PROTOCOL,
                    quitonerror=ERR_RAISE,
                )
                for raw, parsed in sbr:
                    pass
