const std = @import("std");
const clap = @import("clap");
const ziglyph = @import("ziglyph");
const fs = std.fs;
const io = std.io;
const os = std.os;

pub const program_name = "passphrase";
pub const version = std.SemanticVersion{
    .major = 1,
    .minor = 7,
    .patch = 0,
};

pub const params = clap.parseParamsComptime(
    \\  -w, --words     <num> Number of words for passphrase [default: 4]
    \\  -s, --separator <str> String (or '' for none) to put between the words [default: ' ']
    \\  -c, --capitalize      Capitalize each word
    \\  -x, --symbol          Add a symbol after one of the words
    \\  -d, --digit           Add a digit after one of the words
    \\  -f, --file      <str> Use file as wordlist (one word per line)
    \\  -p, --password        Generate a password instead of a passphrase
    \\  -l, --length    <num> Length of password [default: 24]
    \\  -h, --help            Show help and quit
);

pub fn main() !void {
    var file = try fs.cwd().createFile("passphrase.1", .{});

    var doc = document(file.writer());
    try doc.heading(
        1,
        "passphrase",
        "Generate a passphrase",
        std.fmt.comptimePrint("{d}.{d}.{d}", .{
            version.major,
            version.minor,
            version.patch,
        }),
        "November 5, 2023",
    );
    try doc.section("Name");
    try doc.newParagraph();
    try doc.text("passphrase - generate a passphrase");
    try doc.newLine();

    try doc.newParagraph();
    try doc.section("Synopsis");

    try doc.newParagraph();
    try doc.bold("passphrase ");
    try doc.text("[");
    try doc.italic("option");
    try doc.text("]...");
    try doc.newLine();

    try doc.newParagraph();
    try doc.section("Options");

    inline for (params) |p| {
        try doc.newParagraph();
        try doc.indent(0);

        const val: ?[]const u8 = if (p.takes_value != .none)
            " <" ++ comptime p.id.value() ++ ">"
        else
            null;

        if (p.names.short) |s| {
            try doc.bold("-" ++ .{s});

            if (val) |v|
                try doc.italic(v);

            if (p.names.long != null)
                try doc.text(", ");
        }
        if (p.names.long) |s| {
            try doc.bold("--" ++ s);

            if (val) |v|
                try doc.italic(v);
        }
        try doc.newLine();
        try doc.indent(4);
        try doc.text(p.id.description());
        try doc.newLine();
    }
}

const _title_heading = ".TH ";
const _paragraph = ".PP";
const _indent = ".RS";
const _indent_end = ".RE";
const _indented_paragraph = ".IP";
const _section_heading = ".SH {s}";
const _tagged_paragraph = ".TP";

const _bold = "\\fB";
const _italic = "\\fI";
const _previous_font = "\\fP";

pub fn document(writer: anytype) Document(@TypeOf(writer)) {
    return Document(@TypeOf(writer)).init(writer);
}

pub fn Document(comptime Writer: type) type {
    return struct {
        writer: Writer,

        const Self = @This();

        pub fn init(writer: Writer) Self {
            return .{ .writer = writer };
        }

        pub fn indent(self: *Self, amount: usize) !void {
            if (0 < amount) {
                try self.writer.print("{s} {d}", .{ _indent, amount });
            } else {
                try self.writer.writeAll(_indent_end);
            }
            try self.newLine();
        }

        pub fn newParagraph(self: *Self) !void {
            try self.writer.writeAll(_paragraph);
            try self.newLine();
        }

        pub fn heading(
            self: *Self,
            man_sect: u8,
            title: []const u8,
            description: []const u8,
            version_info: []const u8,
            time: anytype,
        ) !void {
            var it = ziglyph.CodePointIterator{
                .bytes = title,
                .i = 0,
            };

            try self.writer.writeAll(_title_heading);

            while (it.next()) |cp| {
                var buf: [4]u8 = undefined;
                const ucp = ziglyph.toUpper(cp.code);
                const n = try std.unicode.utf8Encode(ucp, &buf);
                try self.writer.writeAll(buf[0..n]);
            }

            try self.writer.print(
                " {d} \"{s}\" \"{s}\" \"{s}\"",
                .{
                    man_sect,
                    time,
                    version_info,
                    description,
                },
            );
            try self.newLine();
        }

        pub fn text(self: *Self, s: []const u8) !void {
            for (s) |b|
                if (b == '.')
                    try self.writer.writeAll("\\&.")
                else if (b == '\\')
                    try self.writer.writeAll("\\e")
                else
                    try self.writer.writeByte(b);
        }

        pub fn bold(self: *Self, s: []const u8) !void {
            try self.writer.writeAll(_bold);
            try self.text(s);
            try self.writer.writeAll(_previous_font);
        }

        pub fn italic(self: *Self, s: []const u8) !void {
            try self.writer.writeAll(_italic);
            try self.text(s);
            try self.writer.writeAll(_previous_font);
        }

        pub fn section(self: *Self, s: []const u8) !void {
            try self.writer.print(
                _section_heading,
                .{s},
            );
            try self.newLine();
        }

        pub fn newLine(self: *Self) !void {
            try self.writer.writeByte('\n');
        }
    };
}
