//
// nono
// Copyright (C) 2021 nono project
// Licensed under nono-license.txt
//

#pragma once

#include "header.h"
#include "autofd.h"
#include <array>
#include <map>

// ディスクイメージを実際に処理するドライバ
// (DiskImage クラスの下請けになる)
class DiskImageDriver
{
 public:
	explicit DiskImageDriver(const std::string& pathname_);
	virtual ~DiskImageDriver();

	virtual int IsWriteable() const = 0;
	virtual bool Open(bool read_only) = 0;
	virtual void Close() = 0;
	virtual off_t GetSize() const = 0;
	virtual bool Read(void *buf, off_t offset, size_t size) = 0;
	virtual bool Write(const void *buf, off_t offset, size_t size) = 0;

 protected:
	const std::string& pathname;
};

// ディスクイメージ
class DiskImage
{
 private:
	// Copy-on-Write キャッシュのブロックサイズ
	static const uint BLKSIZE = 4096;

	using SectBuf = std::array<uint8, BLKSIZE>;

 public:
	DiskImage();
	virtual ~DiskImage();

	// パス名を指定して、下位ドライバをオープン。
	// コンストラクタ後最初に実行すること。
	bool CreateHandler(const std::string& pathname_);

	// このディスクイメージが書き込み可能かどうかを返す。
	// Open() 前でも動作する。
	// 書き込み可能なら 1、不可なら 0 を返す。
	// 失敗なら errno をセットして -1 を返す。
	int IsWriteable() const;

	// このディスクイメージを実際のアクセス用にオープンする。
	// read_only == true なら、ディスクイメージを読み込み専用でオープンする。
	// is_cow == true なら、Copy-on-Write モードにする。
	// 失敗すれば warn() でエラーメッセージを表示して false を返す。
	bool Open(bool read_only, bool is_cow);

	// このディスクイメージをクローズする。
	void Close();

	// このディスクイメージのバイト数を返す。
	// オープン後のみ使用可能。
	off_t GetSize() const;

	// このディスクイメージの offset から size バイトを buf に読み出す。
	bool Read(void *buf, off_t offset, size_t size) const;

	// このディスクイメージの offset から size バイトに buf を書き込む。
	bool Write(const void *buf, off_t offset, size_t size);

	// このディスクイメージのパス名を返す。
	const std::string& GetPathName() const { return pathname; }

	// driver があれば true を返す。
	// unique_ptr の (bool) キャストと同じ。
	operator bool() const { return (bool)driver; }

 private:
	// map::find は分かりにくいので。C++20 には map::contains() がある
	bool map_contains(off_t x) const {
		return map.find(x) != map.end();
	}

	std::string pathname {};

	std::unique_ptr<DiskImageDriver> driver {};

	// Copy-on-Write モード
	bool cow_enable {};
	autofd cowfd {};
	std::string cowname {};			// ファイルパス
	// map (連想配列)のキーはオリジナルファイル上のブロック番号(バイト
	// オフセットを BLKSIZE で割ったもの)、対応する値は CoW ファイル上の
	// バイトオフセット。
	std::map<off_t, off_t> map {};
};
