//
// nono
// Copyright (C) 2025 nono project
// Licensed under nono-license.txt
//

//
// HD647180 ASCI (非同期シリアル)
//

#pragma once

#include "device.h"
#include "message.h"
#include <array>

class HostCOMDevice;
class MPU64180Device;

class HD647180ASCIDevice : public Device
{
	using inherited = Device;

	// CNTLA0, CNTLA1
	static constexpr uint32 CNTL_MPE	= 0x80;	// Multi Processor Enable
	static constexpr uint32 CNTL_RE		= 0x40;	// Receive Enable
	static constexpr uint32 CNTL_TE		= 0x20;	// Transmit Enable
	static constexpr uint32 CNTL_nRTS0	= 0x10;	// !RequestToSend
	static constexpr uint32 CNTL_CKA1D	= 0x10;	// CKA1 Disable
	static constexpr uint32 CNTL_EFR	= 0x08;	// Error Flag Reset
	static constexpr uint32 CNTL_DATA8	= 0x04;	// MODE2 (Data bits)
	static constexpr uint32 CNTL_PAREN	= 0x02;	// MODE1 (Parity Enable)
	static constexpr uint32 CNTL_STOP2	= 0x01;	// MODE0 (Stop bits)

	// CNTLB0, CNTLB1
	static constexpr uint32 CNTL_MPBT	= 0x80;	// Multi Processor Bit Transmit
	static constexpr uint32 CNTL_MP		= 0x40;	// Multi Processor
	static constexpr uint32 CNTL_nCTS	= 0x20;	// R: !CTS
	static constexpr uint32 CNTL_PS		= 0x20;	// W: PreScaler
	static constexpr uint32 CNTL_PEO	= 0x10;	// Parity Even or Odd
	static constexpr uint32 CNTL_DR		= 0x08;	// Divide Ratio
	static constexpr uint32 CNTL_SS		= 0x07;	// Clock Source and Speed Select

	// STAT0, STAT1
	static constexpr uint32 STAT_RDRF	= 0x80;	// Receive Data Register Full
	static constexpr uint32 STAT_OVRN	= 0x40;	// Over Run Error
	static constexpr uint32 STAT_PE		= 0x20;	// Parity Error
	static constexpr uint32 STAT_FE		= 0x10;	// Framing Error
	static constexpr uint32 STAT_RIE	= 0x08;	// Receive Interrupt Enable
	static constexpr uint32 STAT_nDCD0	= 0x04;	// R: Data Carrier Detect
	static constexpr uint32 STAT_CTS1E	= 0x04;	// W: !CTS Enable
	static constexpr uint32 STAT_TDRE	= 0x02;	// Transmit Data Register Empty
	static constexpr uint32 STAT_TIE	= 0x01;	// Transmit Interrupt Enable

	// 非同期シリアル1個分
	struct Channel {
		uint id;		// 0 or 1

		bool rx_enable;
		bool tx_enable;
		bool rx_intr_enable;
		bool tx_intr_enable;
		bool data8bit;	// MOD2
		bool parity_en;	// MOD1
		bool stop2bit;	// MOD0
		bool parity_odd;
		bool prescale;
		bool divratio;
		uint8 ss;
		bool nCTS;
		bool overrun;

		uint8 tdr;
		bool tdr_empty;
		uint8 rdr;
		bool rdr_full;
		int32 tsr;		// 負数なら Empty。
		int32 rsr;		// 負数なら Empty。

		uint64 bit12_ns;
		uint32 baudrate;
	};

 public:
	HD647180ASCIDevice();
	~HD647180ASCIDevice() override;

	bool Create() override;
	bool Init() override;
	void ResetHard(bool poweron) override;

	uint32 ReadCNTLA(uint n);
	uint32 ReadCNTLB(uint n);
	uint32 ReadSTAT(uint n);
	uint32 ReadTDR(uint n);
	uint32 ReadRDR(uint n);
	uint32 WriteCNTLA(uint n, uint32 data);
	uint32 WriteCNTLB(uint n, uint32 data);
	uint32 WriteSTAT(uint n, uint32 data);
	uint32 WriteTDR(uint n, uint32 data);
	uint32 WriteRDR(uint n, uint32 data);
	uint32 PeekCNTLA(uint n) const;
	uint32 PeekCNTLB(uint n) const;
	uint32 PeekSTAT(uint n) const;
	uint32 PeekTDR(uint n) const;
	uint32 PeekRDR(uint n) const;

 private:
	DECLARE_MONITOR_SCREEN(MonitorScreen);
	void MonitorScreenChan(TextScreen&, int y, int n) const;

	void ChangeInterrupt(uint n);
	void ChangeBaudRate(uint n);

	void TxCallback(Event *);

	// 受信側
	void HostRxCallback(uint32);
	void RxMessage(MessageID, uint32);
	void RxCallback(Event *);
	bool Rx(uint n, uint32 data);

	// 入力クロックを 160分周した基本周波数とでもいうべきもの、
	// の 12ビット分を [nsec] にしたもの。
	uint64 xc12_ns {};

	std::array<Channel, 2> channels {};
	bool cts1_enable {};
	bool cka1_disable {};

	bool nRTS0 {};
	bool nDCD0 {};

	std::array<Event *, 2> txevent {};
	std::array<Event *, 2> rxevent {};

	std::array<std::unique_ptr<HostCOMDevice>, 2> hostcom /*{}*/;

	Monitor *monitor {};

	MPU64180Device *xp {};
};
