//
// nono
// Copyright (C) 2025 nono project
// Licensed under nono-license.txt
//

//
// SCSI ドメイン
//

#pragma once

#include "device.h"
#include <array>

class SCSIBus;
class SCSIDevice;
class SCSIHostDevice;
class SCSITarget;

//
// 一つの SCSI イニシエータとそのターゲットデバイス一式の管理構造。
// 設定周りとインスタンスの所有。
// バスは含まない (VirtIO SCSI だとバスという構造が存在しないため)。
//
class SCSIDomain : public Device
{
	using inherited = Device;

 public:
	// このドメインで扱うデバイス数。
	static constexpr size_t MaxDevices = 8;

 public:
	SCSIDomain(IODevice *parent_, const char *config_name_);
	~SCSIDomain() override;

	bool Create() override;
	bool Init() override;

	// 設定ファイル上のこのホストデバイスを示す接頭語を返す
	const char *GetConfigName() const noexcept { return config_name; }

	// イニシエータを返す。ないことはないはず。
	SCSIHostDevice *GetInitiator() const noexcept { return initiator.get(); }

	// すべてのターゲットをバスに参加させる。
	void AttachTargets(SCSIBus *bus_);

	// イニシエータに ID を付与してバスに参加させる。
	bool AttachInitiator(SCSIBus *bus_, uint id);

	// 指定 id のターゲットデバイスを返す。なければ NULL を返す。
	SCSITarget *GetTarget(uint id) const noexcept { return target[id].get(); }

	// デバイス (イニシエータ+ターゲット) 配列のコピーを返す。
	void GetDevices(std::array<SCSIDevice *, MaxDevices> *dst) const {
		*dst = device;
	}

	// 指定 id のデバイスを返す。なければ NULL を返す。
	SCSIDevice *GetDevice(uint id) const { return device[id]; }

	// (イニシエータを除く) デバイス数を返す。
	uint GetTargetCount() const;

	// (イニシエータがあれば含む) すべてのデバイスのリストを返す。
	const std::vector<SCSIDevice*>& GetConnectedDevices() const noexcept {
		return connected_devices;
	}

 private:
	DECLARE_MONITOR_SCREEN(MonitorScreen);

	// device[] と connected_devices を更新する。
	bool RefreshDevices();

	// 設定ファイルに使われるこのホストデバイスを示す接頭語。
	// ("spc0" とか)
	const char *config_name {};

	// SCSI イニシエータ (所有)
	std::unique_ptr<SCSIHostDevice> initiator /*{}*/;

	// SCSI ターゲット (所有)。イニシエータは含まない。
	std::array<std::unique_ptr<SCSITarget>, MaxDevices> target /*{}*/;

	// SCSI デバイス (イニシエータ+ターゲット) のポインタ。
	std::array<SCSIDevice *, MaxDevices> device {};

	// 接続中のデバイス(イニシエータ含む)を ID 順に詰めたリスト。
	std::vector<SCSIDevice*> connected_devices {};

	IODevice *parent {};

	Monitor *monitor {};
};
