// vi:set ts=4:

//
// nono
// Copyright (C) 2024 nono project
// Licensed under nono-license.txt
//

#include "runx.h"
#include <stdio.h>
#include <signal.h>
#include <sys/param.h>
#include <sys/sysctl.h>

// ページサイズを取得する。
int
getpgsize()
{
	const int mib[2] = { CTL_HW, HW_PAGESIZE };
	int val;
	size_t len;

	len = sizeof(val);
	if (sysctl(mib, countof(mib), &val, &len, NULL, 0) < 0) {
		err(1, "sysctl");
	}
	return val;
}

// シグナル名(定数名) を返す。なければ "signal %d" を返す。
// スタティック変数を返す場合があるので同時に2回呼ばないこと。
// (Linux には sigabbrev_np() がある)
const char *
signame(int signo)
{
	static char buf[16];
	static const struct {
		int val;
		const char *name;
	} table[] = {
		// <sys/signal.h> から加工した一覧。
		// ターゲットは NetBSD だと分かっているので細かいことは気にしない。
#define E(s)	{ s, #s }
		E(SIGHUP),
		E(SIGINT),
		E(SIGQUIT),
		E(SIGILL),
		E(SIGTRAP),
		E(SIGABRT),
		E(SIGEMT),
		E(SIGFPE),
		E(SIGKILL),
		E(SIGBUS),
		E(SIGSEGV),
		E(SIGSYS),
		E(SIGPIPE),
		E(SIGALRM),
		E(SIGTERM),
		E(SIGURG),
		E(SIGSTOP),
		E(SIGTSTP),
		E(SIGCONT),
		E(SIGCHLD),
		E(SIGTTIN),
		E(SIGTTOU),
		E(SIGIO),
		E(SIGXCPU),
		E(SIGXFSZ),
		E(SIGVTALRM),
		E(SIGPROF),
		E(SIGWINCH),
		E(SIGINFO),
		E(SIGUSR1),
		E(SIGUSR2),
		E(SIGPWR),
#undef E
	};
	for (int i = 0; i < countof(table); i++) {
		if (signo == table[i].val) {
			return table[i].name;
		}
	}
	snprintf(buf, sizeof(buf), "signal %d", signo);
	return buf;
}
