//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

#pragma once

#include <array>
#include <mutex>
#include "fixedqueue.h"

//#define M68030_CUSTOM_ATC
#define ATC_ORDER128

#if defined(ATC_ORDER128)
#include "uint128.h"
#endif

class MPU68030Device;

// SRP, CRP
class m68030RP
{
	//        6                   5                   4               3
	//  3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2
	// +-+-----------------------------+---------------------------+---+
	// |U|           LIMIT             |          unused           |DT |
	// +-+-----------------------------+---------------------------+---+
	//
	//    3                   2                   1                   0
	//  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
	// +-------------------------------------------------------+-------+
	// |           TABLE ADDRESS (PA31-PA4)                    |unused |
	// +-------------------------------------------------------+-------+
 public:
	static const uint32 DT_MASK	= 0x00000003;
	static const uint32 H_MASK	= 0xffff0003;
	static const uint32 L_MASK	= 0xfffffff0;
};

// TT レジスタの各フィールドを束ねた構造体のようなもの。
// MPU68030Device::SetTT(), GetTT() で読み書きする。
class m68030TT
{
	//    3                   2                   1                   0
	//  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
	// +---------------+---------------+-+-+-+-+-+-+-+-+-+-----+-+-----+
	// |LogicalAddrBase|LogicalAddrMask|E|       |C|R|M| | FCB | | FCM |
	// +---------------+---------------+-+-+-+-+-+-+-+-+-+-----+-+-----+

 public:
	static const uint32 MASK		= 0xffff8777;

	static const uint32 LBASE_MASK	= 0xff000000;
	static const uint32 LMASK_MASK	= 0x00ff0000;
	static const uint32 E			= 0x00008000;
	static const uint32 CI			= 0x00000400;
	static const uint32 RW			= 0x00000200;
	static const uint32 RWM			= 0x00000100;
	static const uint32 FCBASE_MASK	= 0x00000070;
	static const uint32 FCMASK_MASK	= 0x00000007;

	bool e {};				// イネーブル
	bool ci {};				// キャッシュ禁止
	uint64 base {};			// ベース
	uint64 mask {};			// マスク

	// アクセスがこの TT と一致するか調べる。
	bool Match(busaddr addr) const;
};

// TC レジスタの各フィールドを束ねた構造体のようなもの。
// m68kcpu::SetTC(), GetTC() で読み書きする。
class m68030TC
{
	//    3                   2                   1                   0
	//  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
	// +-+---------+-+-+-------+-------+-------+-------+-------+-------+
	// |E|    0    |S|F|  PS   |  IS   |  TIA  |  TIB  |  TIC  |  TID  |
	// +-+---------+-+-+-------+-------+-------+-------+-------+-------+
	//  |           | +-- FCL (Function Code Lookup Enable)
	//  |           +---- SRE (Supervisor Root Pointer Enable)
	//  +-- Enable
 public:
	static const uint32 MASK	= 0x83ffffff;
	static const uint32 TC_E	= 0x80000000;
	static const uint32 TC_SRE	= 0x02000000;
	static const uint32 TC_FCL	= 0x01000000;

	// レジスタの各フィールド
	uint32 e {};		// TC_E   の位置
	uint32 sre {};		// TC_SRE の位置
	uint32 fcl {};		// TC_FCL の位置
	uint ps {};
	uint is {};
	union {
		uint tix[4] {};
		struct {
			uint tia;
			uint tib;
			uint tic;
			uint tid;
		} __packed;
	};

	int shift[4] {};		// TIA-TID の各部分のシフト量
	uint32 mask[4] {};		// TIA-TID の各部分のマスク
	uint32 lmask {};		// TIA-TID 全体のマスク
	uint32 pgmask {};		// PS 部分のマスク

 public:
	void Set(uint32);
	bool Check() const;
	void MakeMask();
};

class m68030MMUSR
{
 public:
	static const uint16 MASK = 0xee43;
	static const uint16 B    = 0x8000;	// バスエラー
	static const uint16 L    = 0x4000;	// リミット違反
	static const uint16 S    = 0x2000;	// スーパバイザ専用
	static const uint16 W    = 0x0800;	// 書き込み保護
	static const uint16 I    = 0x0400;	// 無効
	static const uint16 M    = 0x0200;	// 修正
	static const uint16 T    = 0x0040;	// 透過アクセス
	static const uint16 N    = 0x0007;	// レベル数
};

// ATC の1エントリ
struct m68030ATCLine
{
	// tag
	//  LLLLLLLL'LLLLLLLL'LLLLLLLL'0000FFFI
	//
	//  I は Invalid (%1で無効)
	//  L は論理アドレス
	//  F は busaddr 形式の FC (210 の順) で、_Motorola のみで使用。
	//	    _Custom のほうは FC 別のテーブルなので使わない。
	//
	// paddr
	//  PPPPPPPP_PPPPPPPP_PPPPPPPP_00000000 : P は物理アドレス
	//
	// data
	//  00000000_00000000_00000000_0000BCPM
	//                                 |||+-- Modified
	//                                 ||+--- Write Protect
	//                                 |+---- Cache Inhibit
	//                                 +----- Bus Error

	static const uint32 LADDR_MASK	= 0xffffff00;
	static const uint32 INVALID		= 0x00000001;
	static const uint32 PADDR_MASK	= 0xffffff00;
	static const uint32 BUSERROR	= 0x00000008;
	static const uint32 CINHIBIT	= 0x00000004;
	static const uint32 WPROTECT	= 0x00000002;
	static const uint32 MODIFIED    = 0x00000001;

	uint32 tag;
	uint32 paddr;
	uint32 data;
#if defined(M68030_CUSTOM_ATC)
	uint32 age;
#endif

	uint32 GetTag() const		{ return tag; }

	void   Invalidate()			{ tag |= INVALID; }
	bool   IsInvalid() const	{ return tag & INVALID; }
	bool   IsValid() const		{ return !IsInvalid(); }

	uint32 GetLAddr() const		{ return tag & LADDR_MASK; }
#if !defined(M68030_CUSTOM_ATC)
	uint32 GetFC() const		{ return (tag >> 1) & 7; }
#endif
	uint32 GetPAddr() const		{ return paddr; }
	bool   IsBusError() const	{ return data & BUSERROR; }
	bool   IsCInhibit() const	{ return data & CINHIBIT; }
	bool   IsWProtect() const	{ return data & WPROTECT; }
	bool   IsModified() const	{ return data & MODIFIED; }
};

// ATC の統計情報
struct m68030ATCStat
{
	uint32 total {};	// アドレス変換した回数 (TC と TT 両方)
	uint32 tthit {};	// TT にヒットした回数
	uint32 miss {};		// ATC ミスした回数
};

// ATC のテーブル (の共通部分)。
// このクラスは virtual にせず、共通の変数と定数だけを持つこと。
class m68030ATCTable
{
 public:
	static const int LINES = 22;

	int loglevel {};

	// 統計
	m68030ATCStat atcstat {};

 protected:
	MPU68030Device *cpu {};
};

#if defined(M68030_CUSTOM_ATC)
class m68030ATCTable_Custom final : public m68030ATCTable
{
 public:
	// ATC ハッシュ
	enum : uint8 {
		HASH_ATC_BASE	= 0,		// ATC #0-
		HASH_NONE		= 0xff,		// 対応なし
		HASH_TT			= 0xfe,		// TT
		HASH_SUB		= 0xfd,		// PS<4KB
	};

	m68030ATCTable_Custom(MPU68030Device *cpu_, uint fc_);
	~m68030ATCTable_Custom();

	// フラッシュ
	// Flush() はこのテーブルのすべてのエントリが対象。
	// Flush(addr) はこのテーブルのアドレスが一致するエントリが対象。
	void Flush();
	void Flush(uint32 addr);

	// PS<4KB の時に ATC を検索。
	m68030ATCLine *FindSub(const busaddr laddr);

	// テーブルサーチ。
	// テーブルという用語がぶつかってしまったので MMU サーチと呼ぶ。
	m68030ATCLine *MMUSearch(busaddr laddr);

	void Invalidate(m68030ATCLine *a);
	void InvalidateSub(const m68030ATCLine *);

	void MakeHash(const m68030ATCLine *);
	void MoveHead(m68030ATCLine *);
	m68030ATCLine *RemoveTail();

	// デバッグ表示
	std::string LineToStr(const m68030ATCLine *a) const;

	// ATC 実体
	m68030ATCLine line[LINES] {};

	uint32 latest_age {};

	// このテーブルの FC
	uint fc {};

	// PS(ページのビット長) - 12。負数なら PS<4KB
	int ps4k {};

	// 移動平均用バッファ。(こっちは排他制御が必要)
	FixedQueue<m68030ATCStat, 3> atchist {};
	bool atchist_tt_once_hit {};
	std::mutex atchist_mtx {};

	// ATC ハッシュ (他のメンバより後ろに置く?)
	std::array<uint8, 1024 * 1024> hash {};
};
#else
class m68030ATCTable_Motorola final : public m68030ATCTable
{
 public:
	m68030ATCTable_Motorola(MPU68030Device *cpu_);
	~m68030ATCTable_Motorola();

	void Flush();
	void Flush(uint32 fc, uint32 mask);
	void Flush(uint32 fc, uint32 mask, uint32 addr);

	// ATC からサーチ。見つからなければ NULL を返す。
 private:
	template <bool> m68030ATCLine *LookupTempl(const busaddr&);
 public:
	m68030ATCLine *Lookup(const busaddr& laddr);
	m68030ATCLine *Lookup_nostat(const busaddr& laddr);

	// テーブルサーチ。
	// テーブルという用語がぶつかってしまったので MMU サーチと呼ぶ。
	m68030ATCLine *MMUSearch(const busaddr& laddr);

	void Invalidate(m68030ATCLine *a);

	void MoveHead(m68030ATCLine *);
	m68030ATCLine *RemoveTail();

	// デバッグ表示
	std::string LineToStr(const m68030ATCLine *a) const;

	std::array<m68030ATCLine, LINES> line {};

	// エントリの順序。
#if defined(ATC_ORDER128)
	inline uint FindOrder(uint n) const;
	uint128 order {};
#else
	std::array<uint8, LINES> order {};
#endif

	// 統計情報。
	std::array<uint64, LINES> stat_atchit {};
};
#endif // M68030_CUSTOM_ATC

// ATC (Address Translation Cache)
class m68030ATC final
{
 public:
	explicit m68030ATC(MPU68030Device *);
	~m68030ATC();

	// ログレベルを設定する。Object のとは同じ使い方だけど別物。
	void SetLogLevel(int loglevel_);

	// 統計情報、移動平均等をリセットする。
	void ResetStat();

	// フラッシュ (pflush* 命令から呼ばれる)
	void flush();
	void flush(uint32 fc, uint32 mask);
	void flush(uint32 fc, uint32 mask, uint32 addr);

#if defined(M68030_CUSTOM_ATC)
	// ハッシュの、TT にマッチするブロックを TT 印にする。
	void SetToHash(const m68030TT&);
	// ハッシュから TT を削除する。
	void UnsetFromHash(const m68030TT&, const m68030TT&);
	// ページサイズを持っておく必要がある。
	void SetPS(uint32 ps);

	// テーブルの実体。
	std::unique_ptr<m68030ATCTable_Custom> tables[4] {};

	// FC ごとのテーブル。
	// 実際に使うのはこのうち4つだけ。
	std::array<m68030ATCTable_Custom *, 8> fctables {};
#else

	std::unique_ptr<m68030ATCTable_Motorola> table {};
#endif

	int loglevel {};

 private:
	MPU68030Device *cpu {};
};
