/* livepipe.c 
 * Create a character device 
 * Heavily based on teh linux kernel module programmers guide
 *
 * Developed by Scott Manley for use with LiveIce
 * Copyright 1999
 */

/* 
This needs to be compiled as a module and then insmod'd by root, 
Plus if you're going to use it with liveice then you'll need to manually 
setup the device files (as root) .

 */

/* Standard in kernel modules */
#include <linux/kernel.h>   /* We're doing kernel work */
#include <linux/module.h>   /* Specifically, a module */


/* Deal with CONFIG_MODVERSIONS */
#if CONFIG_MODVERSIONS==1
#define MODVERSIONS
#include <linux/modversions.h>
#endif        

/* For character devices */
#include <linux/fs.h>       /* The character device 
                             * definitions are here */
#include <linux/wrapper.h>  
#include <linux/sched.h>
#include <asm/uaccess.h> 
#include <linux/malloc.h>


/* Debugging garbage */
/*#define BREADCRUMBS*/

#define SUCCESS 0

/* Device Declarations **************************** */

/* The name for our device, as it will appear 
 * in /proc/devices */
#define DEVICE_NAME "livepipe"



/* major device number */
static int Major;

/* The maximum length of the message from the device */
#define BUF_LEN 65536
#define ATOMIC_SIZE 8192

struct live_device {
	int read_open;
	int write_open;
	long inpos;
	long outpos;
	long long num_bytes_read;
	long long num_bytes_written;
	struct wait_queue *WaitQ;
	char *live_buffer;
} ;

/* 16 devices should be enough for most people */
static struct live_device live_devices[16];

/* just set all the variables to reasonable values */
int init_minor_live_device(int num){
	if((num<0) || (num>15)){
		printk("only minor numbers 0-15 are supported\n");
		return -1;
	} 
	live_devices[num].read_open=0;
	live_devices[num].write_open=0;
	live_devices[num].inpos=0;
	live_devices[num].outpos=0;
	live_devices[num].num_bytes_read=0;
	live_devices[num].num_bytes_written=0;
	live_devices[num].WaitQ=NULL;
	live_devices[num].live_buffer=NULL;
	return 0;
}





/* our open function */
static int liveice_open(struct inode *inode, struct file *file){
	
	int minnum;
/* get the device minor number */
	minnum=inode->i_rdev & 0xFF;
#ifdef 	BREADCRUMBS
	printk("Device: %d.%d\n", inode->i_rdev >> 8, inode->i_rdev & 0xFF);
#endif
	/* if the minor device number > 15 then return an error */
	if(minnum>16)
		return -ENXIO;
	
	/* only open one end of the pipe at a time */
	/* special case for Read And Write*/
	if( (file->f_mode & FMODE_READ) && (file->f_mode & FMODE_WRITE) ) {
#ifdef 	BREADCRUMBS
		printk("Can't open liveice pipe in read-write mode\n");
#endif
		return -EINVAL;
	}
	/* read end */
	if( file->f_mode & FMODE_READ ) {
		
		if(!live_devices[minnum].read_open){
#ifdef 	BREADCRUMBS
			printk("Opened in Read Only mode\n");
#endif
			live_devices[minnum].read_open++;	 
			live_devices[minnum].outpos=0;
			live_devices[minnum].num_bytes_read=0;
			/* something might be waiting for our connect */
			module_wake_up(&(live_devices[minnum].WaitQ));
			MOD_INC_USE_COUNT;
			return 0;
		} else {
#ifdef 	BREADCRUMBS
			printk("Device is already opened for reading \n");
#endif
			return -EBUSY;
		}
	}
	
/* write end of te pipe */
	if( file->f_mode & FMODE_WRITE  ) {
		if(!live_devices[minnum].write_open){
			/* we might have a prvious chunk of memory alloced */
			if(!live_devices[minnum].live_buffer)
				if( (live_devices[minnum].live_buffer=vmalloc(BUF_LEN) ) == NULL ){
					printk("Help! Can't allocate device buffer!\n");
					return -ENOMEM;
				}
#ifdef 	BREADCRUMBS
			printk("Opened in Write Only mode\n");
#endif
			live_devices[minnum].write_open++;
			live_devices[minnum].inpos=0;
			live_devices[minnum].num_bytes_written=0;
			MOD_INC_USE_COUNT;
			return 0;
		} else {
#ifdef 	BREADCRUMBS
			printk("Device is already opened for writing \n");
#endif
			return -EBUSY;
			
		}
	}
#ifdef BREADCRUMBS
	printk("Doesn't seem to wnat to open in read or write mode... hmmm \n");
#endif
	return -EINVAL;
	
}



/* This function is called when a process closes the 
 * device file. It doesn't have a return value in 
 * version 2.0.x because it can't fail (you must ALWAYS
 * be able to close a device). In version 2.2.x it is 
 * allowed to fail - but we won't let it. 
 */

static int liveice_release(struct inode *inode, 
			   struct file *file)
{
	int minnum;
/* get the device minor number */
	minnum=inode->i_rdev & 0xFF;
	
#ifdef BREADCRUMBS
	printk("Closing liveice device\n");
#endif
	/* if the minor device number > 15 then return an error */
	if(minnum>16)
		return -ENXIO;
	
	if(file->f_mode & FMODE_WRITE){
		live_devices[minnum].write_open=0;
		MOD_DEC_USE_COUNT;
	}
	
	if(file->f_mode & FMODE_READ){
		live_devices[minnum].read_open=0;
		
		MOD_DEC_USE_COUNT;
	}
	
	
	
/* if there is nothing conncted to the device then clear the memory chunk */
	module_wake_up(&(live_devices[minnum].WaitQ));
	
	if(!(live_devices[minnum].read_open || live_devices[minnum].write_open)){
#ifdef BREADCRUMBS
		printk("Freeing chunk of memory\n");
#endif
		if(live_devices[minnum].live_buffer)
			vfree(live_devices[minnum].live_buffer);
		live_devices[minnum].live_buffer=NULL;
	}
	return 0;
}





/* This function is called whenever a process which 
 * have already opened the device file attempts to 
 * read from it. */


static ssize_t liveice_read(struct file *file,
			    char *buffer,    /* The buffer to fill with data */
			    size_t length,   /* The length of the buffer */
			    loff_t *offset)  /* Our offset in the file */
{
	int minnum;
	int available,requested,end_pos,i,is_sig=0;
/* get the device minor number */
	minnum=file->f_dentry->d_inode->i_rdev & 0xFF;
	/* if the minor device number > 15 then return an error */
	if(minnum>16)
		return -ENXIO;
	
	/* check to see if the pointer is valid */
/*	if(verify_area(VERIFY_WRITE,buffer,length))
	return -EFAULT;*/
#ifdef BREADCRUMBS
	printk("Trying to read %d bytes from the pipe\n",length);
#endif
	/*first - check that some input remains */
	if(!live_devices[minnum].live_buffer){
#ifdef BREADCRUMBS		
		printk("No input buffer - waiting for writer\n");
#endif
		if(file->f_flags & O_NONBLOCK){
			/* no waiting here */
			return -EAGAIN;
		} else {
			/* wait until something initialises the other end */
			while(!live_devices[minnum].live_buffer){
				for(i=0; i<_NSIG_WORDS && !is_sig; i++)
					is_sig = current->signal.sig[i] & 
						~current->blocked.sig[i];
				if (is_sig) {
					return -EINTR;
				}
				module_interruptible_sleep_on(&(live_devices[minnum].WaitQ));
				
			}
		}
	}
	/* next trick - check we have enough data for an atomic write */
	/* if the process is in non_blocking mode then return what we have */
	/* otherwise wait until we can fill our request up */
	requested=length;
	if(requested>BUF_LEN-ATOMIC_SIZE)
		requested = BUF_LEN - ATOMIC_SIZE;
	available=live_devices[minnum].inpos-live_devices[minnum].outpos;
	if(available<0)
		available+=BUF_LEN;
	/* if the target is non-blocking then don't wait for more input */
	if(file->f_flags & O_NONBLOCK){
		if(requested>available)
			requested=available;
		if(requested==0)
			return -EAGAIN;
	}
	/* also - if the other process has buggered off then we take 
           what we can get */
	if(!live_devices[minnum].write_open)
		if(requested>available)
			requested=available;
#ifdef BREADCRUMBS			
	printk("requested %d - available %d\n",requested,available);
#endif
	/* now wait in a loop until we have enough data, or the input closes */
	while(requested>available){
		/* check for signals*/ 
		for(i=0; i<_NSIG_WORDS && !is_sig; i++)
			is_sig = current->signal.sig[i] & 
				~current->blocked.sig[i];
		if (is_sig) {
			return -EINTR;
		}
		/* wake up any other processes and go to sleep */
		module_wake_up(&(live_devices[minnum].WaitQ));
		module_interruptible_sleep_on(&(live_devices[minnum].WaitQ));
		/* when we wake up check the amount of data again*/
		available=live_devices[minnum].inpos-live_devices[minnum].outpos;
		if(available<0)
			available+=BUF_LEN;
		if(!live_devices[minnum].write_open)
			if(requested>available)
				requested=available;
#ifdef BREADDRUMBS
		printk("requested %d - available %d\n",requested,available);
#endif
	}
	
	/* now if we've lasted this long then there must be enough 
	   data to satisfy our needs */
	
	/* because we're using a ring buffer we might need to 
	   do this in two writes */
	end_pos = live_devices[minnum].outpos + requested;
#ifdef BREADCRUMBS		
	printk("Now reading %d bytes\n",requested);
#endif
	if(end_pos<BUF_LEN){
		/* easy - one single copy */
		copy_to_user(buffer,&(live_devices[minnum].live_buffer[live_devices[minnum].outpos]),requested);
		live_devices[minnum].outpos+=requested;
		live_devices[minnum].num_bytes_read+=requested;
	} else {
		/* wrap round the end of the buffer */
		copy_to_user(buffer,&(live_devices[minnum].live_buffer[live_devices[minnum].outpos]),BUF_LEN-live_devices[minnum].outpos);
		copy_to_user(&(buffer[BUF_LEN-live_devices[minnum].outpos]),live_devices[minnum].live_buffer,requested-BUF_LEN+live_devices[minnum].outpos);
		live_devices[minnum].outpos=live_devices[minnum].outpos + requested - BUF_LEN;
		live_devices[minnum].num_bytes_read+=requested;
	}
#ifdef BREADCRUMBS		
	printk("%d bytes read from pipe\n",requested);
#endif
	/* wake everyone up in case they're waituing for data */
	module_wake_up(&(live_devices[minnum].WaitQ));
	return requested;
}





/* similarly - this is what we do when a write to the piep is attempted */
static ssize_t liveice_write(struct file *file,
			     const char *buffer,    /* The buffer */
			     size_t length,   /* The length of the buffer */
			     loff_t *offset)  /* Our offset in the file */
{
	int available,requested,end_pos,i,is_sig=0;
	int minnum;
/* get the device minor number */
	minnum=file->f_dentry->d_inode->i_rdev & 0xFF;
	
	/* if the minor device number > 15 then return an error */
	if(minnum>16)
		return -ENXIO;
	
	/* check to see if the pointer is valid */
/*	if(verify_area(VERIFY_READ,buffer,length))
	return -EFAULT;*/
#ifdef BREADCRUMBS
	printk("trying to write %d bytes to the pipe \n",length);
#endif
	/* if this is the start then wait until someone connects 
	   to the other end */
	if(live_devices[minnum].num_bytes_written==0)
		while(!live_devices[minnum].read_open){
			for(i=0; i<_NSIG_WORDS && !is_sig; i++)
				is_sig = current->signal.sig[i] & 
					~current->blocked.sig[i];
			if (is_sig) {
				return -EINTR;
			}
			module_interruptible_sleep_on(&(live_devices[minnum].WaitQ));
		}
	/* OTOH - if the far end has deserted us.... then die */
	if(!live_devices[minnum].read_open){
		send_sig(SIGPIPE,current,0);
		return -EPIPE;
	}
	/* so now we can see if there's enough room to write */
	requested=length;
	if(requested>ATOMIC_SIZE)
		requested = ATOMIC_SIZE;
	available=live_devices[minnum].outpos-live_devices[minnum].inpos-1;
	if(available<=0)
		available+=BUF_LEN;
	
	/* if the target is non-blocking then don't wait for space */
	if(file->f_flags & O_NONBLOCK){
		if(requested>available)
			requested=available;
		if(requested==0)
			return -EAGAIN;
	}
	
	/* now wait in a loop until we have enough room, or the output closes */
	while(requested>available){
		for(i=0; i<_NSIG_WORDS && !is_sig; i++)
			is_sig = current->signal.sig[i] & 
				~current->blocked.sig[i];
		if (is_sig) {
			return -EINTR;
		}
		/* wake up any other processes and go to sleep */
		module_wake_up(&(live_devices[minnum].WaitQ));
#ifdef BREADCRUMBS
		printk("no room in pipe - sleeping\n");
#endif
		module_interruptible_sleep_on(&(live_devices[minnum].WaitQ));
		/* when we wake up check the amount of data again*/
		available=live_devices[minnum].outpos-live_devices[minnum].inpos;
		if(available<=0)
			available+=BUF_LEN;
		/* but if the other process has gone away then give up */
		if(!live_devices[minnum].read_open){
			send_sig(SIGPIPE,current,0);
			return -EPIPE;
		}
	}
	
	/* woohoo - now write our data into the buffer */
	/* because we're using a ring buffer we might need to 
	   do this in two writes */
	end_pos = live_devices[minnum].inpos + requested;
	
	if(end_pos<BUF_LEN){
		/* easy - one single copy */
		copy_from_user(&(live_devices[minnum].live_buffer[live_devices[minnum].inpos]),buffer,requested);
		live_devices[minnum].inpos+=requested;
		live_devices[minnum].num_bytes_written+=requested;
	} else {
		/* wrap round the end of the buffer */
		copy_from_user(&(live_devices[minnum].live_buffer[live_devices[minnum].inpos]),buffer,BUF_LEN-live_devices[minnum].inpos);
		copy_from_user(live_devices[minnum].live_buffer,&(buffer[BUF_LEN-live_devices[minnum].inpos]),requested-BUF_LEN+live_devices[minnum].inpos);
		live_devices[minnum].inpos=live_devices[minnum].inpos + requested - BUF_LEN;
		live_devices[minnum].num_bytes_written+=requested;
	}
#ifdef BREADCRUMBS
	printk("%d bytes written to pipe\n",requested);
#endif
	module_wake_up(&(live_devices[minnum].WaitQ));
	return requested;
}


/* ok.... this is the response to seeking on an open pipe */

static long long liveice_lseek(struct file * file, long long offset, int orig)
{
	int minnum;
	long long pos;
	/* get the device minor number */
	minnum=file->f_dentry->d_inode->i_rdev & 0xFF;
	
	/* if the minor device number > 15 then return an error */
	if(minnum>16)
		return -ENXIO;
	
        if(orig==2)
                return 2147483647;
	if(orig==1){
		if( file->f_mode & FMODE_READ )
			pos=offset+live_devices[minnum].num_bytes_read;
		else 
			pos=offset+live_devices[minnum].num_bytes_written;
	} else {
		pos=offset;
	}
	
        if(pos>=0)
                return pos;
        else
                return 0;
}


/* file operations structure */

struct file_operations liveice_ops = {
	liveice_lseek,   
	liveice_read, 
	liveice_write,
	NULL,   /* readdir */
	NULL,   /* select */
	NULL,   /* ioctl */
	NULL,   /* mmap */
	liveice_open,
	NULL,   /* flush */
	liveice_release  /* a.k.a. close */
};


/* Initialize the module - Register the character device */
int init_module()
{
	int i;
	/* Register the character device (atleast try) */
	Major = module_register_chrdev(0, 
				       DEVICE_NAME,
				       &liveice_ops);
	
	/* Negative values signify an error */
	if (Major < 0) {
		printk ("%s device failed with %d\n",
			"Sorry, registering the character",
			Major);
		return Major;
	}
	/* now clear the data fields on 16 monor numbers */
	for(i=0;i<16;i++){
		init_minor_live_device(i);
	}
	printk ("%s The major device number is %d.\n",
		"Registration is a success.",
		Major);
	printk ("If you want to talk to the device driver,\n");
	printk ("you'll have to create a device file. \n");
	printk ("I suggest you use:\n");
	printk ("mknod <name> c %d <minor>\n", Major);
	printk ("minor number 0-15 are valid for this device\n");
	return 0;
}


/* Cleanup - unregister the appropriate file from /proc */
void cleanup_module()
{
	int ret;
	
	printk("Removing module from kernel\n");
	/* Unregister the device */
	ret = module_unregister_chrdev(Major, DEVICE_NAME);
	
	/* If there's an error, report it */ 
	if (ret < 0)
		printk("Error in unregister_chrdev: %d\n", ret);
}

