// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.

package ntlmssp

import (
	"context"
	"encoding/base64"
	"fmt"
	"io"
	"net/http"
	"os"
	"testing"
	"time"
)

// TestNTLM_E2E tests NTLM authentication against a real IIS server with Windows Authentication enabled
func TestNTLM_E2E(t *testing.T) {
	// Get test configuration from environment variables
	testURL := os.Getenv("NTLM_TEST_URL")
	if testURL == "" {
		testURL = "http://localhost:8080/"
	}

	username := os.Getenv("NTLM_TEST_USER")
	if username == "" {
		username = os.Getenv("USERNAME") // Windows default
		if username == "" {
			t.Skip("No NTLM_TEST_USER or USERNAME environment variable set, skipping e2e test")
		}
	}

	password := os.Getenv("NTLM_TEST_PASSWORD")
	if password == "" {
		t.Skip("No NTLM_TEST_PASSWORD environment variable set, skipping e2e test")
	}

	domain := os.Getenv("NTLM_TEST_DOMAIN")
	if domain == "" {
		domain = os.Getenv("USERDOMAIN") // Windows default
	}

	t.Logf("Testing NTLM authentication against: %s", testURL)
	t.Logf("Using credentials: %s\\%s", domain, username)

	// Test 1: Basic NTLM authentication should succeed
	t.Run("BasicNTLMAuth", func(t *testing.T) {
		client := &http.Client{
			Transport: Negotiator{
				RoundTripper: &http.Transport{},
			},
		}

		// Create request with credentials using test context
		req, err := http.NewRequestWithContext(t.Context(), "GET", testURL, nil)
		if err != nil {
			t.Fatalf("Failed to create request: %v", err)
		}

		// Set basic auth (NTLM negotiator will handle the NTLM handshake)
		if domain != "" {
			req.SetBasicAuth(domain+"\\"+username, password)
		} else {
			req.SetBasicAuth(username, password)
		}

		// Make the request
		resp, err := client.Do(req)
		if err != nil {
			t.Fatalf("NTLM authentication failed: %v", err)
		}
		defer resp.Body.Close()

		// Check that we got a successful response
		if resp.StatusCode != http.StatusOK {
			body, _ := io.ReadAll(resp.Body)
			t.Fatalf("Expected status 200, got %d. Body: %s", resp.StatusCode, string(body))
		}

		// Read response body to ensure everything worked
		body, err := io.ReadAll(resp.Body)
		if err != nil {
			t.Fatalf("Failed to read response body: %v", err)
		}

		t.Logf("NTLM authentication successful! Response body length: %d bytes", len(body))
	})

	// Test 2: Test with UPN format username
	t.Run("UPNFormatAuth", func(t *testing.T) {
		if domain == "" {
			t.Skip("No domain available for UPN format test")
		}

		// Skip UPN test for local computer domains as they typically don't support UPN
		if domain == os.Getenv("COMPUTERNAME") {
			t.Skipf("Skipping UPN test for local computer domain '%s' - UPN format typically not supported in local computer accounts", domain)
		}

		client := &http.Client{
			Transport: Negotiator{
				RoundTripper: &http.Transport{},
			},
		}

		req, err := http.NewRequestWithContext(t.Context(), "GET", testURL, nil)
		if err != nil {
			t.Fatalf("Failed to create request: %v", err)
		}

		// Use UPN format: user@domain.com
		upnUsername := fmt.Sprintf("%s@%s", username, domain)
		req.SetBasicAuth(upnUsername, password)

		t.Logf("Attempting UPN authentication with: %s", upnUsername)

		resp, err := client.Do(req)
		if err != nil {
			t.Fatalf("UPN format NTLM authentication failed: %v", err)
		}
		defer resp.Body.Close()

		if resp.StatusCode != http.StatusOK {
			body, _ := io.ReadAll(resp.Body)

			// UPN might not be supported in all environments, so log this as info rather than failing
			if resp.StatusCode == http.StatusUnauthorized {
				t.Logf("UPN format not supported in this environment (status 401) - this is expected for local computer domains")
				t.Skip("UPN format authentication not supported in this environment")
			} else {
				t.Fatalf("UPN format auth failed with unexpected status %d. Body: %s", resp.StatusCode, string(body))
			}
		}

		t.Log("UPN format NTLM authentication successful!")
	})

	// Test 3: Test authentication failure with wrong credentials
	t.Run("AuthenticationFailure", func(t *testing.T) {
		client := &http.Client{
			Transport: Negotiator{
				RoundTripper: &http.Transport{},
			},
		}

		req, err := http.NewRequestWithContext(t.Context(), "GET", testURL, nil)
		if err != nil {
			t.Fatalf("Failed to create request: %v", err)
		}

		// Use wrong password
		req.SetBasicAuth(username, "wrongpassword")

		resp, err := client.Do(req)
		if err != nil {
			t.Logf("Expected authentication failure occurred: %v", err)
			return // This is expected
		}
		defer resp.Body.Close()

		// Should get 401 Unauthorized
		if resp.StatusCode != http.StatusUnauthorized {
			t.Errorf("Expected 401 Unauthorized with wrong credentials, got %d", resp.StatusCode)
		} else {
			t.Log("Authentication correctly failed with wrong credentials")
		}
	})

	// Test 4: Test server accessibility (without auth)
	t.Run("ServerAccessibility", func(t *testing.T) {
		client := &http.Client{}

		req, err := http.NewRequestWithContext(t.Context(), "GET", testURL, nil)
		if err != nil {
			t.Fatalf("Failed to create request: %v", err)
		}

		resp, err := client.Do(req)
		if err != nil {
			t.Fatalf("Failed to reach server: %v", err)
		}
		defer resp.Body.Close()

		// Should get 401 Unauthorized (server requires auth)
		if resp.StatusCode != http.StatusUnauthorized {
			t.Logf("Unexpected status without auth: %d (expected 401)", resp.StatusCode)
		} else {
			t.Log("Server correctly requires authentication")
		}
	})

	// Test 5: Test with context cancellation
	t.Run("ContextCancellation", func(t *testing.T) {
		client := &http.Client{
			Transport: Negotiator{
				RoundTripper: &http.Transport{},
			},
		}

		ctx, cancel := context.WithTimeout(t.Context(), 1*time.Millisecond)
		defer cancel()

		req, err := http.NewRequestWithContext(ctx, "GET", testURL, nil)
		if err != nil {
			t.Fatalf("Failed to create request: %v", err)
		}

		req.SetBasicAuth(username, password)

		resp, err := client.Do(req)
		if err == nil {
			// If the request somehow succeeded despite the short timeout, close the body
			defer resp.Body.Close()
			t.Error("Expected context cancellation error, but request succeeded")
		} else {
			t.Logf("Context cancellation worked as expected: %v", err)
		}
	})
}

// TestNTLM_E2E_ProcessChallenge tests the ProcessChallenge function directly against real server responses
func TestNTLM_E2E_ProcessChallenge(t *testing.T) {
	testURL := os.Getenv("NTLM_TEST_URL")
	if testURL == "" {
		testURL = "http://localhost:8080/"
	}

	username := os.Getenv("NTLM_TEST_USER")
	if username == "" {
		username = os.Getenv("USERNAME")
		if username == "" {
			t.Skip("No username available for ProcessChallenge test")
		}
	}

	password := os.Getenv("NTLM_TEST_PASSWORD")
	if password == "" {
		t.Skip("No password available for ProcessChallenge test")
	}

	domain := os.Getenv("NTLM_TEST_DOMAIN")
	if domain == "" {
		domain = os.Getenv("USERDOMAIN")
	}

	t.Run("DirectProcessChallenge", func(t *testing.T) {
		// Step 1: Send negotiate message to get challenge
		client := &http.Client{}

		// Create negotiate message
		negotiateMsg, err := NewNegotiateMessage(domain, "")
		if err != nil {
			t.Fatalf("Failed to create negotiate message: %v", err)
		}

		// Send negotiate
		req, err := http.NewRequestWithContext(t.Context(), "GET", testURL, nil)
		if err != nil {
			t.Fatalf("Failed to create request: %v", err)
		}
		req.Header.Set("Authorization", "NTLM "+base64.StdEncoding.EncodeToString(negotiateMsg))

		resp, err := client.Do(req)
		if err != nil {
			t.Fatalf("Failed to send negotiate: %v", err)
		}
		defer resp.Body.Close()

		if resp.StatusCode != http.StatusUnauthorized {
			t.Fatalf("Expected 401 from negotiate, got %d", resp.StatusCode)
		}

		// Extract challenge from response
		authHeader := resp.Header.Get("WWW-Authenticate")
		if authHeader == "" {
			t.Fatal("No WWW-Authenticate header in challenge response")
		}

		t.Logf("Received challenge header: %s", authHeader)

		// Parse challenge (this would need implementation based on your authheader package)
		// For now, we'll test that we got a proper NTLM response
		if len(authHeader) < 10 || authHeader[:4] != "NTLM" {
			t.Fatalf("Invalid NTLM challenge response: %s", authHeader)
		}

		t.Log("Successfully received NTLM challenge from real server")
	})
}

// TestNTLM_E2E_DomainFormats tests different username formats with real NTLM server
func TestNTLM_E2E_DomainFormats(t *testing.T) {
	testURL := os.Getenv("NTLM_TEST_URL")
	if testURL == "" {
		testURL = "http://localhost:8080/"
	}

	username := os.Getenv("NTLM_TEST_USER")
	if username == "" {
		username = os.Getenv("USERNAME")
		if username == "" {
			t.Skip("No username available for domain format test")
		}
	}

	password := os.Getenv("NTLM_TEST_PASSWORD")
	if password == "" {
		t.Skip("No password available for domain format test")
	}

	domain := os.Getenv("NTLM_TEST_DOMAIN")
	if domain == "" {
		domain = os.Getenv("USERDOMAIN")
	}

	t.Logf("Testing domain formats against: %s", testURL)

	t.Run("SAMFormatAuth", func(t *testing.T) {
		if domain == "" {
			t.Skip("No domain available for SAM format test")
		}

		client := &http.Client{
			Transport: Negotiator{
				RoundTripper: &http.Transport{},
			},
		}

		req, err := http.NewRequestWithContext(t.Context(), "GET", testURL, nil)
		if err != nil {
			t.Fatalf("Failed to create request: %v", err)
		}

		// Use SAM format: DOMAIN\user
		samUsername := fmt.Sprintf("%s\\%s", domain, username)
		req.SetBasicAuth(samUsername, password)
		t.Logf("Testing SAM format: %s", samUsername)

		resp, err := client.Do(req)
		if err != nil {
			t.Fatalf("SAM format authentication failed: %v", err)
		}
		defer resp.Body.Close()

		if resp.StatusCode != http.StatusOK {
			body, _ := io.ReadAll(resp.Body)
			t.Fatalf("SAM format auth failed with status %d. Body: %s", resp.StatusCode, string(body))
		}

		t.Log("✓ SAM format (DOMAIN\\user) authentication successful")
	})

	t.Run("PlainUsernameAuth", func(t *testing.T) {
		client := &http.Client{
			Transport: Negotiator{
				RoundTripper: &http.Transport{},
			},
		}

		req, err := http.NewRequestWithContext(t.Context(), "GET", testURL, nil)
		if err != nil {
			t.Fatalf("Failed to create request: %v", err)
		}

		// Use plain username
		req.SetBasicAuth(username, password)
		t.Logf("Testing plain username: %s", username)

		resp, err := client.Do(req)
		if err != nil {
			t.Fatalf("Plain username authentication failed: %v", err)
		}
		defer resp.Body.Close()

		if resp.StatusCode != http.StatusOK {
			body, _ := io.ReadAll(resp.Body)
			t.Fatalf("Plain username auth failed with status %d. Body: %s", resp.StatusCode, string(body))
		}

		t.Log("✓ Plain username authentication successful")
	})
}
